
/******************************************************************************
* MODULE     : tm_scheme.gen.cc
* DESCRIPTION: The TeXmacs-lisp motor
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_scheme.gen.h>
#include <file.gen.h>
#include <analyze.gen.h>

#module code_tm_scheme
#import tm_scheme
#import file
#import analyze

bool eval_scheme_any (string s, string& ret);
bool eval_scheme_bool (string s, bool& ret);
bool eval_scheme_scheme_tree (string s, scheme_tree& ret);

/******************************************************************************
* Constructor and destructor
******************************************************************************/

tm_scheme_rep::tm_scheme_rep () { dialogue_win= NULL; }
tm_scheme_rep::~tm_scheme_rep () {}

/******************************************************************************
* Execution of commands
******************************************************************************/

bool
tm_scheme_rep::exec (string s) {
  string r;
  bool   b= eval_scheme_any (s, r);
  if (b || (r != "")) {
    set_message (r, "execute scheme expression");
    return TRUE;
  }
  return FALSE;
}

bool
tm_scheme_rep::exec (scheme_tree p) {
  return exec (scheme_tree_to_string (p));
}

bool
tm_scheme_rep::exec_file (string dir, string name) {
  string s;
  if (load_string (dir, name, s)) {
    set_message ("Error: file#" * name * "#not found", "execute file");
    return TRUE;
  }
  else {
    string ret;
    bool flag= eval_scheme_any (string ("(begin " * s * ")"), ret);
    if (flag || (ret != "")) {
      cout << "Guile error] " << ret << "\n";
      cout << "           ] in file " << name << "\n";
    }
    return flag;
  }
}

bool
tm_scheme_rep::eval_any (scheme_tree p, string& ret) {
  string s= scheme_tree_to_string (p);
  return eval_scheme_any (s, ret);
}

bool
tm_scheme_rep::eval_bool (scheme_tree p, bool& ret) {
  string s= scheme_tree_to_string (p);
  return eval_scheme_bool (s, ret);
}

bool
tm_scheme_rep::eval_scheme_tree (scheme_tree p, scheme_tree& ret) {
  string s= scheme_tree_to_string (p);
  return eval_scheme_scheme_tree (s, ret);
}

/******************************************************************************
* Delayed execution of commands
******************************************************************************/

void
tm_scheme_rep::exec_delayed (scheme_tree p) {
  cmds << p;
}

void
tm_scheme_rep::exec_pending_commands () {
  while (N(cmds)!=0) {
    int i;
    scheme_tree p= cmds[0];
    array<scheme_tree> a (N(cmds)-1);
    for (i=1; i<N(cmds); i++) a[i-1]= cmds[i];
    cmds= a;
    exec (p);
  }
}

/******************************************************************************
* Dialogues
******************************************************************************/

class dialogue_command_rep: public command_rep {
  server_rep* sv;
  scheme_tree prg;
public:
  dialogue_command_rep (server_rep* sv2, scheme_tree prg2):
    sv (sv2), prg (prg2) {}
  void apply () {
    scheme_tree arg;
    sv->dialogue_inquire (arg);
    if (arg != "cancel") sv->exec_delayed (tree (TUPLE, prg, arg));
    sv->exec_delayed (tree (TUPLE, "dialogue-end")); }
  ostream& print (ostream& out) {
    return out << "Dialogue"; }
};

command
dialogue_command (server_rep* sv, scheme_tree prg) {
  return new dialogue_command_rep (sv, prg);
}

void
tm_scheme_rep::dialogue_start (string name, widget wid, scheme_tree prg) {
  (void) prg;
  if (dialogue_win == NULL) {
    string lan= get_display()->out_lan;
    if (lan == "russian") lan= "english";
    name= get_display()->translate (name, "english", lan);
    char* _name= as_charp (name);
    dialogue_wid= wid;
    dialogue_win= plain_window (dialogue_wid, _name);
    dialogue_win->map ();
    delete[] _name;
  }
}

void
tm_scheme_rep::dialogue_inquire (scheme_tree& arg) {
  string s;
  dialogue_wid << get_string ("input", s);
  arg= string_to_scheme_tree (s);
  if (is_compound (arg)) arg= tree (TUPLE, "quote", arg);
}

void
tm_scheme_rep::dialogue_end () {
  if (dialogue_win != NULL) {
    dialogue_win->unmap ();
    delete dialogue_win;
    dialogue_win= NULL;
    dialogue_wid= widget ();
  }
}

static int
gcd (int i, int j) {
  if (i<j)  return gcd (j, i);
  if (j==0) return i;
  return gcd (j, i%j);
}

void
tm_scheme_rep::choose_file (string title, string type, scheme_tree prg) {
  string magn;
  if (type == "image") {
    editor ed  = get_editor ();
    int dpi    = as_int (ed->get_env_string (DPI));
    int sfactor= ed->get_shrinking_factor ();
    int num    = 75*sfactor;
    int den    = dpi;
    int g      = gcd (num, den);
    num /= g; den /= g;
    if (num != 1) magn << "*" << as_string (num);
    if (den != 1) magn << "/" << as_string (den);
  }

  string name= get_name_buffer ();
  command cb = dialogue_command (get_server(), prg);
  widget  wid= file_chooser_widget (get_display(), cb, type, magn);
  if (!starts (name, "no name")) {
    wid << set_string ("directory", get_radical_file_name (name));
    if ((type != "image") && (type != "postscript"))
      wid << set_string ("file", get_tail_file_name (name));
  }
  else wid << set_string ("directory", ".");
  dialogue_start (title, wid, prg);
  dialogue_win->set_keyboard_focus (dialogue_wid[0]["file"]["input"]);
}

#endmodule // code_tm_scheme
