
/******************************************************************************
* MODULE     : evaluate.gen.cc
* DESCRIPTION: Execution of scheme commands via guile
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tree.gen.h>

#module code_evaluate
#import tree
#include <guile/gh.h>

extern void initialize_glue ();
extern string scm_to_string (SCM p);
extern scheme_tree scm_to_scheme_tree (SCM p);

/******************************************************************************
* Installation of guile and initialization of guile
******************************************************************************/

void
install_guile (int argc, char** argv, void (*call_back) (int, char**)) {
#ifdef DOTS_OK
  gh_enter (argc, argv, (void (*)(...)) ((void*) call_back));
#else
  gh_enter (argc, argv, call_back);
#endif
}

void
initialize_guile () {
  char* init_prg =
    "(define TeXmacs-guile-error-handler
       (lambda args
         (object->string args)))

     (define object->string
       (lambda (obj)
         (call-with-output-string
           (lambda (port) (write obj port)))))

     (define TeXmacs-guile-eval
       (lambda (s)
         (catch
           #t
           (lambda ()
             (object->string (eval-string s)))
           TeXmacs-guile-error-handler)))

     (define TeXmacs-guile-eval-variant
       (lambda (s)
         (catch
           #t
           (lambda () (eval-string s))
           TeXmacs-guile-error-handler)))

     (define buffer-menu '())
     (define project-buffer-menu '())";

  gh_eval_str (init_prg);
  initialize_glue ();
}

/******************************************************************************
* Evaluation of scheme commands
******************************************************************************/

static string
back_slash (string s) {
  int i;
  string r;
  for (i=0; i<N(s); i++)
    switch (s[i]) {
    case '\\': r << "\\\\"; break;
    case '\"': r << "\\\""; break;
    default: r << s[i];
    }
  return r;
}

/*
static string
start (string s) {
  if (N(s)<200) return s;
  else return s(0,200) * "...";
}
*/

static char* (*scm2newstr) (SCM, int*)=
  (char* (*) (SCM, int*)) ((void*) gh_scm2newstr);

bool
eval_scheme_any (string s, string& r) {
  // cout << "Eval any] " << start (s) << "\n";
  string prg= "(TeXmacs-guile-eval \"" * back_slash (s) * "\")";
  char* _s= as_charp (prg);
  SCM result= gh_eval_str (_s);
  delete[] _s;

  int len_r;
  char* _r= scm2newstr (result, &len_r);
  r= _r;
  free (_r);

  if (r == "#<unspecified>") r= "";
  // cout << "Yields] " << start (r) << "\n";
  return FALSE;
}

SCM
eval_scheme (string s) {
  string prg= "(TeXmacs-guile-eval-variant \"" * back_slash (s) * "\")";
  char* _s= as_charp (prg);
  SCM result= gh_eval_str (_s);
  delete[] _s;
  return result;
}

bool
eval_scheme_bool (string s, bool& r) {
  // cout << "Eval bool] " << start (s) << "\n";
  SCM result= eval_scheme (s);
  if (gh_boolean_p (result)) {
    r= gh_scm2bool (result);
    // cout << "Yields] " << r << "\n";
    return FALSE;
  }
  else {
    r= FALSE;
    // cout << "Yields] Type mismatch\n";
    return TRUE;
  }
}

bool
eval_scheme_string (string s, string& r) {
  // cout << "Eval scheme string] " << start (s) << "\n";
  SCM result= eval_scheme (s);
  r= scm_to_string (result);
  // cout << "Yields] " << r << "\n";
  return FALSE;
}

bool
eval_scheme_scheme_tree (string s, scheme_tree& r) {
  // cout << "Eval scheme scheme_tree] " << start (s) << "\n";
  SCM result= eval_scheme (s);
  r= scm_to_scheme_tree (result);
  // cout << "Yields] " << r << "\n";
  return FALSE;
}

bool
eval_scheme_tree (string s, tree& r) {
  // cout << "Eval scheme tree] " << start (s) << "\n";
  SCM result= eval_scheme (s);
  r= scheme_tree_to_tree (scm_to_scheme_tree (result));
  // cout << "Yields] " << r << "\n";
  return FALSE;
}

bool
eval_scheme_func_string_string (string f, string s, string& r) {
  // cout << "Eval scheme string->string] " << f << " (" << start(s) << ")\n";
  string cmd= "(" * f * " \"" * s * "\")";
  return eval_scheme_string (cmd, r);
}

bool
eval_scheme_func_string2_string (string f, string s1, string s2, string& r) {
  // cout << "Eval scheme (string,string)->string] " << f << "\n";
  string cmd= "(" * f * " \"" * s1 * "\" \"" * s2 * "\")";
  return eval_scheme_string (cmd, r);
}

bool
eval_scheme_func_tree_tree (string f, tree t, tree& r) {
  // cout << "Eval scheme tree->tree] " << f << "\n";
  string cmd= "(" * f * " '" * tree_to_scheme (t)  * ")";
  return eval_scheme_tree (cmd, r);
}

bool
eval_scheme_func_tree_string (string f, tree t, string& r) {
  // cout << "Eval scheme tree->string] " << f << "\n";
  string cmd= "(" * f * " '" * tree_to_scheme (t)  * ")";
  return eval_scheme_string (cmd, r);
}

#endmodule // code_evaluate
