
/******************************************************************************
* MODULE     : list
* DESCRIPTION: linked lists with reference counting
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <list.gen.h>

#module code_list (T)
#import list (T)

/******************************************************************************
* output and convertion
******************************************************************************/

ostream&
operator << (ostream& out, list<T> l) {
  out << "[";
  if (!nil (l)) {
    out << " " << l->item;
    l=l->next;
  }
  while (!nil (l)) {
    out << ", " << l->item;
    l=l->next;
  }
  return out << " ]";
}

T&
list<T>::operator [] (int i) {
  if (rep==NULL)
    fatal_error ("list too short", "list<T>::operator []", "list.gen.cc");
  if (i==0) return rep->item;
  return rep->next[i-1];
}

list<T>::operator tree () {
  list<T> l;
  int i, n=N(*this);
  tree t (TUPLE, n);
  for (i=0, l=*this; i<n; i++, l=l->next)
#ifdef no_tree_converter<T>
    t[i]= "?";
#else
    t[i]= (tree) l->item;
#endif
  return t;
}

/******************************************************************************
* insertion and suppression
******************************************************************************/

list<T>&
operator << (list<T>& l, T item) {
  if (nil (l)) l= list<T> (item, list<T> ());
  else l->next << item;
  return l;
}

list<T>&
operator << (list<T>& l1, list<T> l2) {
  if (nil (l1)) l1= l2;
  else l1->next << l2;
  return l1;
}

list<T>&
operator >> (T item, list<T>& l) {
  return (l= list<T> (item, l));
}

list<T>&
operator << (T& item, list<T>& l) {
  item= l->item;
  l   = l->next;
  return l;
}

list<T>&
suppress_last (list<T>& l) {
  if (nil (l)) fatal_error ("empty path", "last_item", "list.gen.cc");
  if (nil (l->next)) l= list<T> ();
  else suppress_last (l->next);
  return l;
}

T&
last_item (list<T> l) {
  if (nil (l)) fatal_error ("empty path", "last_item", "list.gen.cc");
  if (nil (l->next)) return l->item;
  return last_item (l->next);
}

/******************************************************************************
* tests
******************************************************************************/

bool
operator == (list<T> l1, list<T> l2) {
  if (nil (l1) || nil (l2)) return (nil (l1) == nil (l2));
  return (l1->item==l2->item) && (l1->next==l2->next);
}

bool
operator != (list<T> l1, list<T> l2) {
  if (nil (l1) || nil (l2)) return (nil (l1) != nil (l2));
  return (l1->item!=l2->item) || (l1->next!=l2->next);
}

bool
operator < (list<T> l1, list<T> l2) {
  if (nil (l1) || nil (l2)) return !nil (l2);
  return (l1->item==l2->item) && (l1->next<l2->next);
}

bool
operator <= (list<T> l1, list<T> l2) {
  if (nil (l1) || nil (l2)) return nil (l1);
  return (l1->item==l2->item) && (l1->next<=l2->next);
}

/******************************************************************************
* computations with list<T> structures
******************************************************************************/

int
N (list<T> l) {
  if (nil (l)) return 0;
  else return N (l->next) + 1;
}

list<T>
copy (list<T> l) {
  if (nil (l)) return list<T> ();
  else return list<T> (l->item, copy (l->next));
}

list<T>
operator * (list<T> l1, T x) {
  if (nil (l1)) return x;
  else return list<T> (l1->item, l1->next * x);
}

list<T>
operator * (list<T> l1, list<T> l2) {
  if (nil (l1)) return copy (l2);
  else return list<T> (l1->item, l1->next * l2);
}

list<T>
head (list<T> l, int n) {
  if (n==0) return list<T> ();
  if (nil (l)) fatal_error ("list too short", "head", "list.gen.cc");
  return list<T> (l->item, head (l->next, n-1));
}

list<T>
tail (list<T> l, int n) {
  for (; n>0; n--) {
    if (nil (l)) fatal_error ("list too short", "tail", "list.gen.cc");
    l=l->next;
  }
  return l;
}

list<T>
revert (list<T> l) {
  list<T> r;
  while (!nil(l)) {
    r= list<T> (l->item, r);
    l=l->next;
  }
  return r;
}

#endmodule // code_list (T)
