
/******************************************************************************
* MODULE     : string.gen.cc
* DESCRIPTION: Fixed size strings with reference counting.
*              Zero characters can be part of string.
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include "gg_string.h"
#include <stdio.h>

static inline int
round_length (int n) {
  n=(n+3)&(0xfffffffc);
  if (n<24) return n;
  register int i=32;
  while (n>i) i<<=1;
  return i;
}

string_rep::string_rep (int n2):
  ref_count (1), n (n2), a (new char[round_length(n+1)]) { a[n]='\0'; }

string::string (char c) {
  rep= new string_rep(1);
  rep->a[0]=c;
}

string::string (char *a) {
  int i, n=strlen(a);
  rep= new string_rep(n);
  for (i=0; i<n; i++)
    rep->a[i]=a[i];
}

string::string (const char *a) {
  int i, n=strlen(a);
  rep= new string_rep(n);
  for (i=0; i<n; i++)
    rep->a[i]=a[i];
}

string::string (char* a, int n) {
  int i;
  rep= new string_rep(n);
  for (i=0; i<n; i++)
    rep->a[i]=a[i];
}

void
string::resize (int n2) {
  register int n= round_length (rep->n+1);
  register int m= round_length (n2+1);
  if (n!=m) {
    register int i, k=(n2<rep->n?n2:rep->n);
    register char* b= new char[m];
    for (i=0; i<k; i++) b[i]=rep->a[i];
    if (n!=0) delete[] rep->a;
    rep->a=b;
  }
  rep->n    = n2;
  rep->a[n2]= '\0';
}

bool
string::operator == (string a) {
  int i;
  if (rep->n!=a->n) return 0;
  for (i=0; i<rep->n; i++)
    if (rep->a[i]!=a->a[i]) return 0;
  return 1;
}

bool
string::operator != (string a) {
  int i;
  if (rep->n!=a->n) return 1;
  for (i=0; i<rep->n; i++)
    if (rep->a[i]!=a->a[i]) return 1;
  return 0;
}

string
string::operator () (int begin, int end) {
  int i;
  string r (end-begin);
  for (i=begin; i<end; i++) r[i-begin]=rep->a[i];
  return r;
}

bool
string::operator () (char* s) {
  int i,j;
  for (i=0, j=0; (s[i]!='\0') && (j<rep->n); i++, j++)
    if (s[i]!=rep->a[j]) return FALSE;
  return (s[i]=='\0') && (j==rep->n);
}

bool
string::operator () (char* s, int start) {
  int i,j;
  for (i=0, j=start; (s[i]!='\0') && (j<rep->n); i++, j++)
    if (s[i]!=rep->a[j]) return FALSE;
  return s[i]=='\0';
}

string
copy (string s) {
  int i, n=N(s);
  string r (n);
  for (i=0; i<n; i++) r[i]=s[i];
  return r;
}

string&
operator << (string& a, char x) {
  a.resize (N(a)+ 1);
  a [N(a)-1]=x;
  return a;
}

string&
operator << (string& a, string b) {
  int i, k1= N(a), k2=N(b);
  a.resize (k1+k2);
  for (i=0; i<k2; i++) a[i+k1]= b[i];
  return a;
}

string
operator * (string a, string b) {
  int i, n1=N(a), n2=N(b);
  string c(n1+n2);
  for (i=0; i<n1; i++) c[i]=a[i];
  for (i=0; i<n2; i++) c[i+n1]=b[i];
  return c;
}

ostream&
operator << (ostream& out, string a) {
  int i, n=N(a);
  if (n==0) return out;
  for (i=0; i<n; i++) out << a[i];
  return out;
}

int
hash (string s) {
  int i, h=0, n=N(s);
  for (i=0; i<n; i++) {
    h=(h<<9)+(h>>23);
    h=h+((int) s[i]);
  }
  return h;
}

bool
equal (string s, char *a)
{
  int i, n=N(s);
  for (i=0; i<n; i++) {
    if (s[i]!=a[i]) return FALSE;
    if (a[i]=='\0') return FALSE;
  }
  return (a[i]=='\0');
}

int
as_int (string s) {
  int i, n=N(s);
  char buf[n+1];
  for (i=0; i<n; i++) buf[i]=s[i];
  buf[n]='\0';
  sscanf (buf, "%d", &i);
  return i;
}

double
as_double (string s) {
  int i, n= N(s);
  float f;
  char buf[n+1];
  for (i=0; i<n; i++) buf[i]=s[i];
  buf[n]='\0';
  sscanf (buf, "%f", &f);
  return f;
}

char*
as_charp (string s) {
  int i, n=N(s);
  char *s2= new char[n+1];
  for (i=0; i<n; i++) s2[i]=s[i];
  s2[n]= '\0';
  return s2;
}

string
as_string (int i) {
  char buf[64];
  sprintf (buf, "%i\0", i);
  return string(buf);
}

string
as_string (double x) {
  char buf[64];
  sprintf (buf, "%g\0", x);
  return string(buf);
}

string
as_string (char* s) {
  return string (s);
}

bool
is_int (string s) {
  int i=0, n=N(s);
  if (n==0) return FALSE;
  if (s[i]=='+') i++;
  if (s[i]=='-') i++;
  if (i==n) return 0;
  for (; i<n; i++)
    if ((s[i]<'0') || (s[i]>'9')) return 0;
  return 1;
}

bool
is_double (string s) {
  int i=0, n=N(s);
  if (n==0) return FALSE;
  if (s[i]=='+') i++;
  if (s[i]=='-') i++;
  if (i==n) return 0;
  for (; i< n; i++)
    if ((s[i]<'0') || (s[i]>'9')) break;
  if (i==n) return 1;
  if (s[i]=='.') {
    i++;
    if (i==n) return 0;
    for (; i< n; i++)
      if ((s[i]<'0') || (s[i]>'9')) break;
  }
  if (i==n) return 1;
  if (s[i++]!='e') return 0;
  if (s[i]=='+') i++;
  if (s[i]=='-') i++;
  if (i==n) return 0;
  for (; i< n; i++)
    if ((s[i]<'0') || (s[i]>'9')) return 0;
  return 1;
}

bool
is_charp (string s) {
  return TRUE;
}

bool
is_id (string s) {
  int i=0, n=N(s);
  if (n==0) return FALSE;
  for (i=0; i< n; i++) {
    if ((i>0) && (s[i]>='0') && (s[i]<='9')) continue;
    if ((s[i]>='a') && (s[i]<='z')) continue;
    if ((s[i]>='A') && (s[i]<='Z')) continue;
    if (s[i]=='_') continue;
    return 0;
  }
  return 1;
}

void
system (string s) {
  if (info>=2) cout << s << "\n";
  system (s->a);
}

volatile void
fatal_error (char* message, char* routine, char* file) {
  cerr << "\nFatal error: " << message << " in '" << routine << "'\n";
  if (file[0]!='\0') cerr << "See file   : " << file << "\n";
  exit (1);
}
