
/******************************************************************************
* MODULE     : gg_file.gen.cc
* DESCRIPTION: file handling
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include "gg_file.h"

extern string gencc_include;
extern bool   gencc_error;

string
radical (string f) {
  int i= N(f)-1;
  while ((i>0) && (f[i]!='.')) i--;
  return f (0, i);
}

/******************************************************************************
* The ifstream class
******************************************************************************/

bool
ifstream::try_to_open (string s) {
  name= s;
  fp  = fopen (s->a, "r");
  return (fp!=NULL);
}

bool
ifstream::try_to_open (string s, array_string& includes) {
  name= s;
  fp  = NULL;
  if (includes->contains (s)) return (already_included=TRUE);
  fp= fopen (s->a, "r");
  if (fp!=NULL) {
    includes << s;
    return TRUE;
  }
  return FALSE;
}

ifstream::ifstream (string prefix, string s) {
  already_included= FALSE;
  if (N(prefix)==0) try_to_open (s);
  else {
    int i=0;
    while (prefix[i]!='\0') {
      int j=i;
      while ((prefix[j]!=':') && (prefix[j]!='\0')) j++;
      if (try_to_open (prefix (i, j)* "/" * s)) break;
      i=j;
      if (prefix[i]==':') i++;
    }
  }
}

ifstream::ifstream (string prefix, string s, array_string& includes) {
  already_included= FALSE;
  if (N(prefix)==0) try_to_open (s, includes);
  else {
    int i=0;
    while (prefix[i]!='\0') {
      int j=i;
      while ((prefix[j]!=':') && (prefix[j]!='\0')) j++;
      if (try_to_open (prefix (i, j)* "/" * s, includes)) break;
      i=j;
      if (prefix[i]==':') i++;
    }
  }
}

ifstream::~ifstream () {
  if (fp!=NULL) fclose (fp);
}

bool
ifstream::get (char& c) {
  fscanf (fp, "%c", &c);
  return !feof(fp);
}

ifstream::operator bool () {
  return already_included || (fp!=NULL);
}

string
load (string file_name) {
  if (info>=2) cerr << "Info: reading \"" << file_name << "\"\n";
  string the_name;
  ifstream fin ("", file_name);
  if (!fin) {
    cerr << "Fatal error: couldn't open " << file_name << "\n";
    exit (1);
  }
  
  char c;
  string in;
  while (fin.get(c)) in << c;
  return in;
}

/******************************************************************************
* The ofstream class
******************************************************************************/

ofstream::ofstream (string s) {
  fp= fopen (s->a, "w");
}

ofstream::~ofstream () {
  if (fp!=NULL) fclose (fp);
}

ofstream&
ofstream::operator << (char c) {
  fprintf (fp, "%c", c);
  return *this;
}

ofstream&
ofstream::operator << (int i) {
  fprintf (fp, "%i", i);
  return *this;
}

ofstream&
ofstream::operator << (string s) {
  fprintf (fp, "%s", s->a);
  return *this;
}

ofstream::operator bool () {
  return fp!=NULL;
}

void
save (string file_name, string s) {
  if (info>=2) cerr << "Info: writing to \"" << file_name << "\"\n";
  ofstream fout (file_name);
  if (!fout) {
    cerr << "Fatal error: couldn't open " << file_name << "\n";
    exit (1);
  }
  fout << s;
}

/******************************************************************************
* Reading a file with recursive includes
******************************************************************************/

string
relative_name (string base, string f) {
  if ((f[0]=='/') || (f[0]=='$')) return f;
  int i= N(base);
  while ((i>0) && (base[i-1]!='/')) i--;
  while ((i>1) && f ("../", 0) && ((i<3) || base ("../", i-3))) {
    int temp=i;
    i--;
    while ((i>0) && (base[i-1]!='/')) i--;
    if ((base[i]=='$') || (base[i+1]=='$')) {
      i=temp;
      break;
    }
    f=f(3,N(f));
  }
  return base(0,i)*f;
}

void
file_reader::output_line () {
  in << ("#line " * as_string (Input_line) * " \"" * Input_file * "\"\n");
}

void
file_reader::read (string prefix, string file_name) {
  if (info>=2) cerr << "Info: reading \"" << file_name << "\"\n";
  string the_name;
  if (N(prefix)>0) the_name= file_name;
  else the_name= relative_name (Input_file, file_name);
  ifstream fin (prefix, the_name, includes);
  if (!fin) {
    cerr << Input_file;
    if (Input_line!=0) cerr << ":" << Input_line;
    cerr << ": File " << file_name << " not found\n";
    gencc_error= TRUE;
    return;
  }
  file_name= fin.name;
  if (fin.already_included) return;

  string temp_file= Input_file;
  int    temp_line= Input_line;
  bool   os_flag  = TRUE;
  bool   in_module= FALSE;
  Input_file      = file_name;
  Input_line      = 1;
  output_line ();

  char c;
  while (fin.get(c)) {
    in << c;
    if (c=='\n') {
      Input_line++;
      int n   = N(in);
      int last= n-2;
      while ((last>=0) && (in[last]!='\n')) last--;
      last++;
      while ((in[last]==' ') || (in[last]=='\t')) last++;

      /************************ check whether right OS ***********************/
      if (in ("#os ", last)) {
	int l=0;
	os_flag= test_os (in (last+4, N(in)));
	in.resize (last);
	in << '\n';
	continue;
      }
      if (!os_flag) {
	in.resize (last);
	in << '\n';
	continue;
      }

      /************************ handle special keywords **********************/
      if (in ("#option ", last)) continue;
      if (in ("#module ", last)) {
	if (in_module) {
	  cerr << Input_file;
	  if (Input_line!=0) cerr << ":" << Input_line;
	  cerr << ": Recursive modules not allowed\n";
	  gencc_error= TRUE;
	}
	in_module= TRUE;
	output_line ();
	continue;
      }
      if (in ("#endmodule ", last)) {
	output_line ();
	in_module= FALSE;
	continue;
      }
      if (in ("#import ", last)) {
	output_line ();
	continue;
      }
      if (in ("#include ", last)) {
	if (in_module) continue;
      }
      else {
	if (!in_module) {
	  in.resize (last);
	  in << '\n';
	  continue;
	}
	continue;
      }
      
      /*********************** handle recursive includes *********************/
      int end= N(in)-2;
      while ((end>7) && (in[end]!='\n') &&
	     ((in[end]==' ') || (in[end]=='\t'))) end--;
      if ((in[end]!='>') && (in[end]!='\"')) continue; // "

      int start= end-1;
      while ((start>7) && (in[start]!='<') &&
	     (in[start]!='\n') && (in[start]!='\"')) start--; // "
      if ((in[start]!='<') && (in[start]!='\"')) continue; // "
      bool std_flag= ((in[start]=='<') && (in[end]=='>'));
      if (in[start-1]!=' ') continue;
      int k=start-2;
      while ((in[k]==' ') || (in[k]=='\t')) k--;
      if (last+7!=k) continue;
      start++;

      string include_name= in (start, end);
      while ((last>=0) && (in[last]!='\n')) last--;
      in.resize (last+1);
      if (std_flag) read (gencc_include, include_name);
      else          read ("", include_name);
      output_line ();
    }
  }
  in << "\n\n";
  Input_file= temp_file;
  Input_line= temp_line;
}
