
/******************************************************************************
* MODULE     : gg_dep.cc
* DESCRIPTION: analyze dependencies of generic c++ files
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include "gg_file.h"
extern string os;

string gencc_include ("");
string gencc_object  (".");
string gencc_binary  (".");
string gencc_library (".");
bool   gencc_aux   = FALSE;
bool   gencc_shared= FALSE;
bool   gencc_mixed = FALSE;

int    gencc_debug = 0;
bool   gencc_error = FALSE;

string gen_make_name;
int    gen_make_line;

/******************************************************************************
* Some useful subroutines
******************************************************************************/

void
skip_until_CR (string s, int& i) {
  while ((i<N(s)) && (s[i]!='\n')) i++;
  if (i<N(s)) i++;
  gen_make_line++;
}

string
get_argument (string s, int& i) {
  while (s[i]!=' ') i++;
  while ((i<N(s)) && ((s[i]==' ') || (s[i]=='\t'))) i++;
  int start=i;
  while ((i<N(s)) && (s[i]!=' ') && (s[i]!='\t') && (s[i]!='\n')) i++;
  string arg= s (start,i);
  skip_until_CR (s, i);
  return arg;
}

string
get_option (string s, int& i) {
  while (s[i]!=' ') i++;
  while ((i<N(s)) && ((s[i]==' ') || (s[i]=='\t'))) i++;
  int start=i;
  while ((i<N(s)) && (s[i]!='\n')) i++;
  string arg= s (start, i);
  if (i<N(s)) i++;
  gen_make_line++;
  return arg;
}

string
get_option (string global_option, string s, int& i) {
  while ((i<N(s)) && ((s[i]==' ') || (s[i]=='\t') || (s[i]=='\n'))) i++;
  if (s ("#option ", i)) {
    if (N (global_option)==0) return get_option (s, i);
    else return global_option * " " * get_option (s, i);
  }
  return global_option;
}

string
get_phrase (string s, int i) {
  int start=i;
  while ((i<N(s)) && (s[i]!='\n')) i++;
  return s (start, i) * "\n";
}

string
get_file_name (string name, string default_env) {
  if (N(name)==0);
  else if ((name[0]=='<') && (name[N(name)-1]=='>')) {
    name= name (1, N(name)-1);
    if ((N(default_env)>0) &&
	(default_env[N(default_env)-1]=='/')) name= default_env * name;
    else name= default_env * "/" * name;
  }
  else if ((name[0]=='\"') && (name[N(name)-1]=='\"')) // "
    name= name (1, N(name)-1);
  return name;
}

string
concatenate (array_string a) {
  int i;
  string r;
  for (i=0; i<N(a); i++)
    r << (i==0? "": " ") << a[i];
  return r;
}

string
double_dollars (string s) {
  int i;
  string r;
  for (i=0; i<N(s); i++)
    if ((s[i]=='$') && (((i+1)==N(s)) || (s[i+1]!='('))) r << "$$";
    else r << s[i];
  return r;  
}

string
bracket_dollars (string s) {
  int i;
  string r;
  for (i=0; i<N(s); i++)
    if (s[i]=='$') {
      r << "$(";
      i++;
      while ((i<N(s)) && (s[i]!='/')) r << s[i++];
      r << ")";
      if (i<N(s)) r << "/";
    }
    else r << s[i];
  return r;  
}

/******************************************************************************
* handling objects
******************************************************************************/

void
handle_obj_ext (string obj, string s, int& i, string option,
		ofstream& fout, int ext_flag)
{
  option= special_object_option (option);
  file_reader F;
  ofstream cc_out (radical (obj) * ".gen.cc");
  
  for (; i<N(s); ) {
    if (s ("#include ", i)) {
      cc_out << get_phrase (s, i);
      string prefix;
      string name= get_argument (s, i);
      if ((name[0]=='<') && (name[N(name)-1]=='>')) {
	name  = name (1, N(name)-1);
	prefix= gencc_include;
      }
      if ((name[0]=='\"') && (name[N(name)-1]=='\"')) // "
	name= name (1, N(name)-1);
      F.Input_file= gen_make_name;
      F.Input_line= gen_make_line;
      F.read (prefix, name);
      continue;
    }
    if (s ("#code", i)) {
      skip_until_CR (s, i);
      break;
    }
    skip_until_CR (s, i);
  }

  cc_out << "\n#module main\n";
  for (; i<N(s); ) {
    if ((s ("#endobject", i) && (!ext_flag)) ||
	(s ("#endextern", i) && ext_flag)) {
      cc_out << "#endmodule // main\n";
      skip_until_CR (s, i);
      fout << bracket_dollars (obj) << ": "
	   << bracket_dollars (concatenate (F.includes)) << "\n";
      fout << "\t$(GENCC) ";
      if (N(gencc_include)!=0) fout << "-I " << gencc_include << " ";
      if (info!=1) fout << "-i" << info << " ";
      if (ext_flag) {
	if (gencc_debug!=1) fout << "-dg ";
	fout << radical (obj) << ".gen.cc\n\n";
      }
      else {
	string rad= radical (obj);
	if (gencc_debug==2) fout << "-dg ";
	fout << rad << ".gen.cc\n";
	fout << "\t" << make_object (rad * ".cc", rad * ".o", option) << "\n";
	if (!gencc_aux) fout << "\t$(RM) " << rad << ".cc\n";
	fout << "\n";
      }
      return;
    }
    cc_out << get_phrase (s, i);
    skip_until_CR (s, i);
  }

  cerr << gen_make_name << ":" << gen_make_line << ": ";
  cerr << "(fatal) preliminary end in object " << obj << "\n";
  exit (1);
}

/******************************************************************************
* handling binaries and libraries
******************************************************************************/

void
handle_lib_bin (string out, string s, int& i, string option,
		ofstream& fout, bool bin_flag)
{
  bool shared= gencc_shared;
  shared= shared || option_contains (option, "+shared");
  shared= shared && (!option_contains (option, "+static"));
  if (bin_flag) option= special_binary_option (option);
  else option= special_library_option (option);
  if ((option != "") && gencc_mixed && (os != "sun") && (os != "dec")) {
    if (bin_flag) option= "$(LDBSHARED) " * option;
    else option= "$(LDSHARED) " * option;
  }
  if (!bin_flag) {
    if (shared) out << ".so";
    else out << ".a";
  }

  array_string a, b;
  for (; i<N(s); ) {
    if (s ("#object ", i)) {
      string obj= get_file_name (get_argument (s, i), gencc_object);
      a << obj;
      continue;
    }
    if (s ("#uses ", i)) {
      string name= get_argument (s, i);
      if ((N(name)>2) && (name[0]=='<') && (name[N(name)-1]=='>')) {
	name= name (1, N(name)-1);
	option= "-l" * name * " " * option;
	if (gencc_shared) name= "<lib" * name * ".so>";
	else name= "<lib" * name * ".a>";
      }
      else {
	if ((N(name)>2) && (name[0]=='\"') && (name[N(name)-1]=='\"')) // "
	  name= name (1, N(name)-1);
	option= "-l" * name * " " * option;
	name= "\"" * name * "\"";
      }
      if (bin_flag) b << get_file_name (name, gencc_library);
      continue;
    }
    if ((s ("#endlibrary", i) && (!bin_flag)) ||
	(s ("#endbinary", i) && bin_flag)) {
      b << a;
      string in   = concatenate (a);
      string deps = concatenate (b);
      string lib  = gencc_library;
      skip_until_CR (s, i);
      fout << bracket_dollars (out) << ": "
	   << bracket_dollars (deps) << "\n\t";
      in    = double_dollars (in);
      out   = double_dollars (out);
      option= double_dollars (option);
      lib   = double_dollars (lib);
      if (bin_flag) {
	if (shared) fout << make_dynamic_binary (in, out, option, lib);
	else fout << make_static_binary (in, out, option, lib);
      }
      else {
	if (shared) fout << make_dynamic_library (in, out, option, lib);
	else fout << make_static_library (in, out, option, lib);
      }
      fout << "\n\n";
      return;
    }
    skip_until_CR (s, i);
  }

  cerr << gen_make_name << ":" << gen_make_line << ": ";
  cerr << "(fatal) preliminary end in binary or library " << out << "\n";
  exit (1);
}

/******************************************************************************
* generating a makefile from a .gen.make file
******************************************************************************/

void
generate_makefile (string name, string out_name) {
  int i;
  string s= load (name);
  gen_make_name= name;
  gen_make_line= 0;
  bool os_flag=TRUE;
  
  ofstream fout (out_name);
  string obj_option;
  string ext_option;
  string lib_option;
  string bin_option;
  string option;
  array_string targets;

  fout << make_env_variables () << "\n";
  fout << "ALL: END\n\n";
  for (i=0; i<N(s); ) {
    if (s ("#os ", i)) {
      string os= get_option (s, i);
      os_flag= test_os (os);
      continue;
    }
    if (!os_flag) {
      (void) get_option (s, i);
      continue;
    }
    if (s ("#object-option ", i)) {
      obj_option= get_option (s, i);
      continue;
    }
    if (s ("#extern-option ", i)) {
      ext_option= get_option (s, i);
      continue;
    }
    if (s ("#library-option ", i)) {
      lib_option= get_option (s, i);
      continue;
    }
    if (s ("#binary-option ", i)) {
      bin_option= get_option (s, i);
      continue;
    }
    if (s ("#object ", i)) {
      string obj= get_file_name (get_argument (s, i), gencc_object);
      targets << obj;
      option= get_option (obj_option, s, i);
      handle_obj_ext (obj, s, i, option, fout, FALSE);
      continue;
    }
    if (s ("#extern ", i)) {
      string ext= get_file_name (get_argument (s, i), gencc_object);
      targets << ext;
      option= get_option (ext_option, s, i);
      handle_obj_ext (ext, s, i, option, fout, TRUE);
      continue;
    }
    if (s ("#library ", i)) {
      string lib= get_file_name (get_argument (s, i), gencc_library);
      option= get_option (lib_option, s, i);
      handle_lib_bin (lib, s, i, option, fout, FALSE);
      targets << lib;
      continue;
    }
    if (s ("#binary ", i)) {
      string bin= get_file_name (get_argument (s, i), gencc_binary);
      option= get_option (bin_option, s, i);
      handle_lib_bin (bin, s, i, option, fout, TRUE);
      targets << bin;
      continue;
    }
    skip_until_CR (s, i);
  }
  fout << "END: " << bracket_dollars (concatenate (targets)) << "\n";
}

/******************************************************************************
* Main program
******************************************************************************/

int
main (int argc, char **argv)
{
  char* temp;
  temp= getenv ("GENCC_INCLUDE");
  if (temp!=NULL) gencc_include= string(temp);
  temp= getenv ("GENCC_OBJECT");
  if (temp!=NULL) gencc_object = string(temp);
  temp= getenv ("GENCC_LIBRARY");
  if (temp!=NULL) gencc_library= string(temp);
  temp= getenv ("GENCC_BINARY");
  if (temp!=NULL) gencc_binary = string(temp);

  int i, arg;
  if ((argc<=1) || ((argv[1][0]=='-') && (argv[1][1]=='h'))) {
    cout << "--------------------------------------------------------------\n";
    cout << "Generic c, c++ makefile generator options:\n";
    cout << "--------------------------------------------------------------\n";
    cout << "  -I path : Path(s) for include files\n";
    cout << "  -O path : Path where to put object files\n";
    cout << "  -L path : Path where to put libraries\n";
    cout << "  -B path : Path where to put binaries\n";
    cout << "  -o file : Write generated makefile to file\n";
    cout << "  -shared : Create shared libraries\n";
    cout << "  -static : Create static libraries\n";
    cout << "  -ds     : Debugging information for source files\n";
    cout << "  -dg     : Debugging information for generated .cc files\n";
    cout << "  -aux    : Keep generated .cc files\n";
    cout << "  -i0     : Compile silent\n";
    cout << "  -i1     : Display executed processes (default)\n";
    cout << "  -i2     : Maximum information\n";
    cout << "  -h      : display this help message\n";
    cout << "--------------------------------------------------------------\n";
    exit(0);
  }
  
  string out_name;
  for (arg=1; arg<argc; arg++) {
    string name=argv[arg];
    if (name ("-i0")) { info=0; continue; }
    if (name ("-i1")) { info=1; continue; }
    if (name ("-i2")) { info=2; continue; }
    if (name ("-i3")) { info=3; continue; }
    if (name ("-aux")) { gencc_aux= TRUE; continue; }
    if (name ("-ds")) { gencc_debug= 1; continue; }
    if (name ("-dg")) { gencc_debug= 2; continue; }
    if (name ("-shared")) { gencc_shared= TRUE; gencc_mixed= FALSE; continue; }
    if (name ("-mixed")) { gencc_shared= FALSE; gencc_mixed= TRUE; continue; }
    if (name ("-static")) { gencc_shared= gencc_mixed= FALSE; continue; }
    if (arg+1<argc) {
      string next=argv[arg+1];
      if (name ("-I")) {
	if (N(gencc_include)==0) gencc_include= next;
	else gencc_include << ":" << next;
	arg++;
	continue;
      }
      if (name ("-O")) { gencc_object = next; arg++; continue; }
      if (name ("-L")) { gencc_library= next; arg++; continue; }
      if (name ("-B")) { gencc_binary = next; arg++; continue; }
      if (name ("-o")) { out_name = next; arg++; continue; }
    }
    if (N(out_name)==0) out_name= radical (radical (name)) * ".d"; 
    generate_makefile (name, out_name);
    if (gencc_error) system ("$(RM) " * out_name);
    gencc_error= FALSE;
    out_name= "";
  }

  return 0;
}
