/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page.c:
 * Abstract class for all TIPage* classes.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "page.h"
#include "page-priv.h"

#include <glade/glade.h>

#include "util.h"

G_DEFINE_TYPE (TIPage, ti_page, G_TYPE_OBJECT);

enum {
    PROP_LABEL = 1,
    PROP_WIDGET
};

static void
ti_page_dispose (GObject *object)
{
  TIPage *self = TI_PAGE (object);
  TIPagePrivate *priv = TI_PAGE_GET_PRIVATE (self);

  if (priv->disposed)
    return;

  priv->disposed = TRUE;

  if (priv->parent_notebook == NULL)
    {
      g_object_unref (priv->parent_notebook);
      priv->parent_notebook = NULL;
    }

  if (priv->page != NULL)
    {
      g_object_unref (priv->page);
      priv->page = NULL;
    }

  if (priv->page_label != NULL)
    {
      g_object_unref (priv->page_label);
      priv->page_label = NULL;
    }

  G_OBJECT_CLASS (ti_page_parent_class)->dispose (object);
}


static void
ti_page_get_property (GObject *object,
                      guint property_id,
                      GValue *value,
                      GParamSpec *pspec)
{
  TIPage *self = TI_PAGE (object);

  switch (property_id)
    {
    case PROP_LABEL:
      g_value_set_object (value, self->priv->page_label);
      break;
    case PROP_WIDGET:
      g_value_set_object (value, self->priv->page);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_set_property (GObject *object,
                      guint property_id,
                      const GValue *value,
                      GParamSpec *pspec)
{
  TIPage *self = TI_PAGE (object);

  switch (property_id)
    {
    case PROP_LABEL:
      g_return_if_fail (self->priv->page_label == NULL);
      self->priv->page_label = GTK_WIDGET (g_value_dup_object (value));
      break;
    case PROP_WIDGET:
      g_return_if_fail (self->priv->page == NULL);
      self->priv->page = GTK_WIDGET (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_parent_class)->constructed;
  TIPage *self = TI_PAGE (object);

  if (chain_up != NULL)
    chain_up (object);

  (void) self;
}


static void
ti_page_class_init (TIPageClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GParamSpec *param_spec;

  gobject_class->dispose = ti_page_dispose;
  gobject_class->constructed = ti_page_constructed;
  gobject_class->get_property = ti_page_get_property;
  gobject_class->set_property = ti_page_set_property;

  klass->setup_page = NULL;
  klass->restart_page = NULL;

  g_type_class_add_private (klass, sizeof (TIPagePrivate));

  param_spec = g_param_spec_object ("label", "Label", "The label for the tab",
      GTK_TYPE_WIDGET,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (gobject_class, PROP_LABEL, param_spec);

  param_spec = g_param_spec_object ("widget", "Widget",
      "The widget representing the tab's contents", GTK_TYPE_WIDGET,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (gobject_class, PROP_WIDGET, param_spec);
}

static void
ti_page_init (TIPage *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, TI_TYPE_PAGE, TIPagePrivate);

  self->priv->disposed = FALSE;
  self->priv->parent_notebook = NULL;
  self->priv->page = NULL;
  self->priv->page_label = NULL;
}

void
_ti_page_new (TIPage **page,
              GtkNotebook *parent_notebook,
              const gchar *glade_file_name)
{
  TIPagePrivate *priv;
  gint result;
  GladeXML *glade_xml;
  gchar *glade_file_path = NULL;
  TIPageClass *klass = NULL;

  g_assert (parent_notebook != NULL);
  g_assert (GTK_IS_NOTEBOOK (parent_notebook));
  g_assert (page != NULL);
  g_assert (*page != NULL);

  klass = TI_PAGE_GET_CLASS (*page);

  priv = TI_PAGE_GET_PRIVATE (*page);

  priv->parent_notebook = parent_notebook;
  g_object_ref (parent_notebook);

  glade_file_path = ti_get_glade_path (glade_file_name);
  glade_xml = glade_xml_new (glade_file_path, NULL, NULL);

  if (glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (*page);
      *page = NULL;
      goto CLEAN_UP;
    }

  priv->page = glade_xml_get_widget (glade_xml, "page");
  g_assert (priv->page != NULL);
  g_object_ref (priv->page);

  priv->page_label = glade_xml_get_widget (glade_xml, "page_label");
  g_assert (priv->page_label != NULL);
  g_object_ref (priv->page_label);

  g_assert (klass->setup_page != NULL);
  klass->setup_page (*page, glade_xml);

  if (klass->restart_page != NULL)
    {
      klass->restart_page (*page);
    }

  /* Get rid of the glade file and its holder windows. */
  gtk_widget_unparent (priv->page);
  gtk_widget_unparent (priv->page_label);

  g_object_unref (glade_xml);
  glade_xml = NULL;

  /* Append page to notebook */
  result = gtk_notebook_append_page (parent_notebook, priv->page, priv->page_label);
  if (result == -1)
    {
      g_critical ("Unable to append page to notebook.");
      g_object_unref (*page);
      *page = NULL;
      goto CLEAN_UP;
    }

CLEAN_UP:
    g_free (glade_file_path);
}

void
ti_page_detach_from_notebook (TIPage *self)
{
  TIPagePrivate *priv = TI_PAGE_GET_PRIVATE (self);
  gint page_num;

  if (priv->parent_notebook == NULL)
    return;

  page_num = gtk_notebook_page_num (priv->parent_notebook, priv->page);
  g_assert (page_num != -1);

  gtk_notebook_remove_page (priv->parent_notebook, page_num);

  g_object_unref (priv->parent_notebook);
  priv->parent_notebook = NULL;
}

void
ti_page_attach_to_notebook (TIPage *self,
                            GtkNotebook *notebook)
{
  TIPagePrivate *priv = TI_PAGE_GET_PRIVATE (self);
  TIPageClass *klass = TI_PAGE_GET_CLASS (self);

  g_assert (notebook != NULL);
  g_assert (GTK_IS_NOTEBOOK (notebook));

  ti_page_detach_from_notebook (self);

  priv->parent_notebook = notebook;
  g_object_ref (notebook);

  gtk_notebook_append_page (notebook, priv->page, priv->page_label);

  if (klass->restart_page != NULL)
    {
      klass->restart_page (self);
    }
}
