/************************************************************************/
/*									*/
/*  Font Chooser.							*/
/*									*/
/************************************************************************/

#   include	"appFrameConfig.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<stddef.h>
#   include	<string.h>
#   include	<limits.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	<appGeoString.h>
#   include	<utilPropMask.h>

#   include	<appFrame.h>
#   include	<appFontTool.h>
#   include	<appEncodingMenu.h>
#   include	"appDraw.h"

/************************************************************************/
/*									*/
/*  Represents a font chooser.						*/
/*									*/
/************************************************************************/

typedef struct AppFontToolResources
    {
    char *	aftrFamily;
    char *	aftrFace;
    char *	aftrSize;
    char *	aftrRevert;
    char *	aftrSet;

    char *	aftrEncodings[ENCODINGps_COUNT];
    } AppFontToolResources;

typedef struct AppFontChooser
    {
    APP_WIDGET		afcTopWidget;
    APP_WIDGET		afcMainWidget;

    APP_WIDGET		afcFamilyList;
    APP_WIDGET		afcFaceList;
    APP_WIDGET		afcSizeList;
    APP_WIDGET		afcSizeText;

    APP_WIDGET		afcButtonRow;
    APP_WIDGET		afcRevertButton;
    APP_WIDGET		afcSetButton;

    APP_WIDGET		afcSampleDrawing;

    AppEncodingMenu	afcEncodingMenu;

    int			afcFamilyChosen;
    int			afcFaceChosen;
    int			afcBoldChosen;
    int			afcSlantedChosen;
    int			afcSizeChosenHalfPoints;

    int			afcFamilySet;
    int			afcFaceSet;
    int			afcBoldSet;
    int			afcSlantedSet;
    int			afcSizeSetHalfPoints;

    void *		afcTarget;

    AppToolDestroy	afcDestroy;
    FontChooserSetFont	afcSetFont;
    
    AppFontFamily *	afcFamilies;
    int			afcFamilyCount;

    AppDrawingData	afcDrawingData;
    APP_FONT *		afcFont;
    char		afcFamilyFormat[120+1];
    } AppFontChooser;

/************************************************************************/
/*									*/
/*  Value lists that can be used when no particular font is selected.	*/
/*									*/
/************************************************************************/

static int EditFontToolSizesHalfPoints[]=
			    {
			    2* 8,
			    2* 9,
			    2* 10,
			    2* 11,
			    2* 12,
			    2* 14,
			    2* 16,
			    2* 18,
			    2* 24,
			    2* 36,
			    2* 48,
			    2* 64
			    };

static const int EditFontToolSizeCount=
		sizeof(EditFontToolSizesHalfPoints)/sizeof(int);

static AppFontTypeface EditFontToolDummyFace=
	{ "Any", (int *)0, 0, 0, 0, 0, 0 };

static AppFontTypeface EditFontToolDefaultFaces[]=
    {
	{ "Medium", (int *)0, 0, 0, 0, 1, 0 },
	{ "Bold", (int *)0, 0, 1, 0, 1, 0 },
	{ "Italic", (int *)0, 0, 0, 1, 1, 0 },
	{ "Bold Italic", (int *)0, 0, 1, 1, 1, 0 },
    };

static AppFontFamily EditFontToolDummyFamily=
    {
    "Any", (char *)0,
	EditFontToolDefaultFaces,
	sizeof(EditFontToolDefaultFaces)/sizeof(AppFontTypeface),
	0, 1,
    };

/************************************************************************/
/*									*/
/*  Format the size of the font in the Text box below the list.		*/
/*									*/
/************************************************************************/

static void appFontToolFormatSize(	char *		scratch,
					int		sizeHalfPoints )
    {
    if  ( sizeHalfPoints % 2 )
	{ sprintf( scratch, "%d.5", sizeHalfPoints/ 2 );	}
    else{ sprintf( scratch, "%d", sizeHalfPoints/ 2 );		}
    }

/************************************************************************/
/*									*/
/*  Redraw the sample of the font chooser.				*/
/*									*/
/************************************************************************/

static APP_EVENT_HANDLER( appFontRedraw, w, voidafc, exposeEvent )
    {
    AppFontChooser *			afc= (AppFontChooser *)voidafc;
    AppDrawingData *			add= &(afc->afcDrawingData);

    DocumentRectangle			drClip;

    int					x;
    int					y;
    int					wide;
    int					high;

    int					textWide;
    int					fontAscent;
    int					fontDescent;

    const int				ox= 0;
    const int				oy= 0;

    int					l= strlen( afc->afcFamilyFormat );

    appCollectExposures( &drClip, add, ox, oy, exposeEvent );

    appDrawSetForegroundWhite( add );
    appDrawFillRectangle( add, drClip.drX0, drClip.drY0,
					    drClip.drX1- drClip.drX0+ 1,
					    drClip.drY1- drClip.drY0+ 1 );

    if  ( ! afc->afcFont )
	{ SXDEB(afc->afcFamilyFormat,afc->afcFont); return;	}

    appDrawSetForegroundBlack( add );

    appDrawGetSizeOfWidget( &wide, &high, w );

    appDrawTextExtents( &textWide, &fontAscent, &fontDescent,
				add, afc->afcFont, afc->afcFamilyFormat, l );

    y= ( high- fontAscent- fontDescent+ 1 )/ 2;
    y= y+ fontAscent;
    x= wide/ 2- textWide/2;

    appDrawDrawString( add, x, y, afc->afcFamilyFormat, l );

    return;
    }

/************************************************************************/
/*									*/
/*  Show the current font size.						*/
/*									*/
/************************************************************************/

static void appFontShowSizeInList(
				AppFontChooser *	afc,
				int			fontSizeHalfPoints )
    {
    int		pos;
    int		sizeChosen= -1;

    for ( pos= 0; pos < EditFontToolSizeCount; pos++ )
	{
	if  ( fontSizeHalfPoints > 0				&&
	      EditFontToolSizesHalfPoints[pos] ==
					fontSizeHalfPoints	)
	    { sizeChosen= pos; break;	}
	}

    if  ( sizeChosen >= 0 )
	{ appGuiSelectPositionInListWidget( afc->afcSizeList, sizeChosen ); }
    else{ appGuiRemoveSelectionFromListWidget( afc->afcSizeList );	    }

    return;
    }

static void appFontShowSizeInText(
				AppFontChooser *	afc,
				int			fontSizeHalfPoints )
    {
    char	scratch[80];

    if  ( fontSizeHalfPoints > 0 )
	{
	appFontToolFormatSize( scratch, fontSizeHalfPoints );
	appStringToTextWidget( afc->afcSizeText, scratch );
	}
    else{
	appStringToTextWidget( afc->afcSizeText, "" );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Get currently selected font.					*/
/*									*/
/************************************************************************/

static int appFontToolGetSize(	int *			pFontSizeHalfPoints,
				int			anyway,
				AppFontChooser *	afc )
    {
    int			sizeChosenHalfPoints= afc->afcSizeChosenHalfPoints;

    char *		val;
    char *		s;

    const int		minValue= 1;
    const int		adaptToMin= 0;
    const int		maxValue= INT_MAX;
    const int		adaptToMax= 0;

    int			isEmpty;

    s= val= appGetStringFromTextWidget( afc->afcSizeText );
    while( isspace( *s ) )
	{ s++;	}
    isEmpty= ! *s;

    appFreeStringFromTextWidget( val );

    if  ( isEmpty )
	{ sizeChosenHalfPoints= -1;	}
    else{
	int		changed= 0;
	int		sizeTwips= sizeChosenHalfPoints;

	if  ( sizeTwips > 0 )
	    { sizeTwips *= 10;	}

	if  ( appGetLengthFromTextWidget( afc->afcSizeText,
				&sizeTwips, &changed, UNITtyPOINTS,
				minValue, adaptToMin, maxValue, adaptToMax ) )
	    {
	    if  ( anyway )
		{ sizeChosenHalfPoints= -1;	}
	    else{ return -1;			}
	    }
	else{
	    char	scratch[80];

	    sizeChosenHalfPoints= ( sizeTwips+ 5 )/ 10;

	    appFontToolFormatSize( scratch, sizeChosenHalfPoints );

	    val= appGetStringFromTextWidget( afc->afcSizeText );

	    if  ( strcmp( val, scratch ) )
		{ appStringToTextWidget( afc->afcSizeText, scratch );	}

	    appFreeStringFromTextWidget( val );
	    }
	}

    *pFontSizeHalfPoints= sizeChosenHalfPoints;
    return 0;
    }

static int appFontGetCurrent(	AppFontFamily **	pEff,
				AppFontTypeface **	pEft,
				int *			pFontSizeHalfPoints,
				int			anyway,
				AppFontChooser *	afc )
    {
    AppFontFamily *	aff= (AppFontFamily *)0;
    AppFontTypeface *	aft= (AppFontTypeface *)0;

    int			familyChosen= afc->afcFamilyChosen;
    int			faceChosen= afc->afcFaceChosen;
    int			sizeChosenHalfPoints= afc->afcSizeChosenHalfPoints;

    if  ( appFontToolGetSize( &sizeChosenHalfPoints, anyway, afc ) )
	{ return -1;	}

    if  ( familyChosen < 0 && anyway )
	{ familyChosen= 0; }
    if  ( faceChosen < 0 && anyway )
	{ faceChosen= 0; }
    if  ( sizeChosenHalfPoints < 0 && anyway )
	{ sizeChosenHalfPoints= 24; }

    if  ( familyChosen >= 0 )
	{
	aff= afc->afcFamilies+ familyChosen;

	if  ( faceChosen >= 0 )
	    { aft= aff->affFaces+ faceChosen; }
	}
    else{
	aff= &EditFontToolDummyFamily;

	if  ( faceChosen >= 0 )
	    { aft= aff->affFaces+ faceChosen; }
	}

    if  ( pEff )
	{ *pEff= aff;	}
    if  ( pEft )
	{ *pEft= aft;	}
    if  ( pFontSizeHalfPoints )
	{ *pFontSizeHalfPoints= sizeChosenHalfPoints;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Reflect the currently selected font.				*/
/*									*/
/************************************************************************/

static void appFontShowExampleOfCurrent(	AppFontChooser *	afc )
    {
    AppDrawingData *	add= &(afc->afcDrawingData);
    AppEncodingMenu *	aem= &(afc->afcEncodingMenu);

    AppFontFamily *	aff= (AppFontFamily *)0;
    AppFontTypeface *	aft= (AppFontTypeface *)0;
    int			fontSizeHalfPoints= -1;

    char		scratch[120+1];

    if  ( afc->afcFont )
	{
	appDrawFreeFont( &(afc->afcDrawingData), afc->afcFont );
	afc->afcFont= (APP_FONT *)0;
	}

    appFontGetCurrent( &aff, &aft, &fontSizeHalfPoints, 0, afc );

    appFontFormatCurrent( afc->afcFamilyFormat, aff, aft, fontSizeHalfPoints );

    appFontGetCurrent( &aff, &aft, &fontSizeHalfPoints, 1, afc );

    if  ( appFontXFont( scratch, add,
			aff, aff->affDefaultEncoding, aft,
			10* fontSizeHalfPoints, DOCfontREGULAR ) )
	{ SDEB(afc->afcFamilyFormat); return;	}

    afc->afcFont= appDrawOpenFont( add, scratch );
    if  ( ! afc->afcFont )
	{ SXDEB(scratch,afc->afcFont);	}
    else{
	appDrawSetFont( add, afc->afcFont );
	}

    appExposeRectangle( add, 0, 0, 0, 0 );

    appGuiEnableWidget( afc->afcSetButton,
			    afc->afcFamilyChosen >= 0		||
			    aem->aemFontEncodingChosen >= 0	||
			    afc->afcFaceChosen >= 0		||
			    afc->afcSizeChosenHalfPoints >= 0	);

    return;
    }

/************************************************************************/
/*									*/
/*  Refill size list and reselect selected size.			*/
/*									*/
/************************************************************************/

static void appFontFillSizes(	AppFontChooser *	afc,
				AppFontTypeface *	aft,
				int			fontSizeHalfPoints )
    {
    int		pos;

    appGuiEmptyListWidget( afc->afcSizeList );

    for ( pos= 0; pos < EditFontToolSizeCount; pos++ )
	{
	char	scratch[80];

	appFontToolFormatSize( scratch, EditFontToolSizesHalfPoints[pos] );

	appGuiAddValueToListWidget( afc->afcSizeList, scratch );
	}

    appFontShowSizeInList( afc, fontSizeHalfPoints );
    appFontShowSizeInText( afc, fontSizeHalfPoints );

    afc->afcSizeChosenHalfPoints= fontSizeHalfPoints;

    return;
    }

/************************************************************************/
/*									*/
/*  A font family has been chosen, or set.				*/
/*  Try to maintain the properties in the other listboxes.		*/
/*									*/
/************************************************************************/

static void appFontReflectFamily(
				AppFontChooser *	afc,
				int			familyNumber,
				AppFontFamily *		aff,
				int			fontBold,
				int			fontSlanted,
				int			fontSizeHalfPoints )
    {
    AppFontTypeface *		aft;
    int				pos;
    int				faceChosen= -1;

    if  ( afc->afcFamilyChosen != familyNumber )
	{
	appGuiEmptyListWidget( afc->afcFaceList );

	aft= aff->affFaces;
	for ( pos= 0; pos < aff->affFaceCount; aft++, pos++ )
	    {
	    appGuiAddValueToListWidget( afc->afcFaceList, aft->aftFaceName );

	    if  ( fontBold == (int)aft->aftIsBold	&&
		  fontSlanted == (int)aft->aftIsSlanted	)
		{ faceChosen= pos;	}
	    }

	afc->afcFamilyChosen= familyNumber;
	afc->afcFaceChosen= -1;
	}
    else{
	aft= aff->affFaces;
	for ( pos= 0; pos < aff->affFaceCount; aft++, pos++ )
	    {
	    if  ( fontBold == (int)aft->aftIsBold	&&
		  fontSlanted == (int)aft->aftIsSlanted	)
		{ faceChosen= pos;	}
	    }
	}

    appEncodingMenuAdaptToFamilyEncodings( &(afc->afcEncodingMenu), aff );

    if  ( faceChosen >= 0 )
	{
	appGuiSelectPositionInListWidget( afc->afcFaceList, faceChosen );

	if  ( afc->afcFaceChosen != faceChosen )
	    {
	    afc->afcFaceChosen= faceChosen;

	    appFontFillSizes( afc, aff->affFaces+ faceChosen,
							fontSizeHalfPoints );
	    }
	else{
	    appFontShowSizeInList( afc, fontSizeHalfPoints );
	    appFontShowSizeInText( afc, fontSizeHalfPoints );

	    afc->afcSizeChosenHalfPoints= fontSizeHalfPoints;
	    }
	}
    else{
	afc->afcFaceChosen= faceChosen;

	appGuiRemoveSelectionFromListWidget( afc->afcFaceList );

	appFontFillSizes( afc, &EditFontToolDummyFace, fontSizeHalfPoints );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  A font family has been chosen					*/
/*									*/
/************************************************************************/

static APP_LIST_CALLBACK( appFontFamilyChosen, w, voidafc, voidlcs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    AppFontFamily *		aff;
    int				affCount;

    int				fontSize;

    int				familyNumber;
    int				position;

    position= appGuiGetPositionFromListCallback( w, voidlcs );

    if  ( position == afc->afcFamilyChosen )
	{ return;	}

    appFontGetCurrent( NULL, NULL, &fontSize, 0, afc );

    aff= afc->afcFamilies;
    affCount= afc->afcFamilyCount;

    afc->afcFaceChosen= -1;

    familyNumber= position;
    if  ( familyNumber < 0 || familyNumber >= affCount )
	{
	AppEncodingMenu *	aem= &(afc->afcEncodingMenu);

	LLDEB(familyNumber,affCount);
	afc->afcFamilyChosen= -1;
	aem->aemFontEncodingChosen= -1;
	return;
	}

    aff= afc->afcFamilies+ familyNumber;
    appFontReflectFamily( afc, familyNumber, aff,
			afc->afcBoldChosen, afc->afcSlantedChosen, fontSize );

    appFontShowExampleOfCurrent( afc );

    return;
    }

/************************************************************************/
/*									*/
/*  A font style has been chosen					*/
/*									*/
/************************************************************************/

static void appFontEncodingChosen(	APP_WIDGET	w,
					void *		voidafc,
					void *		voidpbcs )
    {
    int				enc;
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    if  ( appEncodingMenuEncodingFromWidget( &enc,
					    &(afc->afcEncodingMenu), w ) )
	{ XDEB(w); return;	}

    if  ( enc != afc->afcEncodingMenu.aemFontEncodingChosen )
	{
	/* ??
	if  ( appSymbolAdaptToEncoding( afc, enc ) )
	    { LDEB(enc);	}
	*/

	afc->afcEncodingMenu.aemFontEncodingChosen= enc;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  A font style has been chosen					*/
/*									*/
/************************************************************************/

static APP_LIST_CALLBACK( appFontFaceChosen, w, voidafc, voidlcs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    AppFontFamily *		aff;
    AppFontTypeface *		aft;
    int				affCount;

    int				fontSize;

    int				position;

    AppFontTypeface *		faceList= EditFontToolDefaultFaces;
    int				faceCount= sizeof(EditFontToolDefaultFaces)/
						    sizeof(AppFontTypeface);

    position= appGuiGetPositionFromListCallback( w, voidlcs );

    if  ( position == afc->afcFaceChosen )
	{ return;	}

    appFontGetCurrent( NULL, NULL, &fontSize, 0, afc );

    aff= afc->afcFamilies;
    affCount= afc->afcFamilyCount;

    afc->afcFaceChosen= -1;

    if  ( afc->afcFamilyChosen >= 0 && afc->afcFamilyChosen < affCount )
	{
	aff= afc->afcFamilies+ afc->afcFamilyChosen;

	faceList= aff->affFaces;
	faceCount= aff->affFaceCount;
	}

    afc->afcFaceChosen= position;
    if  ( afc->afcFaceChosen < 0 || afc->afcFaceChosen >= faceCount )
	{
	LLDEB(afc->afcFaceChosen,aff->affFaceCount);
	afc->afcFaceChosen= -1;
	return;
	}

    aft= aff->affFaces+ afc->afcFaceChosen;

    afc->afcBoldChosen= aft->aftIsBold;
    afc->afcSlantedChosen= aft->aftIsSlanted;

    appFontFillSizes( afc, aft, fontSize );

    appFontShowExampleOfCurrent( afc );

    return;
    }

/************************************************************************/
/*									*/
/*  A font size has been chosen						*/
/*									*/
/************************************************************************/

static APP_LIST_CALLBACK( appFontSizeChosen, w, voidafc, voidlcs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    int				sizeChosen;
    int				position;

    char			scratch[80];

    position= appGuiGetPositionFromListCallback( w, voidlcs );

    sizeChosen= position;
    if  ( sizeChosen < 0 || sizeChosen >= EditFontToolSizeCount )
	{
	LLDEB(sizeChosen,EditFontToolSizeCount);
	afc->afcSizeChosenHalfPoints= -1; return;
	}

    afc->afcSizeChosenHalfPoints= EditFontToolSizesHalfPoints[sizeChosen];

    appFontToolFormatSize( scratch, afc->afcSizeChosenHalfPoints );
    appStringToTextWidget( afc->afcSizeText, scratch );

    appFontShowExampleOfCurrent( afc );

    return;
    }

/************************************************************************/
/*									*/
/*  The user typed 'Enter' in the text box with the font size.		*/
/*									*/
/************************************************************************/

static void appFontToolSizeChanged(	APP_WIDGET	w,
					void *		voidafc,
					void *		voidcbs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    appFontShowExampleOfCurrent( afc );

    return;
    }

static void appFontToolSizeTyped(	APP_WIDGET	w,
					void *		voidafc,
					void *		voidcbs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;
    int				fontSizeHalfPoints= -1;
    int				fontSizeTwips= -1;

    char *			s;

    s= appGetStringFromTextWidget( afc->afcSizeText );

    if  ( ! appGeoLengthFromString( s, UNITtyPOINTS, &fontSizeTwips ) )
	{ fontSizeHalfPoints= ( fontSizeTwips+ 5 )/ 10;	}

    appFreeStringFromTextWidget( s );

    if  ( afc->afcSizeChosenHalfPoints != fontSizeHalfPoints )
	{ appFontShowSizeInList( afc, fontSizeHalfPoints );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Fill the list with families.					*/
/*									*/
/************************************************************************/

static void appFontFillFamilies(	AppFontChooser *	afc )
    {
    int			i;
    AppFontFamily *	aff;

    aff=  afc->afcFamilies;
    for ( i= 0; i < afc->afcFamilyCount; aff++, i++ )
	{
	if  ( aff->affFontFamilyText )
	    {
	    appGuiAddValueToListWidget( afc->afcFamilyList,
						aff->affFontFamilyText );
	    }
	else{
	    appGuiAddValueToListWidget( afc->afcFamilyList,
						aff->affFontFamilyName );
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  A fontchooser must be destroyed.					*/
/*									*/
/************************************************************************/

static APP_CLOSE_CALLBACK( appCloseFontChooser, w, voidafc )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;

    if  ( afc->afcFont )
	{ appDrawFreeFont( &(afc->afcDrawingData), afc->afcFont );	}

    appCleanDrawingData( &(afc->afcDrawingData) );

    if  ( afc->afcDestroy )
	{ (*afc->afcDestroy)( afc->afcTarget );	}

    free( voidafc );

    appDestroyShellWidget( w );

    return;
    }

/************************************************************************/
/*									*/
/*  make the font chooser worm with three listboxes.			*/
/*									*/
/************************************************************************/

static void appFontMakeChooseForm(	APP_WIDGET		parent,
					AppFontToolResources *	aftr,
					AppFontChooser *	afc )
    {
    APP_WIDGET		column;
    APP_WIDGET		label;

    APP_WIDGET		chooseRow;

    const int		heightResizable= 1;

    chooseRow= appMakeRowInColumn( parent, 3, heightResizable );

    /********************************************************************/
    /*									*/
    /*  Parents with listbox and label:					*/
    /*									*/
    /********************************************************************/

    appMakeColumnInRow( &column, chooseRow, 0 );

    appMakeLabelInColumn( &label, column, aftr->aftrFamily );

    appGuiMakeListInColumn( &(afc->afcFamilyList), column, 9,
					    appFontFamilyChosen, (void *)afc );

    /***/

    appMakeColumnInRow( &column, chooseRow, 1 );

    appMakeLabelInColumn( &label, column, aftr->aftrFace );

    appGuiMakeListInColumn( &(afc->afcFaceList), column, 9,
					    appFontFaceChosen, (void *)afc );

    appMakeOptionmenuInColumn( &(afc->afcEncodingMenu.aemEncodingOptionmenu),
								    column );

    {
    const int		textColumns= 5;
    const int		textEnabled= 1;

    appMakeColumnInRow( &column, chooseRow, 2 );

    appMakeLabelInColumn( &label, column, aftr->aftrSize );

    appGuiMakeListInColumn( &(afc->afcSizeList),
				column, 10, appFontSizeChosen, (void *)afc );

    appMakeTextInColumn( &(afc->afcSizeText), column,
						textColumns, textEnabled );

    appGuiSetGotValueCallbackForText( afc->afcSizeText,
					appFontToolSizeChanged, (void *)afc );

    appGuiSetTypingCallbackForText( afc->afcSizeText,
					appFontToolSizeTyped, (void *)afc );
    }

    return;
    }

/************************************************************************/
/*									*/
/*  make the label to preview the font.					*/
/*									*/
/************************************************************************/

static void appFontMakePreviewDrawing(	APP_WIDGET		parent,
					AppFontChooser *	afc,
					double			xfac,
					int			twipsSize )
    {
    const int		wide= -1;
    int			high= 1.3* ( xfac* twipsSize )+ 0.5;
    const int		heightResizable= 0;

    appGuiMakeDrawingAreaInColumn( &(afc->afcSampleDrawing), parent,
		    wide, high, heightResizable, appFontRedraw, (void *)afc );
    return;
    }

/************************************************************************/
/*									*/
/*  The set button of a font tool has been pushed.			*/
/*									*/
/************************************************************************/

static void appFontSetPushed(		APP_WIDGET	w,
					void *		voidafc,
					void *		voidpbcs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;
    AppEncodingMenu *		aem= &(afc->afcEncodingMenu);

    AppFontFamily *		aff;
    AppFontTypeface *		aft;
    int				sizeHalfPoints;

    char *			familyText= (char *)0;
    int				encodingIsDefault= 0;

    if  ( ! afc->afcSetFont )
	{ XDEB(afc->afcSetFont); return;	}

    if  ( appFontGetCurrent( &aff, &aft, &sizeHalfPoints, 0, afc ) )
	{ return;	}

    if  ( aff )
	{
	if  ( aff->affFontFamilyText )
	    { familyText= aff->affFontFamilyText;	}
	else{ familyText= aff->affFontFamilyName;	}

	encodingIsDefault=
			aem->aemFontEncodingChosen == aff->affDefaultEncoding;
	}

    if  ( aft )
	{
	(*afc->afcSetFont)( afc->afcTarget, familyText,
			afc->afcFamilyChosen,
			aem->aemFontEncodingChosen, encodingIsDefault,
			aft->aftIsBold, aft->aftIsSlanted, sizeHalfPoints );
	}
    else{
	(*afc->afcSetFont)( afc->afcTarget, familyText,
			afc->afcFamilyChosen,
			aem->aemFontEncodingChosen, encodingIsDefault,
			-1, -1, sizeHalfPoints );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  The 'Revert' button of a font tool has been pushed.			*/
/*									*/
/************************************************************************/

static void appFontRevertPushed(	APP_WIDGET	w,
					void *		voidafc,
					void *		voidpbcs )
    {
    AppFontChooser *		afc= (AppFontChooser *)voidafc;
    AppEncodingMenu *		aem= &(afc->afcEncodingMenu);

    if  ( afc->afcFamilySet < 0 )
	{ appGuiRemoveSelectionFromListWidget( afc->afcFamilyList ); }
    else{
	AppFontFamily *	aff= afc->afcFamilies+ afc->afcFamilySet;

	appGuiSelectPositionInListWidget( afc->afcFamilyList,
							afc->afcFamilySet );

	appFontReflectFamily( afc, afc->afcFamilySet, aff,
				afc->afcBoldSet, afc->afcSlantedSet,
				afc->afcSizeSetHalfPoints );
	}

    afc->afcFamilyChosen= afc->afcFamilySet;
    aem->aemFontEncodingChosen= aem->aemFontEncodingSet;
    afc->afcFaceChosen= afc->afcFaceSet;
    afc->afcSizeChosenHalfPoints= afc->afcSizeSetHalfPoints;
    afc->afcBoldChosen= afc->afcBoldSet;
    afc->afcSlantedChosen= afc->afcSlantedSet;

    appFontShowExampleOfCurrent( afc );

    return;
    }

/************************************************************************/
/*									*/
/*  Reflect/Remember the current font of the application.		*/
/*									*/
/*  1)  Adapt the old fashioned internals to new style interface.	*/
/*									*/
/************************************************************************/

int appFontSetCurrentFont(	void *			voidafc,
				double			xfac,
				int			fontFamilyNumber,
				int			encoding,
				const PropertyMask *	updMask,
				TextAttribute		taNew )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;
    AppEncodingMenu *	aem= &(afc->afcEncodingMenu);

    int			fontBold= -1;
    int			fontSlanted= -1;
    int			fontSizeHalfPoints= -1;

    /*  1  */
    if  ( PROPmaskISSET( updMask, TApropFONTBOLD ) )
	{ fontBold= taNew.taFontIsBold;	}
    if  ( PROPmaskISSET( updMask, TApropFONTSLANTED ) )
	{ fontSlanted= taNew.taFontIsSlanted;	}
    if  ( PROPmaskISSET( updMask, TApropFONTSIZE ) )
	{ fontSizeHalfPoints= taNew.taFontSizeHalfPoints;	}

    afc->afcFamilySet= fontFamilyNumber;
    afc->afcBoldChosen= afc->afcBoldSet= fontBold;
    afc->afcSlantedChosen= afc->afcSlantedSet= fontSlanted;
    afc->afcSizeSetHalfPoints= fontSizeHalfPoints;

    aem->aemFontEncodingSet= aem->aemFontEncodingChosen= encoding;

    if  ( fontFamilyNumber < 0 )
	{
	appGuiRemoveSelectionFromListWidget( afc->afcFamilyList );

	appFontReflectFamily( afc,
				fontFamilyNumber,
				&EditFontToolDummyFamily,
				fontBold,
				fontSlanted,
				fontSizeHalfPoints );

	afc->afcFaceSet= afc->afcFaceChosen;
	}
    else{
	AppFontFamily *	aff= afc->afcFamilies+ fontFamilyNumber;

	appGuiSelectPositionInListWidget( afc->afcFamilyList,
							fontFamilyNumber );

	appFontReflectFamily( afc,
				fontFamilyNumber,
				aff,
				afc->afcBoldSet,
				afc->afcSlantedSet,
				fontSizeHalfPoints );

	afc->afcFaceSet= afc->afcFaceChosen;
	}

    appSetOptionmenu( &(aem->aemEncodingOptionmenu),
					    aem->aemFontEncodingChosen );

    appFontShowExampleOfCurrent( afc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make the form with the three buttons.				*/
/*									*/
/************************************************************************/

static void appFontMakeButtonForm(	APP_WIDGET		parent,
					AppFontToolResources *	aftr,
					AppFontChooser *	afc	)
    {
    const int	heightResizable= 0;
    const int	showAsDefault= 0;

    afc->afcButtonRow= appMakeRowInColumn( parent, 2, heightResizable );

    appMakeButtonInRow( &(afc->afcRevertButton), afc->afcButtonRow,
			aftr->aftrRevert,
			appFontRevertPushed, (void *)afc, 0, showAsDefault );
    appMakeButtonInRow( &(afc->afcSetButton), afc->afcButtonRow,
			aftr->aftrSet,
			appFontSetPushed, (void *)afc, 1, showAsDefault );

    return;
    }

/************************************************************************/
/*									*/
/*  make a fonts tool.							*/
/*									*/
/************************************************************************/

static AppConfigurableResource APP_FontToolResourceTable[]=
    {
	APP_RESOURCE( "fontToolFamily",
		    offsetof(AppFontToolResources,aftrFamily),
		    "Family" ),
	APP_RESOURCE( "fontToolFace",
		    offsetof(AppFontToolResources,aftrFace),
		    "Typeface" ),
	APP_RESOURCE( "fontToolSize",
		    offsetof(AppFontToolResources,aftrSize),
		    "Size" ),
	APP_RESOURCE( "fontToolRevert",
		    offsetof(AppFontToolResources,aftrRevert),
		    "Revert" ),
	APP_RESOURCE( "fontToolSet",
		    offsetof(AppFontToolResources,aftrSet),
		    "Set" ),
    };

void * appMakeFontsTool(	APP_WIDGET		fontsOption,
				const char *		widgetName,
				APP_BITMAP_IMAGE	iconPixmap,
				APP_BITMAP_MASK		iconMask,
				AppToolDestroy		destroy,
				FontChooserSetFont	setFont,
				EditApplication *	ea )
    {
    AppFontChooser *		afc;
    
    const int			userResizable= 1;

    double			horPixPerMM;
    double			verPixPerMM;
    double			xfac;
    double			yfac;

    static AppFontToolResources	aftr;
    static int			gotResources;

    if  ( ! gotResources )
	{
	appEncodingMenuGetOptionTexts( aftr.aftrEncodings, ea );

	appGuiGetResourceValues( ea, (void *)&aftr, APP_FontToolResourceTable,
					sizeof(APP_FontToolResourceTable)/
					sizeof(AppConfigurableResource) );

	gotResources= 1;
	}

    appGetFactors( ea, &horPixPerMM, &verPixPerMM, &xfac, &yfac );

    afc= (AppFontChooser *)malloc( sizeof(AppFontChooser) );
    if  ( ! afc )
	{ XDEB(afc); return (void *)0;	}

    appInitDrawingData( &(afc->afcDrawingData) );
    afc->afcFont= (APP_FONT *)0;
    afc->afcFamilyFormat[0]= '\0';

    afc->afcTarget= (void *)ea;
    afc->afcDestroy= destroy;
    afc->afcSetFont= setFont;

    appMakeVerticalTool( &(afc->afcTopWidget), &(afc->afcMainWidget), ea,
			    iconPixmap, iconMask, widgetName, userResizable,
			    fontsOption, appCloseFontChooser, (void *)afc );

    {
    int		twipsSize= 20* 48;

    appFontMakePreviewDrawing( afc->afcMainWidget, afc, xfac, twipsSize );

    appFontMakeChooseForm( afc->afcMainWidget, &aftr, afc );

    appFontMakeButtonForm( afc->afcMainWidget, &aftr, afc );
    }

    if  ( psFontCatalog( ea->eaAfmDirectory,
				&afc->afcFamilies, &afc->afcFamilyCount ) )
	{
	SDEB(ea->eaAfmDirectory);
	afc->afcFamilies= (AppFontFamily *)0;
	afc->afcFamilyCount= 0;
	}

    appFontFillFamilies( afc );

    appEncodingMenuFillOptionmenu( aftr.aftrEncodings,
				    appFontEncodingChosen, (void *)afc,
				    &(afc->afcEncodingMenu) );

    afc->afcFamilySet= afc->afcFamilyChosen= -2; /* force fill of styles */
    afc->afcFaceSet= afc->afcFaceChosen= -1;
    afc->afcBoldSet= afc->afcBoldChosen= -1;
    afc->afcSlantedSet= afc->afcSlantedChosen= -1;
    afc->afcSizeSetHalfPoints= afc->afcSizeChosenHalfPoints= -1;

#   ifdef USE_MOTIF
    XtRealizeWidget( afc->afcTopWidget );
#   endif

#   ifdef USE_GTK
    gtk_widget_realize( afc->afcSampleDrawing );
#   endif

    appOptionmenuRefreshWidth( &(afc->afcEncodingMenu.aemEncodingOptionmenu) );

    appSetDrawingDataForWidget( afc->afcSampleDrawing,
				ea->eaMagnification, &(afc->afcDrawingData) );

    appShowShellWidget( afc->afcTopWidget );

    return (void *)afc;
    }

/************************************************************************/
/*									*/
/*  Draw a fonts tool to front.						*/
/*									*/
/************************************************************************/

void appShowFontsTool(		void *	voidafc	)
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    appShowShellWidget( afc->afcTopWidget );
    }

void appEnableFontsTool(	void *	voidafc,
				int	enabled )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    appGuiEnableWidget( afc->afcMainWidget, enabled != 0 );

    return;
    }

void appFontToolTextReadonly(	void *	voidafc,
				int	read_only )
    {
    AppFontChooser *	afc= (AppFontChooser *)voidafc;

    appGuiEnableWidget( afc->afcMainWidget, read_only == 0 );

    return;
    }
