/************************************************************************/
/*									*/
/*  Print PostScript, include PDF marks to preserve links when the	*/
/*  PostScript is converted to PDF.					*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<appImage.h>
#   include	<appWinMeta.h>
#   include	<appMacPict.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>

#   include	"docDraw.h"
#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Printing of borders.						*/
/*									*/
/************************************************************************/

static void psPrintHorizontalBorder(	const BorderProperties *	bp,
					void *				vps,
					int				x0,
					int				x1,
					int				y )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( bp->bpStyle == DOCbsNONE )
	{ return;	}

    fprintf( ps->psFile, "%d %d %d %d rectfill\n",
					x0, y, x1- x0, bp->bpPenWideTwips );

    return;
    }

static void psPrintVerticalBorder(	const BorderProperties *	bp,
					void *				vps,
					int				x,
					int				y0,
					int				y1 )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( bp->bpStyle == DOCbsNONE )
	{ return;	}

    fprintf( ps->psFile, "%d %d %d %d rectfill\n",
					x, y0, bp->bpPenWideTwips, y1- y0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a series of particules with the same attributes.		*/
/*									*/
/************************************************************************/

typedef int (*PLAY_METAFILE)(	FILE *				f,
				SimpleInputStream *		sis,
				const char *			afmDirectory,
				int				mapMode,
				int				xWinExt,
				int				yWinExt,
				int				twipsWide,
				int				twipsHigh );

static int psPrintMetafile(	PrintingState *			ps,
				const InsertedObject *		io,
				const char *			afmDirectory,
				int				x0,
				int				baseline,
				int				scaleX,
				int				scaleY,
				int				xWinExt,
				int				yWinExt,
				int				twipsWide,
				int				twipsHigh )
    {
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisMeta;
    const MemoryBuffer *	mb;

    PostScriptFaceList		psfl;

    int				y0;
    int				i;

    PLAY_METAFILE		playMetafile;
    int				mapMode= 0;

    int				encodingDefined[ENCODINGps_COUNT];

    for ( i= 0; i < ENCODINGps_COUNT; i++ )
	{ encodingDefined[i]= ps->psEncodingDefined[i];	}

    utilInitPostScriptFaceList( &psfl );

    if  ( docPsListObjectFonts( &psfl, io, afmDirectory, "pf" ) )
	{ LDEB(1); return -1;	}

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    mb= &(io->ioObjectData);
	    mapMode= io->ioMapMode;
	    playMetafile= appMetaPlayFilePs;
	    break;

	case DOCokMACPICT:
	    mb= &(io->ioObjectData);
	    playMetafile= appMacPictPlayFilePs;
	    break;

	case DOCokOLEOBJECT:
	    mb= &(io->ioResultData);
	    playMetafile= appMetaPlayFilePs;
	    mapMode= io->ioResultMapMode;
	    break;

	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:
	default:
	    LDEB(io->ioKind); return 0;
	}

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); sioInClose( sisMem ); return -1;	}

    y0= baseline- ( ( scaleY/100.0 )* twipsHigh );

    fprintf( ps->psFile, "100 dict begin\n" );

    appPsFontNames( ps->psFile, &psfl, encodingDefined, /*allFonts=*/ 1 );

    fprintf( ps->psFile, "gsave %d %d translate\n", x0, y0 );

    if  ( scaleX != 100 || scaleY != 100 )
	{ fprintf( ps->psFile, "%f %f scale\n", scaleX/100.0, scaleY/100.0 ); }

    if  ( (*playMetafile)( ps->psFile, sisMeta, afmDirectory, mapMode,
				    xWinExt, yWinExt, twipsWide, twipsHigh ) )
	{ LDEB(1);	}

    fprintf( ps->psFile, "grestore end\n" );

    sioInClose( sisMeta );
    sioInClose( sisMem );

    utilCleanPostScriptFaceList( &psfl );

    return 0;
    }

static int psPrintIncludeEpsObject(	PrintingState *		ps,
					InsertedObject *	io,
					int			x0,
					int			baseLine )
    {
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisHex;

    char			line[512+2];

    sisMem= sioInMemoryOpen( &(io->ioResultData) );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisHex= sioInHexOpen( sisMem );
    if  ( ! sisHex )
	{ XDEB(sisHex); sioInClose( sisMem ); return -1;	}

    appPsBeginEpsObject( ps, x0, baseLine,
				0, 0, io->ioTwipsWide, io->ioTwipsHigh,
				io->ioObjectData.mbBytes );

    while( sioInGetString( line, 512+1, sisHex ) )
	{
	int		emit= 1;
	const char *	s= line;

	while( isspace( *s ) )
	    { s++;	}

	if  ( ! *s || *s == '%' )
	    { emit= 0;	}

	if  ( emit )
	    { fputs( line, ps->psFile ); }

	while( ! strchr( line, '\n' ) )
	    {
	    if  ( ! sioInGetString( line, 512+1, sisHex ) )
		{ break;	}

	    if  ( emit )
		{ fputs( line, ps->psFile ); }
	    }
	}

    sioInClose( sisHex );
    sioInClose( sisMem );

    appPsEndEpsObject( ps );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write links to be picked up by the distiller.			*/
/*									*/
/************************************************************************/

static void psUriLinkDestination(	PrintingState *	ps )
    {
    appPsPrintString( ps->psFile,
	(const unsigned char *)ps->psLinkFile, ps->psLinkFileSize );

    if  ( ps->psLinkMarkSize > 0 )
	{
	putc( '#', ps->psFile );
	appPsPrintString( ps->psFile,
			    (const unsigned char *)ps->psLinkMark,
			    ps->psLinkMarkSize );
	}
    }

static void psWebLinkDestination(	PrintingState *	ps )
    {
    fprintf( ps->psFile, "  /Action << /Subtype /URI /URI (" );

    psUriLinkDestination( ps );

    fprintf( ps->psFile, ") >>\n" );

    return;
    }

static void psFileLinkDestination(	PrintingState *		ps )
    {
    const unsigned char *	file;
    int				size;

    file= (const unsigned char *)ps->psLinkFile;
    size= ps->psLinkFileSize;

    if  ( size > 5 && ! strncmp( (const char *)file, "file:", 5 ) )
	{ file += 5; size -= 5; }
    else{
	while( size > 0 && isalpha( *file ) )
	    { file++; size--;	}

	if  ( size > 0 && *file == ':' )
	    { psWebLinkDestination( ps ); return; }

	file= (const unsigned char *)ps->psLinkFile;
	size= ps->psLinkFileSize;
	}

    fprintf( ps->psFile, "  /Action /Launch /File (" );

    appPsPrintString( ps->psFile, file, size );

    fprintf( ps->psFile, ")\n" );

    if  ( ps->psLinkMarkSize )
	{
	fprintf( ps->psFile, "  /URI (" );
	psUriLinkDestination( ps );
	fprintf( ps->psFile, ")\n" );
	}

    return;
    }

static void psFlushLink(	PrintingState *		ps,
				const ParticuleData *	pd,
				int			lineTop,
				int			lineHeight )
    {
    if  ( ps->psLinkParticulesDone > 0 )
	{
	fprintf( ps->psFile, "[ /Rect [ %d %d %d %d ]\n",
				    ps->psLinkRectLeft, lineTop+ lineHeight, 
				    pd->pdX0+ pd->pdVisibleWidth,
				    lineTop );

	fprintf( ps->psFile, "  /Border [ 0 0 0 ]\n" );

	if  ( ps->psLinkFileSize == 0 )
	    {
	    fprintf( ps->psFile, "  /Action /Goto\n" );
	    fprintf( ps->psFile, "  /Dest /%.*s\n",
					ps->psLinkMarkSize, ps->psLinkMark );
	    }
	else{
	    psFileLinkDestination( ps );
	    }


	fprintf( ps->psFile, "  /Subtype /Link\n" );
	fprintf( ps->psFile, "/LNK pdfmark\n" );

	ps->psLinkParticulesDone= 0;
	ps->psLinkRectLeft= pd->pdX0;
	}

    return;
    }

static int psPrintDrawTab(	PrintingState *		ps,
				int			x0,
				int			x1,
				int			step,
				int			baseLine,
				const char *		tabProc )
    {
    x0= step* ( ( x0+ step- 1 )/ step );
    if  ( x1 <= x0 )
	{ return 0;	}

    if  ( ps->psUseLinkColor )
	{ utilPsSetLinkColor( ps );	}
    else{ utilPsSetTextColor( ps );	}

    fprintf( ps->psFile, "%d %d %d %s\n", x1- x0, x0, baseLine, tabProc );

    return 0;
    }

static int psPrintTab(	PrintingState *			ps,
			DrawingContext *		dc,
			const BufferItem *		bi,
			const TextParticule *		tp,
			ParticuleData *			pd,
			int				baseLine,
			int				lineHeight )
    {
    TabStop *	ts= bi->biParaTabStops+ pd->pdTabNumber;

    int		x0= pd->pdX0+ lineHeight/ 4;
    int		x1= pd->pdX0+ pd->pdWidth- lineHeight/2;

    switch( ts->tsLeader )
	{
	case DOCtlNONE:
	    break;

	case DOCtlDOTS:

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		psPrintDrawTab( ps, x0, x1, 60, baseLine, "dot-tab-bold" );
		}
	    else{
		psPrintDrawTab( ps, x0, x1, 60, baseLine, "dot-tab" );
		}

	    break;

	case DOCtlUNDERLINE:

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		psPrintDrawTab( ps, x0, x1, 20, baseLine, "ul-tab-bold" );
		}
	    else{
		psPrintDrawTab( ps, x0, x1, 20, baseLine, "ul-tab" );
		}

	    break;

	case DOCtlHYPH:

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		psPrintDrawTab( ps, x0, x1, 140, baseLine, "dash-tab-bold" );
		}
	    else{
		psPrintDrawTab( ps, x0, x1, 140, baseLine, "dash-tab" );
		}

	    break;

	case DOCtlTHICK:
	    LDEB(ts->tsLeader);
	    break;

	case DOCtlEQUAL:
	    LDEB(ts->tsLeader);
	    break;

	default:
	    LDEB(ts->tsLeader);
	    break;
	}

    return 0;
    }

static void psPrintObjectBox(	PrintingState *		ps,
				const InsertedObject *	io,
				const ParticuleData *	pd,
				int			baseLine )
    {
    int		high;

    utilPsSetTextColor( ps );

    high= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
    fprintf( ps->psFile, "%d %d moveto ",
			    pd->pdX0, baseLine- high );
    fprintf( ps->psFile, "%d %d lineto ",
			    pd->pdX0+ pd->pdWidth, baseLine- high );
    fprintf( ps->psFile, "%d %d lineto ",
			    pd->pdX0+ pd->pdWidth, baseLine );
    fprintf( ps->psFile, "%d %d lineto ", pd->pdX0, baseLine );
    fprintf( ps->psFile, "closepath stroke\n" );

    return;
    }

static int psPrintParticules(	PrintingState *			ps,
				DrawingContext *		dc,
				const BufferItem *		bi,
				const DocumentFontList *	dfl,
				const char *			afmDirectory,
				int				part,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				count,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    BufferDocument *		bd= dc->dcDocument;

    int				drawn;
    int				i;
    int				len;
    InsertedObject *		io;

    DocumentField *		df;
    int				fontSizeTwips;

    /*  1  */
    switch( tp->tpKind )
	{
	case DOCkindTAB:
	    if  ( pd->pdTabNumber >= 0			&&
		  pd->pdTabNumber < bi->biParaTabCount	)
		{
		if  ( psPrintTab( ps, dc, bi, tp, pd, baseLine, lineHeight ) )
		    { LDEB(1);	}
		}

	    return drawn= 1;

	case DOCkindTEXT:
	    break;

	case DOCkindFIELDSTART:
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( df->dfKind == DOCfkBOOKMARK )
		{
		const char *	refName;
		int		refSize;

		if  ( ! docFieldGetBookmark( df, &refName, &refSize ) )
		    {
		    utilPsDestPdfmark( ps, lineTop, refName, refSize );
		    }

		return drawn= 1;
		}

	    if  ( df->dfKind == DOCfkCHFTN )
		{
		const char *	refName;
		int		refSize;
		const char *	markName;
		int		markSize;

		int		noteIndex;
		DocumentNote *	dn;

		char		ref[25+1];
		char		def[25+1];

		int		cnt;

		cnt= docCountParticulesInField( bi, part,
						    bi->biParaParticuleCount );
		if  ( cnt < 0 )
		    { LDEB(cnt); }

		noteIndex= docGetNote( &dn, bd, bi, tp[cnt+1].tpStroff );
		if  ( noteIndex < 0 )
		    {
		    LLDEB(cnt,tp[cnt+1].tpStroff);
		    SLDEB(docExternalKindStr(bi->biInExternalItem),noteIndex);
		    }

		sprintf( ref, "_NREF_%d", noteIndex+ 1 );
		sprintf( def, "_NDEF_%d", noteIndex+ 1 );

		if  ( bi->biInExternalItem == DOCinBODY )
		    {
		    markName= def;
		    refName=  ref;
		    }
		else{
		    markName= ref;
		    refName=  def;
		    }

		markSize= strlen( markName );
		refSize= strlen( refName );

		utilPsDestPdfmark( ps, lineTop, refName, refSize );

		ps->psLinkFile= (char *)0; ps->psLinkFileSize= 0;
		ps->psLinkMark= markName; ps->psLinkMarkSize= markSize;

		ps->psInsideLink= 1;
		ps->psLinkParticulesDone= 0;
		ps->psLinkRectLeft= pd->pdX0;

		return drawn= 1;
		}

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{
		if  ( ! docFieldGetHyperlink( df,
				&(ps->psLinkFile), &(ps->psLinkFileSize),
				&(ps->psLinkMark), &(ps->psLinkMarkSize) ) )
		    {
		    ps->psInsideLink= 1;
		    ps->psUseLinkColor= 1;
		    ps->psLinkParticulesDone= 0;
		    ps->psLinkRectLeft= pd->pdX0;

		    utilPsSetLinkColor( ps );
		    }

		return drawn= 1;
		}

	    return drawn= 1;

	case DOCkindFIELDEND:
	    if  ( ps->psInsideLink )
		{
		psFlushLink( ps, pd, lineTop, lineHeight );
		ps->psInsideLink= 0;
		ps->psUseLinkColor= 0;
		}

	    utilPsSetTextColor( ps );

	    return drawn= 1;

	case DOCkindXE:
	case DOCkindTC:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	case DOCkindNOTE:
	    return drawn= 1;

	case DOCkindOBJECT:
	    io= bi->biParaObjects+ tp->tpObjectNumber;

	    switch( io->ioKind )
		{
		case DOCokPICTWMETAFILE:
		case DOCokMACPICT:
		    if  ( psPrintMetafile( ps, io,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->io_xWinExt, io->io_yWinExt,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			{ LDEB(1); break;	}

		    ps->psCurrentPhysicalFont= -1;
		    ps->psLinkParticulesDone++;
		    return 1;

		case DOCokPICTJPEGBLIP:
		case DOCokPICTPNGBLIP:

		    if  ( ! io->ioPrivate )
			{
			if  ( docGetBitmapForObject( io ) )
			    { XDEB(io->ioPrivate);	}
			}

		    if  ( io->ioPrivate )
			{
			AppBitmapImage *	abi;
			BitmapDescription *	bmd;

			double			scaleX= io->ioScaleX/ 100.0;
			double			scaleY= io->ioScaleY/ 100.0;

			int			imageWideTwips;
			int			imageHighTwips;

			abi= (AppBitmapImage *)io->ioPrivate;
			bmd= &abi->abiBitmap;

			bmImageSizeTwips( &imageWideTwips, &imageHighTwips,
									bmd );

			if  ( imageWideTwips > 20 )
			    {
			    scaleX= ( scaleX* io->ioTwipsWide )/ imageWideTwips;
			    }
			if  ( imageHighTwips > 20 )
			    {
			    scaleY= ( scaleY* io->ioTwipsHigh )/ imageHighTwips;
			    }

			if  ( bmPsPrintBitmap( ps->psFile, 1,
				    20.0* scaleX, -20.0* scaleY,
				    pd->pdX0, baseLine, 0, 0,
				    bmd->bdPixelsWide, bmd->bdPixelsHigh,
				    bmd, abi->abiBuffer ) )
			    { LDEB(1); return -1; }

			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;

		case DOCokOLEOBJECT:
		    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
			{
			if  ( psPrintMetafile( ps, io,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->io_xWinExt, io->io_yWinExt,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			    { LDEB(1); break;	}

			ps->psCurrentPhysicalFont= -1;
			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;

		case DOCokINCLUDEPICTURE:
		    if  ( io->ioResultKind == DOCokEPSF )
			{
			if  ( psPrintIncludeEpsObject( ps,
						io, pd->pdX0, baseLine ) )
			    { LDEB(1); break;	}

			ps->psCurrentPhysicalFont= -1;
			ps->psLinkParticulesDone++;
			return 1;
			}
		    else{ LLDEB(io->ioKind,io->ioResultKind); break;	}

		default:
		    LDEB(io->ioKind); return 0;
		}

	    psPrintObjectBox( ps, io, pd, baseLine );
	    ps->psLinkParticulesDone++;
	    return 1;

	case DOCkindCHFTNSEP:
	    {
	    int		xHeight;

	    int		x0;
	    int		x1;
	    int		y0;
	    int		h;

	    fontSizeTwips= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

	    xHeight= ( fontSizeTwips+ 1 )/ 2;

	    x0= pd->pdX0;
	    x1= pd->pdX0+ 2800;
	    y0= baseLine- xHeight/2- 15/2;
	    h= 15;

	    fprintf( ps->psFile, "%d %d %d %d rectfill\n", x0, y0, x1- x0, h );
	    }
	    return 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    for ( drawn= 1; drawn < count; drawn++ )
	{
	if  ( tp[drawn].tpPhysicalFont != tp->tpPhysicalFont	||
	      tp[drawn].tpKind != DOCkindTEXT			)
	    { break;	}

	if  ( bi->biParaAlignment == DOCiaJUSTIFIED )
	    { break;	}
	}

    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    if  ( len == 0 )
	{ return drawn;	}

    while( len > 0						&&
	   bi->biParaString[tp->tpStroff+ len- 1] == ' '	)
	{ len--;	}

    fontSizeTwips= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

    if  ( len > 0 )
	{
	int			y;
	int			capHeight;
	int			xHeight;

	if  ( tp->tpPhysicalFont != ps->psCurrentPhysicalFont )
	    {
	    appPsSetFont( ps->psFile, "f", tp->tpTextAttribute );
	    ps->psCurrentPhysicalFont= tp->tpPhysicalFont;
	    }

	if  ( ps->psUseLinkColor )
	    { utilPsSetLinkColor( ps );	}
	else{ utilPsSetTextColor( ps );	}

	putc( '(', ps->psFile );
	appPsPrintString( ps->psFile, bi->biParaString+ tp->tpStroff, len );
	putc( ')', ps->psFile );

	capHeight= ( fontSizeTwips* pd->pdAfi->afiCapHeight+ 500 )/ 1000;
	if  ( capHeight == 0 )
	    { capHeight= ( fontSizeTwips* pd->pdAfi->afiAscender+ 500 )/ 1000; }

	xHeight= ( fontSizeTwips* pd->pdAfi->afiXHeight+ 500 )/ 1000;
	if  ( xHeight == 0 )
	    { xHeight= ( fontSizeTwips+ 1 )/ 2; }

	y= baseLine;

	if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	    { y -= xHeight; }

	if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	    { y += ( 4* capHeight )/ 10; }

	fprintf( ps->psFile, " %d %d mvs\n", pd->pdX0, y );
	}

    i= 0;
    while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= pd[i].pdX0;
	y0= baseLine- ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlinePosition+ 500 )/ 1000;
	h= ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlineThickness+ 500 )/ 1000;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= pd[i].pdX0+ pd[i].pdWidth; i++; }

	if  ( ps->psUseLinkColor )
	    { utilPsSetLinkColor( ps );	}
	else{ utilPsSetTextColor( ps );	}

	fprintf( ps->psFile, "%d %d %d %d rectfill\n", x0, y0, x1- x0, h );
	}

    ps->psLinkParticulesDone += drawn;

    return drawn;
    }

/************************************************************************/
/*									*/
/*  Print one line of output.						*/
/*									*/
/************************************************************************/

static int psPrintTextLine(	PrintingState *			ps,
				DrawingContext *		dc,
				const DocumentFontList *	dfl,
				const char *			afmDirectory,
				const BufferItem *		bi,
				int				part,
				const TextParticule *		tp,
				int				particuleCount,
				ParticuleData *			pd,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    int				done= 0;

    while( done < particuleCount )
	{
	int		drawn;

	drawn= psPrintParticules( ps, dc, bi, dfl, afmDirectory, part, tp, pd,
			particuleCount- done, baseLine, lineTop, lineHeight );

	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn; pd += drawn; part += drawn;
	}

    if  ( done > 0 && ps->psInsideLink )
	{ psFlushLink( ps, pd- 1, lineTop, lineHeight ); }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout and print successive lines of a paragraph.			*/
/*									*/
/************************************************************************/

static int docPsPrintTextLine(	const BufferItem *		bi,
				int				line,
				const ParagraphFrame *		pf,
				void *				vps,
				DrawingContext *		dc )
    {
    BufferDocument *		bd= dc->dcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    AppDrawingData *		add= dc->dcDrawingData;
    PrintingState *		ps= (PrintingState *)vps;
    AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);

    const TextLine *		tl= bi->biParaLines+ line;
    int				part= tl->tlFirstParticule;
    const TextParticule *	tp= bi->biParaParticules+ part;

    int				accepted;
    TextLine			boxLine;
    int				baseline;

    ParticuleData *		pd;

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    boxLine= *tl;
    accepted= docLayoutLineBox( &boxLine, bi, part,
				    dp->dpTabIntervalTwips, &dp->dpFontList,
				    apfl, tp, pd, pf );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( ps->psInsideLink )
	{ ps->psLinkRectLeft= pd->pdX0;	}

    baseline= tl->tlTopPosition.lpPageYTwips+ tl->tlLineAscentTwips;

    psPrintTextLine( ps, dc, &(dp->dpFontList), apfl->apflAfmDirectory,
			bi, part, tp, accepted, pd,
			baseline, tl->tlTopPosition.lpPageYTwips,
			tl->tlLineHeightTwips );

    return accepted;
    }

static int docPsPrintParaTop(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const ParagraphFrame *		pf,
				void *				vps,
				DrawingContext *		dc )
    {
    PrintingState *	ps= (PrintingState *)vps;
    int			x0= pf->pfX0TextLinesTwips;

    if  ( pf->pfX0FirstLineTwips < x0 )
	{ x0= pf->pfX0FirstLineTwips;	}

    utilPsSetTextColor( ps );

    psPrintHorizontalBorder( bi->biParaBorderAboveParagraph,
					vps, x0, pf->pfX1TextLinesTwips,
					bi->biTopPosition.lpPageYTwips+
					bi->biParaSpaceBeforeTwips );

    return 0;
    }

static int docPsPrintParaBottom( const BufferItem *		bi,
				const BorderProperties *	bp,
				const ParagraphFrame *		pf,
				void *				vps,
				DrawingContext *		dc )
    {
    PrintingState *	ps= (PrintingState *)vps;
    int			x0= pf->pfX0TextLinesTwips;

    if  ( pf->pfX0FirstLineTwips < x0 )
	{ x0= pf->pfX0FirstLineTwips;	}

    utilPsSetTextColor( ps );

    psPrintHorizontalBorder( bi->biParaBorderBelowParagraph,
					vps, x0, pf->pfX1TextLinesTwips,
					bi->biBelowPosition.lpPageYTwips-
					bi->biParaSpaceAfterTwips );

    return 0;
    }

static int docPsPrintCellTop(	const BorderProperties *	bp,
				int				asGrid,
				int				x0Twips,
				int				x0Pixels,
				int				x1Twips,
				int				x1Pixels,
				void *				vps,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	utilPsSetTextColor( ps );

	psPrintHorizontalBorder( bp, vps,
				    x0Twips, x1Twips, lpTop->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellBottom( const BorderProperties *	bp,
				int				asGrid,
				int				x0Twips,
				int				x0Pixels,
				int				x1Twips,
				int				x1Pixels,
				void *				vps,
				DrawingContext *		dc,
				const LayoutPosition *		lpBottom )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	utilPsSetTextColor( ps );

	psPrintHorizontalBorder( bp, vps,
				    x0Twips, x1Twips, lpBottom->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellLeft(	const BorderProperties *	bp,
				int				asGrid,
				void *				vps,
				DrawingContext *		dc,
				int				x0Twips,
				int				x0Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	utilPsSetTextColor( ps );

	psPrintVerticalBorder( bp, vps, x0Twips,
			    lpTop->lpPageYTwips, lpBelow->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellRight(	const BorderProperties *	bp,
				int				asGrid,
				void *				vps,
				DrawingContext *		dc,
				int				x1Twips,
				int				x1Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	utilPsSetTextColor( ps );

	psPrintVerticalBorder( bp, vps, x1Twips,
			    lpTop->lpPageYTwips, lpBelow->lpPageYTwips );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static int docPsFinishPage(	void *				vps,
				DrawingContext *		dc,
				int				page,
				int				asLast )
    {
    PrintingState *	ps= (PrintingState *)vps;

    utilPsFinishPage( ps, page, asLast );

    return 0;
    }

static int docPsStartPage(	void *				vps,
				const DocumentGeometry *	dg,
				DrawingContext *		dc,
				int				page )
    {
    PrintingState *	ps= (PrintingState *)vps;

    ps->psCurrentPhysicalFont= -1;

    appPsStartPage( ps, page );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Leave a trace in the PDF document information.			*/
/*									*/
/************************************************************************/

static void psSaveInfo(		const char *		tag,
				const unsigned char *	info,
				FILE *			f )
    {
    if  ( ! info )
	{ return;	}

    fprintf( f, "  %s (", tag );
    appPsPrintString( f, info, strlen( (const char *)info ) );
    fprintf( f, ")\n" );
    }

static void psSaveDate(		const char *		tag,
				const struct tm *	tm,
				FILE *			f )
    {
    char	scratch[40+1];

    if  ( tm->tm_mday == 0 )
	{ return;	}

    if  ( strftime( scratch, sizeof(scratch)- 1, "D:%Y%m%d%H%M", tm ) < 1 )
	{ LDEB(1); return;	}

    psSaveInfo( tag, (const unsigned char *)scratch, f );

    return;
    }

/************************************************************************/
/*									*/
/*  Save procedures to use implement tab leades in PostScript.		*/
/*									*/
/************************************************************************/

static const char *	DOC_PS_dot_tab[]=
    {
    "/dot-tab",
    "  {",
    "  gsave",
    "  10 setlinewidth [ 1 59 ] 0 setdash 1 setlinecap",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static const char *	DOC_PS_dot_tab_bold[]=
    {
    "/dot-tab-bold",
    "  {",
    "  gsave",
    "  16 setlinewidth [ 1 59 ] 0 setdash 1 setlinecap",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static const char *	DOC_PS_dash_tab[]=
    {
    "/dash-tab",
    "  {",
    "  gsave",
    "  10 setlinewidth [ 40 100 ] 0 setdash 1 setlinecap",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static const char *	DOC_PS_dash_tab_bold[]=
    {
    "/dash-tab-bold",
    "  {",
    "  gsave",
    "  16 setlinewidth [ 40 100 ] 0 setdash 1 setlinecap",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static const char *	DOC_PS_ul_tab[]=
    {
    "/ul-tab",
    "  {",
    "  gsave",
    "  10 setlinewidth",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static const char *	DOC_PS_ul_tab_bold[]=
    {
    "/ul-tab-bold",
    "  {",
    "  gsave",
    "  16 setlinewidth",
    "  newpath moveto 0 rlineto stroke",
    "  grestore",
    "  } bind def",
    };

static void docPsSaveTabLeaderProcedures(	FILE *		f )
    {
    utilPsDefineProcedure( f, DOC_PS_dot_tab,
				sizeof(DOC_PS_dot_tab)/sizeof(char *) );

    utilPsDefineProcedure( f, DOC_PS_dot_tab_bold,
				sizeof(DOC_PS_dot_tab_bold)/sizeof(char *) );

    utilPsDefineProcedure( f, DOC_PS_dash_tab,
				sizeof(DOC_PS_dash_tab)/sizeof(char *) );

    utilPsDefineProcedure( f, DOC_PS_dash_tab_bold,
				sizeof(DOC_PS_dash_tab_bold)/sizeof(char *) );

    utilPsDefineProcedure( f, DOC_PS_ul_tab,
				sizeof(DOC_PS_ul_tab)/sizeof(char *) );

    utilPsDefineProcedure( f, DOC_PS_ul_tab_bold,
				sizeof(DOC_PS_ul_tab_bold)/sizeof(char *) );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/************************************************************************/

int docPsPrintDocument(	FILE *				f,
			const char *			title,
			const char *			applicationName,
			const char *			applicationReference,
			AppDrawingData *		add,
			BufferDocument *		bd,
			const PrintGeometry *		pg,
			int				firstPage,
			int				lastPage,
			LAYOUT_EXTERNAL			layoutExternal,
			DOC_CLOSE_OBJECT		closeObject )
    {
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentGeometry *		dg= &(dp->dpGeometry);
    BufferItem *		docBi= &(bd->bdItem);

    PostScriptFaceList		psfl;
    int				i;

    DrawingContext		dc;
    PrintingState		ps;

    int				hasPageHeader= 0;
    int				hasPageFooter= 0;

    utilInitPostScriptFaceList( &psfl );

    docInitDrawingContext( &dc );

    dc.dcDrawTextLine= docPsPrintTextLine;
    dc.dcDrawParaTop= docPsPrintParaTop;
    dc.dcDrawParaBottom= docPsPrintParaBottom;
    dc.dcDrawCellTop= docPsPrintCellTop;
    dc.dcDrawCellBottom= docPsPrintCellBottom;
    dc.dcDrawCellLeft= docPsPrintCellLeft;
    dc.dcDrawCellRight= docPsPrintCellRight;
    dc.dcFinishPage= docPsFinishPage;
    dc.dcStartPage= docPsStartPage;
    dc.dcLayoutExternal= layoutExternal;
    dc.dcCloseObject= closeObject;

    dc.dcDrawingData= add;
    dc.dcDocument= bd;

    dc.dcFirstPage= firstPage;
    dc.dcLastPage= lastPage;
    dc.dcDrawHeadersFooters= 1;

    appPsInitPrintingState( &ps );
    ps.psPrinterGeometry= pg->pgSheetGeometry;
    ps.psFile= f;

    for ( i= 0; i < docBi->biChildCount; i++ )
	{
	int			j;
	BufferItem *		sectBi= docBi->biChildren[i];

	for ( j= 0; j < PAGES__COUNT; j++ )
	    {
	    ExternalItem *	ei;

	    ei= docSectionHeaderFooter( sectBi, DOC_HeaderScopes[j] );
	    if  ( ei && ei->eiItem )
		{ hasPageHeader= 1;	}

	    ei= docSectionHeaderFooter( sectBi, DOC_FooterScopes[j] );
	    if  ( ei && ei->eiItem )
		{ hasPageFooter= 1;	}
	    }
	}

    if  ( dp->dpTitle && dp->dpTitle[0] )
	{ title= (char *)dp->dpTitle;	}

    if  ( appPsSetNup( &ps, dg, pg, hasPageHeader, hasPageFooter ) )
	{ LDEB(pg->pgNup); return -1;	}

    if  ( docPsPrintGetItemFonts( &psfl, docBi,
			(&dp->dpFontList), &(add->addPhysicalFontList) ) )
	{ LDEB(1); return -1;	}

    utilPsStartDSCDocument( &ps, &psfl,
			    title, applicationName, applicationReference );

    fprintf( f, "%%%%BeginProlog\n" );

    fprintf( f, "\n/mvs { moveto show } bind def\n" );

    docPsSaveTabLeaderProcedures( f );

    appPsSetPdfmarkEmulation( f );
    appPsSetRectfillEmulation( f );
    appPsSetSelectfontEmulation( f );

    appPsDefineEpsProcs( f );

    appPsFontNames( f, &psfl, ps.psEncodingDefined, /*allFonts=*/ 0 );

    appMetaDefineProcsetPs( f );

    fprintf( f, "%%%%EndProlog\n" );
    fprintf( f, "%%%%BeginSetup\n" );

    if  ( docHasDocumentInfo( dp ) )
	{
	unsigned char *			scratch;

	scratch= malloc( strlen( applicationName )+ 2+
					strlen( applicationReference )+ 2 );
	if  ( ! scratch )
	    { XDEB(scratch); return -1;	}
	sprintf( (char *)scratch,
			"%s: %s", applicationName, applicationReference );

	fprintf( f, "[\n" );

	psSaveInfo( "/Title",		dp->dpTitle, f );
	psSaveInfo( "/Author",		dp->dpAuthor, f );
	psSaveInfo( "/Subject",		dp->dpSubject, f );
	psSaveInfo( "/Keywords",	dp->dpKeywords, f );
	psSaveInfo( "/Creator",		scratch, f );
	psSaveInfo( "/Producer",	scratch, f );

	psSaveDate( "/ModDate",		&(dp->dpRevtim), f );
	psSaveDate( "/CreationDate",	&(dp->dpCreatim), f );

	fprintf( f, "/DOCINFO pdfmark\n\n" );

	free( scratch );
	}

    fprintf( f, "%%%%EndSetup\n" );

    if  ( firstPage < 0 )
	{ firstPage= 0;	}

    for ( i= 0; i < docBi->biChildCount; i++ )
	{
	if  ( docBi->biChildren[i]->biBelowPosition.lpPage >= firstPage )
	    { break;	}
	}

    if  ( i >= docBi->biChildCount )
	{ LDEB(i); return -1; }

    appPsStartPage( &ps, firstPage );
    docDrawPageHeader( docBi->biChildren[i], (void *)&ps, &dc, firstPage );

    docDrawItem( docBi, (void *)&ps, &dc );

    if  ( lastPage < 0 )
	{ lastPage= docBi->biBelowPosition.lpPage;	}

    for ( i= docBi->biChildCount- 1; i >= 0; i-- )
	{
	if  ( docBi->biChildren[i]->biTopPosition.lpPage <= lastPage )
	    { break;	}
	}

    if  ( i < 0 )
	{ LDEB(i); return -1;	}

    docDrawFootnotesForColumn( lastPage, (void *)&ps, &dc );

    docDrawPageFooter( docBi->biChildren[i], (void *)&ps, &dc, lastPage );

    utilPsFinishPage( &ps, lastPage, /*asLast*/ 1 );

    appPsCleanPrintingState( &ps );

    utilCleanPostScriptFaceList( &psfl );

    return 0;
    }
