[comment {-*- tcl -*- doctools manpage}]
[manpage_begin grammar::fa::op n 0.1.1]
[copyright {2004 Andreas Kupries <andreas_kupries@users.sourceforge.net>}]
[moddesc   {Finite automaton operations and usage}]
[titledesc {Operations on finite automatons}]
[require Tcl 8.4]
[require snit]
[require struct::list]
[require struct::set]
[require grammar::fa::op [opt 0.1.1]]
[description]
[keywords grammar automaton {finite automaton}]
[keywords state {regular expression} {regular grammar}]
[keywords {regular languages} parsing transducer]
[para]

This package provides a number of complex operations on finite
automatons (Short: FA),

as provided by the package [package grammar::fa].

The package does not provide the ability to create and/or manipulate
such FAs, nor the ability to execute a FA for a stream of symbols.

Use the packages [package grammar::fa]
and [package grammar::fa::interpreter] for that.

Another package related to this is [package grammar::fa::compiler]
which turns a FA into an executor class which has the definition of
the FA hardwired into it.

[para]

For more information about what a finite automaton is see section
[emph {FINITE AUTOMATONS}] in package [package grammar::fa].

[section API]

The package exports the API described here.  All commands modify their
first argument. I.e. whatever FA they compute is stored back into
it. Some of the operations will construct an automaton whose states
are all new, but related to the states in the source
automaton(s). These operations take variable names as optional
arguments where they will store mappings which describe the
relationship(s).

The operations can be loosely partitioned into structural and language
operations. The latter are defined in terms of the language the
automaton(s) accept, whereas the former are defined in terms of the
structural properties of the involved automaton(s). Some operations
are both.

[emph {Structure operations}]

[list_begin definitions]

[call [cmd ::grammar::fa::op::reverse] [arg fa]]

Reverses the [arg fa]. This is done by reversing the direction of all
transitions and swapping the sets of [term start] and [term final]
states. The language of [arg fa] changes unpredictably.


[call [cmd ::grammar::fa::op::complete] [arg fa] [opt [arg sink]]]

Completes the [arg fa] [term complete], but nothing is done if the
[arg fa] is already [term complete]. This implies that only the first
in a series of multiple consecutive complete operations on [arg fa]
will perform anything. The remainder will be null operations.

[nl]

The language of [arg fa] is unchanged by this operation.

[nl]

This is done by adding a single new state, the [term sink], and
transitions from all other states to that sink for all symbols they
have no transitions for. The sink itself is made complete by adding
loop transitions for all symbols.

[nl]

Note: When a FA has epsilon-transitions transitions over a symbol for
a state S can be indirect, i.e. not attached directly to S, but to a
state in the epsilon-closure of S. The symbols for such indirect
transitions count when computing completeness of a state. In other
words, these indirectly reached symbols are [emph not] missing.

[nl]

The argument [arg sink] provides the name for the new state and most
not be present in the [arg fa] if specified. If the name is not
specified the command will name the state "sink[var n]", where [var n]
is set so that there are no collisions with existing states.

[nl]

Note that the sink state is [term {not useful}] by definition.  In
other words, while the FA becomes complete, it is also
[term {not useful}] in the strict sense as it has a state from which
no final state can be reached.



[call [cmd ::grammar::fa::op::remove_eps] [arg fa]]

Removes all epsilon-transitions from the [arg fa] in such a manner the
the language of [arg fa] is unchanged. However nothing is done if the
[arg fa] is already [term epsilon-free].

This implies that only the first in a series of multiple consecutive
complete operations on [arg fa] will perform anything. The remainder
will be null operations.

[nl]

[emph Note:] This operation may cause states to become unreachable or
not useful. These states are not removed by this operation.

Use [cmd ::grammar::fa::op::trim] for that instead.


[call [cmd ::grammar::fa::op::trim] [arg fa] [opt [arg what]]]

Removes unwanted baggage from [arg fa].

The legal values for [arg what] are listed below. The command defaults
to [const !reachable|!useful] if no specific argument was given.

[list_begin definitions]
[lst_item [const !reachable]]
Removes all states which are not reachable from a start state.

[lst_item [const !useful]]
Removes all states which are unable to reach a final state.

[lst_item [const !reachable&!useful]]
[lst_item [const !(reachable|useful)]]
Removes all states which are not reachable from a start state and are
unable to reach a final state.

[lst_item [const !reachable|!useful]]
[lst_item [const !(reachable&useful)]]
Removes all states which are not reachable from a start state or are
unable to reach a final state.

[list_end]
[nl]


[call [cmd ::grammar::fa::op::determinize] [arg fa] [opt [arg mapvar]]]

Makes the [arg fa] deterministic without changing the language
accepted by the [arg fa]. However nothing is done if the [arg fa] is
already [term deterministic]. This implies that only the first in a
series of multiple consecutive complete operations on [arg fa] will
perform anything. The remainder will be null operations.

[nl]

The command will store a dictionary describing the relationship
between the new states of the resulting dfa and the states of the
input nfa in [arg mapvar], if it has been specified. Keys of the
dictionary are the handles for the states of the resulting dfa, values
are sets of states from the input nfa.

[nl]

[emph Note]: An empty dictionary signals that the command was able to
make the [arg fa] deterministic without performing a full subset
construction, just by removing states and shuffling transitions around
(As part of making the FA epsilon-free).

[nl]

[emph Note]: The algorithm fails to make the FA deterministic in the
technical sense if the FA has no start state(s), because determinism
requires the FA to have exactly one start states.

In that situation we make a best effort; and the missing start state
will be the only condition preventing the generated result from being
[term deterministic].

It should also be noted that in this case the possibilities for
trimming states from the FA are also severely reduced as we cannot
declare states unreachable.



[call [cmd ::grammar::fa::op::minimize] [arg fa] [opt [arg mapvar]]]

Creates a FA which accepts the same language as [arg fa], but has a
minimal number of states. Uses Brzozowski's method to accomplish this.

[nl]

The command will store a dictionary describing the relationship
between the new states of the resulting minimal fa and the states of
the input fa in [arg mapvar], if it has been specified. Keys of the
dictionary are the handles for the states of the resulting minimal fa,
values are sets of states from the input fa.

[nl]

[emph Note]: An empty dictionary signals that the command was able to
minimize the [arg fa] without having to compute new states. This
should happen if and only if the input FA was already minimal.

[nl]

[emph Note]: If the algorithm has no start or final states to work
with then the result might be technically minimal, but have a very
unexpected structure.

It should also be noted that in this case the possibilities for
trimming states from the FA are also severely reduced as we cannot
declare states unreachable.

[list_end]


[emph {Language operations}]

All operations in this section require that all input FAs have at
least one start and at least one final state. Otherwise the language of
the FAs will not be defined, making the operation senseless (as it
operates on the languages of the FAs in a defined manner).


[list_begin definitions]

[call [cmd ::grammar::fa::op::complement] [arg fa]]

Complements [arg fa]. This is possible if and only if [arg fa] is
[term complete] and [term deterministic]. The resulting FA accepts the
complementary language of [arg fa]. In other words, all inputs not
accepted by the input are accepted by the result, and vice versa.

[nl]

The result will have all states and transitions of the input, and
different final states.


[call [cmd ::grammar::fa::op::kleene] [arg fa]]

Applies Kleene's closure to [arg fa].

The resulting FA accepts all strings [var S] for which we can find a
natural number [var n] (0 inclusive) and strings [var A1] ... [var An]
in the language of [arg fa] such that [var S] is the concatenation of
[var A1] ... [var An].

In other words, the language of the result is the infinite union over
finite length concatenations over the language of [arg fa].

[nl]

The result will have all states and transitions of the input, and new
start and final states.


[call [cmd ::grammar::fa::op::optional] [arg fa]]

Makes the [arg fa] optional. In other words it computes the FA which
accepts the language of [arg fa] and the empty the word (epsilon) as
well.

[nl]

The result will have all states and transitions of the input, and new
start and final states.


[call [cmd ::grammar::fa::op::union] [arg fa] [arg fb] [opt [arg mapvar]]]

Combines the FAs [arg fa] and [arg fb] such that the resulting FA
accepts the union of the languages of the two FAs.

[nl]

The result will have all states and transitions of the two input FAs,
and new start and final states. All states of [arg fb] which exist in
[arg fa] as well will be renamed, and the [arg mapvar] will contain a
mapping from the old states of [arg fb] to the new ones, if present.

[nl]

It should be noted that the result will be non-deterministic, even if
the inputs are deterministic.


[call [cmd ::grammar::fa::op::intersect] [arg fa] [arg fb] [opt [arg mapvar]]]

Combines the FAs [arg fa] and [arg fb] such that the resulting FA
accepts the intersection of the languages of the two FAs. In other
words, the result will accept a word if and only if the word is
accepted by both [arg fa] and [arg fb]. The result will be useful, but
not necessarily deterministic or minimal.

[nl]

The command will store a dictionary describing the relationship
between the new states of the resulting fa and the pairs of states of
the input FAs in [arg mapvar], if it has been specified. Keys of the
dictionary are the handles for the states of the resulting fa, values
are pairs of states from the input FAs. Pairs are represented by
lists. The first element in each pair will be a state in [arg fa], the
second element will be drawn from [arg fb].


[call [cmd ::grammar::fa::op::difference] [arg fa] [arg fb] [opt [arg mapvar]]]

Combines the FAs [arg fa] and [arg fb] such that the resulting FA
accepts the difference of the languages of the two FAs. In other
words, the result will accept a word if and only if the word is
accepted by [arg fa], but not by [arg fb]. This can also be expressed
as the intersection of [arg fa] with the complement of [arg fb]. The
result will be useful, but not necessarily deterministic or minimal.

[nl]

The command will store a dictionary describing the relationship
between the new states of the resulting fa and the pairs of states of
the input FAs in [arg mapvar], if it has been specified. Keys of the
dictionary are the handles for the states of the resulting fa, values
are pairs of states from the input FAs. Pairs are represented by
lists. The first element in each pair will be a state in [arg fa], the
second element will be drawn from [arg fb].




[call [cmd ::grammar::fa::op::concatenate] [arg fa] [arg fb] [opt [arg mapvar]]]

Combines the FAs [arg fa] and [arg fb] such that the resulting FA
accepts the cross-product of the languages of the two FAs. I.e. a word
W will be accepted by the result if there are two words A and B
accepted by [arg fa], and [arg fb] resp. and W is the concatenation of
A and B.

[nl]

The result FA will be non-deterministic.



[call [cmd ::grammar::fa::op::fromRegex] [arg fa] [arg regex] [opt [arg over]]]

Generates a non-deterministic FA which accepts the same language as
the regular expression [arg regex]. If the [arg over] is specified it
is treated as the set of symbols the regular expression and the
automaton are defined over. The command will compute the set from the
"S" constructors in [arg regex] when [arg over] was not
specified. This set is important if and only if the complement
operator "!" is used in [arg regex] as the complementary language of
an FA is quite different for different sets of symbols.

[nl]

The regular expression is represented by a nested list, which forms
a syntax tree. The following structures are legal:

[list_begin definitions]

[lst_item "{S x}"]

Atomic regular expression. Everything else is constructed from
these. Accepts the [const S]ymbol "x".

[lst_item "{. A1 A2 ...}"]

Concatenation operator. Accepts the concatenation of the regular
expressions [var A1], [var A2], etc.

[lst_item "{| A1 A2 ...}"]

Choice operator, also called "Alternative". Accepts all input accepted
by at least one of the regular expressions [var A1], [var A2], etc. In
other words, the union of [var A1], [var A2].

[lst_item "{& A1 A2 ...}"]

Intersection operator, logical and. Accepts all input accepted which
is accepted by all of the regular expressions [var A1], [var A2],
etc. In other words, the intersection of [var A1], [var A2].

[lst_item "{? A}"]

Optionality operator. Accepts the empty word and anything from the
regular expression [var A].

[lst_item "{* A}"]

Kleene closure. Accepts the empty word and any finite concatenation of
words accepted by the regular expression [var A].

[lst_item "{+ A}"]

Positive Kleene closure. Accepts any finite concatenation of words
accepted by the regular expression [var A], but not the empty word.

[lst_item "{! A}"]

Complement operator. Accepts any word not accepted by the regular
expression [var A]. Note that the complement depends on the set of
symbol the result should run over. See the discussion of the argument
[arg over] before.


[list_end]
[list_end]

[para]

[section EXAMPLES]
[para]

[manpage_end]
