/*
 * Serialize a tagged collection to a text file
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tests/test-utils.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/input/string.h>
#include <tagcoll/stream/sink.h>
#include <tagcoll/patch.h>

namespace tut {
using namespace std;
using namespace tagcoll;
using namespace tagcoll::tests;

struct tagcoll_textformat_shar {
};
TESTGRP(tagcoll_textformat);

template<> template<>
void to::test<1>()
{
	input::String coll(
			"a: b, c\n"
			"b:\n"
			"c: \n"
			"d:  c::D, e::F,    f::g\n"
	);
	
	int countItems = 0, countTags = 0;
	textformat::parse(coll, stream::countingSink(countItems, countTags));

	ensure_equals(countItems, 4);
	ensure_equals(countTags, 5);
}

template<> template<>
void to::test<2>()
{
	input::String coll(
			"a: +b, +c\n"
			"b:\n"
			"c: foo\n"
			"d:  +c::D, -e::F,    -f::g\n"
	);
	
	PatchList<string, string> plist = textformat::parsePatch(coll);

	/*
	cerr << "Patchlist[" << plist.size() << "]:" << endl;
	for (PatchList<string, string>::const_iterator i = plist.begin();
			i != plist.end(); i++)
		cerr << " " << i->first << ": " << 
			i->second.added.size()  << "x" <<
			i->second.removed.size()  << endl;
	*/

	ensure_equals(plist.size(), 2u);
	ensure(plist.find("a") != plist.end());
	ensure(plist.find("b") == plist.end());
	ensure(plist.find("c") == plist.end());
	ensure(plist.find("d") != plist.end());

	PatchList<string, string>::const_iterator i = plist.find("a");
	ensure_equals(i->second.added.size(), 2u);
	ensure_equals(i->second.removed.size(), 0u);

	i = plist.find("d");
	ensure_equals(i->second.added.size(), 1u);
	ensure_equals(i->second.removed.size(), 2u);
}

template<> template<>
void to::test<3>()
{
	string reference =
		"foo\n"
		"foo: antani\n"
		"foo: antani, blinda\n"
		"bar, foo: antani, blinda\n";
	char buf[1000];
	FILE* out = fmemopen(buf, 1000, "w");
	ensure(out != 0);

	textformat::StdioWriter writer(out);

	set<string> s1;
	s1.insert("foo");
	*writer = make_pair(s1, wibble::Empty<string>());
	++writer;

	set<string> s2;
	s2.insert("antani");
	*writer = make_pair(s1, s2);
	++writer;

	s2.insert("blinda");
	*writer = make_pair(s1, s2);
	++writer;

	s1.clear();
	s1.insert("foo"); s1.insert("bar");
	*writer = make_pair(s1, s2);
	++writer;

	fclose(out);

	string outstr(buf);
	ensure_equals(reference, outstr);
}

template<> template<>
void to::test<4>()
{
	string reference =
		"foo\n"
		"foo: antani\n"
		"foo: antani, blinda\n"
		"bar, foo: antani, blinda\n";
	stringstream out;

	textformat::OstreamWriter writer(out);

	set<string> s1;
	s1.insert("foo");
	*writer = make_pair(s1, wibble::Empty<string>());
	++writer;

	set<string> s2;
	s2.insert("antani");
	*writer = make_pair(s1, s2);
	++writer;

	s2.insert("blinda");
	*writer = make_pair(s1, s2);
	++writer;

	s1.clear();
	s1.insert("foo"); s1.insert("bar");
	*writer = make_pair(s1, s2);
	++writer;

	ensure_equals(out.str(), reference);
}

template<> template<>
void to::test<5>()
{
	char buf[1000];

	PatchList<string, string> patchlist;
	std::set<string> added; added.insert("add");
	std::set<string> removed; removed.insert("rm");
	patchlist.addPatch(Patch<string, string>("test", added, removed));
	
	FILE* out = fmemopen(buf, 999, "w");
	ensure(out != 0);

	textformat::outputPatch(patchlist, out);
	fclose(out);

	ensure(strcmp(buf, "test: +add, -rm\n") == 0);

	struct {
		std::string operator()(const std::string& str) { return str + "o"; }
	} oer;

	out = fmemopen(buf, 999, "w");
	ensure(out != 0);

#if 0
#warning Enable this test please
	textformat::outputPatch(oer, oer, patchlist, out);
	fclose(out);

	ensure(strcmp(buf, "test: +addo, -rmo\n") == 0);
#endif
}

}

#include <tagcoll/TextFormat.tcc>
#include <tagcoll/stream/filters.tcc>

// vim:set ts=4 sw=4:
