/*
 * Fast index for tag data, based on integer indexes
 *
 * Copyright (C) 2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/MMapIndex.h>
#include <tagcoll/stringf.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

using namespace std;
using namespace Tagcoll;
using namespace stringf;

MasterMMapIndex::MasterMMapIndex() : m_fd(-1), m_buf(0) {}

MasterMMapIndex::MasterMMapIndex(const std::string& filename)
	: m_filename(filename), m_fd(-1), m_buf(0)
{
	// If init throws here, the destructor isn't called (since we're in the
	// constructor), so we do the cleanup and rethrow.
	try {
		init(filename);
	} catch (...) {
		if (m_buf)
			munmap((void*)m_buf, m_size);
		if (m_fd != -1)
			close(m_fd);
		throw;
	}
}

MasterMMapIndex::~MasterMMapIndex()
{
	// Unmap and close the file
	if (m_buf)
		munmap((void*)m_buf, m_size);
	if (m_fd != -1)
		close(m_fd);
}

void MasterMMapIndex::init(const std::string& filename)
{
	m_filename = filename;

	// Open the file
	if ((m_fd = open(m_filename.c_str(), O_RDONLY)) == -1)
		throw SystemException(errno, "opening index file " + m_filename);

	off_t size = lseek(m_fd, 0, SEEK_END);
	if (size == (off_t)-1)
		throw SystemException(errno, "reading the size of index file " + m_filename);
	m_size = size;

	if (m_size == 0)
		throw ConsistencyCheckException("the mmap index file " + filename + " is empty");
	
	// Map the file into memory
	if ((m_buf = (const char*)mmap(0, m_size, PROT_READ, MAP_PRIVATE, m_fd, 0)) == MAP_FAILED)
		throw SystemException(errno, string("mmapping file ") + m_filename);
}


MMapIndex::MMapIndex()
	: m_master(0), m_buf(0), m_size(0) {}

MMapIndex::MMapIndex(const char* buf, int size)
{
	init(buf, size);
}

MMapIndex::MMapIndex(const MasterMMapIndex& master, size_t idx)
{
	init(master, idx);
}

void MMapIndex::init(const char* buf, int size)
{
	m_master = 0;
	m_buf = buf;
	m_size = size;
}

void MMapIndex::init(const MasterMMapIndex& master, size_t idx)
{
	m_master = &master;
	m_buf = master.m_buf;
	m_size = *(int*)m_buf;

	// Jump to the idx-th index
	for (size_t i = 0; i < idx; i++)
	{
		m_buf = m_buf + m_size + sizeof(int);
		if (m_buf > master.m_buf + master.m_size)
			throw NotFoundException("looking for subindex " + fmt(idx) + " in " + master.m_filename);
		m_size = *(int*)m_buf;
	}
	
	// Work with the subindex data, skipping the subindex chain size word
	m_buf += sizeof(int);
}


MasterMMapIndexer::MasterMMapIndexer(const std::string& filename)
	: finalname(filename)
{
	// Create a temporary file next to the target file
	char name[filename.size() + 8];
	memcpy(name, filename.data(), filename.size());
	memcpy(name + filename.size(), ".XXXXXX", 8);

	fd = mkstemp(name);
	tmpname = name;
	if (fd == -1)
		throw SystemException(errno, "creating temporary file " + tmpname);

}

MasterMMapIndexer::~MasterMMapIndexer()
{
	if (fd != -1)
		close(fd);

	if (!tmpname.empty())
		unlink(tmpname.c_str());
}

void MasterMMapIndexer::commit()
{
	// Give the file the right permissions according to umask

	// Read the current umask
	mode_t mask = umask(0);
	umask(mask);
	// Set the file permissions
	if (fchmod(fd, 0666 & ~mask) == -1)
		throw SystemException(errno, string("setting permissions on file ") + tmpname);

	// Close the file
	close(fd);
	fd = -1;

	// Rename to the final file name, performing the atomic update
	if (rename(tmpname.c_str(), finalname.c_str()) == -1)
		throw SystemException(errno, string("renaming file ") + tmpname + " into " + finalname);
	tmpname.clear();
}

void MasterMMapIndexer::append(const MMapIndexer& idx)
{
	// Get the encoded data
	int size = idx.encodedSize();
	char buf[size];
	idx.encode(buf);

	// Write one byte at the end to actually resize the file
	if (::write(fd, &size, sizeof(int)) != sizeof(int))
		throw SystemException(errno, "writing size word to file " + tmpname);
	
	// Write the index data
	if (::write(fd, buf, size) != size)
		throw SystemException(errno, "writing subindex to file " + tmpname);
}


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_tagcoll;

// Very simple test index

class TestIndex : public MMapIndex
{
public:
	TestIndex(MasterMMapIndex& master, int idx) : MMapIndex(master, idx) {}

	const char* get() const { return m_buf; }
	unsigned int size() const { return m_size; }
};

class TestIndexer : public MMapIndexer
{
protected:
	std::string data;

public:
	TestIndexer(const std::string& data) : data(data) {}
	virtual ~TestIndexer() {}

	/// Return the size of the encoded index data (in bytes)
	virtual int encodedSize() const { return MMapIndex::align(data.size() + 1); }

	/// Write the index data in the given buffer, which should be at least
	/// encodedSize bytes
	virtual void encode(char* buf) const { memcpy(buf, data.c_str(), data.size() + 1); }
};

static const char* fname = "tagcoll_mmapindex.tmp";

struct tagcoll_mmapindex_shar {
	tagcoll_mmapindex_shar() {
		TestIndexer index1("pippo");
		TestIndexer index2("pluto");
		TestIndexer index3("paperino");
		MasterMMapIndexer master(fname);
		master.append(index1);
		master.append(index2);
		master.append(index3);
		master.commit();
	}
	~tagcoll_mmapindex_shar() {
		// Delete the test index
		unlink(fname);
	}
};
TESTGRP(tagcoll_mmapindex);

// Check to see if the data read with the index is the same as encoded
template<> template<>
void to::test<1>()
{
	MasterMMapIndex master(fname);

	TestIndex index1(master, 0);
	gen_ensure_equals(index1.size(), MMapIndex::align(6u));
	gen_ensure_equals(string(index1.get()), string("pippo"));

	TestIndex index2(master, 1);
	gen_ensure_equals(index2.size(), MMapIndex::align(6u));
	gen_ensure_equals(string(index2.get()), string("pluto"));

	TestIndex index3(master, 2);
	gen_ensure_equals(index3.size(), MMapIndex::align(9u));
	gen_ensure_equals(string(index3.get()), string("paperino"));
}

}

#endif
// vim:set ts=4 sw=4:
