;;; tc.el --- a Japanese input method with T-Code on Emacs

;; Copyright (C) 1989--2002 Kaoru Maeda, Yasushi Saito and KITAJIMA Akira.

;; Author: Kaoru Maeda <maeda@src.ricoh.co.jp>
;;	Yasushi Saito <yasushi@cs.washington.edu>
;;      KITAJIMA Akira <kitajima@isc.osakac.ac.jp>
;; Maintainer: KITAJIMA Akira
;; Keyword: input method

;; $Id: tc.el,v 1.41 2002/03/27 02:02:24 kitajima Exp $

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.

;;; Code:

(require 'tc-setup)
(require 'tc-sysdep)

;;
;; Version
;;
(defconst tcode-version "2.3pre4")

(defun tcode-version ()
  "TϴĶΥСɽ롣"
  (interactive)
  (if (interactive-p)
      (message (concat "T-Code driver version "
		       (tcode-version)
		       (if (tcode-xemacs-p)
			   " on XEmacs "
			 " on Emacs ")
		       emacs-version
		       (if (boundp 'nemacs-version)
			   (concat "/NEmacs " nemacs-version))
		       (if (boundp 'mule-version)
			   (concat "/Mule " mule-version))))
    tcode-version))

;;
;; Variables for customization
;;

(defcustom tcode-bushu-ready-hook nil
  "Ѵνľ˼¹Ԥեå"
  :type 'hook :group 'tcode)

(defvar tcode-bushu-on-demand 2
  "Ѵ򤤤Ľ뤫
	0 : tc.el ɻ
	1 : Tɥ⡼ɤäȤ
	2 : Ѵ򳫻ϤȤ
            ޤʸΥإפ򸫤褦ȤȤ")

(defcustom tcode-use-postfix-bushu-as-default nil
  "* nil ǤʤȤjfǸַѴ77ַѴԤ
nilλˤϤεա" :type 'boolean :group 'tcode)

(defcustom tcode-use-prefix-mazegaki nil
  "* ַ(跿)θ򤼽Ѵλt, ַ()θ򤼽Ѵλnil

ַǤϡfjϤȡɤϥ⡼ɤꡤ ' 'Ϥ
ѴԤַǤϡfjϤȡݥˤʸȤä
ѴԤ"
  :type 'boolean :group 'tcode)

(defvar tcode-alnum-2-byte nil
  "ѿΥХĹڤ괹ե饰t Ǥ2Хȷϡnil 1Хȷϡ")
(make-variable-buffer-local 'tcode-alnum-2-byte)

(defvar tcode-kuten "" "* ")
(make-variable-buffer-local 'tcode-kuten)
(defvar tcode-touten "" "* ")
(make-variable-buffer-local 'tcode-touten)

(defvar tcode-switch-table-list
  '(((tcode-touten . "")
     (tcode-kuten . ""))

    ((tcode-touten . ", ")
     (tcode-kuten . ". ")))
  "ơ֥ѿͤڤؤ뤿ɽ")

(defvar tcode-stroke-file-name nil "* ȥɽΥѥ̾")
(defvar tcode-table-file-name "tc-tbl" "* TɤΥɽΥѥ̾")
(defcustom tcode-record-file-name "~/.tc-record"
  "* non-nil ΤȤTɤפ򤳤Υե˵Ͽ롣"
  :type 'string :group 'tcode)

(defvar tcode-mode-map nil
  "tcode-mode ΤȤ TɥʳΥΤΥޥåס
`tcode-keymap-table' 򻲾ȡ")

(defcustom tcode-ready-hook nil
  "EmacsΩ夲Ƥǽtcode-modeäȤ¹Ԥեå"
  :type 'hook :group 'tcode)

(defcustom tcode-mode-hook nil
  "ΥХåեǽtcode-modeäȤ¹Ԥեå"
  :type 'hook :group 'tcode)

(defcustom tcode-toggle-hook nil
  "tcode-modeȥ뤹٤˼¹Ԥեå"
  :type 'hook :group 'tcode)

(defcustom tcode-after-load-table-hook nil
  "`tcode-load-table' ˤơ֥ɤ߹٤˼¹Ԥեå"
  :type 'hook :group 'tcode)

(defcustom tcode-before-read-stroke-hook nil
  "2ȥܰʹߤΥȥɤ˼¹Ԥեå"
  :type 'hook :group 'tcode)

(defvar tcode-auto-zap-table nil
  "* non-nil ΤȤȥɽ򼡤ǸǼưŪ˾ä")

(defcustom tcode-auto-help t
  "* non-nil ΤȤϤʸΥإɽưŪɽ롣
ɽΤϡ򤼽ѴѴˤäľϤǤ
Τߡ
ޤͤܥ delete-the-char ΤȤϡإפɽ
˺Ǹ˥إפоݤȤʤäʸõ롣

`input-method-verbose-flag'  nil Ǥʤ t Ȥʤ
Τա"
  :group 'tcode)

(defvar tcode-auto-remove-help-count nil
  "* إɽ̵̤ޤǤ˸ƤФ `tcode-auto-remove-help' β
ؿ `tcode-auto-remove-help' ϡѿβƤФȡ
إɽưŪ˺롣nil ξϼưԤʤ")

(defcustom tcode-adjust-window-for-help nil
  "* non-nil ΤȤإפɽ륦ɥ礭ưŪĴ롣"
  :type 'boolean :group 'tcode)

(defcustom tcode-display-help-delay 2
  "* ľʸϤƤ鲾۸פɽޤǤλ()

`input-method-verbose-flag'  nil Ǥʤ 2 Ȥʤ
Τա"
  :group 'tcode)

(defcustom tcode-verbose-message t
  "* non-nil ΤȤ¿Υåɽ롣

ѿ `input-method-verbose-flag'  nil Ǥʤϡ
ѿͤˤ餺t ꤵƤƱưˤʤΤա"
  :type 'boolean :group 'tcode)

(defvar tcode-mode-help-string nil
  "`tcode-mode-help' ˤäɽʸ
nil ΤȤ `tcode-mode' 롣")

(defvar tcode-keymap-table
;;   0  1  2  3  4  5  6  7  8  9
;;  10 11 12 13 14 15 16 17 18 19
;;  20 21 22 23 24 25 26 27 28 29
;;  30 31 32 33 34 35 36 37 38 39

;;   1  2  3  4  5  6  7  8  9  0
;;   q  w  e  r  t  y  u  i  o  p
;;   a  s  d  f  g  h  j  k  l  ;
;;   z  x  c  v  b  n  m  ,  .  /

;;      !  \"   #   $   %   &   '   (   )   *   +   ,   -   .   /
;;  0   1   2   3   4   5   6   7   8   9   :   ;   <   =   >   ?
;;  @   A   B   C   D   E   F   G   H   I   J   K   L   M   N   O
;;  P   Q   R   S   T   U   V   W   X   Y   Z   [   \   ]   ^   _
;;  `   a   b   c   d   e   f   g   h   i   j   k   l   m   n   o
;;  p   q   r   s   t   u   v   w   x   y   z   {   |   }   ~
  [
   -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  -1  37  -1  38  39
   09  00  01  02  03  04  05  06  07  08  -1  29  -1  -1  -1  -1
   -1  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2
   -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -2  -1  -1  -1  -1  -1
   -1  20  34  32  22  12  23  24  25  17  26  27  28  36  35  18
   19  10  13  21  14  16  33  11  31  15  30  -1  -1  -1  -1
   ]
  "Tɥơ֥ɽΰʸϤȤΰ̣ɽ
0..39:	Tɥ
-1:	ʸ
-2:	бѾʸ
-3:	`tcode-mode-map' ˤäޥɡ
< -3:	- (ʸ)")

(defvar tcode-shift-lowercase nil 
  "TϻΥեȤξˡʸǤϤʤʸϤ롣")

(unless tcode-mode-map
  (setq tcode-mode-map (make-sparse-keymap))
  (define-key tcode-mode-map "?" 'tcode-mode-help)
  (aset tcode-keymap-table (- ?? ? ) -3)
  ;; tcode-mode ΤȤˡ|פϿѴ
  (define-key tcode-mode-map "|" 'tcode-mazegaki-toroku-and-kakutei)
  (aset tcode-keymap-table (- ?| ? ) -3)
  ;; tcode-mode ΤȤˡ!פǺ
  (define-key tcode-mode-map "!" 'tcode-mazegaki-delete-by-last-yomi)
  (aset tcode-keymap-table (- ?! ? ) -3)
  ;; tcode-mode ΤȤˡ=פ䴰
  (define-key tcode-mode-map "=" 'tcode-mazegaki-complete-and-henkan)
  (aset tcode-keymap-table (- ?= ? ) -3))

(defvar tcode-no-wait-display-help-command-list
  '(tcode-mazegaki-select-kouho
    tcode-mazegaki-kakutei-and-self-insert)
  "ڡϤǥإɽΥХåեäǽȤʤޥɤΥꥹȡ
ΥꥹΥޥɤǤϡ `tcode-display-help-buffer' ˤ
إפɽ줿ˡ³ƥڡϤȤǤ
إɽϾäʤ")

(defvar tcode-help-window-height-max 21
  "* إפɽ뤿Υɥι⤵κ")

(defvar tcode-cancel-stroke-list '(?\C-\?)
  "ʸϤŪ˼äΥꥹ")
(defvar tcode-verbose-stroke-list '(? )
  "ʸϤǡޤǤϤ򤹤٤ƤΤޤ륭Υꥹ")
(defvar tcode-another-table nil
  "`tcode-verbose-stroke-list' Υ줿ȤʸΥơ֥롣
ͤ nil ʤФʸvector ǻꤵƤХ˱
ʸ롣")

;;; indicators in modeline
;;; These variables are set in a table file, e.g. tc-tbl.el.
(defvar tcode-transparent-mode-indicator nil)
(defvar tcode-zenkaku-mode-indicator nil)
(defvar tcode-tcode-mode-indicator nil)
(defvar tcode-zenkaku-tcode-mode-indicator nil)
(defvar tcode-hiragana-mode-indicator nil)
(defvar tcode-katakana-mode-indicator nil)

;;
;; Global Variables
;;

;;; ˡ˱Τѿ
(defvar tcode-input-method nil "򤵤Ƥˡ")

;;; ʲѿϡtc-tbl.el ʤɤͤꤵ졢ơ֥
;;; tcode-table ˤƤϿ롣
(defvar tcode-tbl nil)
(defvar tcode-non-2-stroke-char-list nil)
(defvar tcode-special-commands-alist nil)

;;; ϥɤξݻơ֥
(defvar tcode-table nil "ѴѤɽ")
(defvar tcode-stroke-table nil)

;;; ׵ϿѤѿ
(defvar tcode-input-chars 0)
(defvar tcode-number-strokes 0)
(defvar tcode-bushu-occurrence 0)
(defvar tcode-mazegaki-occurrence 0)
(defvar tcode-special-occurrence 0)

(defvar tcode-help-char nil "إפоʸ")
(defvar tcode-auto-remove-help-current-count 0)
(defvar tcode-window-configuration-before-help nil)

(defvar tcode-mode-in-minibuffer nil)

(defvar tcode-dictionaries nil)
;; (Хåե̾ . ե̾)Υꥹȡ

(defconst tcode-stroke-buffer-name " *tcode: stroke*")
(defconst tcode-help-buffer-name "*T-Code Help*")

;;; input-method-verbose-flag 
(defvar tcode-input-method-verbose-flag nil)
(defvar tcode-orig-auto-help nil)
(defvar tcode-orig-verbose-message nil)
(defvar tcode-orig-display-help-delay nil)

(defconst tcode-null-map (make-keymap))

(defvar tcode-use-input-method nil)	; ***experimental***

(defvar tcode-input-filter-functions
  '(((or tcode-katakana-mode tcode-shift-state) . japanese-katakana)
    (tcode-bushu-nest . tcode-do-prefix-bushu)
    (tcode-alnum-2-byte . tcode-1-to-2)))

(defvar tcode-key-shift-table nil)

(defvar tcode-shift-state nil)

;;
;; Buffer Local Variables
;;
(defvar tcode-mode nil "Tɥ⡼ɤΤȤt")
(make-variable-buffer-local 'tcode-mode)

(defvar tcode-zenkaku-mode nil
  "ѥ⡼ɤΤȤttcode-mode  t ΤȤΤͭ")
(make-variable-buffer-local 'tcode-zenkaku-mode)

(defvar tcode-bushu-nest nil "ѴΥͥȥ٥롣")
(make-variable-buffer-local 'tcode-bushu-nest)

(defvar tcode-ready-in-this-buffer nil "ΥХåեTɤνOK")
(make-variable-buffer-local 'tcode-ready-in-this-buffer)

(defvar tcode-current-switch-table 0)
(make-variable-buffer-local 'tcode-current-switch-table)

(defvar tcode-katakana-mode nil "ߥʥ⡼ɤɤ")
(make-variable-buffer-local 'tcode-katakana-mode)

(defvar tcode-message-overlay nil)
(make-variable-buffer-local 'tcode-message-overlay)

;;; ֤

(defvar tcode-key-layout-list
  '(("qwerty" . (?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0
		    ?q ?w ?e ?r ?t ?y ?u ?i ?o ?p
		    ?a ?s ?d ?f ?g ?h ?j ?k ?l ?\;
		    ?z ?x ?c ?v ?b ?n ?m ?, ?. ?/))
    ("qwerty-jis-shift" . (?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0
			      ?q ?w ?e ?r ?t ?y ?u ?i ?o ?p
			      ?a ?s ?d ?f ?g ?h ?j ?k ?l ?\;
			      ?z ?x ?c ?v ?b ?n ?m ?, ?. ?/
			      ?! ?\" ?# ?$ ?% ?& ?' ?( ?) ?~
			      ?Q ?W ?E ?R ?T ?Y ?U ?I ?O ?P
			      ?A ?S ?D ?F ?G ?H ?J ?K ?L ?+
			      ?Z ?X ?C ?V ?B ?N ?m ?< ?> ??))
    ("qwerty-us-shift" . (?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0
			     ?q ?w ?e ?r ?t ?y ?u ?i ?o ?p
			     ?a ?s ?d ?f ?g ?h ?j ?k ?l ?\;
			     ?z ?x ?c ?v ?b ?n ?m ?, ?. ?/
			     ?! ?\@ ?# ?$ ?% ?^ ?& ?* ?( ?)
			     ?Q ?W ?E ?R ?T ?Y ?U ?I ?O ?P
			     ?A ?S ?D ?F ?G ?H ?J ?K ?L ?:
			     ?Z ?X ?C ?V ?B ?N ?m ?< ?> ??))
    ("dvorak" . (?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0
		    ?' ?, ?. ?p ?y ?f ?g ?c ?r ?l
		    ?a ?o ?e ?u ?i ?d ?h ?t ?n ?s
		    ?\; ?q ?j ?k ?x ?b ?m ?w ?v ?z))
    ("dvorak-shift" . (?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?0
			  ?' ?, ?. ?p ?y ?f ?g ?c ?r ?l
			  ?a ?o ?e ?u ?i ?d ?h ?t ?n ?s
			  ?\; ?q ?j ?k ?x ?b ?m ?w ?v ?z
			  ?! ?\@ ?# ?$ ?% ?^ ?& ?* ?( ?)
			  ?\" ?< ?> ?P ?Y ?F ?G ?C ?R ?L
			  ?A ?O ?E ?U ?I ?D ?H ?T ?N ?S
			  ?: ?Q ?J ?K ?X ?B ?M ?W ?V ?Z)))
  "*TɤѤ륭ѤΥǡ
Ūˤϡ̾ȥ¤ӤȤΥꥹȡ
¤ӤǤϡ夫ˡϤʸ40Ĥޤ80¤٤롣
80¤٤硢Ⱦ40ĤϡȾ40ĤΥեȻʸȤư

ޥ`tcode-set-key-layout'Ѥ롣")

(defun tcode-set-key-layout (layout)
  "TɤѤ륭֤ꤹ롣"
  (interactive (let ((layout (completing-read
			      (format 
			       "֤򤷤Ƥ[%s] "
			       (car (car tcode-key-layout-list)))
			      tcode-key-layout-list)))
		 (list layout)))
  (let ((table (make-vector (1+ (- ?~ ? )) -1))
	(list (assoc layout tcode-key-layout-list)))
    (unless list
      (if (or (null layout)
	      (string= layout ""))
	  (unless (setq list (car tcode-key-layout-list))
	    (error "(tcode-key-layout-list)Ƥޤ"))
	(let ((i 0))
	  (while (< i 40)
	    (if (= i 0)
		(message "ܡɤΥ򺸾夫˲Ƥ")
	      (message "%d%d>" (/ i 40) (% i 40)))
	    (setq list (cons (read-char) list)
		  i (1+ i)))
	  (setq list (cons layout (nreverse list))))))
    (let ((i 0))
      (mapcar (lambda (char) 
		(cond ((null char)
		       (aset table (- char ? ) -1))
		      ((and tcode-shift-lowercase
			    (/= (upcase char) char))
		       (aset table (- char ? ) i)
		       (aset table (- (upcase char) ? ) -2))
		      (t
		       (aset table (- char ? ) i)))
		(setq i (1+ i)))
	      (cdr list)))
    (let ((char ? ))
      (while (<= char ?~)
	(if (lookup-key tcode-mode-map (char-to-string char))
	    (aset table (- char ? ) -3))
	(setq char (1+ char))))
    (setq tcode-keymap-table table)
    (setq tcode-key-shift-table nil)
    (if (= (length (cdr list)) 80)
	(let ((l (cdr list)))
	  (while (<= (length l) 40)
	    (setq tcode-key-shift-table (cons (cons (car l)
						    (nth 40 l))
					      tcode-key-shift-table)))))
    list))

(defun tcode-downcase (key)
  (let ((l-key (assq key tcode-key-shift-table)))
    (or (cdr l-key)
	(downcase key))))

;;
;; T-Code main
;;

(defun tcode-on-p ()
  "Tɥ⡼ɤˤʤäƤ뤫ɤ֤"
  (if (window-minibuffer-p (selected-window))
      tcode-mode-in-minibuffer
    tcode-mode))

(defun tcode-function-p (obj)
  (cond ((or (null obj)
	     (arrayp obj))
	 nil)
	((symbolp obj)
	 (fboundp obj))
	((consp obj)
	 (eq (car obj) 'lambda))
	(t
	 nil)))

(defun tcode-get-key-address (c)
  (if (or (< c ? ) (> c ?~))
      -1
    (aref tcode-keymap-table (- c ? ))))

(defun tcode-read-strokes (table &optional not-exec)
  "TABLE η˱ơȥɤࡣ"
  (cond ((null table)			; nil
	 nil)
	((or (vectorp table)		; ޤ
	     (and (consp table)		; ꥹ
		  (not (eq (car table) 'lambda))))
	 (setq tcode-number-strokes (1+ tcode-number-strokes))
	 (run-hooks 'tcode-before-read-stroke-hook)
	 (let ((show-table (sit-for tcode-display-help-delay)))
	   (unwind-protect
	       (let ((echo-keystrokes 0))
		 (if show-table
		     (tcode-display-help-buffer
		      (tcode-draw-current-table table)
		      t))
		 (condition-case nil
		     (tcode-get-action (read-char) table not-exec)
		   (t t)))
	     (if show-table
		 (tcode-auto-remove-help t)))))
	((char-or-string-p table)     ; ʸޤʸ
	 table)
	((and (not (consp table))
	      (not (arrayp table))
	      (commandp table))		; ޥ
	 (if not-exec
	     table
	   (undo-boundary)
	   (setq prefix-arg current-prefix-arg
		 this-command table)
	   (command-execute table)
	   (undo-boundary)
	   t))
	((tcode-function-p table)   ; ޥɤǤʤؿ
	 (if not-exec
	     table
	   (undo-boundary)
	   (funcall table)
	   (undo-boundary)
	   t))
	((boundp table)			; ѿ
	 (tcode-read-strokes (eval table)))
	(t				; ¾
	 nil)))

(defun tcode-get-action (c table &optional not-exec)
  "ʸ C Ȥơɽ TABLE 鳵ư롣
NOT-EXEC  nil ǤʤСưޥɤؿäǤ⡢
¹Ԥʤ"
  (let ((key (tcode-get-key-address c)))
    (if (< key 0)
	;; C is not a key on TABLE
	(cond ((memq c tcode-cancel-stroke-list)
	       t)
	      ((memq c tcode-verbose-stroke-list)
	       (list nil))
	      (t
	       (tcode-get-action-from-table c not-exec)))
      ;; C is a key on TABLE
      (let ((rval (tcode-read-strokes (if (consp table)
					  (cdr (assq (mod key 40) table))
					(aref table (mod key 40)))
				      not-exec)))
	(cond ((or (null rval)
		   (eq rval t)
		   (char-or-string-p rval)
		   (tcode-function-p rval)) ; commandp also
	       rval)
	      ((listp rval)
	       (if (and tcode-another-table
			(>= key 0))
		   (let ((new-rval (aref tcode-another-table
					 (mod key
					      (length tcode-another-table)))))
		     (if (null new-rval)
			 (cons (char-to-string c) rval)
		       (tcode-read-strokes new-rval not-exec)))
		 (cons (char-to-string c) rval))))))))

(defun tcode-get-action-from-table (c &optional not-exec)
  "`tcode-table' 줿ʸ C ϤޤưŪ롣"
  (let ((key (tcode-get-key-address c)))
    (if (< key 0)
	(cond ((= key -1)
	       (if (and (<= ?  c) (<= c ?~))
		   c
		 (if not-exec
		     (list 'lambda nil (list 'tcode-do-command c))
		   (tcode-redo-command c)
		   t)))
	      ((= key -2)
	       (downcase c))
	      ((= key -3)
	       (if not-exec
		   (list 'lambda nil (list 'tcode-do-command c))
		 (let ((save-local-map (current-local-map))
		       (save-global-map (current-global-map))
		       keyseq command)
		   (tcode-redo-command c)
		   (unwind-protect
		       (progn
			 (use-local-map tcode-mode-map)
			 (use-global-map tcode-null-map)
			 (setq keyseq (read-key-sequence nil)
			       command (key-binding keyseq)))
		     (use-local-map save-local-map)
		     (use-global-map save-global-map))
		   (if (null command)
		       (ding)
		     (setq prefix-arg current-prefix-arg
			   this-command command)
		     (command-execute command))
		   t)))
	      (t
	       (- key)))
      ;; C is on table
      (if (< key 40)
	  (tcode-get-action c tcode-table not-exec)
	(setq tcode-shift-state t)
	(tcode-get-action (tcode-downcase c) tcode-table not-exec)))))

(defun tcode-do-prefix-bushu (char)
  ;; tc-bushu.el ɤ߽ФޤǤΥߡؿ
  char)

(defun tcode-apply-filters (list)
  (mapcar (lambda (alist) 
	    (if (eval (car alist))
		(setq list (mapcar (lambda (c) (funcall (cdr alist) c))
				     list))))
	  tcode-input-filter-functions)
  list)

(defun tcode-input-method (ch)
  (setq last-command 'self-insert-command
	last-command-char ch)
  (if input-method-verbose-flag
      (unless tcode-input-method-verbose-flag
	;; push some variables' values
	(setq tcode-orig-auto-help tcode-auto-help
	      tcode-orig-verbose-message tcode-verbose-message
	      tcode-orig-display-help-delay tcode-display-help-delay)
	;; set new values
	(setq tcode-auto-help t
	      tcode-verbose-message t
	      tcode-display-help-delay 2
	      tcode-input-method-verbose-flag t))
    (if tcode-input-method-verbose-flag
	;; pop pushed values
	(setq tcode-auto-help tcode-orig-auto-help
	      tcode-verbose-message tcode-orig-verbose-message
	      tcode-display-help-delay tcode-orig-display-help-delay
	      tcode-input-method-verbose-flag nil)))
  (if (or (and (boundp 'overriding-terminal-local-map)
	       overriding-terminal-local-map)
	  (and (boundp 'overriding-local-map)
	       overriding-local-map))
      (list ch)
    (if tcode-zenkaku-mode
	(list (tcode-1-to-2 ch))
      (setq tcode-number-strokes (1+ tcode-number-strokes))
      (let* ((action (let (input-method-function) ; disable
		       (tcode-get-action-from-table ch)))
	     (result (tcode-apply-filters
		      (cond ((null action)
			     (ding))
			    ((stringp action)
			     (tcode-string-to-char-list action))
			    ((char-or-string-p action)
			     (list action))
			    ((consp action)
			     (mapcar 'tcode-string-to-char
				     (delq nil action)))
			    (t
			     (setq tcode-special-occurrence
				   (1+ tcode-special-occurrence))
			     nil)))))
	(if result
	    (setq tcode-input-chars (+ tcode-input-chars (length result))
		  tcode-help-char (char-to-string (car (reverse result)))))
	(if (not (tcode-on-p)) ; if T-Code is disabled with a command execution
	    (setq input-method-function nil))
	(setq tcode-shift-state nil)
	result))))

(defun tcode-insert (ch)
  "CHХåե롣"
  (unless (stringp ch)
    (setq ch (char-to-string ch)))
  (let* ((p (point))
	 (arg (prefix-numeric-value current-prefix-arg))
	 (n (if (consp current-prefix-arg)
		(/ (car current-prefix-arg) 2)
	      arg)))
    (if (= 1 (length ch))
	;; ascii iso8859-1ξ
	(let ((last-command-char (aref ch 0)))
	  (setq this-command 'self-insert-command)
	  (self-insert-command n))
      (if (and (not (tcode-nemacs-p))
	       (= (chars-in-string ch) 1))
	  (let ((last-command-char (string-to-char ch)))
	    (self-insert-command n))
	(while (> n 0)
	  (insert ch)
	  (setq n (1- n))))
      (unless (and (boundp 'tcode-mazegaki-mode)
		   tcode-mazegaki-mode)
	(if overwrite-mode
	    (let ((str (buffer-substring p (point))))
	      (delete-text-in-column nil (+ (current-column)
					    (string-width str)))))
	(if (and (boundp 'self-insert-after-hook)
		 self-insert-after-hook)
	    (funcall self-insert-after-hook p (point)))
	(tcode-do-auto-fill)
	(run-hooks 'input-method-after-insert-chunk-hook)))))

;;
;; ɽ
;;

(defun tcode-switch-variable (&optional arg)
  "(`tcode-table' ) ѿͤڤؤ롣
ڤؤѿȤͤ `tcode-switch-table-list' ǻꤹ롣
ARG  nil ǤʤȤARG ܤȤڤؤ롣"
  (interactive "P")
  (message
   (mapconcat
    'identity
    (mapcar
     (lambda (elm)
       (set (car elm) (cdr elm)))
     (let ((table (nth (setq tcode-current-switch-table
			     (if arg
				 (1- (prefix-numeric-value arg))
			       (1+ tcode-current-switch-table)))
		       tcode-switch-table-list)))
       (unless table
	 (setq tcode-current-switch-table 0
	       table (car tcode-switch-table-list)))
       table))
    "")))

(defun tcode-set-stroke-property (table sequence)
  (cond ((or (null table)
	     (tcode-function-p table)))
	((stringp table)
	 (put (intern table tcode-stroke-table)
	      'stroke
	      (vconcat sequence)))
	((char-or-string-p table)
	 (put (intern (char-to-string table) tcode-stroke-table)
	      'stroke
	      (vconcat sequence)))
	((consp table)
	 (mapcar (lambda (ent)
		   (tcode-set-stroke-property (cdr ent)
					      (append sequence
						      (list (car ent)))))
		 table))
	((vectorp table)
	 (let ((i 0))
	   (while (< i 40)
	     (tcode-set-stroke-property (aref table i)
					(append sequence (list i)))
	     (setq i (1+ i)))))
	((and (symbolp table)
	      (boundp table))
	 (tcode-set-stroke-property (eval table) sequence))))

(defun tcode-set-action (table sequence value)
  "TABLE Ρ SEQUENCE  VALUE ꤹ롣"
  (let ((key (car sequence))
	(subsequence (cdr sequence))
	(subsubsequence (nthcdr 2 sequence)))
    (cond (subsubsequence
	   (tcode-set-action (if (consp table)
				 (cdr (assq key table))
			       (aref table key))
			     subsequence
			     value))
	  (subsequence
	   (cond ((or (and (char-or-string-p table)
			   (not (stringp table)))
		      (null table))
		  (list key (tcode-set-action nil subsequence value)))
		 ((vectorp table)
		  (aset table key
			(tcode-set-action (aref table key) subsequence value))
		  table)
		 (t
		  (let ((orig table))
		    (if (null (setq table (assq key table)))
			(nconc orig
			       (list (list key
					   (tcode-set-action
					    nil subsequence value))))
		      (setcdr table
			      (tcode-set-action (cdr table) subsequence value))
		      table)))))
	  (t
	   ;; add key
	   (if (listp table)
	       (cond ((null table)
		      (cons key value))
		     ((and (char-or-string-p (car table))
			   (not (stringp (car table)))
			   (not (consp (cdr table))))
		      (list table (cons key value)))
		     (t
		      (let ((old (assq key table)))
			(while old
			  (setq table (delq old table)
				old (assq key table)))
			(if (null table)
			    (cons key value)
			  (nconc (list (cons key value)) table)))))
	     (if (vectorp table)
		 (progn
		   (aset table key value)
		   table)
	       (list (cons key value))))))))

(defun tcode-set-action-to-table (sequence value)
  "Ѥơ֥ SEQUENCE Ф VALUE ꤹ롣
ư(VALUE)ȤƻǤΤϰʲΤȤꡣ
    - ޥ (symbol)		Υޥɤ¹Ԥ롣
    - ؿ (symbol, lambda)	δؿʤǸƤ֡
    - ѿ (symbol)		ɾ̤ưԤ
    - ɽ (vector)		ˤɽ˽äưԤ
    - ꥹ (list)		ˤΥꥹȤ˽äưԤ
    - ʸ (string)		ʸ롣
    - ʸ (char)		ʸ롣

  ϥϤΥꥹȤޤϥϡ
ϤꤹȡǸ SPC 򲡤Ȥưꤹ롣"
  (cond ((consp sequence)
	 (tcode-set-action tcode-table sequence value)
	 (setq tcode-stroke-table (make-vector 511 nil))
	 (tcode-set-stroke-property tcode-table nil))
	((and (char-or-string-p sequence)
	      (not (stringp sequence)))
	 (unless tcode-another-table
	   (setq tcode-another-table (make-vector 40 nil)))
	 (aset tcode-another-table sequence value))
	(t
	 (error "λ̵꤬Ǥ"))))

(defun tcode-replace-part-of-table (table)
  "`tcode-table' ΰ TABLE ֤롣
TABLE ϥΥꥹȤȤư alist"
  (mapcar (lambda (elm)
	    (tcode-set-action tcode-table (car elm) (cdr elm)))
	  table))

;;;###autoload
(defun tcode-load-table (filename)
  (interactive "fLoad T-Code table file: ")
  (let ((k1 0) k2 newval char)
    (load filename)
    (setq tcode-table (make-vector 40 nil))
    (while (< k1 40)
      (aset tcode-table k1 (make-vector 40 nil))
      (setq k1 (1+ k1)))
    (setq k1 0)
    (while (< k1 40)
      (let ((v (aref tcode-tbl k1)))
	(if (null v)
	    ()
	  (setq newval (vconcat (delq ?  (tcode-string-to-char-list v))))
	  (unless (= (length newval) 40)
	    (error "Table corrupted at line %d." (1+ k1)))
	  (setq k2 0)
	  (while (< k2 40)
	    (unless (memq (setq char (aref newval k2))
			  tcode-non-2-stroke-char-list)
	      (aset (aref tcode-table k2) k1 char))
	    (setq k2 (1+ k2)))))
      (setq k1 (1+ k1)))
    (setq tcode-tbl nil)		; free
    ;; ޥɤơ֥Ͽ롣
    (tcode-replace-part-of-table tcode-special-commands-alist)
    (setq tcode-special-commands-alist nil) ; free
    ;; 'stroke property 롣
    (setq tcode-stroke-table (make-vector 511 nil))
    (tcode-set-stroke-property tcode-table nil)
    (if (get-buffer tcode-stroke-buffer-name)
	(kill-buffer tcode-stroke-buffer-name))
    (run-hooks 'tcode-after-load-table-hook)
    (tcode-clear)
    (tcode-mode-line-redisplay)))

(defun tcode-stroke-for-char (ch)
  "CHǤꥹȤ֤ľϤǤʤnil֤"
  (or (append (get (intern-soft ch tcode-stroke-table) 'stroke) nil)
      (let ((ch (char-to-string (tcode-2-to-1 (tcode-string-to-char ch)))))
	(append (get (intern-soft ch tcode-stroke-table) 'stroke) nil))))

(defun tcode-encode-sequence (sequence table)
  (cond ((or (null table)
	     (null sequence))
	 table)
	((consp table)
	 (tcode-encode-sequence (cdr sequence) (assq (car sequence) table)))
	((vectorp table)
	 (tcode-encode-sequence (cdr sequence) (aref table (car sequence))))
	(t
	 (error "bad data structure"))))

;;
;; tcode-self-insert-command ط
;;

(defvar tcode-self-inserting-commands
  '((egg-self-insert-command . "p")
    (canna-self-insert-command . "*p")
    (electric-c-brace . "P")
    (electric-c-semi . "P")
    (electric-c-terminator . "P")
    (c-electric-semi&comma . "P")
    (c-electric-slash . "*P")
    (electric-perl-terminator . "P")) "\
tcode-mode ξˤ `tcode-self-insert-command' ˲륳ޥɤalist
\(command . arg\) η򤷤Ƥ롣
㤨СC⡼ɤ `;'(`electric-c-semi')Τ褦ʥޥɤ㡣
arg  interactive 2ѿƱʸ")

(defvar tcode-self-insert-non-undo-count 0)
(defvar tcode-cancel-undo-boundary-commands
  '(tcode-self-insert-command-maybe self-insert-command))

(defun tcode-cancel-undo-boundary ()
  "ʸȤˡޤȤ undo Ǥ褦Ĵ롣"
  (if (or (not (memq last-command
		     (if (memq this-command
			       tcode-cancel-undo-boundary-commands)
			 tcode-cancel-undo-boundary-commands
		       (setq tcode-cancel-undo-boundary-commands
			     (cons this-command
				   tcode-cancel-undo-boundary-commands)))))
	  (>= tcode-self-insert-non-undo-count (if (tcode-on-p) 10 20)))
      (progn
	(undo-boundary)
	(setq tcode-self-insert-non-undo-count 1))
    (cancel-undo-boundary)
    (setq tcode-self-insert-non-undo-count
	  (1+ tcode-self-insert-non-undo-count))))

(defun tcode-self-insert-command (&optional arg)
  "Encode Tcode character and insert."
  (interactive)
  (tcode-cancel-undo-boundary)
  (let ((s (mapconcat 'char-to-string
		      (delq nil (tcode-input-method last-command-char))
		      "")))
    (unless (string= s "")
      (tcode-insert s))))

(defun tcode-self-insert-command-maybe (arg)
  (interactive "p")
  (if (tcode-on-p)
      (tcode-self-insert-command arg)
    (tcode-cancel-undo-boundary)
    (self-insert-command arg)))

(defun tcode-substitute-self-insertings ()
  "`tcode-self-inserting-commands' ˥ꥹȤƤ륳ޥɤ̵롣
δؿfuncϡorig:funcȤ̾Ѥ롣
ơfuncϡTɥ⡼ǤϡTʸϤ򡤥⡼ɳǤ
orig:funcƤӽФؿ֤롣"
  (let ((commands tcode-self-inserting-commands)
	com symbol havearg tcode-func orig-func)
    (while commands
      (setq com (car commands)
	    commands (cdr commands)
	    symbol (car com)
	    havearg (cdr com)
	    tcode-func (intern (concat "tcode:" (symbol-name symbol)))
	    orig-func (intern (concat "orig:" (symbol-name symbol))))
      (when (and (not (fboundp tcode-func))
		 (fboundp symbol))
	(fset orig-func (symbol-function symbol))
	(eval (list 'defun tcode-func (if havearg '(arg))
		    (documentation orig-func)
		    (cons 'interactive (if havearg (list havearg)))
		    (list 'if 'tcode-mode '(tcode-self-insert-command)
			  (cons orig-func (if (cdr com) '(arg))))))
	(fset symbol tcode-func)))))

;;
;; 
;;

(defun tcode-buffer-init ()
  "ХåեȤTɤνԤ`tcode-mode-hook'Ƥ֡"
  (unless tcode-ready-in-this-buffer
    (unless tcode-use-input-method
      (tcode-substitute-self-insertings))
    (setq tcode-ready-in-this-buffer t)
    (run-hooks 'tcode-mode-hook)))

(defun tcode-substitute-command-keys (string)
  "`substitute-command-keys'  `tcode-mode-map' ΤȤŬѤ롣"
  (let ((orig-map (current-local-map)))
    (prog2
	(use-local-map tcode-mode-map)
	(substitute-command-keys string)
      (use-local-map orig-map))))

(defun tcode-bushu-init (level)
  "ѴνԤ
LEVELѿ`tcode-bushu-on-demand'꾮äԤʤ"
  (interactive (list 999))
  (when (>= level tcode-bushu-on-demand)
    (require 'tc-bushu)
    (tcode-bushu-load-dictionary)
    (run-hooks 'tcode-bushu-ready-hook)))

(defun tcode-init ()
  "EmacsưƤǽtcode-modeǼ¹Ԥ롣
ɽ򤼽ѴνԤ
`tcode-ready-hook' Ƥ֡"
  (unless (file-exists-p tcode-init-file-name)
    (if (y-or-n-p (format "ե%sޤ󡣺ޤ?"
			  tcode-init-file-name))
	(tcode-install)
      (call-interactively 'tcode-set-key-layout)))
  (unless tcode-table
    (tcode-load-table tcode-table-file-name))
  (tcode-bushu-init 1)
  (if tcode-use-isearch
      (require tcode-isearch-type))
  (unless tcode-use-input-method
    (substitute-key-definition 'self-insert-command
			       'tcode-self-insert-command-maybe
			       global-map))
  (add-hook 'minibuffer-exit-hook 'tcode-exit-minibuffer)
  (when (fboundp 'inactivate-input-method)
    (defadvice inactivate-input-method (after
					turn-off-current-input-method-title
					activate)
      "Turn off `current-input-method-title' for mode line."
      (and (string= (car input-method-history)
		    tcode-current-package)
	   (setq current-input-method-title
		 tcode-transparent-mode-indicator)))
    (defadvice activate-input-method (after
				      turn-on-current-input-method-title
				      activate)
      "Turn on `current-input-method-title' for mode line."
      (and (string= (car input-method-history)
		    tcode-current-package)
	   (tcode-mode-line-redisplay))))
  (provide 'tcode-ready)
  (run-hooks 'tcode-ready-hook)
  (tcode-verbose-message
   (tcode-substitute-command-keys
    "Tɥ⡼ɤǤϡ\\[tcode-mode-help]פǥإפɽޤ")))

;;
;; ⡼ɤڤؤ
;;

(defun tcode-mode-line-redisplay ()
  (setq current-input-method-title
	(if (tcode-on-p)
	    (if tcode-zenkaku-mode
		tcode-zenkaku-mode-indicator
	      (concat (if tcode-alnum-2-byte
			  tcode-zenkaku-tcode-mode-indicator
			tcode-tcode-mode-indicator)
		      (if tcode-katakana-mode
			  tcode-katakana-mode-indicator
			tcode-hiragana-mode-indicator)))
	  tcode-transparent-mode-indicator))
  (and (window-minibuffer-p (selected-window))
       (boundp 'minibuffer-preprompt)
       (setq minibuffer-preprompt
	     (list "[" (eval tcode-mode-indicator) "]")))
  (set-buffer-modified-p (buffer-modified-p)))

(defun tcode-clear (&optional help-also)
  "䤳⡼ɤäƤΤꥢ롣"
  (interactive)
  (and (boundp 'tcode-mazegaki-mode)
       tcode-mazegaki-mode
       (tcode-mazegaki-kakutei))
  (and (boundp 'tcode-mazegaki-in-prefix)
       tcode-mazegaki-in-prefix
       (tcode-mazegaki-kakutei))
  (if (not (tcode-on-p))
      (tcode-auto-remove-help)
    ;; tcode on
    (while tcode-bushu-nest
      (tcode-bushu-end))
    (if help-also
	(tcode-auto-remove-help t))))

(defun tcode-activate (&optional arg)
  "Tɥ⡼ɤͭˤ롣ARGΤȤ̵ˤ롣

Tɥ⡼ɤˤĤƤϡ\\[tcode-mode-help] ɽإפ򻲾ȡ"
  (if (and arg
	   (< (prefix-numeric-value arg) 0))
      ;; inactivate T-Code mode
      (unwind-protect
	  (progn
	    (if (window-minibuffer-p (selected-window))
		(setq tcode-mode-in-minibuffer nil))
	    (setq tcode-mode nil
		  tcode-self-insert-non-undo-count 1
		  tcode-zenkaku-mode nil)
	    (tcode-clear)
	    (run-hooks 'input-method-inactivate-hook))
	(setq input-method-function nil))
    ;; activate T-Code mode
    (if (window-minibuffer-p (selected-window))
	(setq tcode-mode-in-minibuffer t))
    (setq tcode-mode t
	  tcode-self-insert-non-undo-count 1)
    (unless (featurep 'tcode-ready)
      (tcode-init))
    (unless tcode-ready-in-this-buffer
      (tcode-buffer-init))
    (run-hooks 'input-method-activate-hook)
    (when tcode-use-input-method
      (set (make-local-variable 'input-method-function)
	   'tcode-input-method)))
  (run-hooks 'tcode-toggle-hook)
  (tcode-mode-line-redisplay))

(defun tcode-inactivate ()
  "Tɥ⡼ɤ̵ˤ롣"
  (tcode-activate -1))

(defun tcode-exit-minibuffer ()
  (tcode-inactivate)
  (setq current-input-method nil)
  (if (boundp 'minibuffer-preprompt)
      (setq minibuffer-preprompt nil)))

(defun tcode-toggle-katakana-mode (arg)
  "ʥ⡼ɤڤؤ롣"
  (interactive "P")
  (setq tcode-katakana-mode (if (null arg)
				(not tcode-katakana-mode)
			      (>= (prefix-numeric-value arg) 0)))
  (tcode-mode-line-redisplay))

;;
;; 2Хȱѿ
;;

(defvar tcode-alnum-1-to-2-table
  (concat "ɡǡʡˡܡݡ䡩"
	  "£ãģţƣǣȣɣʣˣ̣ͣΣϣУѣңӣԣգ֣ףأ٣ڡΡϡ"
	  "ƣСáѡ")
  "1Хȱѿ ' '..'~' 2ХȱѿѴ/Ѵ뤿Υơ֥")

(defun tcode-check-alnum-1-to-2-table ()
  (if (stringp tcode-alnum-1-to-2-table)
      (setq tcode-alnum-1-to-2-table
	    (vconcat (tcode-string-to-char-list tcode-alnum-1-to-2-table)))))

(defun tcode-1-to-2-region (beg end)
  "꡼1Хȱѿ2ХȤѴ롣"
  (interactive "*r")
  (tcode-check-alnum-1-to-2-table)
  (save-excursion
    (save-restriction
      (goto-char beg)
      (narrow-to-region beg end)
      (let (char)
	(while (progn (skip-chars-forward "^!-~" (point-max))
		      (< (point) (point-max)))
	  (setq char (following-char))
	  (tcode-delete-char 1)
	  (insert (char-to-string (tcode-1-to-2 char))))))))

(defun tcode-1-to-2 (char)
  "1ХȱѿCHAR2ХȤѴ롣"
  (tcode-check-alnum-1-to-2-table)
  (if (and (<= ?! char) (<= char ?~))
      (aref tcode-alnum-1-to-2-table (- char ? ))
    char))

(defun tcode-2-to-1-region (beg end)
  "꡼2Хȱѿ1ХȤѴ롣"
  (interactive "*r")
  (tcode-check-alnum-1-to-2-table)
  (save-excursion
    (save-restriction
      (goto-char beg)
      (narrow-to-region beg end)
      (let ((alnum-2byte-regexp (concat "["
					(mapconcat 'char-to-string
						   tcode-alnum-1-to-2-table "")
					"]+"))
	    str)
	(while (re-search-forward alnum-2byte-regexp nil t)
	  (setq str (buffer-substring (match-beginning 0) (match-end 0)))
	  (delete-region (match-beginning 0) (match-end 0))
	  (insert (mapconcat (lambda (c)
			       (char-to-string (tcode-2-to-1 c)))
			     (tcode-string-to-char-list str)
			     "")))))))

(defun tcode-2-to-1 (char)
  "2ХȱѿCHAR1ХȤѴ롣"
  (tcode-check-alnum-1-to-2-table)
  (let ((ch 0))
    (catch 'found
      (while (< ch 95)
	(if (= (aref tcode-alnum-1-to-2-table ch) char)
	    (throw 'found (+ ch 32)))
	(setq ch (1+ ch)))
      char)))

;;
;; Help
;;

(defun tcode-action-to-printable (action)
  (cond ((or (null action)
	     (stringp action))
	 action)
	((char-or-string-p action)
	 (char-to-string action))
	((and (symbolp action)
	      (boundp action))
	 (tcode-action-to-printable (eval action)))
	(t
	 "*")))

(defun tcode-draw-current-table (table)
  "TABLE 顢Ϥˤʸɽɽ"
  (tcode-draw-table
   (if (vectorp table)
       (let ((draw-table (copy-sequence table))
	     (i 0))
	 (while (< i 40)
	   (aset draw-table i (tcode-action-to-printable (aref draw-table i)))
	   (setq i (1+ i)))
	 draw-table)
     ;; table ϥꥹ
     (let ((draw-table (make-vector 40 nil)))
       (mapcar (lambda (elm)
		 (aset draw-table
		       (car elm)
		       (tcode-action-to-printable (cdr elm))))
	       table)
       draw-table))
   1 1))

(defun tcode-verbose-message (message &optional non-verbose-message)
  "ѿ `tcode-verbose-message'  non-nil ξˤϡ MESSAGE ɽ롣
ǤʤȤ NON-VERBOSE-MESSAGE Сɽ롣"
  (if (or tcode-verbose-message
	  non-verbose-message)
      (message (if tcode-verbose-message message non-verbose-message))))

(defun tcode-format-list (param list)
  "PARAM ˽äʸΥꥹ LIST Ĥʤ롣
PARAM ˻ǤΤϡʸȿǡ LIST Ǥ
ɽ뤫ɽ"
  (mapconcat
   (lambda (elm)
     (cond ((stringp elm)
	    elm)
	   ((integerp elm)
	    (let* ((str (car list))
		   (width (apply '+
				 (mapcar (lambda (c)
					   (tcode-char-width c))
					 (tcode-string-to-char-list str))))
		   (pad-width (- (abs elm) width))
		   (pad (if (> pad-width 0)
			    (make-string pad-width ? )
			  "")))
	      (setq list (cdr list))
	      (if (< elm 0)
		  (concat str pad)
		(concat pad str))))))
   param
   ""))

(defconst tcode-table-line-format-1
  '(" " -5 -5 -5 -5 " " -5 " " 5 " " 5 5 5 5 "\n"))
(defconst tcode-table-line-format-2
  '("[" -5 -5 -5 -4 "] " -5 "  " 4 " [" 4 5 5 5 "]\n"))

(defun tcode-draw-table (table page whole-page)
  "ɽ TABLE ˴ŤɽϤʤ"
  (let ((buf (get-buffer-create " *tcode: table*")))
    (save-excursion
      (set-buffer buf)
      (erase-buffer)
      (mapcar
       (lambda (key-list)
	 (let ((kouho-list (mapcar (lambda (n) (or (aref table n) "-"))
				   key-list)))
	   (insert
	    (tcode-format-list
	     (if (= (car key-list) 0)
		 tcode-table-line-format-1
	       tcode-table-line-format-2)
	     kouho-list))))
       '(( 0  1  2  3  4  5  6  7  8  9)
	 (10 11 12 13 14 15 16 17 18 19)
	 (20 21 22 23 24 25 26 27 28 29)
	 (30 31 32 33 34 35 36 37 38 39)))
      (unless (= whole-page 1)
	(backward-char 1)
	(insert (format "     (%d/%d)" page whole-page))))
    buf))

(defun tcode-display-help-buffer (buffer &optional display-only append)
  "\"*T-Code Help*\" ȤХåե BUFFER Ƥɽ롣
ɽľ˶Ϥȡ DISPLAY-ONLY  nil ʤФΥХåե
õ롣 APPEND  nil ǤʤȤϡƤɲäɽ롣"
  ;; ɥ¸
  (unless (get-buffer-window tcode-help-buffer-name)
    (setq tcode-window-configuration-before-help
	  (if (one-window-p)
	      nil
	    (current-window-configuration))))
  ;; ɽƤɽ
  (let (previous-contents)
    (if append
	(let ((buf (get-buffer tcode-help-buffer-name)))
	  (setq previous-contents (and buf
				       (save-excursion
					 (set-buffer buf)
					 (buffer-string))))))
    (with-output-to-temp-buffer tcode-help-buffer-name
      (when previous-contents
	(princ previous-contents)
	(princ "\n"))
      (princ (save-excursion (set-buffer buffer) (buffer-string))))
    (if (fboundp 'help-mode)
	(save-excursion
	  (set-buffer (get-buffer tcode-help-buffer-name))
	  (help-mode))))
  ;; ɥ礭Ĵ
  (let ((orig-win (selected-window))
	(new-win (get-buffer-window tcode-help-buffer-name))
	(window-min-height 2))
    (when new-win
      (select-window new-win)
      (if (and (or (not tcode-window-configuration-before-help)
		   tcode-adjust-window-for-help)
	       (= (frame-width) (window-width)))
	  (enlarge-window (- (1+ (min (count-lines (point-min) (point-max))
				      tcode-help-window-height-max))
			     (window-height))))
      (set-window-start (selected-window) (point-min))
      (unless (one-window-p)
	(select-window orig-win))))
  ;; ɽν
  (setq tcode-auto-remove-help-current-count 0)
  (unless (or display-only
	      (memq this-command tcode-no-wait-display-help-command-list))
    (tcode-verbose-message "ڡǥإפäޤ" " ")
    (let ((ch (read-char)))
      (if (/= ch ? )
	  (tcode-redo-command ch)
	(tcode-auto-remove-help t))
      (message ""))))

(defun tcode-auto-remove-help (&optional immediate)
  "إפưŪ˾õ롣
õΤϡإפɽƤ
δؿ `tcode-auto-remove-help-count' ƤФ줿Ȥ"
  (when (or immediate
	    (and tcode-auto-remove-help-count
		 (>= (setq tcode-auto-remove-help-current-count
			   (1+ tcode-auto-remove-help-current-count))
		     tcode-auto-remove-help-count)))
    (let ((help-buf (get-buffer tcode-help-buffer-name))
	  help-win)
      (and help-buf
	   (not (eq help-buf (current-buffer)))
	   (setq help-win (get-buffer-window help-buf))
	   (cond (tcode-window-configuration-before-help
		  (let ((orig-win (selected-window))
			(orig-buf (current-buffer))
			(orig-pos (point)))
		    (if tcode-adjust-window-for-help
			(set-window-configuration
			 tcode-window-configuration-before-help))
		    (unless (one-window-p)
		      (select-window orig-win))
		    (set-window-buffer (selected-window) orig-buf)
		    (goto-char orig-pos)))
		 ((not (one-window-p))
		  (delete-window help-win))))
      (and help-buf
	   (or immediate
	       (not (eq help-buf (current-buffer))))
	   (kill-buffer help-buf)))))

;;;
;;; ƥ⥸塼ǻѤѴؿ
;;;

(defun tcode-unmap-key (c)
  "ֹ椫бʸ롣"
  (let ((max (length tcode-keymap-table))
	(i 0))
    (catch 'found
      (while (< i max)
	(if (= c (aref tcode-keymap-table i))
	    (throw 'found t))
	(setq i (1+ i))))
    (+ i ? )))

(defun tcode-removable-fill-prefix-p ()
  "Ƥ褤 fill-prefix 
Ƥ褤 fill-prefix Ȥϡ
Ƭ point ޤǤ fill-prefix Ǥꡢ
ιԤ fill-prefix ǻϤޤäƤ򤤤"
  (and fill-prefix
       (and (string= fill-prefix
		     (buffer-substring (save-excursion
					 (beginning-of-line)
					 (point))
				       (point)))
	    (save-excursion
	      (and (= (forward-line -1) 0)
		   (looking-at (regexp-quote fill-prefix)))))))

(defun tcode-skip-blank-backward ()
  "Ƭ point ޤǤʤ point ιԤι˰ư롣
`fill-prefix'ꤵƤȤϡʸ̵뤹롣"
  (let ((p (point))
	(fill-prefix-end (and fill-prefix
			      (save-excursion
				(beginning-of-line)
				(and (looking-at (regexp-quote fill-prefix))
				     (match-end 0))))))
    (cond ((bobp)
	   nil)
	  ((bolp)
	   ;; ιԤ
	   (forward-line -1)
	   (end-of-line)
	   (if (bobp)
	       nil
	     (point)))
	  ((null fill-prefix-end) ; fill-prefix ʤ硣
	   (if (save-excursion
		 (beginning-of-line)
		 (or (not (re-search-forward "^\\s +" p t))
		     (/= (point) p)))
	       p
	     ;; ƬζȤФ
	     (forward-line -1)
	     (end-of-line)
	     (if (bobp)
		 nil
	       (point))))
	  ((not (save-excursion
		  (goto-char fill-prefix-end)
		  (tcode-removable-fill-prefix-p)))
	   ;; ľιԤ fill-prefix ǻϤޤäƤʤ硣
	   (if (= fill-prefix-end p)
	       nil    ; ƤϤʤ fill-prefix
	     p))
	  ((<= p fill-prefix-end) ; fill-prefix ˤ硣
	   (forward-line -1)
	   (end-of-line)
	   (if (bobp)
	       nil
	     (point)))
	  (t		  ; fill-prefix + ʸξ硣
	   (if (save-excursion
		 (beginning-of-line)
		 (or (not (re-search-forward
			   (concat "^" (regexp-quote fill-prefix) "\\s +")
			   p t))
		     (/= (point) p)))
	       p
	     ;; fill-prefix + ȤФ
	     (forward-line -1)
	     (end-of-line)
	     (point))))))

(defun tcode-get-context (max &optional terminate-char-list)
  " point ƬˤܸޤϱñĤΥꥹȤ֤
ꥹȤǤ(POINT . \"ʸ\")Ǥ롣
ǡʸפϡܸʸξ1ʸʸξ1ñǤ롣
ꥹȤν֤ȤƤϡХåեƬ˶ᤤʸƬ¦ˤʤ롣
ꥹȤĹϺ MAX ʸǤ롣"
  (save-excursion
    (let (ch context)
      (while (and (< (length context) max)
		  (tcode-skip-blank-backward)
		  (or (not (memq (setq ch (tcode-char-before (point)))
				 terminate-char-list))
		      (null context))
		  (not (bobp)))
	(if (> ch 255)
	    (progn
	      ;; 1
	      (tcode-forward-char -1)
	      (setq context (cons (cons (point) (char-to-string ch))
				  context)))
	    ;; ե٥åȤξ
	    (let ((end (point))
		  (beg (progn
			 (if (= (char-syntax ch) ?w)
			     ;; 
			     (while (and (not (bobp))
					 (= (char-syntax
					     (setq ch (tcode-char-before
						       (point))))
					    ?w)
					 (<= ch 255))
			       (tcode-forward-char -1))
			   ;; 
			   (tcode-forward-char -1))
			 (point))))
	      (setq context (cons
			     (cons beg
				   (buffer-substring-no-properties beg end))
			     context)))))
      context)))

(defun tcode-katakana-to-hiragana-string (str)
  "ʸ STR ΥʤҤ餬ʤѴ롣"
  (mapconcat (lambda (c)
	       (char-to-string (japanese-hiragana c)))
	     (tcode-string-to-char-list str)
	     ""))

(defun tcode-set-work-buffer (bufname filename &optional force noerror)
  "ԽоݥХåեFILENAME ƤĥХåե BUFNAME ˤ롣
ե FILENAME ޤɤ߹ޤƤʤˤɤ߹ࡣ
֤ͤȤơꤵ줿Хåե֤

FORCE  nil ǤʤȤϡɤ߹ߤԤ
NOERROR  nil ǤʤȤϡFILENAME ꤵƤʤ
ե뤬¸ߤʤǤ⡢ΥХåե롣"
  (let ((buffer (get-buffer bufname))
	(file (expand-file-name filename)))
    (if (and buffer
	     (not force))
	(set-buffer buffer)
      (if (and file (file-exists-p file))
	  (prog2
	      (message "ե %s ɤ߹..." file)
	      (set-buffer (get-buffer-create bufname))
	    (erase-buffer)
	    (insert-file-contents file)
	    (set-buffer-modified-p nil)
	    (message "ե %s ɤ߹...λ" file))
	(if noerror
	    (set-buffer (get-buffer-create bufname))
	  (error "ե %s ¸ߤޤ" file))))))

(defun tcode-save-buffer (bufname filename &optional backup-inhibited)
  "BUFNAMEΥХåեѹƤFILENAMEΥե¸롣
BACKUP-INHIBITED  nil ǤʤϡХååץեκ
Ԥʤ"
  (let ((buffer (get-buffer bufname)))
    (when (and buffer
	       (buffer-modified-p buffer)
	       (file-writable-p filename))
      (save-excursion
	(set-buffer buffer)
	(unless (or backup-inhibited
		    (not (file-exists-p filename)))
	  (rename-file filename (make-backup-file-name filename) t))
	(write-region (point-min) (point-max) filename)
	(set-buffer-modified-p nil)))))

(defun tcode-overlay-message (str)
  "overlayѤơߤιԤ1Բ˥å(STR)ɽ롣"
  (let ((point (save-excursion
		 (forward-line 1)
		 (point)))
	(nol (apply '+ (mapcar (lambda (c)
				 (if (= c ?\n)
				     1
				   0))
			       (string-to-char-list str)))))
    (setq tcode-message-overlay
	  (if (overlayp tcode-message-overlay)
	      (move-overlay tcode-message-overlay point point)
	    (make-overlay point point)))
    (if (= (point) (point-max))
	(setq str (concat "\n" str)))
    (overlay-put tcode-message-overlay 'before-string str)
    ;; ɽȱϺɽ
    (if (>= (+ (count-lines (window-start) (point)) nol 1)
	    (1- (window-height)))
	(recenter (1- (- nol))))))

(defun tcode-delete-overlay-message ()
  "`tcode-overlay-message'ɽ줿åä"
  (interactive)
  (when (overlayp tcode-message-overlay)
    (delete-overlay tcode-message-overlay)
    (redraw-frame (selected-frame))))

;;
;; ĥޥ
;;

(defun tcode-insert-register (reg arg)
  "`insert-register' ƱݥȤȥޡΰ֤ա"
  (interactive "cInsert register: \nP")
  (insert-register reg (not arg)))

(defun tcode-transpose-strokes (arg)
  "ݥȰ֤ʸΥȥ줫롣"
  (interactive "*P")
  (if (not (tcode-on-p))
      (transpose-chars arg)
    (if (eolp) (tcode-forward-char -1))
    (let* ((ch (buffer-substring (point)
				(save-excursion (tcode-forward-char 1)
						(point))))
	   (stroke (tcode-stroke-for-char ch)))
      (when (and (= (length stroke) 2)
		 (setq ch (tcode-action-to-printable
			   (tcode-encode-sequence (reverse stroke)
						  tcode-table))))
	(tcode-delete-char 1)
	(insert ch)))))

;;
;; EmacsλΥǡ¸
;;

(defun tcode-save-dictionaries (&optional backup-inhibited)
  "TɤѤ뼭ѹƤ¸롣"
  (interactive)
  (mapcar (lambda (dic)
	    (let ((bufname (car dic))
		  (filename (cdr dic)))
	      (tcode-save-buffer bufname filename
				 backup-inhibited)))
	  tcode-dictionaries))

(defun tcode-kill-emacs-function ()
  (tcode-save-dictionaries)
  (tcode-record))

(add-hook 'kill-emacs-hook 'tcode-kill-emacs-function)

(defun tcode-record ()
  (when (and tcode-record-file-name
	     (> tcode-number-strokes 0))
    (let ((bufname " *tcode: record*"))
      (save-excursion
	(tcode-set-work-buffer bufname tcode-record-file-name nil t)
	(goto-char (point-max))
	(insert
	 (format (concat "%s  ʸ: %4d  : %3d(%d%%)  "
			 "򤼽: %3d(%d%%)  ǽ: %3d(%d%%)\n")
		 (let ((time (current-time-string)))
		   (if (not (string-match "^... \\(.*:.*\\):" time))
		       ""
		     (substring time (match-beginning 1) (match-end 1))))
		 tcode-input-chars
		 tcode-bushu-occurrence
		 (/ (* 100 tcode-bushu-occurrence) tcode-number-strokes)
		 tcode-mazegaki-occurrence
		 (/ (* 100 tcode-mazegaki-occurrence) tcode-number-strokes)
		 tcode-special-occurrence
		 (/ (* 100 tcode-special-occurrence) tcode-number-strokes)))
	(tcode-save-buffer bufname tcode-record-file-name t)))))

;;;
;;; Ѵν
;;;

(provide 'tc)

(unless (or (featurep 'tc-bushu)
	    (< 0 tcode-bushu-on-demand))
  (require 'tc-bushu)
  (tcode-bushu-init 999))

;;; tc.el ends here
