# -*- coding: utf-8 -*-
"""
This module contains functions for computing 
integrals both symbolically and numerically.
"""

__author__ = "Kent-Andre Mardal and Martin Sandve Alnes"
__date__   = "2007-11-07 -- 2009-03-19"
__copyright__ = "(C) 2007-2009 Martin Sandve Alnes and Simula Resarch Laboratory"
__license__  = "GNU GPL Version 2, or (at your option) any later version"

import swiginac
import SyFi

from sfc.common import sfc_error
from sfc.quadrature import find_quadrature_rule
from sfc.geometry import UFCCell, geometry_mapping
from sfc.symbolic_utils.symbolic_utils import symbol, symbols

def symbolic_integral(polygon, function):
    """
    This function computes the integral of the specified function over
    the specified polygon symbolically.
    The function may take as arguments the global coordinates (x,y,z)
    as well as coordinates on the reference polygon (xi0, xi1, xi2).
    """
    nsd = polygon.no_space_dim()
    G, x0 = geometry_mapping(polygon) # FIXME: passing Rectangle (or Box) here will yield an affine mapping only correct for straight rectangles

    # TODO: Get symbols as input
    x, y, z = symbols(["x", "y", "z"])
    xi0, xi1, xi2 =  symbols(["xi0", "xi1", "xi2"])
    p = swiginac.matrix(nsd, 1, [x,y,z])
    Ginv = G.inverse()

    xi = Ginv*(p-x0)
    ss = function.subs(xi0 == xi[0,0])
    if xi.rows() > 1: ss = ss.subs(xi1 == xi[1,0])
    if xi.rows() > 2: ss = ss.subs(xi2 == xi[2,0])
    
    return polygon.integrate(ss).evalf()

def numeric_integral(polygon, function, order):
    """
    This function computes the integral of the specified function over
    the specified polygon numerically.
    The function may take as arguments the global coordinates (x,y,z)
    as well as coordinates on the reference polygon (xi0, xi1, xi2).
    """

    # TODO: Get symbols as input

    nsd = polygon.no_space_dim()
#   UFCCell is only defined on reference domains ? 
#   OLD code
#    cell = UFCCell(polygon)
#    table = find_quadrature_rule(cell.shape, order, family="gauss")
    shape = ""
    if polygon.str().find("Triangle") >= 0:  
        shape = "triangle"
    if polygon.str().find("Line") >= 0:  
        shape = "interval"
    if polygon.str().find("Tetrahedron") >= 0:  
        shape = "tetrahedron"
    table = find_quadrature_rule(shape, order, family="gauss")
 
    x, y, z = symbols(["x", "y", "z"])
    xi0, xi1, xi2 =  symbols(["xi0", "xi1", "xi2"])
    
    G, x0 = geometry_mapping(polygon) # FIXME: passing Rectangle (or Box) here will yield an affine mapping only correct for straight rectangles
    xi = swiginac.matrix(nsd, 1, [xi0, xi1, xi2])
    global_point = G*xi + x0
    ss = function.subs(x == global_point[0,0])
    if global_point.rows() > 1: ss = ss.subs(y == global_point[1,0])
    if global_point.rows() > 2: ss = ss.subs(z == global_point[2,0])

    sum = 0.0
    for k in range(table.num_points):
        p = table.points[k]
        w = table.weights[k]
        s = ss.subs(xi0 == p[0])
        if len(p) > 1: s = s.subs(xi1 == p[1])
        if len(p) > 2: s = s.subs(xi2 == p[2])
        sum += s*w
    sum *= G.determinant()
    return sum

def integral(polygon, function, integration_mode=True, integration_order=None):
    """
    A small wrapper on top of the functions
    symbolic_integral and numeric_integral.
    """
    # TODO: Support Taylor series expansion as well
    if integration_mode == "symbolic":
        return symbolic_integral(polygon, function)
    elif integration_mode == "quadrature":
        return numeric_integral(polygon, function, integration_order)
    sfc_error("Not implemented integration mode %s" % integration_mode)
