#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

/*

GVerb algorithm designed and implemented by Juhana Sadeharju.
LADSPA implementation and GVerb speeds ups by Steve Harris.

Comments and suggestions should be mailed to Juhana Sadeharju
(kouhia at nic funet fi).

*/

      #include "gverb/gverbdsp.h"
      #include "gverb/gverb.h"
      #include "ladspa-util.h"

#define GVERB_ROOMSIZE                 0
#define GVERB_REVTIME                  1
#define GVERB_DAMPING                  2
#define GVERB_INPUTBANDWIDTH           3
#define GVERB_DRYLEVEL                 4
#define GVERB_EARLYLEVEL               5
#define GVERB_TAILLEVEL                6
#define GVERB_INPUT                    7
#define GVERB_OUTL                     8
#define GVERB_OUTR                     9

LADSPA_Descriptor *gverbDescriptor = NULL;

typedef struct {
	LADSPA_Data *roomsize;
	LADSPA_Data *revtime;
	LADSPA_Data *damping;
	LADSPA_Data *inputbandwidth;
	LADSPA_Data *drylevel;
	LADSPA_Data *earlylevel;
	LADSPA_Data *taillevel;
	LADSPA_Data *input;
	LADSPA_Data *outl;
	LADSPA_Data *outr;
	LADSPA_Data  last_damping;
	LADSPA_Data  last_earlylevel;
	LADSPA_Data  last_inputbandwidth;
	LADSPA_Data  last_revtime;
	LADSPA_Data  last_roomsize;
	LADSPA_Data  last_taillevel;
	ty_gverb *   verb;
	LADSPA_Data run_adding_gain;
} Gverb;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return gverbDescriptor;
	default:
		return NULL;
	}
}

void cleanupGverb(LADSPA_Handle instance) {
	Gverb *plugin_data = (Gverb *)instance;
	gverb_free(plugin_data->verb);
	free(instance);
}

void connectPortGverb(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Gverb *plugin;

	plugin = (Gverb *)instance;
	switch (port) {
	case GVERB_ROOMSIZE:
		plugin->roomsize = data;
		break;
	case GVERB_REVTIME:
		plugin->revtime = data;
		break;
	case GVERB_DAMPING:
		plugin->damping = data;
		break;
	case GVERB_INPUTBANDWIDTH:
		plugin->inputbandwidth = data;
		break;
	case GVERB_DRYLEVEL:
		plugin->drylevel = data;
		break;
	case GVERB_EARLYLEVEL:
		plugin->earlylevel = data;
		break;
	case GVERB_TAILLEVEL:
		plugin->taillevel = data;
		break;
	case GVERB_INPUT:
		plugin->input = data;
		break;
	case GVERB_OUTL:
		plugin->outl = data;
		break;
	case GVERB_OUTR:
		plugin->outr = data;
		break;
	}
}

LADSPA_Handle instantiateGverb(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Gverb *plugin_data = (Gverb *)malloc(sizeof(Gverb));
	LADSPA_Data last_damping;
	LADSPA_Data last_earlylevel;
	LADSPA_Data last_inputbandwidth;
	LADSPA_Data last_revtime;
	LADSPA_Data last_roomsize;
	LADSPA_Data last_taillevel;
	ty_gverb *verb = NULL;

	last_roomsize = 50.0f;
	last_revtime = 7.0f;
	last_damping = 0.5f;
	last_inputbandwidth = 0.5f;
	last_earlylevel = -1.9382f;
	last_taillevel = 0.0f;
	verb = gverb_new(s_rate, 300.0f, last_roomsize, last_revtime, last_damping, 15.0f, last_inputbandwidth, DB_CO(last_earlylevel), DB_CO(last_taillevel));

	plugin_data->last_damping = last_damping;
	plugin_data->last_earlylevel = last_earlylevel;
	plugin_data->last_inputbandwidth = last_inputbandwidth;
	plugin_data->last_revtime = last_revtime;
	plugin_data->last_roomsize = last_roomsize;
	plugin_data->last_taillevel = last_taillevel;
	plugin_data->verb = verb;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runGverb(LADSPA_Handle instance, unsigned long sample_count) {
	Gverb *plugin_data = (Gverb *)instance;

	/* Roomsize (m) (float value) */
	LADSPA_Data roomsize = *(plugin_data->roomsize);

	/* Reverb time (s) (float value) */
	LADSPA_Data revtime = *(plugin_data->revtime);

	/* Damping (float value) */
	LADSPA_Data damping = *(plugin_data->damping);

	/* Input bandwidth (float value) */
	LADSPA_Data inputbandwidth = *(plugin_data->inputbandwidth);

	/* Dry signal level (dB) (float value) */
	LADSPA_Data drylevel = *(plugin_data->drylevel);

	/* Early reflection level (dB) (float value) */
	LADSPA_Data earlylevel = *(plugin_data->earlylevel);

	/* Tail level (dB) (float value) */
	LADSPA_Data taillevel = *(plugin_data->taillevel);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Left output (array of floats of length sample_count) */
	LADSPA_Data *outl = plugin_data->outl;

	/* Right output (array of floats of length sample_count) */
	LADSPA_Data *outr = plugin_data->outr;
	LADSPA_Data last_damping = plugin_data->last_damping;
	LADSPA_Data last_earlylevel = plugin_data->last_earlylevel;
	LADSPA_Data last_inputbandwidth = plugin_data->last_inputbandwidth;
	LADSPA_Data last_revtime = plugin_data->last_revtime;
	LADSPA_Data last_roomsize = plugin_data->last_roomsize;
	LADSPA_Data last_taillevel = plugin_data->last_taillevel;
	ty_gverb * verb = plugin_data->verb;

	unsigned long pos;
	float l, r;
	float dryc = DB_CO(drylevel);

	// Parameter watching, saves slightly on small blocksizes
	if (last_roomsize != roomsize) {
	  gverb_set_roomsize(verb, roomsize);
	}
	if (last_revtime != revtime) {
	  gverb_set_revtime(verb, revtime);
	}
	if (last_damping != damping) {
	  gverb_set_damping(verb, damping);
	}
	if (last_inputbandwidth != inputbandwidth) {
	  gverb_set_inputbandwidth(verb, inputbandwidth);
	}
	if (last_earlylevel != earlylevel) {
	  gverb_set_earlylevel(verb, DB_CO(earlylevel));
	}
	if (last_taillevel != taillevel) {
	  gverb_set_taillevel(verb, DB_CO(taillevel));
	}

	for (pos = 0; pos < sample_count; pos++) {
	  gverb_do(verb, input[pos], &l, &r);
	  buffer_write(outl[pos], l + input[pos] * dryc);
	  buffer_write(outr[pos], r + input[pos] * dryc);
	}

	plugin_data->last_roomsize = roomsize;
	plugin_data->last_revtime = revtime;
	plugin_data->last_damping = damping;
	plugin_data->last_inputbandwidth = inputbandwidth;
	plugin_data->last_earlylevel = earlylevel;
	plugin_data->last_taillevel = taillevel;
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainGverb(LADSPA_Handle instance, LADSPA_Data gain) {
	((Gverb *)instance)->run_adding_gain = gain;
}

void runAddingGverb(LADSPA_Handle instance, unsigned long sample_count) {
	Gverb *plugin_data = (Gverb *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Roomsize (m) (float value) */
	LADSPA_Data roomsize = *(plugin_data->roomsize);

	/* Reverb time (s) (float value) */
	LADSPA_Data revtime = *(plugin_data->revtime);

	/* Damping (float value) */
	LADSPA_Data damping = *(plugin_data->damping);

	/* Input bandwidth (float value) */
	LADSPA_Data inputbandwidth = *(plugin_data->inputbandwidth);

	/* Dry signal level (dB) (float value) */
	LADSPA_Data drylevel = *(plugin_data->drylevel);

	/* Early reflection level (dB) (float value) */
	LADSPA_Data earlylevel = *(plugin_data->earlylevel);

	/* Tail level (dB) (float value) */
	LADSPA_Data taillevel = *(plugin_data->taillevel);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Left output (array of floats of length sample_count) */
	LADSPA_Data *outl = plugin_data->outl;

	/* Right output (array of floats of length sample_count) */
	LADSPA_Data *outr = plugin_data->outr;
	LADSPA_Data last_damping = plugin_data->last_damping;
	LADSPA_Data last_earlylevel = plugin_data->last_earlylevel;
	LADSPA_Data last_inputbandwidth = plugin_data->last_inputbandwidth;
	LADSPA_Data last_revtime = plugin_data->last_revtime;
	LADSPA_Data last_roomsize = plugin_data->last_roomsize;
	LADSPA_Data last_taillevel = plugin_data->last_taillevel;
	ty_gverb * verb = plugin_data->verb;

	unsigned long pos;
	float l, r;
	float dryc = DB_CO(drylevel);

	// Parameter watching, saves slightly on small blocksizes
	if (last_roomsize != roomsize) {
	  gverb_set_roomsize(verb, roomsize);
	}
	if (last_revtime != revtime) {
	  gverb_set_revtime(verb, revtime);
	}
	if (last_damping != damping) {
	  gverb_set_damping(verb, damping);
	}
	if (last_inputbandwidth != inputbandwidth) {
	  gverb_set_inputbandwidth(verb, inputbandwidth);
	}
	if (last_earlylevel != earlylevel) {
	  gverb_set_earlylevel(verb, DB_CO(earlylevel));
	}
	if (last_taillevel != taillevel) {
	  gverb_set_taillevel(verb, DB_CO(taillevel));
	}

	for (pos = 0; pos < sample_count; pos++) {
	  gverb_do(verb, input[pos], &l, &r);
	  buffer_write(outl[pos], l + input[pos] * dryc);
	  buffer_write(outr[pos], r + input[pos] * dryc);
	}

	plugin_data->last_roomsize = roomsize;
	plugin_data->last_revtime = revtime;
	plugin_data->last_damping = damping;
	plugin_data->last_inputbandwidth = inputbandwidth;
	plugin_data->last_earlylevel = earlylevel;
	plugin_data->last_taillevel = taillevel;
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	gverbDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (gverbDescriptor) {
		gverbDescriptor->UniqueID = 1216;
		gverbDescriptor->Label = strdup("gverb");
		gverbDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		gverbDescriptor->Name =
		 strdup("GVerb");
		gverbDescriptor->Maker =
		 strdup("Juhana Sadeharju <kouhia at nic.funet.fi>, LADSPAification by Steve Harris <steve@plugin.org.uk>");
		gverbDescriptor->Copyright =
		 strdup("GPL");
		gverbDescriptor->PortCount = 10;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(10,
		 sizeof(LADSPA_PortDescriptor));
		gverbDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(10,
		 sizeof(LADSPA_PortRangeHint));
		gverbDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(10, sizeof(char*));
		gverbDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Roomsize (m) */
		port_descriptors[GVERB_ROOMSIZE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_ROOMSIZE] =
		 strdup("Roomsize (m)");
		port_range_hints[GVERB_ROOMSIZE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_ROOMSIZE].LowerBound = 1;
		port_range_hints[GVERB_ROOMSIZE].UpperBound = 300;

		/* Parameters for Reverb time (s) */
		port_descriptors[GVERB_REVTIME] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_REVTIME] =
		 strdup("Reverb time (s)");
		port_range_hints[GVERB_REVTIME].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_REVTIME].LowerBound = 0.1;
		port_range_hints[GVERB_REVTIME].UpperBound = 360;

		/* Parameters for Damping */
		port_descriptors[GVERB_DAMPING] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_DAMPING] =
		 strdup("Damping");
		port_range_hints[GVERB_DAMPING].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_DAMPING].LowerBound = 0;
		port_range_hints[GVERB_DAMPING].UpperBound = 1;

		/* Parameters for Input bandwidth */
		port_descriptors[GVERB_INPUTBANDWIDTH] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_INPUTBANDWIDTH] =
		 strdup("Input bandwidth");
		port_range_hints[GVERB_INPUTBANDWIDTH].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_INPUTBANDWIDTH].LowerBound = 0;
		port_range_hints[GVERB_INPUTBANDWIDTH].UpperBound = 1;

		/* Parameters for Dry signal level (dB) */
		port_descriptors[GVERB_DRYLEVEL] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_DRYLEVEL] =
		 strdup("Dry signal level (dB)");
		port_range_hints[GVERB_DRYLEVEL].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_DRYLEVEL].LowerBound = -70;
		port_range_hints[GVERB_DRYLEVEL].UpperBound = 0;

		/* Parameters for Early reflection level (dB) */
		port_descriptors[GVERB_EARLYLEVEL] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_EARLYLEVEL] =
		 strdup("Early reflection level (dB)");
		port_range_hints[GVERB_EARLYLEVEL].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_EARLYLEVEL].LowerBound = -70;
		port_range_hints[GVERB_EARLYLEVEL].UpperBound = 0;

		/* Parameters for Tail level (dB) */
		port_descriptors[GVERB_TAILLEVEL] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[GVERB_TAILLEVEL] =
		 strdup("Tail level (dB)");
		port_range_hints[GVERB_TAILLEVEL].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[GVERB_TAILLEVEL].LowerBound = -70;
		port_range_hints[GVERB_TAILLEVEL].UpperBound = 0;

		/* Parameters for Input */
		port_descriptors[GVERB_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[GVERB_INPUT] =
		 strdup("Input");
		port_range_hints[GVERB_INPUT].HintDescriptor = 0;

		/* Parameters for Left output */
		port_descriptors[GVERB_OUTL] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[GVERB_OUTL] =
		 strdup("Left output");
		port_range_hints[GVERB_OUTL].HintDescriptor = 0;

		/* Parameters for Right output */
		port_descriptors[GVERB_OUTR] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[GVERB_OUTR] =
		 strdup("Right output");
		port_range_hints[GVERB_OUTR].HintDescriptor = 0;

		gverbDescriptor->activate = NULL;
		gverbDescriptor->cleanup = cleanupGverb;
		gverbDescriptor->connect_port = connectPortGverb;
		gverbDescriptor->deactivate = NULL;
		gverbDescriptor->instantiate = instantiateGverb;
		gverbDescriptor->run = runGverb;
		gverbDescriptor->run_adding = runAddingGverb;
		gverbDescriptor->set_run_adding_gain = setRunAddingGainGverb;
	}
}

void _fini() {
	int i;

	if (gverbDescriptor) {
		free((char *)gverbDescriptor->Label);
		free((char *)gverbDescriptor->Name);
		free((char *)gverbDescriptor->Maker);
		free((char *)gverbDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)gverbDescriptor->PortDescriptors);
		for (i = 0; i < gverbDescriptor->PortCount; i++)
			free((char *)(gverbDescriptor->PortNames[i]));
		free((char **)gverbDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)gverbDescriptor->PortRangeHints);
		free(gverbDescriptor);
	}

}
