/*
 * Keyboard stuff
 */

#include <stdio.h>

#include <vncviewer.h>
#include <vgakeyboard.h>
#include <X11/keysym.h>

#define NR_KEYS 128
static int KeyState [NR_KEYS]; 

/*
 * The idea behind the modifications is:  Since the server cannot
 * handle all the nice characters on my keyboard ( - yes, letters
 * with two dots above, the last character is called 'sharp s' and, to
 * complete it, has no corresponding uppercase letter) and the server
 * has a different understanding where which character on my keyboard is
 * we will solve the problem for him.
 *
 * This means: (1) We define our own keyboard layout in keys.h and
 *             (2) We do all the handling of the AltGr key (the right
 *		   side Alt) which is used for characters like '@',
 *		   '\' and '~' on my german keyboard.
 *
 * Yes, (2) might not be what you need for your keyboard.
 *
 * (2) starts with defining a new modifier state: altgr -- 27JUL99wzk
 */

enum { mod_shift, mod_lock, mod_control, mod_mod1, mod_mod2, mod_mod3,
       mod_mod4, mod_mod5, mod_altgr, num_modifiers };

static int state[num_modifiers] = {0,};

/*
 * No longer only normal and shifted keys.  The new keymap included
 * in keys.h supports up to four keystates: normal, shifted, altgr
 * and I don't know.  Linux keymaps support also four.  Perhaps it
 * makes sense for some kind of keyboard.  Look into keys.h for
 * details -- 27JUL99wzk
 */

#include "keys.h"

/*
 * This is source from the original code.
 *
 *	static KeySym keynorm[NR_KEYS], keyshift[NR_KEYS];
 *
 * keys.h was included later in the original version -- 27JUL99wzk
 */


/*
 * While I was editing the code I inserted some messages going to
 * stderr.  Define DEBUG(x) as x to get the messages -- 27JUL99wzk
 */
 
#define	DEBUG(x)


static KeySym keycode2keysym (int);
static void keyboard_handler (int, int);

/*
 * for debugging; this indicates whether the normal or the shifted
 * keysym is selected by keycode2keysym. check keytest.c.
 */
static int shifted;

Bool
svncKeyboardInit(void)
{
    int i;

    if (keyboard_init ()) {
	fprintf (stderr, "Couldn't open keyboard !\n");
	return False;
    }
    keyboard_translatekeys (DONT_CATCH_CTRLC);
    keyboard_seteventhandler (keyboard_handler);

    for (i = 0; i < NR_KEYS; i++)
        KeyState[i] = -1;

    return True;
}

static void keyboard_handler (int, int);


/*
 * HandleKeyboardEvent 
 */
void HandleKeyboardEvent (void)
{
    int key;
    KeySym ks;

    for (key = 0; key < NR_KEYS; key++) {
        if (KeyState[key] != -1) {
            ks = keycode2keysym (key);
            if (ks == XK_BackSpace && state[mod_control] && state[mod_mod1]) {
                ShutdownSvga();
                exit (0);
            }

	    /*
	     * This is important:
	     * We do not send `NoSymbol's to the server.  It can deal with
	     * it, gets worried and won't play with us.  Took me a day to
	     * find it out -- 27JUL99wzk
	     */

	    if (ks != NoSymbol) {
	            SendKeyEvent (keycode2keysym(key), (KeyState[key] == KEY_EVENTPRESS));
       		    KeyState[key] = -1;
		    }
        }
    }
}

static KeySym 
keycode2keysym(int code)
{
    int	m, key, updown;

    updown = KeyState[code];

    /* This is by executing xmodmap -pm and -pke */
    switch(keytable[code][KT_NORMAL]) {
        case XK_Shift_L:	m = mod_shift;		break;
        case XK_Shift_R:	m = mod_shift;		break;
        case XK_Caps_Lock:	m = mod_lock;		break;
        case XK_Control_L:	m = mod_control;	break;
        case XK_Control_R:	m = mod_control;	break;
        case XK_Alt_L:		m = mod_mod1;		break;
        case XK_Alt_R:		m = mod_altgr;		break;
        case XK_Num_Lock:	m = mod_mod2;		break;
        case XK_Scroll_Lock:	m = mod_mod5;		break;
        default:		m = -1;			break;
    }


	DEBUG(
		if (m != -1)
			fprintf (stderr, "code= %d, event= %d\n", code, KeyState[code]);
		)

	
    if (KeyState[code] == KEY_EVENTPRESS) {
        switch (m) {
            case mod_lock:              /* caps */
            case mod_mod2:              /* num */
                if (!(state[m] & 2)) { state[m] ^= 1 ; state[m] |= 2; }
                break;

            case mod_control:           /* control */
            case mod_shift:             /* shift */  
            case mod_mod1:              /* alt */    
                state[m] = 1;
                break;
		
	    case mod_altgr:		/* altgr or alt_r */
                state[m] = 1;
                break;
        }
    } else {
        switch (m) {
            case mod_lock:              /* caps */
            case mod_mod2:              /* num */
                state[m] &= ~2;
                break;
		
            case mod_control:           /* control */
            case mod_shift:             /* shift */
            case mod_mod1:              /* alt */
                state[m] = 0;
                break;
		
	    case mod_altgr:		/* altgr or alt_r */
                state[m] = 0;
                break;
        }
    }
    

    /*
     * The AltGr (or Alt-Right) key is the problem.  We do not send it
     * to the server -- 27JUL99wzk
     */

    key = keytable[code][KT_NORMAL];
    if (key == XK_Alt_L) {
    	DEBUG( fprintf (stderr, "dropping alt key: %d, %04X (%d/%d)\n", key, key, code, updown); )
    	return (NoSymbol);
	}


    /*
     * I'm not sure if like the code too (see below), but ... wow, yeah,
     * looks nice.  Perhaps a little bit difficult to read, but who cares
     * if it works?
     *
     * AltGr keys are however not shifted -- 27JUL99wzk
     */

    if ((state[mod_altgr] == 1)  ||  (state[mod_control] && state[mod_mod1])) {
    	int	key;

	key = keytable[code][KT_ALTGR];
	DEBUG( fprintf (stderr, "altgr key: %d, %04X (%d/%d)\n", key, key, code, updown); )

	return (key);
	}


    /*
     * WOW! I LIKE THIS CODE!!!
     *
     * funda: keynorm[] and keyshift[] give the normal and shifted
     * KeySyms of a keycode respectively. This shifted sense is reversed
     * by caps lock for a-z and numlock for keypad keys 0-9 and dot.
     *
     * Hence  ((numlock & keypad) OR (caps & a-z)) XOR shift
     * means you select the shifted KeySym, else select the normal one.
     *
     * All in one expression!
     */

    return 
        (shifted = ((((state[mod_mod2] & 1)
            & (((keytable[code][KT_SHIFTED] >= XK_KP_0) & (keytable[code][KT_SHIFTED] <= XK_KP_9))
		| (keytable[code][KT_SHIFTED] == XK_KP_Decimal)))
        | ((state[mod_lock] & 1)
            & (keytable[code][KT_NORMAL] >= XK_a) & (keytable[code][KT_NORMAL] <= XK_z)))
        ^ state[mod_shift])) ? keytable[code][KT_SHIFTED] : keytable[code][KT_NORMAL];
}

static void 
keyboard_handler (int keycode, int newstate)
{
    KeyState[keycode] = newstate;
}

