/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleSubViewer2.h"
#include <iostream>
#include <fstream>
#include <glibmm/ustring.h>
#include "utility.h"
#include "RegEx.h"

/*
 *
 */
Glib::ustring SubtitleSubViewer2::get_name()
{
	return "SubViewer 2.0";
}

/*
 *
 */
Glib::ustring SubtitleSubViewer2::get_extension()
{
	return "sub";
}

/*
 *
 */
bool SubtitleSubViewer2::check(const std::string &line)
{
	static RegEx ex(
				"[0-9][0-9]:[0-9][0-9]:[0-9][0-9].[0-9]{2}"
				","
				"[0-9][0-9]:[0-9][0-9]:[0-9][0-9].[0-9]{2}");

	return ex.exec(line);
}

/*
 *
 */
SubtitleSubViewer2::SubtitleSubViewer2(Document* doc)
:SubtitleFormat(doc)
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}

/*
 *
 */
SubtitleSubViewer2::~SubtitleSubViewer2()
{
	se_debug(SE_DEBUG_LOADER | SE_DEBUG_SAVER);
}


/*
 *
 */
bool SubtitleSubViewer2::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_LOADER);

	SubtitleFormat::open(filename,encoding);
	
	std::ifstream file(filename.c_str());
	if(!file)
	{
		throw SubtitleException("SubtitleSubViewer2", _("I can't open this file."));
	}

	std::string line;
	while(!file.eof() && std::getline(file, line))
	{
		// lecture du temps
		if(std::getline(file, line))
		{
			int start[4], end[4];

			if(sscanf(line.c_str(), "%d:%d:%d.%d,%d:%d:%d.%d",
						&start[0], &start[1], &start[2], &start[3], 
						&end[0], &end[1], &end[2], &end[3]) == 8)
			{
				Glib::ustring text;
				int count = 0; // permet de compter le nombre de ligne

				while(std::getline(file, line))
				{
					line = check_end_char(charset_to_utf8(line));

					if(line.size()==0)
						break;
					else
					{
						if(count > 0) // s'il y a deja une ligne alors on utlise le | pour le retour a la ligne
							text += get_newline();

						text += line;
						++count;
					}
				}
		
				SubtitleModifier subtitle(m_subtitleModel->append());

				characters_to_newline(text, "[br]");

				subtitle.set_text(text);
				subtitle.set_start(SubtitleTime(start[0], start[1], start[2], start[3]));
				subtitle.set_end(SubtitleTime(end[0], end[1], end[2], end[3]));
			}
		}
	}

	file.close();

	return true;
}

/*
 *
 */
bool SubtitleSubViewer2::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	se_debug(SE_DEBUG_SAVER);

	SubtitleFormat::save(filename,encoding);

	std::ofstream file(filename.c_str());
	if(!file)
		throw SubtitleException("SubtitleSubViewer2", _("I can't open this file."));

	file << "[INFORMATION]" << std::endl;
	file << "[TITLE]" << utf8_to_charset(m_scriptInfo->Title) << std::endl;
	file << "[AUTHOR]" << utf8_to_charset(m_scriptInfo->OriginalEditing) << std::endl;
	file << "[COMMENT]" << utf8_to_charset(m_scriptInfo->Comment) << std::endl;
	file << "[END INFORMATION]" << std::endl;
	file << "[SUBTITLE]" << std::endl;

	SubtitleColumnRecorder column;
	Glib::ustring text;
	
	Gtk::TreeNodeChildren rows = m_subtitleModel->children();
	for(Gtk::TreeIter it = rows.begin(); it; ++it)
	{
		SubtitleModifier subtitle(it);
		SubtitleTime start = subtitle.get_start();
		SubtitleTime end = subtitle.get_end();

		gchar *tmp = g_strdup_printf("%.2i:%.2i:%.2i.%.2i,%.2i:%.2i:%.2i.%.2i",
				start.hours, start.mins, start.secs, start.msecs,
				end.hours, end.mins, end.secs, end.msecs);

		file << tmp << std::endl;

		g_free(tmp);

		text = subtitle.get_text();

		newline_to_characters(text, "[br]");

		file << utf8_to_charset(text) << std::endl;
		file << std::endl;
	}
	
	file.close();
	return true;
}
