# Feed wrapper module
# Copyright (c) 2002 Juri Pakaste
# You may use and distribute this software under the terms of the
# GNU General Public License, version 2 or later

import Event
import FeedItems
import FeedDataRouter
import Config
from error import log

class Feed(object, Event.SignalEmitter):
    "A Feed object stores information set by user about a RSS feed."

    DEFAULT = -1
    STATUS_IDLE = 0
    STATUS_POLLING = 1

    __slots__ = ('_title', '_location', '_username', '_password', '_parsed',
                 '__save_fields', '_items', '_slots',
                 '_id', '_channel_description',
                 '_channel_title', '_channel_link', '_channel_copyright',
                 'channel_lbd', 'channel_editor', 'channel_webmaster',
                 'channel_creator', '_previous_etag',
                 '_error', '_process_status', 'router', 'sticky', '_parent',
                 '_items_stored', '_poll_freq', '_last_poll', '_n_items_unread')

    __save_fields = (('_title', ""), ('_location', ""), ('_username', ""),
                     ('_password', ""), ('_id', ""),
                     ('_channel_description', ""), ('_channel_title', ""),
                     ('_channel_link', ""), ('_channel_copyright', ""),
                     ('channel_creator', ""),
                     ('_previous_etag', ""), ('_error', None),
                     ('_items_stored', DEFAULT),
                     ('_poll_freq', DEFAULT),
                     ('_last_poll', 0),
                     ('_n_items_unread', 0))


    # use one of the factory functions below instead of this directly
    def __init__(self, title="", location="", username="", password=""):
        Event.SignalEmitter.__init__(self)
        self.initialize_slots(Event.NewItemsSignal,
                              Event.ItemsAddedSignal,
                              Event.ItemReadSignal,
                              Event.ItemStickySignal,
                              Event.ItemDeletedSignal,
                              Event.AllItemsReadSignal,
                              Event.RefreshFeedDisplaySignal,
                              Event.FeedsChangedSignal,
                              Event.FeedPolledSignal,
                              Event.FeedStatusChangedSignal,
                              Event.FeedErrorStatusChangedSignal,
                              Event.SaveFeedsSignal)
        self._title = title
        self._channel_description = ""
        self._channel_title = ""
        self._channel_link = ""
        self._channel_copyright = ""
        self.channel_lbd = None
        self.channel_editor = ""
        self.channel_webmaster = ""
        self.channel_creator = ""
        self._location = location
        self._username = username
        self._password = password
        self._parsed = None
        self._items = FeedItems.FeedItems(self)
        self._previous_etag = None
        self._error = None
        self._process_status = self.STATUS_IDLE
        self.router = FeedDataRouter.FeedDataRouter(self)
        self._parent = None
        self._items_stored = Feed.DEFAULT
        self._poll_freq = Feed.DEFAULT
        self._last_poll = 0
        self._n_items_unread = 0
        return

    def set_parsed(self, parsed):
        self._parsed = parsed
        return

    def get_parsed(self):
        return self._parsed

    # ooh, properties are crunchy
    parsed = property(get_parsed, set_parsed, None,
                      "A ParsedSummary object generated from the summary file")

    def get_id(self):
        return self._id

    id = property(get_id, None, None, "")
    
    def get_title(self):
        return self._title

    def set_title(self, title):
        self._title = title
        self.emit_signal(Event.FeedsChangedSignal(self, self))
        return

    title = property(get_title, set_title, None,
                     "The title of this Feed (as defined by user)")

    def set_access_info(self, (location, username, password)):
        self._location = location
        self._username = username
        self._password = password
        self.emit_signal(Event.FeedsChangedSignal(self, self))
        return

    def get_access_info(self):
        return (self._location, self._username, self._password)

    access_info = property(get_access_info, set_access_info, None,
                           "A tuple of location, username, password")

    def set_location(self, location):
        self._location = location

    def get_location(self):
        return self._location

    location = property(get_location, set_location, None, None)

    def get_channel_title(self):
        return self._channel_title

    def set_channel_title(self, t):
        changed = self._channel_title != t
        self._channel_title = t
        if changed:
            self.emit_signal(Event.FeedsChangedSignal(self, self))

    channel_title = property(get_channel_title, set_channel_title, None, "")

    def get_channel_description(self):
        return self._channel_description

    def set_channel_description(self, t):
        changed = self._channel_description != t
        self._channel_description = t
        if changed:
            self.emit_signal(Event.FeedsChangedSignal(self, self))

    channel_description = property(get_channel_description, set_channel_description, None, "")

    def get_channel_link(self):
        return self._channel_link

    def set_channel_link(self, t):
        changed = self._channel_link != t
        self._channel_link = t
        if changed:
            self.emit_signal(Event.FeedsChangedSignal(self, self))

    channel_link = property(get_channel_link, set_channel_link, None, "")

    def get_channel_copyright(self):
        return self._channel_copyright

    def set_channel_copyright(self, t):
        changed = self._channel_copyright != t
        self._channel_copyright = t
        if changed:
            self.emit_signal(Event.FeedsChangedSignal(self, self))

    channel_copyright = property(get_channel_copyright, set_channel_copyright, None, "")

    def get_number_of_items_stored(self):
        return self._items_stored

    def set_number_of_items_stored(self, num=None):
        if self._items_stored != num:
            self._items_stored = num
            self.emit_signal(Event.FeedsChangedSignal(self, self))
        return

    number_of_items_stored = property(get_number_of_items_stored,
                                      set_number_of_items_stored)

    def set_poll_frequency(self, poll_frequency):
        if self._poll_freq != poll_frequency:
            self._poll_freq = poll_frequency
            self.emit_signal(Event.FeedsChangedSignal(self, self))
        return

    def get_poll_frequency(self):
        return self._poll_freq

    poll_frequency = property(get_poll_frequency, set_poll_frequency)

    def get_last_poll(self):
        return self._last_poll

    def set_last_poll(self, time):
        self._last_poll = time
        self.emit_signal(Event.FeedsChangedSignal(self, self))

    last_poll = property(get_last_poll, set_last_poll)

    def get_next_refresh(self):
        """ return the feed's next refresh (time)"""
        if self._poll_freq == self.DEFAULT:
            increment = Config.get_instance().poll_frequency
        else:
            increment = self._poll_freq
        if increment > 0:
            return self.last_poll + increment
        else:
            return None

    next_refresh = property(get_next_refresh)

    def get_n_items_unread(self):
        # currently used by FeedItems
        return self._n_items_unread

    def set_n_items_unread(self, n):
        # currently used by FeedItems
        self._n_items_unread = n
        self.emit_signal(Event.SaveFeedsSignal(self))

    n_items_unread = property(get_n_items_unread, set_n_items_unread, None)

    def dump(self):
        fl = {}
        for f, default in self.__save_fields:
            fl[f] = self.__getattribute__(f)
        return fl

    def undump(self, dict):
        for f, default in self.__save_fields:
            self.__setattr__(f, dict.get(f, default))
        return

    def poll_done(self):
        self.emit_signal(Event.FeedPolledSignal(self))

    ###
    # These are for the use of FeedItems

    def signal_new_items(self, items):
        self.emit_signal(Event.NewItemsSignal(self, items))

    def signal_added_items(self, items):
        #log("-------------------------------------------- enter")
        #log("--------------------------------------------------")
        self.emit_signal(Event.ItemsAddedSignal(self, items))

    def signal_deleted_item(self, item):
        self.emit_signal(Event.ItemDeletedSignal(self, item))

    def signal_refresh_display(self):
        self.emit_signal(Event.RefreshFeedDisplaySignal(self))

    def signal_all_items_read(self, items):
        self.emit_signal(Event.AllItemsReadSignal(self, items))

    def forward_signal(self, signal):
        self.emit_signal(signal)

    # End of interface for FeedItems
    ######

    ###
    # And these proxy FeedItems methods

    def add_items(self, items):
        return self._items.add_items(items)

    def restore_items(self, items):
        return self._items.restore_items(items)

    def get_items(self):
        return self._items.get_items()

    def get_next_item(self, item):
        return self._items.get_next_item(item)

    def get_previous_item(self, item):
        return self._items.get_previous_item(item)

    def get_first_item(self):
        return self._items.get_first_item()

    def get_last_item(self):
        return self._items.get_last_item()

    def get_item(self, id):
        return self._items.get_item(id)

    def get_item_index(self, item):
        return self._items.get_item_index(item)

    items = property(get_items, None, None, "Stored items, newest first")

    def get_number_of_unread(self):
        return self._items.number_of_unread

    def get_number_of_items(self):
        return self._items.number_of_items

    number_of_unread = property(get_number_of_unread, None, None, "")
    number_of_items = property(get_number_of_items, None, None, "")

    def mark_all_read(self):
        return self._items.mark_all_read()

    # End of FeedItemsProxying
    ##############
    
    def __str__(self):
        return "Feed '%s' from %s" % (self._title, self._location)

    def get_previous_etag(self):
        return self._previous_etag

    def set_previous_etag(self, etag):
        self._previous_etag = etag

    previous_etag = property(get_previous_etag, set_previous_etag, None, "")

    def set_error(self, error):
        self._error = error
        self.emit_signal(Event.FeedErrorStatusChangedSignal(self))

    def get_error(self):
        return self._error

    error = property(get_error, set_error, None, "")

    def set_process_status(self, status):
        if status != self._process_status:
            self._process_status = status
            self.emit_signal(Event.FeedStatusChangedSignal(self))

    def get_process_status(self):
        return self._process_status

    process_status = property(get_process_status, set_process_status, None, "")

    def get_parent(self):
        return self._parent

    def set_parent(self, parent):
        self._parent = parent

    parent = property(get_parent, set_parent)

    def create_new_feed(klass, title, location="", username="", password=""):
        f = klass()
        f._title = title
        f._location = location
        f._id = Config.get_instance().next_feed_id_seq()
        f._username = username
        f._password = password
        return f
    create_new_feed = classmethod(create_new_feed)

    def create_empty_feed(klass):
        f = klass()
        return f
    create_empty_feed = classmethod(create_empty_feed)
