#!@@PERL@@
#
# WWW based spong acknowledgment tool.  When a spong event occurs (or will
# occur), you can use this tool to acknowledge that you know there is a
# problem.  You can provide text that will be seen by others looking at the
# even (via a spong display program).  You can specify a time limit that the
# problem will occur.  If a problem has been acknowledged, you will no longer
# receive notifications of the problem, and the display programs will show the
# status of the service as "blue"
#
# History
# (1) Created, pulled out of www-spong program (Ed 07-28-97)
#
# $Id: www-spong-ack.pl,v 1.9 2001/04/17 02:58:12 sljohnson Exp $

use CGI;
use Sys::Hostname;
use Socket;
use POSIX;
use Time::Local;

# Load our configuration variables, including the user specified configuration
# information (spong.conf, spong.hosts, and spong.groups files).

$|++;
$conf_file   = "@@ETCDIR@@/spong.conf";
$hosts_file  = "@@ETCDIR@@/spong.hosts";
$groups_file = "@@ETCDIR@@/spong.groups";
($HOST)      = gethostbyname(&Sys::Hostname::hostname());
$HOST        =~ tr/A-Z/a-z/;

%HUMANS      = ();
%HOSTS       = ();
%GROUPS      = ();
 
&load_config_files();  # Loads the user specified configuration information

# Pull apart the commands that we are given as part of our URL path, and call
# the appropriate function.

$cmd = $ENV{'PATH_INFO'};

if( $cmd =~ m!^/delete/(.+?)-(\w+)-(\d+)$! ) { &remove( $1, $2, $3 ); exit; }

if( $cmd =~ m!^/help$! )        { &help();        exit; }
if( $cmd =~ m!^/ack-doit$! )    { &ack_doit();    exit; }
if( $cmd =~ m!^/(.+)/(.+)$! )   { &ack( $1, $2 ); exit; }
if( $cmd =~ m!^/(.+)/$! )       { &ack( $1 );     exit; }
if( $cmd =~ m!^/(.+)$! )        { &ack( $1 );     exit; }
if( $cmd eq "" || $cmd eq "/" ) { &ack();         exit; }



# This brings up a page that allows you to acknowledge problems.  This changes
# the color of an acknowledged service or host to blue, and allows you to
# specify when the machine will be ready, and provide a text message indicating
# what the problem is.

sub ack {
   my( $group, $service ) = @_;
   my( @hostlist ) = hostlist( $group );
   my $host;

   $group = "all" unless $group;

   header();
   print "<form action=\"$main::WWWACK/ack-doit\" method=get>\n";
   print "<input type=hidden name=group value=\"$group\">\n";
   print "<font size=+2><b>Acknowledge Problems</b></font>\n<hr>\n";

   print "Fill out the form below to acknowledge a problem or set of ";
   print "problems on a given host.  <a href=\"$main::WWWACK/help\">Help</a> ";
   print "is provided on the correct format for the information you need ";
   print "to supply in the fields below.<p>\n";

   print "<b><font size=+1>Current Acknowledgements</font></b><p>";

   print &acklist( $group ), "<p>\n";

   print "<b><font size=+1>New Acknowledgement</font></b><p>";

   print "<b>Host:</b> <font size=-1>(The fully qualified domain name of ";
   print "the host you are acknowledging)</font><br>\n";

   if( $#hostlist >= 1 ) {
      print "<select name=\"host\">\n";
      foreach $host ( @hostlist ) { print "<option>$host\n";}
      print "</select></td></tr><tr>\n";
   } else {
      print "<input type=text name=host size=30 value=\"$hostlist[0]\">\n";
      print "</td></tr><tr>\n";
   }

   print "<p><b>Service:</b> <font size=-1>(The service you are acknowledging";
   print ", or \"<i>all</i>\" to signify all services)</font><br>\n";
   print "<input type=text name=services size=30 value=\"$service\">";

   print "<p><b>Duration:</b> <font size=-1>(When service will be ";
   print "available, +5h, +2d, 18:00, 7/28/97)</font><br>\n";
   print "<input type=text name=duration size=30>\n";

   print "<p><b>Email:</b> <font size=-1>(Email address of the contact person";
   print ")</font><br>\n";
   print "<input type=text name=user size=30>\n";

   print "<p><b>Message:</b> <font size=-1>(Description of the ";
   print "problem, details about the solution)</font><br>\n";
   print "<textarea name=message wrap=virtual cols=40 rows=3></textarea>\n";

   print "</center><br><input type=submit value=\" Create \">\n";
   print "<input type=reset value=\" Reset \">\n";
   print "<hr></form>\n";

   footer();
}

# This is the action part of the above form, it takes the information that is
# supplied, and calls the spong-ack program with that information.  I have a 
# lot of smarts here that should probably be pulled out.

sub ack_doit {
   my $cgi      = new CGI;
   my $host     = $cgi->param( 'host' );
   my $services = $cgi->param( 'services' );
   my $date     = $cgi->param( 'duration' );
   my $user     = $cgi->param( 'user' );
   my $message  = $cgi->param( 'message' );
   my $group    = $cgi->param( 'group' );

   if( $services eq "" ) { &error( "Service name not supplied." ); }
   if( $host eq "" ) { &error( "Host name not supplied." ); }
   if( $date eq "" ) { &error( "Duration not supplied." ); }

   my $time = 0;
   my( $sec, $min, $hour ) = ( 0, 0, 0 );
   my( $mday, $mon, $year ) = (localtime(time()))[3,4,5];
   my %secs = ( 'm', 60, 'h', 3600, 'd', 86400, 'w', 604800 );
   
   if( $date =~ /^(\d+)$/ ) { $time eq $date; $ok = 1; } # WTF!!!

   # Check for a duration in the format +X(mhdw)...
   if( $date =~ /^\+\s*(\d+)\s*([mhdw])$/ ) {
      $time = time() + ( $1 * $secs{$2} ); $ok = 1; } 

   # Check for a duration in the format HH:MM:SS...
   if( $date =~ /\b(\d+):(\d+):(\d+)\b/ ) {
      ( $hour, $min, $sec ) = ( $1, $2, $3 ); $ok = 1; }
   
   # Check for a duration in the format MM/DD/YYYY...
   if( $date =~ /\b(\d+)\/(\d+)\/(\d+)\b/ ) {
      ( $mon, $mday, $year ) = ( $1, $2, $3 );
      if( $year < 100 ) { 
	 &error( "Dates must be expressed in the format MM/DD/YYYY" ); }
      $mon--; $year -= 1900; $ok = 1;
   } 

   if( $ok ) {
      $time = timelocal( $sec, $min, $hour, $mday, $mon, $year) unless $time;
   } else {
      &error( "Invalid duration specifier, view the " .
	      "<a href=\"$main::WWWACK/help\">Help</a> screen for more " .
	      "information on valid formats." );
   }

   my $r = &ack_send( $SPONGSERVER, $host, $services, $time, $message, $user );

   # Print a message for the user letting them know if they were successful in
   # their acknowledgment.

   if( $r ne "ok" ) {
      &error( "Could not acknowledge $host/$service!<p>Message: $results");
   } else {
      &header();
      print "<font size=+2><b>Acknowledge Problems - Success</b></font><hr>\n";
      print "Your acknowledgment has been successfully registered.  Please ";
      print "press <a href=\"$main::WWWSPONG/group/$group\" target=_top>here";
      print "</a> to refresh the spong display.<p>\n";
   }
}



# When we delete an Acknowledgement, we just do our thing, and if it was
# removed successfully we just redirect them back to the page they came
# from, otherwise we give them an error message.

sub remove {
   my( $host, $service, $end ) = @_;
   my $id = "$host-$service-$end";

   my $r = &del_send( $SPONGSERVER, $id );

   # Print a message for the user letting them know if they were successful in
   # their acknowledgment.

   if( $r ne "ok" ) {
      &error( "Could not acknowledge $host/$service!<p>Message: $r");
   } else {

      # If we can tell where they came from, then just redirect them back,
      # otherwise just show them a screen which says things are ok.

      if( $ENV{'HTTP_REFERER'} ) {
	 print "Location: $ENV{'HTTP_REFERER'}\n\n"; 
      } else {
	 &header();
	 print "<font size=+2><b>Acknowledge Problems - Success</b></font>";
	 print "<hr>\n";
	 print "The acknowledgment has been successfully deleted.  ";
	 print "Please press <a href=\"$main::WWWSPONG\" ";
	 print "target=_top>here</a> to refresh the spong display.<p>\n";
      }
   }
}


# This prints out a help message which describes all the fields.

sub help {
   &header();

print <<'_EOF_';

<font size=+2><b>Acknowledge Problems: Help</b></font>
<hr>

This program sends an acknowledgment to the spong server so that others can
tell that a specific problem is being worked on, or at least known more about
the problem.  Here is a description of the fields that you need to supply for
the acknowledgment to be registered.<p>

<ul>
<table width=90% border=0 cellpadding=1 cellspacing=1>
<tr><td align=left valign=top><b>Host</b> &nbsp</td>
<td align=left valign=top>
The fully qualified (e.g. strobe.weeg.uiowa.edu, not strobe) domain name of
the host having the problems you are acknowledging.<p>
</td></tr>

<tr><td align=left valign=top><b>Service</b> &nbsp</td>
<td align=left valign=top>
The service or services (separated by ",") or all services (represented by the
string 'all') that you are acknowledging.<p>
</td></tr>

<tr><td align=left valign=top><b>Duration</b> &nbsp</td>
<td align=left valign=top>
The time that the acknowledgment will last.  This can be either an offset
"+1h, +3d, +1w", or an absolute date and/or time indicator "12/25/1997
14:00:00".  The date needs to be a year 2000 valid string, and the time needs
to be in 24 hour format.<p>
</td></tr>

<tr><td align=left valign=top><b>Email</b> &nbsp</td>
<td align=left valign=top>
The email address of the person that can be contacted if there are further
questions.<p>
</td></tr>

<tr><td align=left valign=top><b>Message</b> &nbsp</td>
<td align=left valign=top>
A message that will appear to those viewing the acknowledgement, this can
contain any text and generally would provide additional details about the
problem.<p>
</td></tr></table>
</ul>
<hr>

_EOF_

}

# This prints an error message when the user enters some invalid data, or when
# the spong-server returns an error to this problem when the acknowledgment is
# sent.

sub error {
   my $message = shift;

   &header();
   print "<font size=+2><b>Acknowledge Problems: Error!</b></font>\n<hr>\n";
   print "An error has occurred.  The text of the error message appears ";
   print "below.<p>";
   print "<b>Error:</b> $message<p>\n";
   print "If possible, please return to the previous screen and correct the ";
   print "problem.  If you are unable to correct the problem, please contact ";
   print "the Spong administrator.\n<hr>\n";

   exit(0);
}

# ---------------------------------------------------------------------------
# Private/Internal functions
# ---------------------------------------------------------------------------

# This function just loads in all the configuration information from the 
# spong.conf, spong.hosts, and spong.groups files.

sub load_config_files {
   my( $evalme, $inhosts, $ingroups );

   require $conf_file || die "Can't load $conf_file: $!";
   if( -f "$conf_file.$HOST" ) {
      require "$conf_file.$HOST" || die "Can't load $conf_file.$HOST: $!";
   } else {
      my $tmp = (split( /\./, $HOST ))[0];
      if( -f "$conf_file.$tmp" ) { # for lazy typist
	 require "$conf_file.$tmp" || die "Can't load $conf_file.$tmp: $!";
      }
   }

   # Read in the spong.hosts file.  We are a little nasty here in that we do
   # some junk to scan through the file so that we can maintain the order of
   # the hosts as they appear in the file.

   open( HOSTS, $hosts_file ) || die "Can't load $hosts_file: $!";
   while( <HOSTS> ) {
      $evalme .= $_;
      if( /^\s*%HOSTS\s*=\s*\(/ ) { $inhosts = 1; }
      if( $inhosts && /^\s*[\'\"]?([^\s\'\"]+)[\'\"]?\s*\=\>\s*\{/ ) {
	 push( @HOSTS_LIST, $1 ); }
   }
   close( HOSTS );
   eval $evalme || die "Invalid spong.hosts file: $@";

   # Fallback, if we didn't read things correctly...
   
   if( sort ( @HOSTS_LIST ) != sort ( keys %HOSTS ) ) { 
      @HOSTS_LIST = sort keys %HOSTS; }

   # Do the same thing for the groups file.

   $evalme = "";
   open( GROUPS, $groups_file ) || die "Can't load $groups_file: $!";
   while( <GROUPS> ) {
      $evalme .= $_;
      if( /^\s*%GROUPS\s*=\s*\(/ ) { $ingroups = 1; }
      if( $ingroups && /^\s*[\'\"]?([^\s\'\"]+)[\'\"]?\s*\=\>\s*\{/ ) {
	 push( @GROUPS_LIST, $1 ); }
   }
   close( GROUPS );
   eval $evalme || die "Invalid spong.groups file: $@";

   if( sort ( @GROUPS_LIST ) != sort ( keys %GROUPS ) ) { 
      @GROUPS_LIST = sort keys %GROUPS; }
}


# ----------------------------------------------------------------------------
# Display helper functions
# ----------------------------------------------------------------------------

# These allow users to easily customize some aspects of spong, by providing 
# their own header and footer information for each page.

sub header { 
   print "Content-type: text/html\n\n"; 
   &show( "header" ) if -f "$main::WWWHTML/header.html"; }

sub footer { 
   &show( "footer" ) if -f "$main::WWWHTML/footer.html"; }

sub show {
   my $file = shift;
   my $show = $main::WWWSPONG . "/help";

   if( -f "$main::WWWHTML/$file.html" ) {
      open( FILE, "$main::WWWHTML/$file.html" );
      while( <FILE> ) { s/!!WWWSHOW!!/$show/g; print $_; }
      close( FILE );
   }
}


# Converts a command line representation of a host list to a list of hostnames

sub hostlist {
   my $hosts = shift;
   my @hostlist;

   if( $hosts eq "all" || $hosts eq "" ) {
      @hostlist = @HOSTS_LIST;
   } elsif( ref( $GROUPS{$hosts} ) eq "HASH" ) {
      foreach( @{$GROUPS{$hosts}->{'members'}} ) { push( @hostlist, $_ ); }
   } else {
      @hostlist = split( /\,/, $hosts );
   }
   
   return @hostlist;
}

# This finds out all of the outstanding acknowledgements, and builds a little
# HTML table that allows the user to either delete or update an Ack.

sub acklist {
   my( $group ) = @_;
   my $results = &query( $SPONGSERVER, "acks", $group, "text", "special" ); 
   my $ack;

   if( $results =~ /^\s*$/ ) { return "No current acknowledgments."; }

   my $str = "<table width=100% border=0 cellspacing=0 cellpadding=0>";
   foreach $ack ( split( /\n/, $results ) ) {
      ($host, $service, $time) = (split( /:/, $ack ));
      my $id = "$host-$service-$time";

      $str .= "<tr><td valign=top><a href=\"";
      $str .= $main::WWWACK . "/delete/$id\">Delete</a></td>";

      $str .= "<td>$host/$service</td>";
      $str .= "<td>" . POSIX::strftime( "%D, %H:%M", localtime($time) );
      $str .= "</td></tr>";
   }

   $str .= "</table>\n";

   return $str;
}


# ----------------------------------------------------------------------------
# Networking functions...
# ----------------------------------------------------------------------------

# This function sends a query to the spong server.  It takes the results it
# gets back based on the user's query and returns the string back to the 
# code that called this function.

sub query {
   my( $addr, $query, $hosts, $display, $view, $other ) = @_;
   my( $iaddr, $paddr, $proto, $line, $ip, $ok, $msg );

   if( $addr =~ /^\s*((\d+\.){3}\d+)\s*$/ ) {
      $ip = $addr;
   } else {
      my( @addrs ) = (gethostbyname($addr))[4];
      my( $a, $b, $c, $d ) = unpack( 'C4', $addrs[0] );
      $ip = "$a.$b.$c.$d";
   }

   $iaddr = inet_aton( $ip ) || die "no host: $host\n";
   $paddr = sockaddr_in( $SPONG_QUERY_PORT, $iaddr );
   $proto = getprotobyname( 'tcp' );
   
   # Set an alarm so that if we can't connect "immediately" it times out.

   $SIG{'ALRM'} = sub { die };
   alarm(30);

   eval <<'_EOM_';
   socket( SOCK, PF_INET, SOCK_STREAM, $proto ) || die "socket: $!";
   connect( SOCK, $paddr )                      || die "connect: $!";
   select((select(SOCK), $| = 1)[0]);
   print SOCK "$query [$hosts] $display $view $other\n";
   while( <SOCK> ) { $msg .= $_; }
   close( SOCK )                                || die "close: $!";
   $ok = 1;
_EOM_

   alarm(0);
   
   return $msg if $ok;
   return "Can't connect to spong server!\n" if ! $ok;
}


# This function sends an acknowledgment message to the Spong server.  It
# reports what host, and what services on that host (*) for all services that
# are to be acknowledge.  It supplies a time (in int format) that the
# acknowledgment is good until.  It provides the user who made the
# acknowledgment, and last a message that can be viewed by others that might
# provide additional information about the problem/acknowledgment.

sub ack_send {
   my( $addr, $host, $service, $time, $message, $user ) = @_;
   my( $iaddr, $paddr, $proto, $line, $ip, $ok );
   my $results = "ok";
   my $now = time();

   if( $addr =~ /^\s*((\d+\.){3}\d+)\s*$/ ) {
      $ip = $addr;
   } else {
      my( @addrs ) = (gethostbyname($addr))[4];
      my( $a, $b, $c, $d ) = unpack( 'C4', $addrs[0] );
      $ip = "$a.$b.$c.$d";
   }

   $iaddr = inet_aton( $ip ) || die "no host: $host\n";
   $paddr = sockaddr_in( $SPONG_UPDATE_PORT, $iaddr );
   $proto = getprotobyname( 'tcp' );
   
   # Set an alarm so that if we can't connect "immediately" it times out.

   $SIG{'ALRM'} = sub { die };
   alarm(30);

   eval <<'_EOM_';
   socket( SOCK, PF_INET, SOCK_STREAM, $proto ) || die "socket: $!";
   connect( SOCK, $paddr )                      || die "connect: $!";
   select((select(SOCK), $| = 1)[0]);
   print SOCK "ack $host $service $now $time $user\n";
   print SOCK "$message\n";
   close( SOCK )                                || die "close: $!";
   $ok = 1;
_EOM_

   alarm(0);
   return $results if $ok;
   return "can not connect to spong server" if ! $ok;
}

# This function sends an delete-acknowledgment message to the Spong server.  
# It provides the spong server an ID which indicates what acknowledgement that
# we want deleted.

sub del_send {
   my( $addr, $id ) = @_;
   my( $iaddr, $paddr, $proto, $line, $ip, $ok );
   my $results = "ok";
   my $now = time();

   if( $addr =~ /^\s*((\d+\.){3}\d+)\s*$/ ) {
      $ip = $addr;
   } else {
      my( @addrs ) = (gethostbyname($addr))[4];
      my( $a, $b, $c, $d ) = unpack( 'C4', $addrs[0] );
      $ip = "$a.$b.$c.$d";
   }

   $iaddr = inet_aton( $ip ) || die "no host: $addr\n";
   $paddr = sockaddr_in( $SPONG_UPDATE_PORT, $iaddr );
   $proto = getprotobyname( 'tcp' );
   
   # Set an alarm so that if we can't connect "immediately" it times out.

   $SIG{'ALRM'} = sub { die };
   alarm(30);

   eval <<'_EOM_';
   socket( SOCK, PF_INET, SOCK_STREAM, $proto ) || die "socket: $!";
   connect( SOCK, $paddr )                      || die "connect: $!";
   select((select(SOCK), $| = 1)[0]);
   print SOCK "ack-del $id\n";
   close( SOCK )                                || die "close: $!";
   $ok = 1;
_EOM_

   alarm(0);
   return $results if $ok;
   return "can not connect to spong server" if ! $ok;
}

