/* specter_LOGEMU.c
 *
 * specter output target for syslog logging emulation
 *
 * This target produces a file which looks the same like the syslog-entries
 * of the LOG target.
 *
 * (C) 2000-2001 by Harald Welte <laforge@gnumonks.org>
 *
 *  11 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      Fixed for specter.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include "printpkt.h"


#ifndef LOGEMU_DEFAULT
#define LOGEMU_DEFAULT	"/var/log/specter.logemu"
#endif


static config_entry_t my_config[] = {
	{ .key = "sync", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "logfile", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE,
		.u = { .string = LOGEMU_DEFAULT } },
	{ .key = "ip_options", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "tcp_options", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "tcp_seq", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "mac_header", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
};


struct my_data {
	FILE *of;
	int flags;
};



static void fini_logemu(config_entry_t *ce, void *data)
{
	struct my_data *md = data;

	if (md->of != stdout && md->of != stderr) {
		fclose(md->of);
	}

	free(data);
}


static void *init_logemu(config_entry_t *ce)
{
	struct my_data *data;

	if (printpkt_init() == -1) {
		specter_log(SPECTER_FATAL, "printpkt module init failure.\n");
		exit(EXIT_FAILURE);
	}

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	/* open logfile */
	if (!strcmp(GET_CE(ce,1)->u.string, "stdout")) {
		data->of = stdout;
	}
	else if (!strcmp(GET_CE(ce,1)->u.string, "stderr")) {
		data->of = stderr;
	}
	else {
		data->of = fopen(GET_CE(ce,1)->u.string, "a");
		if (!data->of) {
			specter_log(SPECTER_FATAL, "Couldn't open \"%s\": %s.\n", 
				GET_CE(ce,1)->u.string, strerror(errno));
			return NULL;
		}
	}

	/* set flags for quicker reference later */
	data->flags = 0x0;
	if (GET_CE(ce, 2)->u.value)
		data->flags |= PRINTPKT_IP_OPT;
	if (GET_CE(ce, 3)->u.value)
		data->flags |= PRINTPKT_TCP_OPT;
	if (GET_CE(ce, 4)->u.value)
		data->flags |= PRINTPKT_TCP_SEQ;
	if (GET_CE(ce, 5)->u.value)
		data->flags |= PRINTPKT_MAC_HDR;

	return data;
}


static int output_logemu(config_entry_t *ce, void *data)
{
	struct my_data *md = data;
	static char buf[4096];

	if (printpkt_print(buf, PRINTPKT_PREFIX|md->flags))
		fprintf(md->of, "%s", buf);

	if (GET_CE(ce,0)->u.value)
		fflush(md->of);

	return 0;
}


static int sighup_handler(config_entry_t *ce, void *data, int signal)
{
	struct my_data *md = data;

	switch (signal) {
	case SIGHUP:
		specter_log(SPECTER_INFO, "Reopening logfile.\n");

		if (md->of == stderr || md->of == stdout)
			break;

		fclose(md->of);
		md->of = fopen(GET_CE(ce,1)->u.string, "a");
		if (!md->of) {
			specter_log(SPECTER_FATAL, "Couldn't open \"%s\": %s.\n",
				GET_CE(ce,1)->u.string, strerror(errno));
			return -1;
		}
		break;
	default:
		break;
	}

	return 0;
}


static specter_output_t logemu_op = {
	.name = "logemu",
	.ce_base = my_config,
	.ce_num = 6,
	.init = &init_logemu,
	.fini = &fini_logemu,
	.output = &output_logemu,
	.signal = &sighup_handler
};


void _init(void)
{
	if (register_output(&logemu_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

