/*
 * SpanDSP - a series of DSP components for telephony
 *
 * v29rx.c - ITU V.29 modem receive part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2003 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: v29rx.c,v 1.66 2006/01/21 05:10:28 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <inttypes.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/async.h"
#include "spandsp/power_meter.h"
#include "spandsp/arctan2.h"
#include "spandsp/complex.h"
#include "spandsp/dds.h"
#include "spandsp/complex_filters.h"

#include "spandsp/v29rx.h"

/* Segments of the training sequence */
#define V29_TRAINING_SEG_2_LEN      128
#define V29_TRAINING_SEG_3_LEN      384
#define V29_TRAINING_SEG_4_LEN      48

#define EQUALIZER_DELTA             (0.25*2.0)

enum
{
    TRAINING_STAGE_NORMAL_OPERATION = 0,
    TRAINING_STAGE_SYMBOL_ACQUISITION,
    TRAINING_STAGE_LOG_PHASE,
    TRAINING_STAGE_WAIT_FOR_CDCD,
    TRAINING_STAGE_TRAIN_ON_CDCD,
    TRAINING_STAGE_TRAIN_ON_CDCD_AND_TEST,
    TRAINING_STAGE_TEST_ONES,
    TRAINING_STAGE_PARKED
};

static const complex_t v29_constellation[16] =
{
    { 3.0,  0.0},   /*   0deg low  */
    { 1.0,  1.0},   /*  45deg low  */
    { 0.0,  3.0},   /*  90deg low  */
    {-1.0,  1.0},   /* 135deg low  */
    {-3.0,  0.0},   /* 180deg low  */
    {-1.0, -1.0},   /* 225deg low  */
    { 0.0, -3.0},   /* 270deg low  */
    { 1.0, -1.0},   /* 315deg low  */
    { 5.0,  0.0},   /*   0deg high */
    { 3.0,  3.0},   /*  45deg high */
    { 0.0,  5.0},   /*  90deg high */
    {-3.0,  3.0},   /* 135deg high */
    {-5.0,  0.0},   /* 180deg high */
    {-3.0, -3.0},   /* 225deg high */
    { 0.0, -5.0},   /* 270deg high */
    { 3.0, -3.0}    /* 315deg high */
};

static const uint8_t space_map_9600[20][20] =
{
    {13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11},
    {13, 13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 11},
    {13, 13, 13, 13, 13, 13, 13,  4,  4,  4,  4,  4,  4, 11, 11, 11, 11, 11, 11, 11},
    {13, 13, 13, 13, 13, 13, 13,  4,  4,  4,  4,  4,  4, 11, 11, 11, 11, 11, 11, 11},
    {13, 13, 13, 13, 13, 13, 13,  4,  4,  4,  4,  4,  4, 11, 11, 11, 11, 11, 11, 11},
    {13, 13, 13, 13, 13, 13, 13,  5,  4,  4,  4,  4,  3, 11, 11, 11, 11, 11, 11, 11},
    {14, 13, 13, 13, 13, 13,  5,  5,  5,  5,  3,  3,  3,  3, 11, 11, 11, 11, 11, 10},
    {14, 14,  6,  6,  6,  5,  5,  5,  5,  5,  3,  3,  3,  3,  3,  2,  2,  2, 10, 10},
    {14, 14,  6,  6,  6,  6,  5,  5,  5,  5,  3,  3,  3,  3,  2,  2,  2,  2, 10, 10},
    {14, 14,  6,  6,  6,  6,  5,  5,  5,  5,  3,  3,  3,  3,  2,  2,  2,  2, 10, 10},
    {14, 14,  6,  6,  6,  6,  7,  7,  7,  7,  1,  1,  1,  1,  2,  2,  2,  2, 10, 10},
    {14, 14,  6,  6,  6,  6,  7,  7,  7,  7,  1,  1,  1,  1,  2,  2,  2,  2, 10, 10},
    {14, 14,  6,  6,  6,  7,  7,  7,  7,  7,  1,  1,  1,  1,  1,  2,  2,  2, 10, 10},
    {14, 15, 15, 15, 15, 15,  7,  7,  7,  7,  1,  1,  1,  1,  9,  9,  9,  9,  9, 10},
    {15, 15, 15, 15, 15, 15, 15,  7,  0,  0,  0,  0,  1,  9,  9,  9,  9,  9,  9,  9},
    {15, 15, 15, 15, 15, 15, 15,  0,  0,  0,  0,  0,  0,  9,  9,  9,  9,  9,  9,  9},
    {15, 15, 15, 15, 15, 15, 15,  0,  0,  0,  0,  0,  0,  9,  9,  9,  9,  9,  9,  9},
    {15, 15, 15, 15, 15, 15, 15,  0,  0,  0,  0,  0,  0,  9,  9,  9,  9,  9,  9,  9},
    {15, 15, 15, 15, 15, 15, 15,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9},
    {15, 15, 15, 15, 15, 15,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9}
};

#if 0
/* Raised root cosine pulse shaping; Beta = 0.25; 4 symbols either
   side of the centre. We cannot simplify this by using only half
   the filter, as each variant are each skewed by n/PULSESHAPER_COEFF_SETS of a
   sample. Only one is symmetric. */
#define PULSESHAPER_GAIN        (1.5964543012e+02/16.0)
#define PULSESHAPER_COEFF_SETS  48
static const float pulseshaper[PULSESHAPER_COEFF_SETS][V29RX_FILTER_STEPS] =
{
    {
         0.0281155862,          /* Filter 0 */
         0.0138439626,
        -0.0285615173,
        -0.0521677994,
        -0.0162777823,
         0.0603982517,
         0.0999497832,
         0.0342926437,
        -0.1119970377,
        -0.2077231549,
        -0.1023409733,
         0.2441526325,
         0.6974182428,
         1.0170945955,
         1.0170945955,
         0.6974182428,
         0.2441526325,
        -0.1023409733,
        -0.2077231549,
        -0.1119970377,
         0.0342926437,
         0.0999497832,
         0.0603982517,
        -0.0162777823,
        -0.0521677994,
        -0.0285615173,
         0.0138439626
    },
    {
         0.0282352761,          /* Filter 1 */
         0.0131357839,
        -0.0294413322,
        -0.0520897765,
        -0.0149001870,
         0.0619195582,
         0.0997733266,
         0.0317675166,
        -0.1150869926,
        -0.2079908873,
        -0.0974155706,
         0.2532528331,
         0.7063015080,
         1.0207295896,
         1.0133133958,
         0.6884748817,
         0.2351031244,
        -0.1071546901,
        -0.2073637831,
        -0.1088860894,
         0.0367783473,
         0.1000768493,
         0.0588605710,
        -0.0176351372,
        -0.0522175536,
        -0.0276733679,
         0.0145381864
    },
    {
         0.0283381118,          /* Filter 2 */
         0.0124139562,
        -0.0303122695,
        -0.0519833054,
        -0.0135028846,
         0.0634236139,
         0.0995471438,
         0.0292037719,
        -0.1181543779,
        -0.2081658535,
        -0.0923789224,
         0.2624017105,
         0.7151223517,
         1.0242173737,
         1.0093870344,
         0.6794737613,
         0.2261063011,
        -0.1118563180,
        -0.2069139221,
        -0.1057557185,
         0.0392238457,
         0.1001548808,
         0.0573073940,
        -0.0189717322,
        -0.0522392321,
        -0.0267774280,
         0.0152181591
    },
    {
         0.0284239271,          /* Filter 3 */
         0.0116787947,
        -0.0311737872,
        -0.0518482195,
        -0.0120864223,
         0.0649095437,
         0.0992709191,
         0.0266022387,
        -0.1211976129,
        -0.2082469503,
        -0.0872315071,
         0.2715972261,
         0.7238784604,
         1.0275569840,
         1.0053165950,
         0.6704172289,
         0.2171641311,
        -0.1164454920,
        -0.2063747447,
        -0.1026074897,
         0.0416283812,
         0.1001842531,
         0.0557395992,
        -0.0202870615,
        -0.0522330408,
        -0.0258742422,
         0.0158835939
    },
    {
         0.0284925649,          /* Filter 4 */
         0.0109306237,
        -0.0320253446,
        -0.0516843660,
        -0.0106513604,
         0.0663764748,
         0.0989443574,
         0.0239637700,
        -0.1242151125,
        -0.2082330984,
        -0.0819738404,
         0.2808373192,
         0.7325675332,
         1.0307474970,
         1.0011032002,
         0.6613076413,
         0.2082785582,
        -0.1209218845,
        -0.2057474457,
        -0.0994429615,
         0.0439912202,
         0.1001653608,
         0.0541580658,
        -0.0215806336,
        -0.0521991983,
        -0.0249643553,
         0.0165342138
    },
    {
         0.0285438766,          /* Filter 5 */
         0.0101697770,
        -0.0328664028,
        -0.0514916056,
        -0.0091982726,
         0.0678235367,
         0.0985671844,
         0.0212892419,
        -0.1272052882,
        -0.2081232437,
        -0.0766064761,
         0.2901199075,
         0.7411872828,
         1.0337880300,
         0.9967480115,
         0.6521473646,
         0.1994515012,
        -0.1252852054,
        -0.2050332414,
        -0.0962636853,
         0.0463116534,
         0.1000986179,
         0.0525636728,
        -0.0228519713,
        -0.0521379359,
        -0.0240483120,
         0.0171697517
    },
    {
         0.0285777227,          /* Filter 6 */
         0.0093965971,
        -0.0336964248,
        -0.0512698128,
        -0.0077277457,
         0.0692498618,
         0.0981391468,
         0.0185795538,
        -0.1301665484,
        -0.2079163571,
        -0.0711300052,
         0.2994428875,
         0.7497354363,
         1.0366777417,
         0.9922522288,
         0.6429387725,
         0.1906848531,
        -0.1295352022,
        -0.2042333694,
        -0.0930712050,
         0.0485889958,
         0.0999844569,
         0.0509572988,
        -0.0241006115,
        -0.0520494973,
        -0.0231266569,
         0.0177899501
    },
    {
         0.0285939730,          /* Filter 7 */
         0.0086114355,
        -0.0345148761,
        -0.0510188762,
        -0.0062403794,
         0.0706545858,
         0.0976600127,
         0.0158356279,
        -0.1330972988,
        -0.2076114352,
        -0.0655450567,
         0.3088041356,
         0.7582097354,
         1.0394158319,
         0.9876170899,
         0.6336842466,
         0.1819804810,
        -0.1336716600,
        -0.2033490873,
        -0.0898670563,
         0.0508225871,
         0.0998233285,
         0.0493398215,
        -0.0253261060,
        -0.0519341383,
        -0.0221999338,
         0.0183945618
    },
    {
         0.0285925065,          /* Filter 8 */
         0.0078146525,
        -0.0353212243,
        -0.0507386984,
        -0.0047367858,
         0.0720368483,
         0.0971295716,
         0.0130584092,
        -0.1359959437,
        -0.2072075009,
        -0.0598522969,
         0.3182015083,
         0.7666079380,
         1.0420015423,
         0.9828438702,
         0.6243861749,
         0.1733402251,
        -0.1376944010,
        -0.2023816729,
        -0.0866527665,
         0.0530117917,
         0.0996157019,
         0.0477121176,
        -0.0265280208,
        -0.0517921269,
        -0.0212686859,
         0.0189833495
    },
    {
         0.0285732113,          /* Filter 9 */
         0.0070066169,
        -0.0361149404,
        -0.0504291963,
        -0.0032175899,
         0.0733957928,
         0.0965476351,
         0.0102488652,
        -0.1388608857,
        -0.2067036039,
        -0.0540524296,
         0.3276328431,
         0.7749278181,
         1.0444341565,
         0.9779338822,
         0.6150469512,
         0.1647658985,
        -0.1416032847,
        -0.2013324232,
        -0.0834298535,
         0.0551559984,
         0.0993620635,
         0.0460750618,
        -0.0277059371,
        -0.0516237428,
        -0.0203334550,
         0.0195560859
    },
    {
         0.0285359852,          /* Filter 10 */
         0.0061877060,
        -0.0368954982,
        -0.0500903010,
        -0.0016834284,
         0.0747305678,
         0.0959140366,
         0.0074079856,
        -0.1416905272,
        -0.2060988208,
        -0.0481461961,
         0.3370959589,
         0.7831671671,
         1.0467130004,
         0.9728884750,
         0.6056689744,
         0.1562592865,
        -0.1453982077,
        -0.2002026540,
        -0.0801998256,
         0.0572546211,
         0.0990629176,
         0.0444295272,
        -0.0288594506,
        -0.0514292773,
        -0.0193947816,
         0.0201125542
    },
    {
         0.0284807357,          /* Filter 11 */
         0.0053583054,
        -0.0376623750,
        -0.0497219582,
        -0.0001349506,
         0.0760403263,
         0.0952286318,
         0.0045367821,
        -0.1444832701,
        -0.2053922561,
        -0.0421343748,
         0.3465886571,
         0.7913237942,
         1.0488374421,
         0.9677090342,
         0.5962546478,
         0.1478221462,
        -0.1490791036,
        -0.1989936992,
        -0.0769641811,
         0.0593070986,
         0.0987187853,
         0.0427763841,
        -0.0299881721,
        -0.0512090332,
        -0.0184532046,
         0.0206525475
    },
    {
         0.0284073795,          /* Filter 12 */
         0.0045188089,
        -0.0384150518,
        -0.0493241283,
         0.0014271830,
         0.0773242271,
         0.0944912991,
         0.0016362882,
        -0.1472375170,
        -0.2045830423,
        -0.0360177816,
         0.3561087217,
         0.7993955274,
         1.0508068925,
         0.9623969809,
         0.5868063781,
         0.1394562060,
        -0.1526459430,
        -0.1977069103,
        -0.0737244073,
         0.0613128943,
         0.0983302043,
         0.0411165002,
        -0.0310917276,
        -0.0509633241,
        -0.0175092610,
         0.0211758696
    },
    {
         0.0283158435,          /* Filter 13 */
         0.0036696181,
        -0.0391530137,
        -0.0488967864,
         0.0030022999,
         0.0785814349,
         0.0937019393,
        -0.0012924414,
        -0.1499516718,
        -0.2036703407,
        -0.0297972690,
         0.3656539202,
         0.8073802140,
         1.0526208053,
         0.9569537715,
         0.5773265749,
         0.1311631652,
        -0.1560987329,
        -0.1963436557,
        -0.0704819805,
         0.0632714968,
         0.0978977287,
         0.0394507399,
        -0.0321697579,
        -0.0506924750,
        -0.0165634853,
         0.0216823343
    },
    {
         0.0282060643,          /* Filter 14 */
         0.0028111425,
        -0.0398757500,
        -0.0484399222,
         0.0045897167,
         0.0798111203,
         0.0928604764,
        -0.0042483303,
        -0.1526241400,
        -0.2026533412,
        -0.0234737269,
         0.3752220046,
         0.8152757214,
         1.0542786771,
         0.9513808974,
         0.5678176496,
         0.1229446934,
        -0.1594375171,
        -0.1949053204,
        -0.0672383651,
         0.0651824199,
         0.0974219288,
         0.0377799642,
        -0.0332219194,
        -0.0503968213,
        -0.0156164098,
         0.0221717660
    },
    {
         0.0280779883,          /* Filter 15 */
         0.0019437993,
        -0.0405827546,
        -0.0479535407,
         0.0061887386,
         0.0810124611,
         0.0919668574,
        -0.0072302817,
        -0.1552533293,
        -0.2015312634,
        -0.0170480815,
         0.3848107115,
         0.8230799377,
         1.0557800480,
         0.9456798841,
         0.5582820148,
         0.1148024303,
        -0.1626623755,
        -0.1933933051,
        -0.0639950135,
         0.0670452021,
         0.0969033901,
         0.0361050299,
        -0.0342478835,
        -0.0500767089,
        -0.0146685640,
         0.0226439994
    },
    {
         0.0279315720,          /* Filter 16 */
         0.0010680132,
        -0.0412735264,
        -0.0474376615,
         0.0077986603,
         0.0821846419,
         0.0910210527,
        -0.0102371780,
        -0.1578376507,
        -0.2003033567,
        -0.0105212960,
         0.3944177633,
         0.8307907726,
         1.0571245011,
         0.9398522908,
         0.5487220838,
         0.1067379848,
        -0.1657734244,
        -0.1918090257,
        -0.0607533654,
         0.0688594072,
         0.0963427137,
         0.0344267896,
        -0.0352473370,
        -0.0497324940,
        -0.0137204743,
         0.0230988800
    },
    {
         0.0277667821,          /* Filter 17 */
         0.0001842159,
        -0.0419475693,
        -0.0468923197,
         0.0094187658,
         0.0833268551,
         0.0900230559,
        -0.0132678812,
        -0.1603755186,
        -0.1989689008,
        -0.0038943696,
         0.4040408685,
         0.8384061579,
         1.0583116633,
         0.9338997104,
         0.5391402692,
         0.0987529352,
        -0.1687708158,
        -0.1901539128,
        -0.0575148472,
         0.0706246240,
         0.0957405154,
         0.0327460911,
        -0.0362199823,
        -0.0493645428,
        -0.0127726640,
         0.0235362635
    },
    {
         0.0275835953,          /* Filter 18 */
        -0.0007071536,
        -0.0426043926,
        -0.0463175652,
         0.0110483291,
         0.0844383008,
         0.0889728845,
        -0.0163212335,
        -0.1628653515,
        -0.1975272061,
         0.0028316618,
         0.4136777228,
         0.8459240484,
         1.0593412048,
         0.9278237679,
         0.5295389827,
         0.0908488281,
        -0.1716547376,
        -0.1884294110,
        -0.0542808717,
         0.0723404663,
         0.0950974255,
         0.0310637773,
        -0.0371655369,
        -0.0489732311,
        -0.0118256527,
         0.0239560162
    },
    {
         0.0273819985,          /* Filter 19 */
        -0.0016056495,
        -0.0432435116,
        -0.0457134635,
         0.0126866141,
         0.0855181878,
         0.0878705798,
        -0.0193960573,
        -0.1653055730,
        -0.1959776139,
         0.0096557267,
         0.4233260095,
         0.8533424222,
         1.0602128397,
         0.9216261210,
         0.5199206341,
         0.0830271786,
        -0.1744254131,
        -0.1866369781,
        -0.0510528377,
         0.0740065730,
         0.0944140884,
         0.0293806855,
        -0.0380837342,
        -0.0485589444,
        -0.0108799564,
         0.0243580149
    },
    {
         0.0271619890,          /* Filter 20 */
        -0.0025108192,
        -0.0438644473,
        -0.0450800952,
         0.0143328749,
         0.0865657336,
         0.0867162070,
        -0.0224911559,
        -0.1676946117,
        -0.1943194973,
         0.0165767175,
         0.4329834002,
         0.8606592820,
         1.0609263260,
         0.9153084588,
         0.5102876305,
         0.0752894697,
        -0.1770831009,
        -0.1847780851,
        -0.0478321295,
         0.0756226081,
         0.0936911623,
         0.0276976473,
        -0.0389743227,
        -0.0481220773,
        -0.0099360869,
         0.0247421472
    },
    {
         0.0269235742,          /* Filter 21 */
        -0.0034222037,
        -0.0444667272,
        -0.0444175563,
         0.0159863564,
         0.0875801649,
         0.0855098554,
        -0.0256053134,
        -0.1700309027,
        -0.1925522609,
         0.0235934914,
         0.4426475558,
         0.8678726550,
         1.0614814654,
         0.9088725014,
         0.5006423756,
         0.0676371518,
        -0.1796280945,
        -0.1828542149,
        -0.0446201163,
         0.0771882605,
         0.0929293185,
         0.0260154882,
        -0.0398370668,
        -0.0476630334,
        -0.0089945521,
         0.0251083110
    },
    {
         0.0266667722,          /* Filter 22 */
        -0.0043393378,
        -0.0450498853,
        -0.0437259583,
         0.0176462945,
         0.0885607182,
         0.0842516385,
        -0.0287372955,
        -0.1723128874,
        -0.1906753418,
         0.0307048699,
         0.4523161271,
         0.8749805941,
         1.0618781038,
         0.9023199995,
         0.4909872689,
         0.0600716423,
        -0.1820607223,
        -0.1808668621,
        -0.0414181519,
         0.0787032441,
         0.0921292414,
         0.0243350270,
        -0.0406717462,
        -0.0471822251,
        -0.0080558550,
         0.0254564149
    },
    {
         0.0263916112,          /* Filter 23 */
        -0.0052617500,
        -0.0456134625,
        -0.0430054281,
         0.0193119161,
         0.0895066400,
         0.0829416942,
        -0.0318858496,
        -0.1745390147,
        -0.1886882093,
         0.0379096396,
         0.4619867553,
         0.8819811784,
         1.0621161309,
         0.8956527337,
         0.4813247051,
         0.0525943255,
        -0.1843813471,
        -0.1788175322,
        -0.0382275743,
         0.0801672978,
         0.0912916280,
         0.0226570758,
        -0.0414781564,
        -0.0466800732,
        -0.0071204942,
         0.0257863781
    },
    {
         0.0260981301,          /* Filter 24 */
        -0.0061889631,
        -0.0461570066,
        -0.0422561082,
         0.0209824397,
         0.0904171873,
         0.0815801850,
        -0.0350497051,
        -0.1767077412,
        -0.1865903658,
         0.0452065524,
         0.4716570731,
         0.8888725138,
         1.0621954806,
         0.8888725138,
         0.4716570731,
         0.0452065524,
        -0.1865903658,
        -0.1767077412,
        -0.0350497051,
         0.0815801850,
         0.0904171873,
         0.0209824397,
        -0.0422561082,
        -0.0461570066,
        -0.0061889631,
         0.0260981301
    },
    {
         0.0257863781,          /* Filter 25 */
        -0.0071204942,
        -0.0466800732,
        -0.0414781564,
         0.0226570758,
         0.0912916280,
         0.0801672978,
        -0.0382275743,
        -0.1788175322,
        -0.1843813471,
         0.0525943255,
         0.4813247051,
         0.8956527337,
         1.0621161309,
         0.8819811784,
         0.4619867553,
         0.0379096396,
        -0.1886882093,
        -0.1745390147,
        -0.0318858496,
         0.0829416942,
         0.0895066400,
         0.0193119161,
        -0.0430054281,
        -0.0456134625,
        -0.0052617500,
         0.0263916112
    },
    {
         0.0254564149,          /* Filter 26 */
        -0.0080558550,
        -0.0471822251,
        -0.0406717462,
         0.0243350270,
         0.0921292414,
         0.0787032441,
        -0.0414181519,
        -0.1808668621,
        -0.1820607223,
         0.0600716423,
         0.4909872689,
         0.9023199995,
         1.0618781038,
         0.8749805941,
         0.4523161271,
         0.0307048699,
        -0.1906753418,
        -0.1723128874,
        -0.0287372955,
         0.0842516385,
         0.0885607182,
         0.0176462945,
        -0.0437259583,
        -0.0450498853,
        -0.0043393378,
         0.0266667722
    },
    {
         0.0251083110,          /* Filter 27 */
        -0.0089945521,
        -0.0476630334,
        -0.0398370668,
         0.0260154882,
         0.0929293185,
         0.0771882605,
        -0.0446201163,
        -0.1828542149,
        -0.1796280945,
         0.0676371518,
         0.5006423756,
         0.9088725014,
         1.0614814654,
         0.8678726550,
         0.4426475558,
         0.0235934914,
        -0.1925522609,
        -0.1700309027,
        -0.0256053134,
         0.0855098554,
         0.0875801649,
         0.0159863564,
        -0.0444175563,
        -0.0444667272,
        -0.0034222037,
         0.0269235742
    },
    {
         0.0247421472,          /* Filter 28 */
        -0.0099360869,
        -0.0481220773,
        -0.0389743227,
         0.0276976473,
         0.0936911623,
         0.0756226081,
        -0.0478321295,
        -0.1847780851,
        -0.1770831009,
         0.0752894697,
         0.5102876305,
         0.9153084588,
         1.0609263260,
         0.8606592820,
         0.4329834002,
         0.0165767175,
        -0.1943194973,
        -0.1676946117,
        -0.0224911559,
         0.0867162070,
         0.0865657336,
         0.0143328749,
        -0.0450800952,
        -0.0438644473,
        -0.0025108192,
         0.0271619890
    },
    {
         0.0243580149,          /* Filter 29 */
        -0.0108799564,
        -0.0485589444,
        -0.0380837342,
         0.0293806855,
         0.0944140884,
         0.0740065730,
        -0.0510528377,
        -0.1866369781,
        -0.1744254131,
         0.0830271786,
         0.5199206341,
         0.9216261210,
         1.0602128397,
         0.8533424222,
         0.4233260095,
         0.0096557267,
        -0.1959776139,
        -0.1653055730,
        -0.0193960573,
         0.0878705798,
         0.0855181878,
         0.0126866141,
        -0.0457134635,
        -0.0432435116,
        -0.0016056495,
         0.0273819985
    },
    {
         0.0239560162,          /* Filter 30 */
        -0.0118256527,
        -0.0489732311,
        -0.0371655369,
         0.0310637773,
         0.0950974255,
         0.0723404663,
        -0.0542808717,
        -0.1884294110,
        -0.1716547376,
         0.0908488281,
         0.5295389827,
         0.9278237679,
         1.0593412048,
         0.8459240484,
         0.4136777228,
         0.0028316618,
        -0.1975272061,
        -0.1628653515,
        -0.0163212335,
         0.0889728845,
         0.0844383008,
         0.0110483291,
        -0.0463175652,
        -0.0426043926,
        -0.0007071536,
         0.0275835953
    },
    {
         0.0235362635,          /* Filter 31 */
        -0.0127726640,
        -0.0493645428,
        -0.0362199823,
         0.0327460911,
         0.0957405154,
         0.0706246240,
        -0.0575148472,
        -0.1901539128,
        -0.1687708158,
         0.0987529352,
         0.5391402692,
         0.9338997104,
         1.0583116633,
         0.8384061579,
         0.4040408685,
        -0.0038943696,
        -0.1989689008,
        -0.1603755186,
        -0.0132678812,
         0.0900230559,
         0.0833268551,
         0.0094187658,
        -0.0468923197,
        -0.0419475693,
         0.0001842159,
         0.0277667821
    },
    {
         0.0230988800,          /* Filter 32 */
        -0.0137204743,
        -0.0497324940,
        -0.0352473370,
         0.0344267896,
         0.0963427137,
         0.0688594072,
        -0.0607533654,
        -0.1918090257,
        -0.1657734244,
         0.1067379848,
         0.5487220838,
         0.9398522908,
         1.0571245011,
         0.8307907726,
         0.3944177633,
        -0.0105212960,
        -0.2003033567,
        -0.1578376507,
        -0.0102371780,
         0.0910210527,
         0.0821846419,
         0.0077986603,
        -0.0474376615,
        -0.0412735264,
         0.0010680132,
         0.0279315720
    },
    {
         0.0226439994,          /* Filter 33 */
        -0.0146685640,
        -0.0500767089,
        -0.0342478835,
         0.0361050299,
         0.0969033901,
         0.0670452021,
        -0.0639950135,
        -0.1933933051,
        -0.1626623755,
         0.1148024303,
         0.5582820148,
         0.9456798841,
         1.0557800480,
         0.8230799377,
         0.3848107115,
        -0.0170480815,
        -0.2015312634,
        -0.1552533293,
        -0.0072302817,
         0.0919668574,
         0.0810124611,
         0.0061887386,
        -0.0479535407,
        -0.0405827546,
         0.0019437993,
         0.0280779883
    },
    {
         0.0221717660,          /* Filter 34 */
        -0.0156164098,
        -0.0503968213,
        -0.0332219194,
         0.0377799642,
         0.0974219288,
         0.0651824199,
        -0.0672383651,
        -0.1949053204,
        -0.1594375171,
         0.1229446934,
         0.5678176496,
         0.9513808974,
         1.0542786771,
         0.8152757214,
         0.3752220046,
        -0.0234737269,
        -0.2026533412,
        -0.1526241400,
        -0.0042483303,
         0.0928604764,
         0.0798111203,
         0.0045897167,
        -0.0484399222,
        -0.0398757500,
         0.0028111425,
         0.0282060643
    },
    {
         0.0216823343,          /* Filter 35 */
        -0.0165634853,
        -0.0506924750,
        -0.0321697579,
         0.0394507399,
         0.0978977287,
         0.0632714968,
        -0.0704819805,
        -0.1963436557,
        -0.1560987329,
         0.1311631652,
         0.5773265749,
         0.9569537715,
         1.0526208053,
         0.8073802140,
         0.3656539202,
        -0.0297972690,
        -0.2036703407,
        -0.1499516718,
        -0.0012924414,
         0.0937019393,
         0.0785814349,
         0.0030022999,
        -0.0488967864,
        -0.0391530137,
         0.0036696181,
         0.0283158435
    },
    {
         0.0211758696,          /* Filter 36 */
        -0.0175092610,
        -0.0509633241,
        -0.0310917276,
         0.0411165002,
         0.0983302043,
         0.0613128943,
        -0.0737244073,
        -0.1977069103,
        -0.1526459430,
         0.1394562060,
         0.5868063781,
         0.9623969809,
         1.0508068925,
         0.7993955274,
         0.3561087217,
        -0.0360177816,
        -0.2045830423,
        -0.1472375170,
         0.0016362882,
         0.0944912991,
         0.0773242271,
         0.0014271830,
        -0.0493241283,
        -0.0384150518,
         0.0045188089,
         0.0284073795
    },
    {
         0.0206525475,          /* Filter 37 */
        -0.0184532046,
        -0.0512090332,
        -0.0299881721,
         0.0427763841,
         0.0987187853,
         0.0593070986,
        -0.0769641811,
        -0.1989936992,
        -0.1490791036,
         0.1478221462,
         0.5962546478,
         0.9677090342,
         1.0488374421,
         0.7913237942,
         0.3465886571,
        -0.0421343748,
        -0.2053922561,
        -0.1444832701,
         0.0045367821,
         0.0952286318,
         0.0760403263,
        -0.0001349506,
        -0.0497219582,
        -0.0376623750,
         0.0053583054,
         0.0284807357
    },
    {
         0.0201125542,          /* Filter 38 */
        -0.0193947816,
        -0.0514292773,
        -0.0288594506,
         0.0444295272,
         0.0990629176,
         0.0572546211,
        -0.0801998256,
        -0.2002026540,
        -0.1453982077,
         0.1562592865,
         0.6056689744,
         0.9728884750,
         1.0467130004,
         0.7831671671,
         0.3370959589,
        -0.0481461961,
        -0.2060988208,
        -0.1416905272,
         0.0074079856,
         0.0959140366,
         0.0747305678,
        -0.0016834284,
        -0.0500903010,
        -0.0368954982,
         0.0061877060,
         0.0285359852
    },
    {
         0.0195560859,          /* Filter 39 */
        -0.0203334550,
        -0.0516237428,
        -0.0277059371,
         0.0460750618,
         0.0993620635,
         0.0551559984,
        -0.0834298535,
        -0.2013324232,
        -0.1416032847,
         0.1647658985,
         0.6150469512,
         0.9779338822,
         1.0444341565,
         0.7749278181,
         0.3276328431,
        -0.0540524296,
        -0.2067036039,
        -0.1388608857,
         0.0102488652,
         0.0965476351,
         0.0733957928,
        -0.0032175899,
        -0.0504291963,
        -0.0361149404,
         0.0070066169,
         0.0285732113
    },
    {
         0.0189833495,          /* Filter 40 */
        -0.0212686859,
        -0.0517921269,
        -0.0265280208,
         0.0477121176,
         0.0996157019,
         0.0530117917,
        -0.0866527665,
        -0.2023816729,
        -0.1376944010,
         0.1733402251,
         0.6243861749,
         0.9828438702,
         1.0420015423,
         0.7666079380,
         0.3182015083,
        -0.0598522969,
        -0.2072075009,
        -0.1359959437,
         0.0130584092,
         0.0971295716,
         0.0720368483,
        -0.0047367858,
        -0.0507386984,
        -0.0353212243,
         0.0078146525,
         0.0285925065
    },
    {
         0.0183945618,          /* Filter 41 */
        -0.0221999338,
        -0.0519341383,
        -0.0253261060,
         0.0493398215,
         0.0998233285,
         0.0508225871,
        -0.0898670563,
        -0.2033490873,
        -0.1336716600,
         0.1819804810,
         0.6336842466,
         0.9876170899,
         1.0394158319,
         0.7582097354,
         0.3088041356,
        -0.0655450567,
        -0.2076114352,
        -0.1330972988,
         0.0158356279,
         0.0976600127,
         0.0706545858,
        -0.0062403794,
        -0.0510188762,
        -0.0345148761,
         0.0086114355,
         0.0285939730
    },
    {
         0.0177899501,          /* Filter 42 */
        -0.0231266569,
        -0.0520494973,
        -0.0241006115,
         0.0509572988,
         0.0999844569,
         0.0485889958,
        -0.0930712050,
        -0.2042333694,
        -0.1295352022,
         0.1906848531,
         0.6429387725,
         0.9922522288,
         1.0366777417,
         0.7497354363,
         0.2994428875,
        -0.0711300052,
        -0.2079163571,
        -0.1301665484,
         0.0185795538,
         0.0981391468,
         0.0692498618,
        -0.0077277457,
        -0.0512698128,
        -0.0336964248,
         0.0093965971,
         0.0285777227
    },
    {
         0.0171697517,          /* Filter 43 */
        -0.0240483120,
        -0.0521379359,
        -0.0228519713,
         0.0525636728,
         0.1000986179,
         0.0463116534,
        -0.0962636853,
        -0.2050332414,
        -0.1252852054,
         0.1994515012,
         0.6521473646,
         0.9967480115,
         1.0337880300,
         0.7411872828,
         0.2901199075,
        -0.0766064761,
        -0.2081232437,
        -0.1272052882,
         0.0212892419,
         0.0985671844,
         0.0678235367,
        -0.0091982726,
        -0.0514916056,
        -0.0328664028,
         0.0101697770,
         0.0285438766
    },
    {
         0.0165342138,          /* Filter 44 */
        -0.0249643553,
        -0.0521991983,
        -0.0215806336,
         0.0541580658,
         0.1001653608,
         0.0439912202,
        -0.0994429615,
        -0.2057474457,
        -0.1209218845,
         0.2082785582,
         0.6613076413,
         1.0011032002,
         1.0307474970,
         0.7325675332,
         0.2808373192,
        -0.0819738404,
        -0.2082330984,
        -0.1242151125,
         0.0239637700,
         0.0989443574,
         0.0663764748,
        -0.0106513604,
        -0.0516843660,
        -0.0320253446,
         0.0109306237,
         0.0284925649
    },
    {
         0.0158835939,          /* Filter 45 */
        -0.0258742422,
        -0.0522330408,
        -0.0202870615,
         0.0557395992,
         0.1001842531,
         0.0416283812,
        -0.1026074897,
        -0.2063747447,
        -0.1164454920,
         0.2171641311,
         0.6704172289,
         1.0053165950,
         1.0275569840,
         0.7238784604,
         0.2715972261,
        -0.0872315071,
        -0.2082469503,
        -0.1211976129,
         0.0266022387,
         0.0992709191,
         0.0649095437,
        -0.0120864223,
        -0.0518482195,
        -0.0311737872,
         0.0116787947,
         0.0284239271
    },
    {
         0.0152181591,          /* Filter 46 */
        -0.0267774280,
        -0.0522392321,
        -0.0189717322,
         0.0573073940,
         0.1001548808,
         0.0392238457,
        -0.1057557185,
        -0.2069139221,
        -0.1118563180,
         0.2261063011,
         0.6794737613,
         1.0093870344,
         1.0242173737,
         0.7151223517,
         0.2624017105,
        -0.0923789224,
        -0.2081658535,
        -0.1181543779,
         0.0292037719,
         0.0995471438,
         0.0634236139,
        -0.0135028846,
        -0.0519833054,
        -0.0303122695,
         0.0124139562,
         0.0283381118
    },
    {
         0.0145381864,          /* Filter 47 */
        -0.0276733679,
        -0.0522175536,
        -0.0176351372,
         0.0588605710,
         0.1000768493,
         0.0367783473,
        -0.1088860894,
        -0.2073637831,
        -0.1071546901,
         0.2351031244,
         0.6884748817,
         1.0133133958,
         1.0207295896,
         0.7063015080,
         0.2532528331,
        -0.0974155706,
        -0.2079908873,
        -0.1150869926,
         0.0317675166,
         0.0997733266,
         0.0619195582,
        -0.0149001870,
        -0.0520897765,
        -0.0294413322,
         0.0131357839,
         0.0282352761
    },
};
#else
/* Raised root cosine pulse shaping; Beta = 0.5; 4 symbols either
   side of the centre. We cannot simplify this by using only half
   the filter, as each variant are each skewed by n/PULSESHAPER_COEFF_SETS of a
   sample. Only one is symmetric. */
#define PULSESHAPER_GAIN        10.00736638
#define PULSESHAPER_COEFF_SETS  48
static const float pulseshaper[PULSESHAPER_COEFF_SETS][V29RX_FILTER_STEPS] =
{
    {
        -0.0077172773,    /* Filter 0 */
        -0.0038694321,
         0.0112115904,
         0.0124836134,
        -0.0082281633,
        -0.0167092864,
         0.0146132015,
         0.0413429837,
        -0.0147924454,
        -0.1313410544,
        -0.1320188601,
         0.1554566471,
         0.6664080627,
         1.0773601675,
         1.0773601675,
         0.6664080627,
         0.1554566471,
        -0.1320188601,
        -0.1313410544,
        -0.0147924454,
         0.0413429837,
         0.0146132015,
        -0.0167092864,
        -0.0082281633,
         0.0124836134,
         0.0112115904,
        -0.0038694321
    },
    {
        -0.0041244119,    /* Filter 1 */
        -0.0036083922,
         0.0114733973,
         0.0122241026,
        -0.0086953937,
        -0.0164559194,
         0.0155079202,
         0.0412120072,
        -0.0169863104,
        -0.1333196873,
        -0.1292784439,
         0.1646043954,
         0.6773475777,
         1.0822215889,
         1.0723073790,
         0.6554248879,
         0.1464170119,
        -0.1346244304,
        -0.1293125105,
        -0.0126316456,
         0.0414339250,
         0.0137183623,
        -0.0169421543,
        -0.0077555379,
         0.0127315677,
         0.0109431119,
        -0.0041244119
    },
    {
        -0.0043731652,    /* Filter 2 */
        -0.0033414650,
         0.0117282579,
         0.0119531772,
        -0.0091567523,
        -0.0161820543,
         0.0164017016,
         0.0410404216,
        -0.0192122152,
        -0.1352462396,
        -0.1264022918,
         0.1738581306,
         0.6882397074,
         1.0868898157,
         1.0670651207,
         0.6444017799,
         0.1374875558,
        -0.1370960973,
        -0.1272362227,
        -0.0105048989,
         0.0414854268,
         0.0128242090,
        -0.0171545394,
        -0.0072779944,
         0.0129678349,
         0.0106682397,
        -0.0043731652
    },
    {
        -0.0046155325,    /* Filter 3 */
        -0.0030688299,
         0.0119759017,
         0.0116709905,
        -0.0096117632,
        -0.0158877103,
         0.0172937187,
         0.0408276754,
        -0.0214690972,
        -0.1371185400,
        -0.1233895677,
         0.1832156667,
         0.6990807290,
         1.0913630919,
         1.0616353597,
         0.6333424645,
         0.1286702834,
        -0.1394348551,
        -0.1251143537,
        -0.0084131553,
         0.0414981056,
         0.0119315364,
        -0.0173464758,
        -0.0067960093,
         0.0131922955,
         0.0103872554,
        -0.0046155325
    },
    {
        -0.0048513612,    /* Filter 4 */
        -0.0027906730,
         0.0122160622,
         0.0113777066,
        -0.0100599512,
        -0.0155729245,
         0.0181831345,
         0.0405732405,
        -0.0237558566,
        -0.1389344167,
        -0.1202394894,
         0.1926747584,
         0.7098669242,
         1.0956397332,
         1.0560201310,
         0.6222506654,
         0.1199671379,
        -0.1416417489,
        -0.1229490616,
        -0.0063573274,
         0.0414725987,
         0.0110411278,
        -0.0175180145,
        -0.0063100588,
         0.0134048412,
         0.0101004434,
        -0.0048513612
    },
    {
        -0.0050805055,    /* Filter 5 */
        -0.0025071863,
         0.0124484774,
         0.0110735009,
        -0.0105008428,
        -0.0152377524,
         0.0190691031,
         0.0402766124,
        -0.0260713566,
        -0.1406916989,
        -0.1169513302,
         0.2022331022,
         0.7205945806,
         1.0997181284,
         1.0502215375,
         0.6111301022,
         0.1113800000,
        -0.1437178738,
        -0.1207424980,
        -0.0043382895,
         0.0414095629,
         0.0101537543,
        -0.0176692240,
        -0.0058206179,
         0.0136053750,
         0.0098080911,
        -0.0050805055
    },
    {
        -0.0053028267,    /* Filter 6 */
        -0.0022185685,
         0.0126728899,
         0.0107585598,
        -0.0109339660,
        -0.0148822675,
         0.0199507698,
         0.0399373110,
        -0.0284144241,
        -0.1423882175,
        -0.1135244193,
         0.2118883368,
         0.7312599938,
         1.1035967397,
         1.0442417482,
         0.5999844889,
         0.1029106879,
        -0.1456643740,
        -0.1184968073,
        -0.0023568781,
         0.0413096740,
         0.0092701743,
        -0.0178001892,
        -0.0053281603,
         0.0137938111,
         0.0095104884,
        -0.0053028267
    },
    {
        -0.0055181930,    /* Filter 7 */
        -0.0019250239,
         0.0128890474,
         0.0104330806,
        -0.0113588515,
        -0.0145065623,
         0.0208272719,
         0.0395548815,
        -0.0307838494,
        -0.1440218066,
        -0.1099581426,
         0.2216380443,
         0.7418594687,
         1.1072741035,
         1.0380829975,
         0.5888175322,
         0.0945609563,
        -0.1474824414,
        -0.1162141254,
        -0.0004138916,
         0.0411736260,
         0.0083911333,
        -0.0179110118,
        -0.0048331579,
         0.0139700749,
         0.0092079274,
        -0.0055181930
    },
    {
        -0.0057264799,    /* Filter 8 */
        -0.0016267630,
         0.0130967024,
         0.0100972717,
        -0.0117750324,
        -0.0141107475,
         0.0216977392,
         0.0391288948,
        -0.0331783869,
        -0.1455903047,
        -0.1062519435,
         0.2314797506,
         0.7523893215,
         1.1107488316,
         1.0317475841,
         0.5776329299,
         0.0863324965,
        -0.1491733154,
        -0.1138965787,
         0.0014899097,
         0.0410021305,
         0.0075173632,
        -0.0180018097,
        -0.0043360803,
         0.0141341030,
         0.0089007023,
        -0.0057264799
    },
    {
        -0.0059275699,    /* Filter 9 */
        -0.0013240021,
         0.0132956134,
         0.0097513521,
        -0.0121820448,
        -0.0136949529,
         0.0225612946,
         0.0386589480,
        -0.0355967552,
        -0.1470915558,
        -0.1024053236,
         0.2414109265,
         0.7628458810,
         1.1140196110,
         1.0252378699,
         0.5664343693,
         0.0782269355,
        -0.1507382808,
        -0.1115462830,
         0.0033538037,
         0.0407959159,
         0.0066495821,
        -0.0180727168,
        -0.0038373950,
         0.0142858431,
         0.0085891090,
        -0.0059275699
    },
    {
        -0.0061213530,    /* Filter 10 */
        -0.0010169632,
         0.0134855441,
         0.0093955517,
        -0.0125794285,
        -0.0132593270,
         0.0234170544,
         0.0381446654,
        -0.0380376377,
        -0.1485234109,
        -0.0984178433,
         0.2514289886,
         0.7732254908,
         1.1170852055,
         1.0185562789,
         0.5552255254,
         0.0702458358,
        -0.1521786680,
        -0.1091653424,
         0.0051771063,
         0.0405557266,
         0.0057884938,
        -0.0181238828,
        -0.0033375660,
         0.0144252539,
         0.0082734449,
        -0.0061213530
    },
    {
        -0.0063077263,    /* Filter 11 */
        -0.0007058736,
         0.0136662644,
         0.0090301109,
        -0.0129667273,
        -0.0128040373,
         0.0242641295,
         0.0375856989,
        -0.0404996829,
        -0.1498837288,
        -0.0942891228,
         0.2615313001,
         0.7835245105,
         1.1199444556,
         1.0117052957,
         0.5440100595,
         0.0623906951,
        -0.1534958509,
        -0.1067558483,
         0.0069591716,
         0.0402823226,
         0.0049347874,
        -0.0181554728,
        -0.0028370545,
         0.0145523054,
         0.0079540084,
        -0.0063077263
    },
    {
        -0.0064865941,    /* Filter 12 */
        -0.0003909662,
         0.0138375505,
         0.0086552808,
        -0.0133434892,
        -0.0123292702,
         0.0251016255,
         0.0369817283,
        -0.0429815045,
        -0.1511703778,
        -0.0900188421,
         0.2717151717,
         0.7937393177,
         1.1225962793,
         1.0046874648,
         0.5327916175,
         0.0546629459,
        -0.1546912466,
        -0.1043198784,
         0.0086993916,
         0.0399764783,
         0.0040891370,
        -0.0181676672,
        -0.0023363179,
         0.0146669781,
         0.0076310991,
        -0.0064865941
    },
    {
        -0.0066578683,    /* Filter 13 */
        -0.0000724789,
         0.0139991847,
         0.0082713225,
        -0.0137092670,
        -0.0118352313,
         0.0259286435,
         0.0363324626,
        -0.0454816823,
        -0.1523812363,
        -0.0856067424,
         0.2819778628,
         0.8038663095,
         1.1250396727,
         0.9975053893,
         0.5215738282,
         0.0470639554,
        -0.1557663137,
        -0.1018594956,
         0.0103971963,
         0.0396389819,
         0.0032522013,
        -0.0181606611,
        -0.0018358095,
         0.0147692639,
         0.0073050171,
        -0.0066578683
    },
    {
        -0.0068214680,    /* Filter 14 */
         0.0002493454,
         0.0141509564,
         0.0078785078,
        -0.0140636189,
        -0.0113221454,
         0.0267442806,
         0.0356376397,
        -0.0479987626,
        -0.1535141950,
        -0.0810526259,
         0.2923165823,
         0.8139019043,
         1.1272737104,
         0.9901617293,
         0.5103603020,
         0.0395950250,
        -0.1567225517,
        -0.0993767470,
         0.0120520536,
         0.0392706350,
         0.0024246234,
        -0.0181346643,
        -0.0013359786,
         0.0148591651,
         0.0069760629,
        -0.0068214680
    },
    {
        -0.0069773197,    /* Filter 15 */
         0.0005742587,
         0.0142926615,
         0.0074771184,
        -0.0144061086,
        -0.0107902564,
         0.0275476309,
         0.0348970277,
        -0.0505312582,
        -0.1545671571,
        -0.0763563569,
         0.3027284900,
         0.8238425431,
         1.1292975460,
         0.9826592009,
         0.4991546291,
         0.0322573902,
        -0.1575614998,
        -0.0968736629,
         0.0136634691,
         0.0388722515,
         0.0016070303,
        -0.0180899006,
        -0.0008372696,
         0.0149366950,
         0.0066445373,
        -0.0069773197
    },
    {
        -0.0071253572,    /* Filter 16 */
         0.0009020085,
         0.0144241033,
         0.0070674458,
        -0.0147363057,
        -0.0102398272,
         0.0283377856,
         0.0341104250,
        -0.0530776494,
        -0.1555380404,
        -0.0715178621,
         0.3132106970,
         0.8336846919,
         1.1311104123,
         0.9750005749,
         0.4879603781,
         0.0250522204,
        -0.1582847354,
        -0.0943522562,
         0.0152309858,
         0.0384446569,
         0.0008000327,
        -0.0180266078,
        -0.0003401218,
         0.0150018773,
         0.0063107407,
        -0.0071253572
    },
    {
        -0.0072655220,    /* Filter 17 */
         0.0012323380,
         0.0145450926,
         0.0066497916,
        -0.0150537867,
        -0.0096711403,
         0.0291138344,
         0.0332776610,
        -0.0556363845,
        -0.1564247779,
        -0.0665371312,
         0.3237602674,
         0.8434248424,
         1.1327116219,
         0.9671886754,
         0.4767810943,
         0.0179806190,
        -0.1588938735,
        -0.0918145207,
         0.0167541843,
         0.0379886875,
         0.0000042245,
        -0.0179450374,
         0.0001550306,
         0.0150547463,
         0.0059749734,
        -0.0072655220
    },
    {
        -0.0073977626,    /* Filter 18 */
         0.0015649858,
         0.0146554474,
         0.0062244669,
        -0.0153581346,
        -0.0090844968,
         0.0298748652,
         0.0323985964,
        -0.0582058801,
        -0.1572253195,
        -0.0614142172,
         0.3343742193,
         0.8530595144,
         1.1341005677,
         0.9592263784,
         0.4656202984,
         0.0110436227,
        -0.1593905652,
        -0.0892624310,
         0.0182326825,
         0.0375051901,
        -0.0007798170,
        -0.0178454537,
         0.0006477590,
         0.0150953467,
         0.0056375349,
        -0.0073977626
    },
    {
        -0.0075220352,    /* Filter 19 */
         0.0018996866,
         0.0147549939,
         0.0057917921,
        -0.0156489401,
        -0.0084802176,
         0.0306199657,
         0.0314731241,
        -0.0607845218,
        -0.1579376330,
        -0.0561492367,
         0.3450495257,
         0.8625852570,
         1.1352767227,
         0.9511166107,
         0.4544814848,
         0.0042422023,
        -0.1597764966,
        -0.0866979409,
         0.0196661350,
         0.0369950206,
        -0.0015515318,
        -0.0177281343,
         0.0011376411,
         0.0151237335,
         0.0052987238,
        -0.0075220352
    },
    {
        -0.0076383031,    /* Filter 20 */
         0.0022361711,
         0.0148435662,
         0.0053520969,
        -0.0159258012,
        -0.0078586421,
         0.0313482235,
         0.0305011692,
        -0.0633706649,
        -0.1585597055,
        -0.0507423709,
         0.3557831160,
         0.8719986504,
         1.1362396407,
         0.9428623482,
         0.4433681204,
        -0.0024227381,
        -0.1600533878,
        -0.0841229830,
         0.0210542334,
         0.0364590439,
        -0.0023103774,
        -0.0175933690,
         0.0016242604,
         0.0151399716,
         0.0049588376,
        -0.0076383031
    },
    {
        -0.0077465371,    /* Filter 21 */
         0.0025741665,
         0.0149210065,
         0.0049057203,
        -0.0161883245,
        -0.0072201291,
         0.0320587270,
         0.0294826899,
        -0.0659626349,
        -0.1590895446,
        -0.0451938654,
         0.3665718769,
         0.8812963074,
         1.1369889562,
         0.9344666146,
         0.4322836427,
        -0.0089503609,
        -0.1602229915,
        -0.0815394677,
         0.0223967058,
         0.0358981326,
        -0.0030558287,
        -0.0174414598,
         0.0021072073,
         0.0151441363,
         0.0046181723,
        -0.0077465371
    },
    {
        -0.0078467154,    /* Filter 22 */
         0.0029133963,
         0.0149871656,
         0.0044530098,
        -0.0164361249,
        -0.0065650562,
         0.0327505660,
         0.0284176775,
        -0.0685587283,
        -0.1595251795,
        -0.0395040305,
         0.3774126540,
         0.8904748751,
         1.1375243848,
         0.9259324800,
         0.4212314588,
        -0.0153398946,
        -0.1602870918,
        -0.0789492822,
         0.0236933163,
         0.0353131667,
        -0.0037873786,
        -0.0172727204,
         0.0025860790,
         0.0151363124,
         0.0042770226,
        -0.0078467154
    },
    {
        -0.0079388232,    /* Filter 23 */
         0.0032535809,
         0.0150419030,
         0.0039943217,
        -0.0166688264,
        -0.0058938201,
         0.0334228328,
         0.0273061573,
        -0.0711572128,
        -0.1598646624,
        -0.0336732419,
         0.3883022526,
         0.8995310364,
         1.1378457233,
         0.9172630595,
         0.4102149439,
        -0.0215906342,
        -0.1602475034,
        -0.0763542897,
         0.0249438653,
         0.0347050324,
        -0.0045045381,
        -0.0170874759,
         0.0030604795,
         0.0151165945,
         0.0039356808,
        -0.0079388232
    },
    {
        -0.0080228532,    /* Filter 24 */
         0.0035944376,
         0.0150850869,
         0.0035300207,
        -0.0168860623,
        -0.0052068361,
         0.0340746221,
         0.0261481886,
        -0.0737563289,
        -0.1601060697,
        -0.0277019409,
         0.3992374395,
         0.9084615115,
         1.1379528497,
         0.9084615115,
         0.3992374395,
        -0.0277019409,
        -0.1601060697,
        -0.0737563289,
         0.0261481886,
         0.0340746221,
        -0.0052068361,
        -0.0168860623,
         0.0035300207,
         0.0150850869,
         0.0035944376,
        -0.0080228532
    },
    {
        -0.0080988052,    /* Filter 25 */
         0.0039356808,
         0.0151165945,
         0.0030604795,
        -0.0170874759,
        -0.0045045381,
         0.0347050324,
         0.0249438653,
        -0.0763542897,
        -0.1602475034,
        -0.0215906342,
         0.4102149439,
         0.9172630595,
         1.1378457233,
         0.8995310364,
         0.3883022526,
        -0.0336732419,
        -0.1598646624,
        -0.0711572128,
         0.0273061573,
         0.0334228328,
        -0.0058938201,
        -0.0166688264,
         0.0039943217,
         0.0150419030,
         0.0032535809,
        -0.0080988052
    },
    {
        -0.0081666861,    /* Filter 26 */
         0.0042770226,
         0.0151363124,
         0.0025860790,
        -0.0172727204,
        -0.0037873786,
         0.0353131667,
         0.0236933163,
        -0.0789492822,
        -0.1602870918,
        -0.0153398946,
         0.4212314588,
         0.9259324800,
         1.1375243848,
         0.8904748751,
         0.3774126540,
        -0.0395040305,
        -0.1595251795,
        -0.0685587283,
         0.0284176775,
         0.0327505660,
        -0.0065650562,
        -0.0164361249,
         0.0044530098,
         0.0149871656,
         0.0029133963,
        -0.0081666861
    },
    {
        -0.0082265098,    /* Filter 27 */
         0.0046181723,
         0.0151441363,
         0.0021072073,
        -0.0174414598,
        -0.0030558287,
         0.0358981326,
         0.0223967058,
        -0.0815394677,
        -0.1602229915,
        -0.0089503609,
         0.4322836427,
         0.9344666146,
         1.1369889562,
         0.8812963074,
         0.3665718769,
        -0.0451938654,
        -0.1590895446,
        -0.0659626349,
         0.0294826899,
         0.0320587270,
        -0.0072201291,
        -0.0161883245,
         0.0049057203,
         0.0149210065,
         0.0025741665,
        -0.0082265098
    },
    {
        -0.0082782975,    /* Filter 28 */
         0.0049588376,
         0.0151399716,
         0.0016242604,
        -0.0175933690,
        -0.0023103774,
         0.0364590439,
         0.0210542334,
        -0.0841229830,
        -0.1600533878,
        -0.0024227381,
         0.4433681204,
         0.9428623482,
         1.1362396407,
         0.8719986504,
         0.3557831160,
        -0.0507423709,
        -0.1585597055,
        -0.0633706649,
         0.0305011692,
         0.0313482235,
        -0.0078586421,
        -0.0159258012,
         0.0053520969,
         0.0148435662,
         0.0022361711,
        -0.0082782975
    },
    {
        -0.0083220771,    /* Filter 29 */
         0.0052987238,
         0.0151237335,
         0.0011376411,
        -0.0177281343,
        -0.0015515318,
         0.0369950206,
         0.0196661350,
        -0.0866979409,
        -0.1597764966,
         0.0042422023,
         0.4544814848,
         0.9511166107,
         1.1352767227,
         0.8625852570,
         0.3450495257,
        -0.0561492367,
        -0.1579376330,
        -0.0607845218,
         0.0314731241,
         0.0306199657,
        -0.0084802176,
        -0.0156489401,
         0.0057917921,
         0.0147549939,
         0.0018996866,
        -0.0083220771
    },
    {
        -0.0083578834,    /* Filter 30 */
         0.0056375349,
         0.0150953467,
         0.0006477590,
        -0.0178454537,
        -0.0007798170,
         0.0375051901,
         0.0182326825,
        -0.0892624310,
        -0.1593905652,
         0.0110436227,
         0.4656202984,
         0.9592263784,
         1.1341005677,
         0.8530595144,
         0.3343742193,
        -0.0614142172,
        -0.1572253195,
        -0.0582058801,
         0.0323985964,
         0.0298748652,
        -0.0090844968,
        -0.0153581346,
         0.0062244669,
         0.0146554474,
         0.0015649858,
        -0.0083578834
    },
    {
        -0.0083857580,    /* Filter 31 */
         0.0059749734,
         0.0150547463,
         0.0001550306,
        -0.0179450374,
         0.0000042245,
         0.0379886875,
         0.0167541843,
        -0.0918145207,
        -0.1588938735,
         0.0179806190,
         0.4767810943,
         0.9671886754,
         1.1327116219,
         0.8434248424,
         0.3237602674,
        -0.0665371312,
        -0.1564247779,
        -0.0556363845,
         0.0332776610,
         0.0291138344,
        -0.0096711403,
        -0.0150537867,
         0.0066497916,
         0.0145450926,
         0.0012323380,
        -0.0083857580
    },
    {
        -0.0084057493,    /* Filter 32 */
         0.0063107407,
         0.0150018773,
        -0.0003401218,
        -0.0180266078,
         0.0008000327,
         0.0384446569,
         0.0152309858,
        -0.0943522562,
        -0.1582847354,
         0.0250522204,
         0.4879603781,
         0.9750005749,
         1.1311104123,
         0.8336846919,
         0.3132106970,
        -0.0715178621,
        -0.1555380404,
        -0.0530776494,
         0.0341104250,
         0.0283377856,
        -0.0102398272,
        -0.0147363057,
         0.0070674458,
         0.0144241033,
         0.0009020085,
        -0.0084057493
    },
    {
        -0.0084179123,    /* Filter 33 */
         0.0066445373,
         0.0149366950,
        -0.0008372696,
        -0.0180899006,
         0.0016070303,
         0.0388722515,
         0.0136634691,
        -0.0968736629,
        -0.1575614998,
         0.0322573902,
         0.4991546291,
         0.9826592009,
         1.1292975460,
         0.8238425431,
         0.3027284900,
        -0.0763563569,
        -0.1545671571,
        -0.0505312582,
         0.0348970277,
         0.0275476309,
        -0.0107902564,
        -0.0144061086,
         0.0074771184,
         0.0142926615,
         0.0005742587,
        -0.0084179123
    },
    {
        -0.0084223083,    /* Filter 34 */
         0.0069760629,
         0.0148591651,
        -0.0013359786,
        -0.0181346643,
         0.0024246234,
         0.0392706350,
         0.0120520536,
        -0.0993767470,
        -0.1567225517,
         0.0395950250,
         0.5103603020,
         0.9901617293,
         1.1272737104,
         0.8139019043,
         0.2923165823,
        -0.0810526259,
        -0.1535141950,
        -0.0479987626,
         0.0356376397,
         0.0267442806,
        -0.0113221454,
        -0.0140636189,
         0.0078785078,
         0.0141509564,
         0.0002493454,
        -0.0084223083
    },
    {
        -0.0084190052,    /* Filter 35 */
         0.0073050171,
         0.0147692639,
        -0.0018358095,
        -0.0181606611,
         0.0032522013,
         0.0396389819,
         0.0103971963,
        -0.1018594956,
        -0.1557663137,
         0.0470639554,
         0.5215738282,
         0.9975053893,
         1.1250396727,
         0.8038663095,
         0.2819778628,
        -0.0856067424,
        -0.1523812363,
        -0.0454816823,
         0.0363324626,
         0.0259286435,
        -0.0118352313,
        -0.0137092670,
         0.0082713225,
         0.0139991847,
        -0.0000724789,
        -0.0084190052
    },
    {
        -0.0084080772,    /* Filter 36 */
         0.0076310991,
         0.0146669781,
        -0.0023363179,
        -0.0181676672,
         0.0040891370,
         0.0399764783,
         0.0086993916,
        -0.1043198784,
        -0.1546912466,
         0.0546629459,
         0.5327916175,
         1.0046874648,
         1.1225962793,
         0.7937393177,
         0.2717151717,
        -0.0900188421,
        -0.1511703778,
        -0.0429815045,
         0.0369817283,
         0.0251016255,
        -0.0123292702,
        -0.0133434892,
         0.0086552808,
         0.0138375505,
        -0.0003909662,
        -0.0084080772
    },
    {
        -0.0083896047,    /* Filter 37 */
         0.0079540084,
         0.0145523054,
        -0.0028370545,
        -0.0181554728,
         0.0049347874,
         0.0402823226,
         0.0069591716,
        -0.1067558483,
        -0.1534958509,
         0.0623906951,
         0.5440100595,
         1.0117052957,
         1.1199444556,
         0.7835245105,
         0.2615313001,
        -0.0942891228,
        -0.1498837288,
        -0.0404996829,
         0.0375856989,
         0.0242641295,
        -0.0128040373,
        -0.0129667273,
         0.0090301109,
         0.0136662644,
        -0.0007058736,
        -0.0083896047
    },
    {
        -0.0083636741,    /* Filter 38 */
         0.0082734449,
         0.0144252539,
        -0.0033375660,
        -0.0181238828,
         0.0057884938,
         0.0405557266,
         0.0051771063,
        -0.1091653424,
        -0.1521786680,
         0.0702458358,
         0.5552255254,
         1.0185562789,
         1.1170852055,
         0.7732254908,
         0.2514289886,
        -0.0984178433,
        -0.1485234109,
        -0.0380376377,
         0.0381446654,
         0.0234170544,
        -0.0132593270,
        -0.0125794285,
         0.0093955517,
         0.0134855441,
        -0.0010169632,
        -0.0083636741
    },
    {
        -0.0083303778,    /* Filter 39 */
         0.0085891090,
         0.0142858431,
        -0.0038373950,
        -0.0180727168,
         0.0066495821,
         0.0407959159,
         0.0033538037,
        -0.1115462830,
        -0.1507382808,
         0.0782269355,
         0.5664343693,
         1.0252378699,
         1.1140196110,
         0.7628458810,
         0.2414109265,
        -0.1024053236,
        -0.1470915558,
        -0.0355967552,
         0.0386589480,
         0.0225612946,
        -0.0136949529,
        -0.0121820448,
         0.0097513521,
         0.0132956134,
        -0.0013240021,
        -0.0083303778
    },
    {
        -0.0082898140,    /* Filter 40 */
         0.0089007023,
         0.0141341030,
        -0.0043360803,
        -0.0180018097,
         0.0075173632,
         0.0410021305,
         0.0014899097,
        -0.1138965787,
        -0.1491733154,
         0.0863324965,
         0.5776329299,
         1.0317475841,
         1.1107488316,
         0.7523893215,
         0.2314797506,
        -0.1062519435,
        -0.1455903047,
        -0.0331783869,
         0.0391288948,
         0.0216977392,
        -0.0141107475,
        -0.0117750324,
         0.0100972717,
         0.0130967024,
        -0.0016267630,
        -0.0082898140
    },
    {
        -0.0082420866,    /* Filter 41 */
         0.0092079274,
         0.0139700749,
        -0.0048331579,
        -0.0179110118,
         0.0083911333,
         0.0411736260,
        -0.0004138916,
        -0.1162141254,
        -0.1474824414,
         0.0945609563,
         0.5888175322,
         1.0380829975,
         1.1072741035,
         0.7418594687,
         0.2216380443,
        -0.1099581426,
        -0.1440218066,
        -0.0307838494,
         0.0395548815,
         0.0208272719,
        -0.0145065623,
        -0.0113588515,
         0.0104330806,
         0.0128890474,
        -0.0019250239,
        -0.0082420866
    },
    {
        -0.0081873051,    /* Filter 42 */
         0.0095104884,
         0.0137938111,
        -0.0053281603,
        -0.0178001892,
         0.0092701743,
         0.0413096740,
        -0.0023568781,
        -0.1184968073,
        -0.1456643740,
         0.1029106879,
         0.5999844889,
         1.0442417482,
         1.1035967397,
         0.7312599938,
         0.2118883368,
        -0.1135244193,
        -0.1423882175,
        -0.0284144241,
         0.0399373110,
         0.0199507698,
        -0.0148822675,
        -0.0109339660,
         0.0107585598,
         0.0126728899,
        -0.0022185685,
        -0.0081873051
    },
    {
        -0.0081255844,    /* Filter 43 */
         0.0098080911,
         0.0136053750,
        -0.0058206179,
        -0.0176692240,
         0.0101537543,
         0.0414095629,
        -0.0043382895,
        -0.1207424980,
        -0.1437178738,
         0.1113800000,
         0.6111301022,
         1.0502215375,
         1.0997181284,
         0.7205945806,
         0.2022331022,
        -0.1169513302,
        -0.1406916989,
        -0.0260713566,
         0.0402766124,
         0.0190691031,
        -0.0152377524,
        -0.0105008428,
         0.0110735009,
         0.0124484774,
        -0.0025071863,
        -0.0081255844
    },
    {
        -0.0080570447,    /* Filter 44 */
         0.0101004434,
         0.0134048412,
        -0.0063100588,
        -0.0175180145,
         0.0110411278,
         0.0414725987,
        -0.0063573274,
        -0.1229490616,
        -0.1416417489,
         0.1199671379,
         0.6222506654,
         1.0560201310,
         1.0956397332,
         0.7098669242,
         0.1926747584,
        -0.1202394894,
        -0.1389344167,
        -0.0237558566,
         0.0405732405,
         0.0181831345,
        -0.0155729245,
        -0.0100599512,
         0.0113777066,
         0.0122160622,
        -0.0027906730,
        -0.0080570447
    },
    {
        -0.0079818115,    /* Filter 45 */
         0.0103872554,
         0.0131922955,
        -0.0067960093,
        -0.0173464758,
         0.0119315364,
         0.0414981056,
        -0.0084131553,
        -0.1251143537,
        -0.1394348551,
         0.1286702834,
         0.6333424645,
         1.0616353597,
         1.0913630919,
         0.6990807290,
         0.1832156667,
        -0.1233895677,
        -0.1371185400,
        -0.0214690972,
         0.0408276754,
         0.0172937187,
        -0.0158877103,
        -0.0096117632,
         0.0116709905,
         0.0119759017,
        -0.0030688299,
        -0.0079818115
    },
    {
        -0.0079000149,    /* Filter 46 */
         0.0106682397,
         0.0129678349,
        -0.0072779944,
        -0.0171545394,
         0.0128242090,
         0.0414854268,
        -0.0105048989,
        -0.1272362227,
        -0.1370960973,
         0.1374875558,
         0.6444017799,
         1.0670651207,
         1.0868898157,
         0.6882397074,
         0.1738581306,
        -0.1264022918,
        -0.1352462396,
        -0.0192122152,
         0.0410404216,
         0.0164017016,
        -0.0161820543,
        -0.0091567523,
         0.0119531772,
         0.0117282579,
        -0.0033414650,
        -0.0079000149
    },
    {
        -0.0078117902,    /* Filter 47 */
         0.0109431119,
         0.0127315677,
        -0.0077555379,
        -0.0169421543,
         0.0137183623,
         0.0414339250,
        -0.0126316456,
        -0.1293125105,
        -0.1346244304,
         0.1464170119,
         0.6554248879,
         1.0723073790,
         1.0822215889,
         0.6773475777,
         0.1646043954,
        -0.1292784439,
        -0.1333196873,
        -0.0169863104,
         0.0412120072,
         0.0155079202,
        -0.0164559194,
        -0.0086953937,
         0.0122241026,
         0.0114733973,
        -0.0036083922,
        -0.0078117902
    },
};
#endif

float v29_rx_carrier_frequency(v29_rx_state_t *s)
{
    return s->carrier_phase_rate*(float) SAMPLE_RATE/(65536.0*65536.0);
}
/*- End of function --------------------------------------------------------*/

float v29_rx_symbol_timing_correction(v29_rx_state_t *s)
{
    return s->gardner_total_correction;
}
/*- End of function --------------------------------------------------------*/

void v29_rx_signal_cutoff(v29_rx_state_t *s, float cutoff)
{
    s->carrier_on_power = power_meter_level_dbm0(cutoff - 2.5);
    s->carrier_off_power = power_meter_level_dbm0(cutoff + 2.5);
}
/*- End of function --------------------------------------------------------*/

float v29_rx_signal_power(v29_rx_state_t *s)
{
    return power_meter_dbm0(&s->power);
}
/*- End of function --------------------------------------------------------*/

int v29_rx_equalizer_state(v29_rx_state_t *s, complex_t **coeffs)
{
    *coeffs = s->eq_coeff;
    return V29_EQUALIZER_PRE_LEN + 1 + V29_EQUALIZER_POST_LEN;
}
/*- End of function --------------------------------------------------------*/

static void equalizer_reset(v29_rx_state_t *s)
{
    int i;

    /* Start with an equalizer based on everything being perfect */
    for (i = 0;  i < V29_EQUALIZER_PRE_LEN + 1 + V29_EQUALIZER_POST_LEN;  i++)
        s->eq_coeff[i] = complex_set(0.0, 0.0);
    s->eq_coeff[V29_EQUALIZER_PRE_LEN] = complex_set(3.0, 0.0);
    for (i = 0;  i <= V29_EQUALIZER_MASK;  i++)
        s->eq_buf[i] = complex_set(0.0, 0.0);

    s->eq_put_step = PULSESHAPER_COEFF_SETS*10/(3*2) - 1;
    s->eq_step = 0;
    s->eq_delta = EQUALIZER_DELTA/(V29_EQUALIZER_PRE_LEN + 1 + V29_EQUALIZER_POST_LEN);
}
/*- End of function --------------------------------------------------------*/

static __inline__ complex_t equalizer_get(v29_rx_state_t *s)
{
    int i;
    int p;
    complex_t z;
    complex_t z1;

    /* Get the next equalized value. */
    z = complex_set(0.0, 0.0);
    p = s->eq_step - 1;
    for (i = 0;  i < V29_EQUALIZER_PRE_LEN + 1 + V29_EQUALIZER_POST_LEN;  i++)
    {
        p = (p - 1) & V29_EQUALIZER_MASK;
        z1 = complex_mul(&s->eq_coeff[i], &s->eq_buf[p]);
        z = complex_add(&z, &z1);
    }
    return z;
}
/*- End of function --------------------------------------------------------*/

static void tune_equalizer(v29_rx_state_t *s, const complex_t *z, const complex_t *target)
{
    int i;
    int p;
    complex_t ez;
    complex_t z1;

    /* Find the x and y mismatch from the exact constellation position. */
    ez = complex_sub(target, z);
    ez.re *= s->eq_delta;
    ez.im *= s->eq_delta;

    p = s->eq_step - 1;
    for (i = 0;  i < V29_EQUALIZER_PRE_LEN + 1 + V29_EQUALIZER_POST_LEN;  i++)
    {
        p = (p - 1) & V29_EQUALIZER_MASK;
        z1 = complex_conj(&s->eq_buf[p]);
        z1 = complex_mul(&ez, &z1);
        s->eq_coeff[i] = complex_add(&s->eq_coeff[i], &z1);
        /* Leak a little to tame uncontrolled wandering */
        s->eq_coeff[i].re *= 0.9999;
        s->eq_coeff[i].im *= 0.9999;
    }
}
/*- End of function --------------------------------------------------------*/

static int scrambled_training_bit(v29_rx_state_t *s)
{
    int bit;

    /* Segment 3 of the training sequence - the scrambled CDCD part. */
    /* Apply the 1 + x^-6 + x^-7 scrambler */
    bit = s->training_scramble_reg & 1;
    s->training_scramble_reg >>= 1;
    if (bit ^ (s->training_scramble_reg & 1))
        s->training_scramble_reg |= 0x40;
    return bit;
}
/*- End of function --------------------------------------------------------*/

static __inline__ int find_quadrant(const complex_t *z)
{
    int b1;
    int b2;

    /* Split the space along the two diagonals. */
    b1 = (z->im > z->re);
    b2 = (z->im < -z->re);
    return (b2 << 1) | (b1 ^ b2);
}
/*- End of function --------------------------------------------------------*/

static __inline__ void track_carrier(v29_rx_state_t *s, const complex_t *z, const complex_t *target)
{
    complex_t zz;

    /* The initial coarse carrier frequency and phase estimation should have
       got us in the right ballpark. Now we need to fine tune fairly quickly,
       to get the receovered carrier more precisely on target. Then we need to
       fine tune in a more damped way to keep us on target. The goal is to have
       things running really well by the time the training is complete. 
       We assume the frequency of the oscillators at the two ends drift only
       very slowly. The PSTN has rather limited doppler problems. :-) Any
       remaining FDM in the network should also drift slowly. */
    /* For small errors the imaginary part of zz is now proportional to the phase
       error, for any particular target. However, the different amplitudes of
       the various target positions scale things. This isn't all bad, as the
       angular error for the larger amplitude constellation points is probably
       a more reliable indicator, and we are weighting it as such. */
    zz = complex_conj(target);
    zz = complex_mul(z, &zz);
    
    /* Use a proportional-integral approach to tracking the carrier. The PI
       parameters are coarser at first, until we get precisely on target. Then,
       the filter will be damped more to keep us on target. */
    s->carrier_phase_rate += s->carrier_track_i*zz.im;
    s->carrier_phase += s->carrier_track_p*zz.im;
    //span_log(&s->logging, SPAN_LOG_FLOW, "Im = %15.5f   f = %15.5f\n", zz.im, (float) s->carrier_phase_rate*SAMPLE_RATE/(65536.0*65536.0));
}
/*- End of function --------------------------------------------------------*/

static __inline__ void put_bit(v29_rx_state_t *s, int bit)
{
    int out_bit;

    bit &= 1;

    /* Descramble the bit */
    out_bit = (bit ^ (s->scramble_reg >> 17) ^ (s->scramble_reg >> 22)) & 1;
    s->scramble_reg = (s->scramble_reg << 1) | bit;

    /* We need to strip the last part of the training - the test period of all 1s -
       before we let data go to the application. */
    if (s->in_training == TRAINING_STAGE_NORMAL_OPERATION)
    {
        s->put_bit(s->user_data, out_bit);
    }
    else
    {
        //span_log(&s->logging, SPAN_LOG_FLOW, "bit %5d %d\n", s->training_cd, out_bit);
        /* The bits during the final stage of training should be all ones. However,
           buggy modems mean you cannot rely on this. Therefore we don't bother
           testing for ones, but just rely on a constellation mismatch measurement. */
    }
}
/*- End of function --------------------------------------------------------*/

static void decode_baud(v29_rx_state_t *s, complex_t *z)
{
    static const uint8_t phase_steps_9600[8] =
    {
        4, 0, 2, 6, 7, 3, 1, 5
    };
    static const uint8_t phase_steps_4800[4] =
    {
        0, 2, 3, 1
    };
    int nearest;
    int raw_bits;
    int i;
    int re;
    int im;

    switch (s->bit_rate)
    {
    case 9600:
        re = (z->re + 5.0)*2.0;
        if (re > 19)
            re = 19;
        else if (re < 0)
            re = 0;
        im = (z->im + 5.0)*2.0;
        if (im > 19)
            im = 19;
        else if (im < 0)
            im = 0;
        nearest = space_map_9600[re][im];
        /* Deal with the amplitude bit */
        put_bit(s, nearest >> 3);
        raw_bits = phase_steps_9600[(nearest - s->constellation_state) & 7];
        for (i = 0;  i < 3;  i++)
        {
            put_bit(s, raw_bits);
            raw_bits >>= 1;
        }
        break;
    case 7200:
        /* We can reuse the space map for 9600, but drop the top bit */
        re = (z->re + 5.0)*2.0;
        if (re > 19)
            re = 19;
        else if (re < 0)
            re = 0;
        im = (z->im + 5.0)*2.0;
        if (im > 19)
            im = 19;
        else if (im < 0)
            im = 0;
        nearest = space_map_9600[re][im] & 7;
        raw_bits = phase_steps_9600[(nearest - s->constellation_state) & 7];
        for (i = 0;  i < 3;  i++)
        {
            put_bit(s, raw_bits);
            raw_bits >>= 1;
        }
        break;
    case 4800:
        nearest = find_quadrant(z) << 1;
        raw_bits = phase_steps_4800[((nearest - s->constellation_state) >> 1) & 3];
        put_bit(s, raw_bits);
        put_bit(s, raw_bits >> 1);
        break;
    }
    track_carrier(s, z, &v29_constellation[nearest]);
    if (--s->eq_skip <= 0)
    {
        /* Once we are in the data the equalization should not need updating.
           However, the line characteristics may slowly drift. We, therefore,
           tune up on the occassional sample, keeping the compute down. */
        s->eq_skip = 10;
        tune_equalizer(s, z, &v29_constellation[nearest]);
    }
    s->constellation_state = nearest;
}
/*- End of function --------------------------------------------------------*/

static void process_baud(v29_rx_state_t *s, const complex_t *sample)
{
    static const int cdcd_pos[6] =
    {
        0, 11,
        0,  3,
        0,  2
    };
    complex_t z;
    complex_t zz;
    const complex_t *target;
    float p;
    float q;
    int bit;
    int i;
    int j;
    float phase;
    int32_t angle;
    int32_t ang;

    s->rrc_filter[s->rrc_filter_step] =
    s->rrc_filter[s->rrc_filter_step + V29RX_FILTER_STEPS] = *sample;
    if (++s->rrc_filter_step >= V29RX_FILTER_STEPS)
        s->rrc_filter_step = 0;
    /* Put things into the equalization buffer at T/2 rate. The Gardner algorithm
       will fiddle the step to align this with the bits. */
    if ((s->eq_put_step -= PULSESHAPER_COEFF_SETS) > 0)
    {
        //span_log(&s->logging, SPAN_LOG_FLOW, "Samp, %f, %f, %f, 0, 0x%X\n", z.re, z.im, sqrt(z.re*z.re + z.im*z.im), s->eq_put_step);
        return;
    }

    /* This is our interpolation filter and phase shifter, as well as our demod filter. */
    j = -s->eq_put_step;
    if (j > PULSESHAPER_COEFF_SETS - 1)
        j = PULSESHAPER_COEFF_SETS - 1;
    z = complex_set(0.0, 0.0);
    for (i = 0;  i < V29RX_FILTER_STEPS;  i++)
    {
        z.re += pulseshaper[j][i]*s->rrc_filter[i + s->rrc_filter_step].re;
        z.im += pulseshaper[j][i]*s->rrc_filter[i + s->rrc_filter_step].im;
    }
    z.re *= 1.0/PULSESHAPER_GAIN;
    z.im *= 1.0/PULSESHAPER_GAIN;

    s->eq_put_step += PULSESHAPER_COEFF_SETS*10/(3*2);

    /* Add a sample to the equalizer's circular buffer, but don't calculate anything
       at this time. */
    s->eq_buf[s->eq_step] = z;
    s->eq_step = (s->eq_step + 1) & V29_EQUALIZER_MASK;

    /* On alternate insertions we have a whole baud, and must process it. */
    if ((s->baud_phase ^= 1))
        return;

    /* Perform a Gardner test for baud alignment on the three most recent samples. */
    p = s->eq_buf[(s->eq_step - 3) & V29_EQUALIZER_MASK].re
      - s->eq_buf[(s->eq_step - 1) & V29_EQUALIZER_MASK].re;
    p *= s->eq_buf[(s->eq_step - 2) & V29_EQUALIZER_MASK].re;

    q = s->eq_buf[(s->eq_step - 3) & V29_EQUALIZER_MASK].im
      - s->eq_buf[(s->eq_step - 1) & V29_EQUALIZER_MASK].im;
    q *= s->eq_buf[(s->eq_step - 2) & V29_EQUALIZER_MASK].im;

    s->gardner_integrate += (p + q > 0.0)  ?  s->gardner_step  :  -s->gardner_step;

    if (abs(s->gardner_integrate) >= 8)
    {
        /* This integrate and dump approach avoids rapid changes of the equalizer put step.
           Rapid changes, without hysteresis, are bad. They degrade the equalizer performance
           when the true symbol boundary is close to a sample boundary. */
        s->eq_put_step += (s->gardner_integrate/8);
        s->gardner_total_correction += (s->gardner_integrate/8);
        if (s->qam_report)
            s->qam_report(s->qam_user_data, NULL, NULL, s->gardner_integrate);
        s->gardner_integrate = 0;
    }

    z = equalizer_get(s);

    switch (s->in_training)
    {
    case TRAINING_STAGE_NORMAL_OPERATION:
        /* Normal operation. */
        decode_baud(s, &z);
        target = &v29_constellation[s->constellation_state];
        break;
    case TRAINING_STAGE_SYMBOL_ACQUISITION:
        /* Allow time for the Gardner algorithm to settle the symbol timing. */
        target = &z;
        if (++s->training_count >= 60)
        {
            /* Record the current phase angle */
            s->in_training = TRAINING_STAGE_LOG_PHASE;
            s->angles[0] =
            s->start_angles[0] = arctan2(z.im, z.re);
        }
        break;
    case TRAINING_STAGE_LOG_PHASE:
        /* Record the current alternate phase angle */
        target = &z;
        s->angles[1] =
        s->start_angles[1] = arctan2(z.im, z.re);
        s->training_count = 1;
        s->in_training = TRAINING_STAGE_WAIT_FOR_CDCD;
        break;
    case TRAINING_STAGE_WAIT_FOR_CDCD:
        target = &z;
        angle = arctan2(z.im, z.re);
        /* Look for the initial ABAB sequence to display a phase reversal, which will
           signal the start of the scrambled CDCD segment */
        ang = angle - s->angles[(s->training_count - 1) & 0xF];
        s->angles[(s->training_count + 1) & 0xF] = angle;
        if ((ang > 0x20000000  ||  ang < -0x20000000)  &&  s->training_count >= 13)
        {
            /* We seem to have a phase reversal */
            /* Slam the carrier frequency into line, based on the total phase drift over the last
               section. Use the shift from the odd bits and the shift from the even bits to get
               better jitter suppression. We need to scale here, or at the maximum specified
               frequency deviation we could overflow, and get a silly answer. */
            /* Step back a few symbols so we don't get ISI distorting things. */
            i = (s->training_count - 8) & ~1;
            /* Avoid the possibility of a divide by zero */
            if (i)
            {
                j = i & 0xF;
                ang = (s->angles[j] - s->start_angles[0])/i
                    + (s->angles[j | 0x1] - s->start_angles[1])/i;
                s->carrier_phase_rate += 3*(ang/20);
            }
            span_log(&s->logging, SPAN_LOG_FLOW, "Coarse carrier frequency %7.2f (%d)\n", s->carrier_phase_rate*8000.0/(65536.0*65536.0), i);

            /* Make a step shift in the phase, to pull it into line. We need to rotate the RRC filter
               buffer and the equalizer buffer, as well as the carrier phase, for this to play out
               nicely. */
            p = angle*2.0*3.14159/(65536.0*65536.0);
            zz = complex_set(cos(p), -sin(p));
            for (i = 0;  i < 2*V29RX_FILTER_STEPS;  i++)
                s->rrc_filter[i] = complex_mul(&s->rrc_filter[i], &zz);
            for (i = 0;  i <= V29_EQUALIZER_MASK;  i++)
                s->eq_buf[i] = complex_mul(&s->eq_buf[i], &zz);
            s->carrier_phase += angle;

            /* QAM and Gardner only play nicely with heavy damping, so we need to change to
               a slow rate of symbol timing adaption. However, it must not be so slow that it
               cannot track the worst case timing error specified in V.29. This should be 0.01%,
               but since we might be off in the opposite direction from the source, the total
               error could be higher. */
            s->gardner_step = 2;
            /* We have just seen the first bit of the scrambled sequence, so skip it. */
            bit = scrambled_training_bit(s);
            s->training_count = 1;
            s->in_training = TRAINING_STAGE_TRAIN_ON_CDCD;
        }
        else
        {
            if (++s->training_count > V29_TRAINING_SEG_2_LEN)
            {
                /* This is bogus. There are not this many bauds in this section
                   of a real training sequence. */
                span_log(&s->logging, SPAN_LOG_FLOW, "Training failed (sequence failed)\n");
                /* Park this modem */
                s->in_training = TRAINING_STAGE_PARKED;
                s->put_bit(s->user_data, PUTBIT_TRAINING_FAILED);
            }
        }
        break;
    case TRAINING_STAGE_TRAIN_ON_CDCD:
        /* Train on the scrambled CDCD section. */
        bit = scrambled_training_bit(s);
        //span_log(&s->logging, SPAN_LOG_FLOW, "%5d %15.5f, %15.5f     %15.5f, %15.5f\n", s->training_count, z.re, z.im, v29_constellation[cdcd_pos[s->training_cd + bit]].re, v29_constellation[cdcd_pos[s->training_cd + bit]].im);
        s->constellation_state = cdcd_pos[s->training_cd + bit];
        target = &v29_constellation[s->constellation_state];
        track_carrier(s, &z, target);
        tune_equalizer(s, &z, target);
        if (++s->training_count >= V29_TRAINING_SEG_3_LEN - 48)
        {
            s->in_training = TRAINING_STAGE_TRAIN_ON_CDCD_AND_TEST;
            s->training_error = 0.0;
            s->carrier_track_i = 200.0;
            s->carrier_track_p = 1000000.0;
        }
        break;
    case TRAINING_STAGE_TRAIN_ON_CDCD_AND_TEST:
        /* Continue training on the scrambled CDCD section, but measure the quality of training too. */
        bit = scrambled_training_bit(s);
        //span_log(&s->logging, SPAN_LOG_FLOW, "%5d %15.5f, %15.5f     %15.5f, %15.5f\n", s->training_count, z.re, z.im, v29_constellation[cdcd_pos[s->training_cd + bit]].re, v29_constellation[cdcd_pos[s->training_cd + bit]].im);
        s->constellation_state = cdcd_pos[s->training_cd + bit];
        target = &v29_constellation[s->constellation_state];
        track_carrier(s, &z, target);
        tune_equalizer(s, &z, target);
        /* Measure the training error */
        zz = complex_sub(&z, target);
        s->training_error += power(&zz);
        if (++s->training_count >= V29_TRAINING_SEG_3_LEN)
        {
            span_log(&s->logging, SPAN_LOG_FLOW, "Training error %f\n", s->training_error);
            if (s->training_error < 100.0)
            {
                s->training_count = 0;
                s->training_cd = 0;
                s->training_error = 0.0;
                s->constellation_state = 0;
                s->in_training = TRAINING_STAGE_TEST_ONES;
            }
            else
            {
                span_log(&s->logging, SPAN_LOG_FLOW, "Training failed (convergence failed)\n");
                /* Park this modem */
                s->in_training = TRAINING_STAGE_PARKED;
                s->put_bit(s->user_data, PUTBIT_TRAINING_FAILED);
            }
        }
        break;
    case TRAINING_STAGE_TEST_ONES:
        /* We are in the test phase, where we check that we can receive reliably.
           We should get a run of 1's, 48 symbols (192 bits at 9600bps) long. */
        //span_log(&s->logging, SPAN_LOG_FLOW, "%5d %15.5f, %15.5f\n", s->training_count, z.re, z.im);
        decode_baud(s, &z);
        target = &v29_constellation[s->constellation_state];
        /* Measure the training error */
        zz = complex_sub(&z, target);
        s->training_error += power(&zz);
        if (++s->training_count >= V29_TRAINING_SEG_4_LEN)
        {
            if (s->training_error < 50.0)
            {
                /* We are up and running */
                span_log(&s->logging, SPAN_LOG_FLOW, "Training succeeded (constellation mismatch %f)\n", s->training_error);
                s->in_training = TRAINING_STAGE_NORMAL_OPERATION;
                s->put_bit(s->user_data, PUTBIT_TRAINING_SUCCEEDED);
            }
            else
            {
                /* Training has failed */
                span_log(&s->logging, SPAN_LOG_FLOW, "Training failed (constellation mismatch %f)\n", s->training_error);
                /* Park this modem */
                s->in_training = TRAINING_STAGE_PARKED;
                s->put_bit(s->user_data, PUTBIT_TRAINING_FAILED);
            }
        }
        break;
    case TRAINING_STAGE_PARKED:
    default:
        /* We failed to train! */
        /* Park here until the carrier drops. */
        target = &z;
        break;
    }
    if (s->qam_report)
        s->qam_report(s->qam_user_data, &z, target, s->constellation_state);
}
/*- End of function --------------------------------------------------------*/

int v29_rx(v29_rx_state_t *s, const int16_t *amp, int len)
{
    int i;
    int16_t sample;
    complex_t z;
    int32_t power;
    float x;

    for (i = 0;  i < len;  i++)
    {
        sample = amp[i];
        power = power_meter_update(&(s->power), sample);
        if (s->carrier_present)
        {
            /* Look for power below turnoff threshold to turn the carrier off */
            if (power < s->carrier_off_power)
            {
                v29_rx_restart(s, s->bit_rate);
                s->put_bit(s->user_data, PUTBIT_CARRIER_DOWN);
                continue;
            }
        }
        else
        {
            /* Look for power exceeding turnon threshold to turn the carrier on */
            if (power < s->carrier_on_power)
                continue;
            s->carrier_present = TRUE;
            s->put_bit(s->user_data, PUTBIT_CARRIER_UP);
        }
        if (s->in_training != TRAINING_STAGE_PARKED)
        {
            /* Only spend effort processing this data if the modem is not
               parked, after training failure. */
            if (s->in_training == TRAINING_STAGE_SYMBOL_ACQUISITION)
            {
                /* Only AGC during the initial training */
                s->agc_scaling = 5.0*0.72/sqrt(power);
            }
            x = sample*s->agc_scaling;
            /* Shift to baseband */
            z = dds_complexf(&(s->carrier_phase), s->carrier_phase_rate);
            z.re *= x;
            z.im *= x;
            process_baud(s, &z);
        }
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

void v29_rx_set_put_bit(v29_rx_state_t *s, put_bit_func_t put_bit, void *user_data)
{
    s->put_bit = put_bit;
    s->user_data = user_data;
}
/*- End of function --------------------------------------------------------*/

int v29_rx_restart(v29_rx_state_t *s, int rate)
{
    int i;

    switch (rate)
    {
    case 9600:
        s->training_cd = 0;
        break;
    case 7200:
        s->training_cd = 2;
        break;
    case 4800:
        s->training_cd = 4;
        break;
    default:
        return -1;
    }
    s->bit_rate = rate;

    memset(s->rrc_filter, 0, sizeof(s->rrc_filter));
    s->rrc_filter_step = 0;

    s->scramble_reg = 0;
    s->training_scramble_reg = 0x2A;
    s->in_training = TRAINING_STAGE_SYMBOL_ACQUISITION;
    s->training_count = 0;
    s->carrier_present = FALSE;

    s->carrier_phase_rate = dds_phase_stepf(1700.0);
    s->carrier_phase = 0;
    s->carrier_track_i = 1000.0;
    //s->carrier_track_i = 100000.0;
    s->carrier_track_p = 4000000.0;

    power_meter_init(&(s->power), 4);
    s->agc_scaling = 0.0005;

    s->constellation_state = 0;

    equalizer_reset(s);
    s->eq_skip = 0;

    s->gardner_integrate = 0;
    s->gardner_total_correction = 0;
    /* Start with very big Gardner steps, until we start to pull in the symbol timing. */
    s->gardner_step = 64;
    s->baud_phase = 0;
    return 0;
}
/*- End of function --------------------------------------------------------*/

v29_rx_state_t *v29_rx_init(v29_rx_state_t *s, int rate, put_bit_func_t put_bit, void *user_data)
{
    int i;
    int j;
    int k;
    int best;
    float distance;
    float best_distance;
    float x;
    float y;

    if (s == NULL)
    {
        if ((s = (v29_rx_state_t *) malloc(sizeof(*s))) == NULL)
            return NULL;
    }

    memset(s, 0, sizeof(*s));
    s->put_bit = put_bit;
    s->user_data = user_data;
    s->carrier_on_power = power_meter_level_dbm0(-26);
    s->carrier_off_power = power_meter_level_dbm0(-31);
    span_log_init(&s->logging, SPAN_LOG_NONE, NULL);
    span_log_set_protocol(&s->logging, "V.29");

    v29_rx_restart(s, rate);
    return s;
}
/*- End of function --------------------------------------------------------*/

int v29_rx_release(v29_rx_state_t *s)
{
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/

void v29_rx_set_qam_report_handler(v29_rx_state_t *s, qam_report_handler_t *handler, void *user_data)
{
    s->qam_report = handler;
    s->qam_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
