/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * wsdl-stubs.c: Emit stub routines for clients
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "wsdl-parse.h"
#include "wsdl-soap-stubs.h"
#include "wsdl-trace.h"
#include "wsdl-soap-emit.h"

static void
wsdl_emit_soap_stubs_binding_operation (
				FILE                                *out, 
				const guchar                        *opns,
				const guchar                        *opnsuri,
				const wsdl_binding_operation * const op)
{
	wsdl_porttype_operation *porttype_op;

	g_assert (op->name != NULL);
	g_assert (op->soap_operation != NULL);

	fprintf (out, "/* BEGIN Binding Operation %s */\n\n", op->name);

	porttype_op = op->thread_soap_porttype_operation;
	g_assert (porttype_op != NULL);
	g_assert (porttype_op->input != NULL);

	if (op->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", op->documentation->str);
	}

	if (porttype_op->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", porttype_op->documentation->str);
	}

	/* Emit the callback */
	fprintf (out,
		 "static void %s_%s_soup_callback(SoupMessage *req, SoupErrorCode err, gpointer user_data)\n",
		 opns, 
		 op->name);
	fprintf (out, "{\n");
	fprintf (out,
		 "\t_wsdl_stubs_callback_data_t *cb_data = (_wsdl_stubs_callback_data_t *) user_data;\n");
	fprintf (out, 
		 "\t%s_%s_callback callback;\n", 
		 opns, 
		 op->name);

	if (porttype_op->output != NULL) {
		fprintf (out, 
			 "\t/* more args here based on output parts */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t%t _out_%p;\n");
	}

	fprintf (out, "\twsdl_param params[]={\n");
	if (porttype_op->output != NULL) {
		fprintf (out, "\t\t/* params here based on output parts */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t\t{\"%p\", &_out_%p, &WSDL_TC_%n_%N_struct},\n");
	}
	fprintf (out, "\t\t{NULL, NULL, NULL},\n");
	fprintf (out, "\t};\n");

	fprintf (out, "\n");
	fprintf (out, "\tswitch(err) {\n");
	fprintf (out, "\tcase SOUP_ERROR_NONE:\n");
	fprintf (out, "\t\t/* Handle results */\n");
	/* Are response headers interesting to clients? */

	fprintf (out,
		 "\t\t/* parse req->response.body into callback params */\n");
	fprintf (out,
		 "\t\twsdl_soap_parse(req->response.body, \"%s\", params, cb_data->env, WSDL_SOAP_FLAGS_RESPONSE);\n",
		 op->name);

	fprintf (out, "\t\tbreak;\n");
	fprintf (out, "\tcase SOUP_ERROR_CANCELLED:\n");
	fprintf (out, "\tcase SOUP_ERROR_CANT_CONNECT:\n");
	fprintf (out, "\tcase SOUP_ERROR_IO:\n");
	fprintf (out, "\tcase SOUP_ERROR_MALFORMED_HEADER:\n");
	fprintf (out, "\tcase SOUP_ERROR_CANT_AUTHENTICATE:\n");
	fprintf (out, "\tcase SOUP_ERROR_HANDLER:\n");
	fprintf (out, "\t\t/* Handle errors */\n");
	fprintf (out, "\t\tbreak;\n");
	fprintf (out, "\t}\n");
	fprintf (out, "\n");
	fprintf (out, "\t/* SOUP_ERROR_NONE == 0 */\n");
	fprintf (out, "\tif(cb_data!=NULL) {\n");
	fprintf (out, "\t\tcallback = (%s_%s_callback)cb_data->callback;\n",
		 opns, 
		 op->name);
	fprintf (out, "\t\tif (callback!=NULL) {\n");
	fprintf (out,
		 "\t\t\tg_hash_table_foreach(req->response_headers, (GHFunc) add_response_header, cb_data->env);\n");
	fprintf (out,
		 "\t\t\tcallback(err, req->response_code, req->response_phrase, cb_data->env,\n");

	if (porttype_op->output != NULL) {
		fprintf (out, "\t\t\t\t/* output args go here */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t\t\t\t _out_%p,\n");
	}

	fprintf (out, "\t\t\t\tcb_data->user_data);\n");
	fprintf (out, "\t\t}\n");
	fprintf (out, "\t\tg_free((gpointer)cb_data);\n");
	fprintf (out, "\t}\n");

	fprintf (out, "}\n");
	fprintf (out, "\n");

	/* And now do the client stub */
	fprintf (out, "void %s_%s(SoupEnv *env,\n", opns, op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* parameters go here */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t%t _in_%p,\n");
	}

	fprintf (out, "\t%s_%s_callback user_callback,\n", opns, op->name);
	fprintf (out, "\tgpointer user_data)\n");

	fprintf (out, "{\n");
	fprintf (out, "\tSoupContext *_soup_context;\n");
	fprintf (out, "\tSoupMessage *_message;\n");
	fprintf (out, "\t_wsdl_stubs_callback_data_t *_cb_data;\n");

	fprintf (out, "\twsdl_param _in_params[]={\n");
	if (porttype_op->input != NULL) {
		fprintf (out, "\t\t/* params here based on input parts */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t\t{\"%p\", &_in_%p, &WSDL_TC_%n_%N_struct},\n");
	}
	fprintf (out, "\t\t{NULL, NULL, NULL},\n");
	fprintf (out, "\t};\n");

	fprintf (out, "\n");
	fprintf (out, "\tif (soup_env_get_address(env) != NULL)\n");
	fprintf (out,
		 "\t\t_soup_context=soup_context_get(soup_env_get_address(env));\n");
	fprintf (out, "\telse\n");
	fprintf (out, 
		 "\t\t_soup_context=soup_context_get(\"%s\");\n",
		 op->soap_operation->soapAction);

	fprintf (out, 
		 "\t_message=soup_message_new(_soup_context, \"%s\");\n",
		 op->name);
	fprintf (out, "\tsoup_context_unref(_soup_context);\n");

	fprintf (out, "\t_cb_data = g_new0(_wsdl_stubs_callback_data_t, 1);\n");
	fprintf (out, "\t_cb_data->callback=(gpointer)user_callback;\n");
	fprintf (out, "\t_cb_data->user_data=user_data;\n");
	fprintf (out, "\t_cb_data->env=env;\n");

	fprintf (out,
		 "\twsdl_soap_marshal(\"%s\", \"%s\", \"%s\", _in_params, &_message->request, env, WSDL_SOAP_FLAGS_REQUEST);\n",
		 op->name, 
		 opns, 
		 opnsuri);
	fprintf (out,
		 "\tsoup_message_queue(_message, %s_%s_soup_callback, _cb_data);\n",
		 opns, 
		 op->name);

	fprintf (out, "}\n\n");

	/* And now do the synchronous client stub */
	fprintf (out, "void %s_%s_sync(SoupEnv *env\n", opns, op->name);

	if (porttype_op->input != NULL) {
		fprintf (out, "\t/* input parameters go here */\n");
		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t,%t _in_%p\n");
	}
	if (porttype_op->output != NULL) {
		fprintf (out, "\t/* output parameters go here */\n");
		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t,%t * _out_%p\n");
	}

	fprintf (out, "\t)\n");

	fprintf (out, "{\n");
	fprintf (out, "\tSoupContext *_soup_context;\n");
	fprintf (out, "\tSoupMessage *_message;\n");

	fprintf (out, "\twsdl_param _in_params[]={\n");
	if (porttype_op->input != NULL) {
		fprintf (out, "\t\t/* params here based on input parts */\n");

		wsdl_emit_part_list (out, 
				     porttype_op->input->thread_soap_parts,
				     "\t\t{\"%p\", &_in_%p, &WSDL_TC_%n_%N_struct},\n");
	}
	fprintf (out, "\t\t{NULL, NULL, NULL},\n");
	fprintf (out, "\t};\n");

	fprintf (out, "\twsdl_param _out_params[]={\n");
	if (porttype_op->output != NULL) {
		fprintf (out, "\t\t/* params here based on output parts */\n");

		wsdl_emit_part_list (out,
				     porttype_op->output->thread_soap_parts,
				     "\t\t{\"%p\", _out_%p, &WSDL_TC_%n_%N_struct},\n");
	}
	fprintf (out, "\t\t{NULL, NULL, NULL},\n");
	fprintf (out, "\t};\n");

	fprintf (out, "\n");
	fprintf (out, "\tif (soup_env_get_address(env) != NULL)\n");
	fprintf (out,
		 "\t\t_soup_context=soup_context_get(soup_env_get_address(env));\n");
	fprintf (out, "\telse\n");
	fprintf (out, "\t\t_soup_context=soup_context_get(\"%s\");\n",
		 op->soap_operation->soapAction);

	fprintf (out, "\t_message=soup_message_new(_soup_context, \"%s\");\n",
		 op->name);
	fprintf (out, "\tsoup_context_unref(_soup_context);\n");

	fprintf (out,
		 "\twsdl_soap_marshal(\"%s\", \"%s\", \"%s\", _in_params, &_message->request, env, WSDL_SOAP_FLAGS_REQUEST);\n",
		 op->name, 
		 opns, 
		 opnsuri);
	fprintf (out, "\tsoup_message_send(_message);\n");
	fprintf (out,
		 "\twsdl_soap_parse(_message->response.body, \"%s\", _out_params, env, WSDL_SOAP_FLAGS_RESPONSE);\n",
		 op->name);

	fprintf (out, "}\n\n");

	fprintf (out, "/* END Binding Operation %s */\n\n", op->name);
}

static void
wsdl_emit_soap_stubs_binding (FILE                      *out, 
			      const guchar              *opns,
			      const guchar              *opnsuri,
			      const wsdl_binding * const binding)
{
	GSList *iter;
	wsdl_porttype *porttype;

	g_assert (binding->name != NULL);

	fprintf (out, "/* BEGIN Binding %s */\n\n", binding->name);

	if (binding->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", binding->documentation->str);
	}

	porttype = binding->thread_soap_porttype;
	g_assert (porttype != NULL);

	/* For each binding operation, output a function */
	iter = binding->operations;
	while (iter != NULL) {
		wsdl_emit_soap_stubs_binding_operation (out, 
							opns, 
							opnsuri,
							iter->data);

		iter = iter->next;
	}

	fprintf (out, "/* END Binding %s */\n\n", binding->name);
}

static void
wsdl_emit_soap_stubs_service (FILE                      *out, 
			      const guchar              *opns,
			      const guchar              *opnsuri,
			      const wsdl_service * const service)
{
	GSList *iter;

	g_assert (service->name != NULL);

	fprintf (out, "/* BEGIN Service %s */\n\n", service->name);

	if (service->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", service->documentation->str);
	}

	iter = service->thread_soap_ports;
	while (iter != NULL) {
		wsdl_service_port *port = iter->data;

		g_assert (port->thread_soap_binding != NULL);

		wsdl_emit_soap_stubs_binding (out, 
					      opns, 
					      opnsuri,
					      port->thread_soap_binding);

		iter = iter->next;
	}

	fprintf (out, "/* END Service %s */\n\n", service->name);
}

/**
 * wsdl_emit_soap_stubs:
 * @outdir: a string containing the path to a directory.  This
 * function expects the string to have a trailing '/'.
 * @fileroot: a string containing the root of a filename.  "-stubs.c"
 * will be appended to this name.
 * @definitions: a pointer to a #wsdl_definitions structure,
 * containing a set of WSDL elements.
 *
 * Creates the file @outdir/@fileroot-stubs.c, and writes C code
 * containing client stubs.
 */
void
wsdl_emit_soap_stubs (const guchar * outdir, const guchar * fileroot,
		      const wsdl_definitions * const definitions)
{
	FILE *out;
	GSList *iter;
	guchar *filename;
	const guchar *opns;

	filename = g_strconcat (outdir, fileroot, "-stubs.c", NULL);
	wsdl_debug (WSDL_LOG_DOMAIN_STUBS, 
		    G_LOG_LEVEL_DEBUG, 
		    "file: [%s]",
		    filename);

	out = fopen (filename, "w");
	g_free (filename);

	if (out == NULL) {
		g_warning ("Couldn't open %s for writing: %s", filename,
			   strerror (errno));
		return;
	}

	fprintf (out, "/*\n");
	if (definitions->name != NULL) {
		fprintf (out, " * %s\n", definitions->name);
		opns = definitions->name;
	} else {
		opns = "m";
	}

	fprintf (out, " *\n");
	fprintf (out, " * Automatically generated by soup-wsdl.\n");
	fprintf (out, " */\n");
	fprintf (out, "\n");
	fprintf (out, "#include <glib.h>\n");
	fprintf (out, "#include <libsoup/soup.h>\n");
	fprintf (out, "#include <libwsdl/wsdl.h>\n");
	fprintf (out, "#include \"%s.h\"\n\n", fileroot);

	if (definitions->documentation != NULL) {
		fprintf (out, "/* %s */\n\n", definitions->documentation->str);
	}

	/* Declare structure for passing callback data */
	fprintf (out, "typedef struct {\n");
	fprintf (out, "\tgpointer callback;\n");
	fprintf (out, "\tgpointer user_data;\n");
	fprintf (out, "\tSoupEnv *env;\n");
	fprintf (out, "} _wsdl_stubs_callback_data_t;\n\n");

	/* Private functions */
	fprintf (out,
		 "/* g_hash_table_foreach function to add a request header */\n");
	fprintf (out,
		 "static void add_response_header (gpointer key, gpointer value, gpointer data)\n");
	fprintf (out, "{\n");
	fprintf (out, "\tSoupEnv *env = (SoupEnv *) data;\n");
	fprintf (out, "\n");
	fprintf (out,
		 "\tsoup_env_set_response_header(env, (const gchar *) key, (const gchar *) data);\n");
	fprintf (out, "}\n\n");

	iter = definitions->thread_soap_services;
	while (iter != NULL) {
		wsdl_emit_soap_stubs_service (out, 
					      opns,
					      definitions->targetNamespace,
					      iter->data);

		iter = iter->next;
	}

	fclose (out);
}
