<?php
/**
 * Forwards_Driver_ldap:: implements the Forwards_Driver API for LDAP driven
 * mail servers.
 *
 * $Horde: forwards/lib/Driver/ldap.php,v 1.10 2006/02/16 16:34:43 jan Exp $
 *
 * Copyright 2001-2006 Eric Rostetter <eric.rostetter@physics.utexas.edu>
 *
 * See the enclosed file LICENSE for license information (ASL). If you
 * did not receive this file, see http://www.horde.org/licenses/asl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @package Forwards
 */
class Forwards_Driver_ldap extends Forwards_Driver {

    /**
     * Pointer to the ldap connection.
     *
     * @var resource
     */
    var $_ds;

    /**
     * Check if the realm has a specific configuration.
     *
     * If not, try to fall back on the default configuration.  If
     * still not a valid configuration then exit with an error.
     *
     * @param string $realm  The realm of the user, or "default" if none.
     *                       Note: passed by reference so we can change
     *                       its value!
     */
    function checkConfiguration(&$realm)
    {
        // If no realm passed in, or no host config for the realm
        // passed in, then we fall back to the default realm.
        if (empty($realm) || empty($this->_params[$realm]['server'])) {
            $realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$realm]['host']) ||
            empty($this->_params[$realm]['port']) ) {
            $this->_error = _("The module is not properly configured!");
            return false;
        }
        return true;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox. (Ignored)
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm, $password, $target, $keeplocal)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Get the user's DN.
        if (isset($this->_params[$realm]['userdn'])) {
            $userdn = $this->_params[$realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn($user, $realm);
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $res = $this->_connect($userdn, $password, $realm);
        if (is_a($res, 'PEAR_Error')) {
            $this->_error = $res->getMessage();
            $this->_error .= ' - ' .  _("Check your password");
            return false;
        }

        // Change the user's forwards.
        $newDetails[$this->_params[$realm]['forwards']] = $target;
        $res = ldap_mod_replace($this->_ds, $userdn, $newDetails);
        if (!$res) {
            $res = PEAR::raiseError(ldap_error($this->_ds));
        }

        // Disconnect from the ldap server.
        $this->_disconnect();

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm, $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Get the user's DN.
        if (isset($this->_params[$realm]['userdn'])) {
            $userdn = $this->_params[$realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn($user, $realm);
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $res = $this->_connect($userdn, $password, $realm);
        if (is_a($res, 'PEAR_Error')) {
            $this->_error = $res->getMessage();
            $this->_error .= ' - ' .  _("Check your password");
            return false;
        }

        // Delete the user's forwards.
        $attribs = array($this->_params[$realm]['forwards']);
        $value = $this->_getForwards($userdn, $attribs);
        if (!$value) {
            // Nothing to delete, so treat as success.
            return true;
        }

        $newDetails[$this->_params[$realm]['forwards']] = $value;
        $res = ldap_mod_del($this->_ds, $userdn, $newDetails);
        if (!$res) {
            $res = PEAR::raiseError(ldap_error($this->_ds));
        }

        // Disconnect from the ldap server.
        $this->_disconnect();

        return $res;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false otherwise.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Get the user's DN.
        if (isset($this->_params[$realm]['userdn'])) {
            $userdn = $this->_params[$realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn($user, $realm);
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $res = $this->_connect($userdn, $password, $realm);
        if (is_a($res, 'PEAR_Error')) {
            $this->_disconnect();
            if ($res->getMessage() == _("Could not bind to ldap server")) {
                return PEAR::raiseError(_("Incorrect Password"));
            }
            return $res;
        }

        $attribs = array($this->_params[$realm]['forwards']);
        if ($this->_getForwards($userdn, $attribs)) {
            $this->_disconnect();
            return 'Y';
        } else {
            $this->_disconnect();
            return 'N';
        }
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false on error.
     */
    function currentTarget($user, $realm = 'default', $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Get the user's dn.
        if (isset($this->_params[$realm]['userdn'])) {
            $userdn = $this->_params[$realm]['userdn'];
        } else {
            $userdn = $this->_lookupdn($user, $realm);
            if (is_a($userdn, 'PEAR_Error')) {
                return $userdn;
            }
        }

        // Connect as the user.
        $res = $this->_connect($userdn, $password, $realm);
        if (is_a($res, 'PEAR_Error')) {
            $this->_disconnect();
            if ($res->getMessage() == _("Could not bind to ldap server")) {
                return PEAR::raiseError(_("Incorrect Password"));
            }
            return $res;
        }

        $attribs = array($this->_params[$realm]['forwards']);

        return $this->_getForwards($userdn, $attribs);
    }

    /**
     * Lookup and return the user's dn.
     *
     * @param string $user   The username of the user.
     * @param string $realm  The realm of the user.
     *
     * @return string  The ldap dn for the user.
     */
    function _lookupdn($user, $realm)
    {
        // Bind as guest.
        $this->_connect();

        // Construct search.
        $search = $this->_params[$realm]['uid'] . '=' . $user;
        if (!empty($this->_params[$realm]['realm'])) {
            $search .= '@' . $this->_params[$realm]['realm'];
        }

        // Get userdn.
        $result = ldap_search($this->_ds, $this->_params[$realm]['basedn'], $search);
        $entry = ldap_first_entry($this->_ds, $result);
        if ($entry === false) {
            $this->_disconnect();
            return PEAR::raiseError(_("User not found."));
        }
        $userdn = ldap_get_dn($this->_ds, $entry);

        // Disconnect from ldap server.
        $this->_disconnect();

        return $userdn;
    }

    function _getForwards($userdn, $attribs)
    {
        $sr = ldap_search($this->_ds, $userdn, 'uid=*', $attribs);
        $entry = ldap_first_entry($this->_ds, $sr);
        $res = ldap_get_attributes($this->_ds, $entry);
        if ($res['count'] == 0) {
            return false;
        }

        $values = ldap_get_values($this->_ds, $entry, $attribs[0]);
        return $values[0];
    }

    /**
     * Connects to an LDAP server and binds as the guest user or as the
     * optional userdn.
     *
     * @param string $userdn    The DN to use when binding non-anonymously.
     * @param string $password  The password for $userdn.
     * @param string $realm     The realm of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function _connect($userdn = null, $password = null, $realm = 'default')
    {
        $this->_ds = ldap_connect($this->_params[$realm]['host'], $this->_params[$realm]['port']);
        if (!$this->_ds) {
            return PEAR::raiseError(_("Could not connect to ldap server"));
        }
        if (isset($this->_params[$realm]['version'])) {
            ldap_set_option($this->_ds, LDAP_OPT_PROTOCOL_VERSION,
                            $this->_params[$realm]['version']);
        }

        if (!is_null($userdn)) {
            $result = @ldap_bind($this->_ds, $userdn, $password);
        } else {
            $result = @ldap_bind($this->_ds);
        }

        if (!$result) {
            return PEAR::raiseError(_("Could not bind to ldap server"));
        }

        return true;
    }

    /**
     * Close the ldap connection.
     */
    function _disconnect()
    {
        @ldap_close($this->_ds);
    }

}
