/* Somaplayer_controller - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                       <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#include <controller/somaplayer.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <gtk/gtk.h>

#define NAME_TOOL "somaplayer_controller"

typedef struct
{
  GtkWidget *v;
  GtkWidget *b;
  GtkWidget *sb;
  GtkWidget *tv;

  int sb_id;

  int no_rec;
  int no_timer;

  somaplayer_controller *c;

} vb_scales;

static struct termios t1, t2;
static vb_scales *scales;
static GtkWidget *popup;
static char *drag_data = NULL;
static int text_mode = 0;

void playlist_remove (GtkWidget *, somaplayer_controller *);
void playlist_play (GtkWidget *, somaplayer_controller *);

void
echo_off (void)
{
  tcgetattr (STDIN_FILENO, &t1);
  t2 = t1;

  t2.c_lflag &= ~(ICANON | ECHO);
  tcsetattr (STDIN_FILENO, TCSADRAIN, &t2);
}

void
echo_on (void)
{
  tcsetattr (STDIN_FILENO, TCSADRAIN, &t1);
}

void
quit (void)
{
  puts ("Error: server connection.");
  if (text_mode)
    echo_on ();
  exit (1);
}

void
b_prev (GtkWidget * w, somaplayer_controller * c)
{
  if (somaplayer_prev (c))
    quit ();
}

void
b_next (GtkWidget * w, somaplayer_controller * c)
{
  if (somaplayer_next (c))
    quit ();
}

void
drag_begin (GtkWidget * w, GdkDragContext * context,
	    somaplayer_controller * c)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GList *list;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (scales->tv));
  list = gtk_tree_selection_get_selected_rows (selection, &model);
  if (list)
    {
      gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, list->data);
      gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, 0, &drag_data, -1);
      gtk_tree_path_free (list->data);
    }
  else
    drag_data = NULL;
}

void
drag_end (GtkWidget * w, GdkDragContext * context, somaplayer_controller * c)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  int id, i;
  char *file;

  if (!drag_data)
    return;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (w));
  if (gtk_tree_model_get_iter_first (model, &iter) == FALSE)
    return;

  i = 0;
  do
    {
      gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, 0, &file, 1, &id,
			  -1);

      if (!strcmp (drag_data, file))
	{
	  somaplayer_move_playlist (c, id, i ? i - 1 : i);
	  break;
	}

      i++;
    }
  while (gtk_tree_model_iter_next (GTK_TREE_MODEL (model), &iter));

}

void
b_quit (GtkWidget * w, somaplayer_controller * c)
{
  if (somaplayer_quit (c))
    quit ();

  exit (0);
}

void
b_pause (GtkWidget * w, somaplayer_controller * c)
{
  if (somaplayer_pause (c))
    quit ();
}

void
s_volume (GtkWidget * w, somaplayer_controller * c)
{
  GtkAdjustment *v;

  if (scales->no_rec)
    return;

  scales->no_timer = 1;

  v = gtk_range_get_adjustment (GTK_RANGE (w));

  if (somaplayer_set_volume (c, (int) v->value))
    quit ();

  scales->no_timer = 0;
}

void
s_balance (GtkWidget * w, somaplayer_controller * c)
{
  GtkAdjustment *v;

  if (scales->no_rec)
    return;

  scales->no_timer = 1;

  v = gtk_range_get_adjustment (GTK_RANGE (w));

  if (v->value > 45 && v->value < 55)
    {
      gtk_adjustment_set_value (GTK_ADJUSTMENT (v), 50);
      if (somaplayer_set_balance (c, 0))
	quit ();

      scales->no_timer = 0;
      return;
    }

  if (somaplayer_set_balance (c, ((int) v->value * 2) - 100))
    quit ();

  scales->no_timer = 0;
}

GtkWidget *
create_menu (somaplayer_controller * c)
{
  GtkWidget *menu;
  GtkWidget *item;

  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_mnemonic ("Play");
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (menu), item);
  g_signal_connect ((gpointer) item, "activate", G_CALLBACK (playlist_play),
		    c);

  item = gtk_menu_item_new_with_mnemonic ("Remove");
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (menu), item);
  g_signal_connect ((gpointer) item, "activate", G_CALLBACK (playlist_remove),
		    c);

  return menu;
}

gboolean
show_popup (GtkWidget * w, GdkEventButton * event, gpointer menu)
{
  if (event->type == GDK_2BUTTON_PRESS)
    {
      GtkTreeSelection *selection;
      GtkTreeModel *model;
      GtkTreeIter iter;
      GList *list;
      int id;

      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (scales->tv));
      list = gtk_tree_selection_get_selected_rows (selection, &model);
      if (list)
	{
	  gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, list->data);
	  gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, 1, &id, -1);
	  gtk_tree_path_free (list->data);

	  somaplayer_play_playlist (scales->c, id);

	  g_list_free (list);

	}
    }

  else if (event->button == 3)
    {
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, event->button,
		      event->time);

      return TRUE;
    }

  return FALSE;
}

void
create_window (somaplayer_controller * c, int v, int b)
{
  GtkWidget *window;
  GtkWidget *box;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *scale;
  GtkAdjustment *adj;
  GtkWidget *statusbar;
  GtkWidget *tv;
  GtkListStore *model;
  GtkTreeSelection *selection;
  GtkCellRenderer *cell;
  GtkTreeViewColumn *column;
  GtkWidget *sw;

  scales->no_rec = 1;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Somaplayer Controller");
  g_signal_connect ((gpointer) window, "destroy", G_CALLBACK (gtk_main_quit),
		    NULL);

  box = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (box);
  gtk_container_add (GTK_CONTAINER (window), box);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (box), hbox, TRUE, TRUE, 0);

  frame = gtk_frame_new ("Somaplayer Controller");
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  popup = create_menu (c);

  sw = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (sw);
  gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_IN);

  model = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_INT);
  tv = gtk_tree_view_new ();
  gtk_widget_set_size_request (tv, 300, -1);
  gtk_widget_show (tv);
  gtk_tree_view_set_model (GTK_TREE_VIEW (tv), GTK_TREE_MODEL (model));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tv), TRUE);
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tv), TRUE);
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tv), FALSE);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);
  gtk_container_add (GTK_CONTAINER (sw), tv);
  g_signal_connect (tv, "button-press-event", G_CALLBACK (show_popup), popup);
  g_signal_connect (tv, "drag_end", G_CALLBACK (drag_end), c);
  g_signal_connect (tv, "drag_begin", G_CALLBACK (drag_begin), c);

  scales->tv = tv;

  g_object_unref (model);

  cell = gtk_cell_renderer_text_new ();
  column =
    gtk_tree_view_column_new_with_attributes ("Playlist", cell, "text", 0,
					      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv),
			       GTK_TREE_VIEW_COLUMN (column));

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_mnemonic ("Prev");
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) button, "clicked", G_CALLBACK (b_prev), c);

  button = gtk_button_new_with_mnemonic ("Next");
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) button, "clicked", G_CALLBACK (b_next), c);

  button = gtk_button_new_with_mnemonic ("Pause");
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) button, "clicked", G_CALLBACK (b_pause), c);

  button = gtk_button_new_with_mnemonic ("Quit");
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) button, "clicked", G_CALLBACK (b_quit), c);

  adj = (GtkAdjustment *) gtk_adjustment_new ((gdouble) v, 0, 100, 5, 5, 0);
  scale = gtk_hscale_new (adj);
  gtk_widget_show (scale);
  gtk_box_pack_start (GTK_BOX (vbox), scale, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) scale, "value_changed", G_CALLBACK (s_volume),
		    c);

  scales->v = scale;

  adj =
    (GtkAdjustment *)
    gtk_adjustment_new ((gdouble) ((b + 100) != 0 ? (b + 100) / 2 : 0), 0,
			100, 5, 5, 0);
  scale = gtk_hscale_new (adj);
  gtk_widget_show (scale);
  gtk_box_pack_start (GTK_BOX (vbox), scale, FALSE, FALSE, 0);
  g_signal_connect ((gpointer) scale, "value_changed", G_CALLBACK (s_balance),
		    c);

  scales->b = scale;

  statusbar = gtk_statusbar_new ();
  gtk_widget_show (statusbar);
  gtk_box_pack_start (GTK_BOX (box), statusbar, FALSE, FALSE, 0);

  scales->sb = statusbar;
  scales->sb_id =
    gtk_statusbar_get_context_id (GTK_STATUSBAR (statusbar), "");

  gtk_widget_show (window);
}

void
timeout_refresh (somaplayer_controller * c)
{
  GtkListStore *model;
  GtkTreeIter iter;
  char **pls;
  static char **prev = NULL;
  int i = 0;
  int ref = 0;

  if (!scales->tv)
    return;

  if (!(pls = somaplayer_get_playlist (c)))
    {
      if (prev != NULL)
	{
	  gtk_widget_hide (popup);

	  model =
	    (GtkListStore *)
	    gtk_tree_view_get_model (GTK_TREE_VIEW (scales->tv));
	  while (gtk_tree_model_iter_nth_child
		 (GTK_TREE_MODEL (model), &iter, NULL, 0))
	    gtk_list_store_remove (model, &iter);

	  free (prev);
	  prev = NULL;

	}

      return;
    }

  if (prev)
    {
      while (pls[i])
	{
	  if (!prev || !prev[i] || strcmp (pls[i], prev[i]))
	    ref = 1;

	  i++;
	}
    }
  else
    ref = 1;

  if (!ref)
    {
      free (pls);
      return;
    }

  gtk_widget_hide (popup);

  model =
    (GtkListStore *) gtk_tree_view_get_model (GTK_TREE_VIEW (scales->tv));
  while (gtk_tree_model_iter_nth_child
	 (GTK_TREE_MODEL (model), &iter, NULL, 0))
    gtk_list_store_remove (model, &iter);

  i = 0;
  while (pls[i])
    {
      gtk_list_store_append (model, &iter),
	gtk_list_store_set (model, &iter, 0, pls[i], 1, i, -1);

      i++;
    }

  if (prev)
    free (prev);

  prev = pls;
}

gint
timeout (somaplayer_controller * c)
{
  int vol, bal;
  double v, b;
  GtkAdjustment *adj;
  char *file;

  static int k = 0;

  if (k == 0)
    {
      k = 0;

      timeout_refresh (c);
    }
  else if (k == 10)
    k = 0;

  else
    k++;

  if (scales->no_timer)
    return TRUE;

  if (somaplayer_get_volume (c, &vol))
    quit ();

  if (somaplayer_get_balance (c, &bal))
    quit ();

  if (!(file = somaplayer_this (c)))
    quit ();

  v = (double) vol;
  b = (double) bal;

  adj = gtk_range_get_adjustment (GTK_RANGE (scales->v));
  if (adj->value != v)
    {
      scales->no_rec = 1;
      gtk_adjustment_set_value (adj, v);
      scales->no_rec = 0;
    }

  adj = gtk_range_get_adjustment (GTK_RANGE (scales->b));

  if (adj->value != ((b + 100) != 0 ? (b + 100) / 2 : 0))
    {
      scales->no_rec = 1;
      gtk_adjustment_set_value (adj,
				((b + 100) / 2) != 0 ? (b + 100) / 2 : 0);
      scales->no_rec = 0;
    }

  if (file)
    {
      gtk_statusbar_pop (GTK_STATUSBAR (scales->sb), scales->sb_id);
      scales->sb_id =
	gtk_statusbar_push (GTK_STATUSBAR (scales->sb), scales->sb_id, file);
      free (file);
    }

  return TRUE;
}

int
usage (void)
{
  fprintf (stderr,
	   "Usage:\n\n"
	   "\t%s [unix://file] or [tcp://server[:port]] (options)\n\n"
	   "You can run:\n\n" "\t%s [unix] or [tcp]\n"
	   "for generic connections.\n\nOptions:\n\t-t or --text\t\tfor text mode\n\n",
	   NAME_TOOL, NAME_TOOL);
  return 1;
}

somaplayer_controller *
parse (char *a)
{

  if (!strcmp (a, "unix"))
    return somaplayer_open_unix (NULL);

  if (!strcmp (a, "tcp"))
    return somaplayer_open_tcp (NULL, 0);

  if (!strncmp (a, "unix://", 7))
    {
      char *file;
      file = a + 7;

      return somaplayer_open_unix (file);
    }

  if (!strncmp (a, "tcp://", 6))
    {
      char *server;
      int i, port = 0;

      server = a + 6;

      for (i = 0; i < strlen (server); i++)
	{
	  if (server[i] == ':')
	    {

	      if (i + 1 < strlen (server))
		{
		  port = atoi (&server[i + 1]);
		}

	      server[i] = 0;
	    }
	}

      return somaplayer_open_tcp (server, port);
    }

  return NULL;
}

int
main (int argc, char *argv[])
{
  somaplayer_controller *c;
  int v, b;
  int i;
  char *data = NULL;

  for (i = 1; i < argc; i++)
    {
      if (!strcmp (argv[i], "-h") || !strcmp (argv[i], "--help"))
	return usage ();
      else if (!strcmp (argv[i], "-t") || !strcmp (argv[i], "--text"))
	text_mode = 1;
      else if (!data)
	data = argv[i];
      else
	return usage ();
    }

  if (!data)
    {
      puts ("Argoment error!");
      return 1;
    }

  if(!getenv("DISPLAY")) text_mode=1;

  if (!(c = parse (data)))
    {
      puts ("Argoment error!");
      return 1;
    }

  if (!text_mode)
    {
      if (!(scales = (vb_scales *) malloc (sizeof (vb_scales))))
	{
	  puts ("Error: memory.");
	  return 1;
	}

      if (somaplayer_get_volume (c, &v))
	quit ();
      if (somaplayer_get_balance (c, &b))
	quit ();

      gtk_init (NULL, NULL);
      create_window (c, v, b);

      scales->no_rec = 0;
      scales->no_timer = 0;
      scales->c = c;

      g_timeout_add (800, timeout, c);

      gtk_main ();

      if (somaplayer_close (c))
	quit ();

    }
  else
    {
      char buf[80];
      int q = 0;
      char *file;
      fd_set fd_read;
      struct timeval tv;
      char ch;
      int flag;

      flag = fcntl (1, F_GETFL, 0);
      fcntl (1, F_SETFL, flag | O_NONBLOCK);

      echo_off();

      v = snprintf (buf, 80, NAME_TOOL);
      v = (80 - v) ? (80 - v) / 2 : 0;
      for (b = 0; b < v; b++)
	putchar (' ');
      fprintf (stdout, "%s", buf);
      for (b = 0; b < v; b++)
	putchar (' ');
      putchar ('\n');

      for (b = 0; b < 80; b++)
	putchar ('-');
      putchar ('\n');

      v =
	snprintf (buf, 80,
		  "Enter = exit | Space = pause | q = quit | n = next | p = prev");
      v = (80 - v) ? (80 - v) / 2 : 0;
      for (b = 0; b < v; b++)
	putchar (' ');
      fprintf (stdout, "%s", buf);
      for (b = 0; b < v; b++)
	putchar (' ');
      putchar ('\n');

      v =
	snprintf (buf, 80,
		  "v <volume | V>volume | b<balance | B>balance | c = center");
      v = (80 - v) ? (80 - v) / 2 : 0;
      for (b = 0; b < v; b++)
	putchar (' ');
      fprintf (stdout, "%s", buf);
      for (b = 0; b < v; b++)
	putchar (' ');
      putchar ('\n');

      v = snprintf (buf, 80, "l = show list | r = remove item");
      v = (80 - v) ? (80 - v) / 2 : 0;
      for (b = 0; b < v; b++)
	putchar (' ');
      fprintf (stdout, "%s", buf);
      for (b = 0; b < v; b++)
	putchar (' ');
      putchar ('\n');

      for (b = 0; b < 80; b++)
	putchar ('-');
      putchar ('\n');

      while (!q)
	{
	  if (somaplayer_get_volume (c, &v))
	    quit ();
	  if (somaplayer_get_balance (c, &b))
	    quit ();

	  file = somaplayer_this (c);

	  i = snprintf (buf, 79, "[ V: %3d - B: %4d ] - %s", v, b,
			file ? file : "no playing");
	  for (; i < 80; i++)
	    buf[i] = ' ';

	  buf[79] = '\r';

	  write (1, buf, 80);

	  free (file);

	  FD_ZERO (&fd_read);
	  FD_SET (0, &fd_read);

	  tv.tv_sec = 0;
	  tv.tv_usec = 500000;

	  select (1, &fd_read, NULL, NULL, &tv);

	  if (read (0, &ch, 1) != 1)
	    continue;

	  switch (ch)
	    {
	    case '\n':
	      q = 1;
	      break;

	    case 'q':
	    case 'Q':
	      somaplayer_quit (c);
	      q = 1;
	      break;

	    case ' ':
	      somaplayer_pause (c);
	      break;

	    case 'p':
	    case 'P':
	      somaplayer_prev (c);
	      break;

	    case 'n':
	    case 'N':
	      somaplayer_next (c);
	      break;

	    case 'v':
	      somaplayer_set_volume (c, v - 5);
	      break;

	    case 'V':
	      somaplayer_set_volume (c, v + 5);
	      break;

	    case 'b':
	      somaplayer_set_balance (c, b - 5);
	      break;

	    case 'B':
	      somaplayer_set_balance (c, b + 5);
	      break;

	    case 'c':
	    case 'C':
	      somaplayer_set_balance (c, 0);
	      break;

	    case 'l':
	    case 'L':
	      {
		char **playlist, **old;
		int n = 0;

		playlist = somaplayer_get_playlist (c);

		fprintf (stdout, "\n\nPlaylist:\n");

		if (!playlist)
		  fprintf (stdout, "\n\nNo Playlist!\n\n");

		else
		  {
		    old = playlist;

		    while (*playlist)
		      {
			n++;
			fprintf (stdout, "%d - %s\n", n, *playlist);
			playlist++;
		      }

		    free (old);
		    fprintf (stdout, "\n\n");
		  }

		break;
	      }

	    case 'r':
	    case 'R':
	      {
		int n;

		fprintf (stdout, "\n\nItem: ");
		echo_on ();
		fscanf (stdin, "%d", &n);
		echo_off ();

		somaplayer_del_playlist (c, n);

		break;
	      }

	    case 'm':
	    case 'M':
	      {
		int n, before;

		fprintf (stdout, "\n\nItem and where: ");
		echo_on ();
		fscanf (stdin, "%d %d", &n, &before);
		echo_off ();

		somaplayer_move_playlist (c, n, before);

		break;
	      }

	    }
	}

      fcntl (1, F_SETFL, flag);

      if (somaplayer_close (c))
	quit ();

      echo_on ();

      return 0;
    }

  return 0;
}

void
playlist_remove (GtkWidget * w, somaplayer_controller * c)
{
  GList *list, *old;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  int k, j = 0;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (scales->tv));
  if (!(list = gtk_tree_selection_get_selected_rows (selection, &model)))
    return;

  old = list;

  while (list)
    {
      k = gtk_tree_path_get_indices (list->data)[0];
      gtk_tree_path_free (list->data);

      somaplayer_del_playlist (c, k - j);
      j++;

      list = list->next;
    }

  timeout_refresh (c);

  g_list_free (old);
}

void
playlist_play (GtkWidget * w, somaplayer_controller * c)
{
  GList *list;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  int k;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (scales->tv));
  if (!(list = gtk_tree_selection_get_selected_rows (selection, &model)))
    return;

  k = gtk_tree_path_get_indices (list->data)[0];
  gtk_tree_path_free (list->data);

  somaplayer_play_playlist (c, k);

  g_list_free (list);
}
