#include <string.h>
#include <stdio.h>
#include <stdint.h>
#include <cpu_ppc.h>
#include <idecode_ppc.h>
#include <setjmp.h>
#include <cycletimer.h>
#include "mmu_ppc.h"

uint32_t cpu_signals;

void
PpcSetMsr(uint32_t value) {
	gcpu.msr = value;
	gcpu.msr_ee = value & MSR_EE; 	// External Interrupts
        gcpu.msr_pr = value & MSR_PR; 	// Privilege level
        gcpu.msr_me = value & MSR_ME; 	// Machine Check Exceptions enable
        gcpu.msr_fe = ((value & MSR_FE0) >> (MSR_FE0_SHIFT-1))
			| ((value & MSR_FE1) >> MSR_FE1_SHIFT); // floating point exception mode
        gcpu.msr_ip = value & MSR_IP; 	// exception prefix
        gcpu.msr_ir = value & MSR_IR;  // translate instruction
        gcpu.msr_dr = value & MSR_DR;  // translate data
	if(value & (1 | (1<<16))) {
		fprintf(stderr,"MSR: Little endian mode not supported\n");
	}	
	return;
}
/*
 * --------------------------------------
 * Create a pointer table for the SPRs
 * --------------------------------------
 */
static void
PpcCpu_CreateSprs(PpcCpu *cpu) 
{
	cpu->spr[1] = &cpu->xer;	
	cpu->spr[8] = &cpu->lr;	
	cpu->spr[9] = &cpu->ctr;	
	cpu->spr[18] = &cpu->dsisr;
	cpu->spr[19] = &cpu->dar;
	cpu->spr[22] = &cpu->dec;
	cpu->spr[25] = &cpu->sdr1;
	cpu->spr[26] = &cpu->srr0;
	cpu->spr[27] = &cpu->srr1;
	cpu->spr[268] = &cpu->tbl;
	cpu->spr[269] = &cpu->tbu;

	cpu->spr[272] = &cpu->sprg0;	
	cpu->spr[273] = &cpu->sprg1;	
	cpu->spr[274] = &cpu->sprg2;	
	cpu->spr[275] = &cpu->sprg3;	
	cpu->spr[282] = &cpu->ear;
	cpu->spr[287] = &cpu->pvr;
	cpu->spr[528] = &cpu->ibat0u;
	cpu->spr[529] = &cpu->ibat0l;
	cpu->spr[530] = &cpu->ibat1u;
	cpu->spr[531] = &cpu->ibat1l;
	cpu->spr[532] = &cpu->ibat2u;
	cpu->spr[533] = &cpu->ibat2l;
	cpu->spr[534] = &cpu->ibat3u;
	cpu->spr[535] = &cpu->ibat3l;

	cpu->spr[536] = &cpu->dbat0u;
	cpu->spr[537] = &cpu->dbat0l;
	cpu->spr[538] = &cpu->dbat1u;
	cpu->spr[539] = &cpu->dbat1l;
	cpu->spr[540] = &cpu->dbat2u;
	cpu->spr[541] = &cpu->dbat2l;
	cpu->spr[542] = &cpu->dbat3u;
	cpu->spr[543] = &cpu->dbat3l;
	cpu->spr[1013] = &cpu->dabr;
	cpu->spr[1022] = &cpu->fpecr;
	cpu->spr[1023] = &cpu->pir;
		
}

PpcCpu gcpu;
/*
 * -----------------------------
 * Constructor for the CPU
 * -----------------------------
 */
PpcCpu *
PpcCpu_New(int cpu_type,uint32_t initial_msr) {
	PpcCpu *cpu = &gcpu; /* global is faster */
	if(!cpu) {
		fprintf(stderr,"Out of memory allocating PowerPC CPU\n");
		exit(345);
	}
	memset(cpu,0,sizeof(PpcCpu));	
	/*  exception Vectors at 0xfffxxxxx p.77      */
	PpcSetMsr(initial_msr); 
	switch(cpu_type) {
		case CPU_MPC852T:
			PVR=0x00500000; /* Processor Version from MPC866UM p.134 3-18 */
			break;
		default:
			fprintf(stderr,"CPU-type %d not implemented\n",cpu_type);
			exit(175);
	}
	PpcCpu_CreateSprs(cpu); 
	IDecoder_New(cpu_type);
	return cpu;
}

static inline void
CheckSignals() {

}

/*
 * --------------------------------------------------
 * Register a Read and a write function for a SPR
 * --------------------------------------------------
 */

void 
Ppc_RegisterSprHandler(PpcCpu *cpu,unsigned int spr,SPR_ReadProc *rproc,SPR_WriteProc *wproc, void *cd) 
{
	if(spr<1024) {
		cpu->spr_read[spr] = rproc;
		cpu->spr_write[spr] = wproc;
		cpu->spr_clientData[spr] = cd;
	}
}

void
PpcCpu_Run(uint32_t start_addr) 
{
        InstructionProc *iproc;
        uint32_t icode;
        fprintf(stderr,"Starting PPC-CPU at %08x\n",start_addr);
        gettimeofday(&gcpu.starttime,NULL);
        NIA=start_addr;
        /* Exceptions use goto (longjmp) */
        setjmp(gcpu.abort_jump);
        while(1) {
		icode=MMU_IFetch(NIA);
		//fprintf(stderr,"addr %08x icode %08x ",NIA,icode);
		NIA+=4;
                iproc=InstructionProcFind(icode);
                iproc(icode);
                CycleCounter+=2;
                CycleTimers_Check();
                CheckSignals();
        }
}

void
_init(void) {
	fprintf(stderr,"PowerPC Emulation Module loaded\n");
}
