/*
 * ----------------------------------------------------
 *
 * Command interpreter for debugging language
 *
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * State: 
 *	nothing is working
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <interpreter.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <xy_hash.h>

/* typedefed in header file */
struct DbgInterp {
	XY_HashTable cmdhash;	
};

/*
 * -----------------------------------
 * Split a script into args 
 * -----------------------------------
 */
static int
split_args(char *script,int maxargc,char *argv[])
{
        int i,argc=0;
	int state=1;
        argv[0]=script;
        for(i=0;script[i];i++) {
		if((state==1) && (script[i]==' ')) {
			continue;
		}
		if(state==1) {
			state=0;
			argv[argc++]=script+i;
			if(argc>=maxargc) {
				return argc;
			}
			continue;
		}
		if((state==0) && (script[i]==' ')) {
			script[i]=0;
			state = 1;
		}
        }
        return argc;
}

/*
 * -------------------------------------------------------------
 * Constructor for debug Interpreter
 * Creates a Hashtable for commands 
 * -------------------------------------------------------------
 */
DbgInterp *
DbgInterpr_New() {
	DbgInterp *interp = malloc(sizeof(DbgInterp));	
	if(!interp) {
		fprintf(stderr,"Out of memory\n");
		exit(5367);
	}
	if(XY_InitHashTable(&interp->cmdhash,XY_STRING_KEYS,8192)<0) {
		fprintf(stderr,"Hash table creation failed\n");
		exit(5377);
	}
	return interp;	
}

void
DbgInterp_Del(DbgInterp *interp) 
{
	XY_DeleteHashTable(&interp->cmdhash);
	free(interp);
}

DbgCmd *
DbgInterp_CreateCmd(DbgInterp *interp,char *cmdname,DbgCmdProc *proc,void *clientData,Dbg_CmdDeleteProc *delproc) 
{
	int isnew;
	DbgCmd *cmd;
	XY_HashEntry *entryPtr;
	cmd = malloc(sizeof(DbgCmd));	
	if(!cmd) {
		fprintf(stderr,"Out of memory\n");
		exit(43249);
	}
	entryPtr = XY_CreateHashEntry(&interp->cmdhash,cmdname,&isnew);
	if(!isnew) {
		// doesn't matter, I overwrite it
	}
	cmd->cmdProc = proc;
	cmd->delProc = delproc;
	XY_SetHashValue(entryPtr,cmd);	
	return cmd;
}

#define MAXARGS (20)
int
DbgInterp_Eval(DbgInterp *interp,char *script) 
{
	XY_HashEntry *entryPtr;
	DbgCmd *cmd;
	char *argv[MAXARGS];
	int argc;
	argc = split_args(script,MAXARGS,argv);
	if(argc<1) {
		return -2;
	}
	entryPtr = XY_FindHashEntry(&interp->cmdhash,argv[0]); 
	if(!entryPtr) {
		fprintf(stderr,"cmd not found\n");
		return -1;
	}
	cmd = XY_GetHashValue(entryPtr);
	return cmd->cmdProc(cmd->clientData,interp,argc,argv);
}
