/*==================================================================
 * util.c - Utility functions
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include "util.h"
#include "pixmap.h"
#include "i18n.h"
#include "widgets/popdog.h"

static GString *log_buf = NULL;
static gint log_maxsize = GBUF_MAXSIZE;

gboolean log_viewactive = FALSE;	/* log window active? */
gint log_poplevel = LogBad;

gint log_groups = 0;		/* # of nested log groups */
gint log_grp_popup_lvl;		/* lowest log level to cause popup */
gint log_grp_highest_lvl;	/* highest occured log level in open group */

static GtkWidget *log_view_widg = NULL;	/* currently active error view widg */

/* unique dialog system data */
typedef struct {
  GtkWidget *dialog;
  gchar *strkey;
  int key2;
} UniqueDialogKey;

gboolean unique_dialog_inited = FALSE;
GArray *unique_dialog_array;

static void util_cb_quick_popup_btn_clicked (GtkWidget *btn,
					     gpointer userdata);
static void util_unique_dialog_init (void);
static void util_cb_waitfor_widget_destroyed (GtkWidget *widg, gpointer data);
static void log_view_cb_destroy (void);

/* pops up a dialog for user input, for simple questions
The arguments should be:
msg, btn1label, btn1func (UtilQuickFunc *), btn1data, btn2label, btn2func...
if btn1label = NULL then a single "OK" button is created,
otherwise a button is created for each label that is provided with a
callback to btnNfunc, if btnNfunc is NULL then gtk_widget_destroy is used */
GtkWidget *
util_quick_popup (gchar * msg, gchar * btn1, ...)
{
  va_list args;
  GtkWidget *popdog;
  GtkWidget *lbl;
  GtkWidget *btn;
  gchar *s;
  UtilQuickFunc func;
  gpointer userdata;
  gint index = 1;

  popdog = gtk_popdog_new (NULL);
  gtk_window_set_modal (GTK_WINDOW (popdog), TRUE);

  lbl = gtk_label_new (msg);
  gtk_widget_show (lbl);
  gtk_box_pack_start (GTK_BOX (GTK_POPDOG (popdog)->vbox), lbl, FALSE, FALSE,
    0);

  va_start (args, btn1);

  if (!btn1)
    {
      btn = gtk_button_new_with_label (_("OK"));
      gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
	(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (popdog));
      gtk_widget_show (btn);
      gtk_box_pack_start (GTK_BOX (GTK_POPDOG (popdog)->action_area), btn,
	FALSE, FALSE, 0);
    }
  else
    {
      s = btn1;
      do
	{
	  btn = gtk_button_new_with_label (s);
	  func = va_arg (args, UtilQuickFunc);
	  userdata = va_arg (args, gpointer);

	  if (!func)		/* use destroy (popdog) as callback if !func */
	    {
	      func = (UtilQuickFunc)gtk_widget_destroy;
	      userdata = popdog;
	    }

	  gtk_object_set_data (GTK_OBJECT (btn), "parent", popdog);
	  gtk_object_set_data (GTK_OBJECT (btn), "func", func);
	  gtk_object_set_data (GTK_OBJECT (btn), "index",
			       GINT_TO_POINTER (index++));
	  gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			      util_cb_quick_popup_btn_clicked, userdata);

	  gtk_widget_show (btn);
	  gtk_box_pack_start (GTK_BOX (GTK_POPDOG (popdog)->action_area), btn,
	    FALSE, FALSE, 0);
	}
      while ((s = va_arg (args, gchar *)));
    }

  va_end (args);

  gtk_widget_show (popdog);
  return (popdog);
}

static void
util_cb_quick_popup_btn_clicked (GtkWidget *btn, gpointer userdata)
{
  GtkWidget *parent;
  UtilQuickFunc func;
  gint index;

  index = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (btn), "index"));
  util_widget_action (btn, GINT_TO_POINTER (index));

  func = gtk_object_get_data (GTK_OBJECT (btn), "func");
  parent = gtk_object_get_data (GTK_OBJECT (btn), "parent");
  (*func) (userdata, parent);
}

/* Unique dialog system is for allowing unique non-modal dialogs for
   resources identified by a string key and an optional additional
   integer key, attempting to open up a second dialog for the same
   resource will cause the first dialog to be brought to the front of
   view and no additional dialog will be created */
static void
util_unique_dialog_init (void)
{
  unique_dialog_array = g_array_new (FALSE, FALSE, sizeof (UniqueDialogKey));
  unique_dialog_inited = TRUE;
}

/* looks up a unique dialog widget by its keys, returns the widget or NULL */
GtkWidget *
util_lookup_unique_dialog (gchar *strkey, gint key2)
{
  UniqueDialogKey *udkeyp;
  gint i;

  if (!unique_dialog_inited)
    util_unique_dialog_init ();

  for (i = unique_dialog_array->len - 1; i >= 0; i--)
    {
      udkeyp = &g_array_index (unique_dialog_array, UniqueDialogKey, i);
      if ((udkeyp->strkey == strkey || strcmp (udkeyp->strkey, strkey) == 0)
	  && udkeyp->key2 == key2)
	return (udkeyp->dialog);
    }

  return (NULL);
}

/* register a unique dialog, if a dialog already exists with the same keys,
   then activate the existing dialog and return FALSE, otherwise register the
   new dialog and return TRUE */
gboolean
util_register_unique_dialog (GtkWidget *dialog, gchar *strkey, gint key2)
{
  UniqueDialogKey udkey;
  GtkWidget *widg;

  if (!unique_dialog_inited)
    util_unique_dialog_init ();

  if ((widg = util_lookup_unique_dialog (strkey, key2)))
    {
      gtk_widget_activate (widg);
      return (FALSE);
    }

  /* tell window manager to remember dialog state (position etc) */
  gtk_window_set_wmclass (GTK_WINDOW (dialog), strkey, "Smurf");

  udkey.dialog = dialog;
  udkey.strkey = strkey;
  udkey.key2 = key2;

  g_array_append_val (unique_dialog_array, udkey);

  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      (GtkSignalFunc)util_unregister_unique_dialog, NULL);

  return (TRUE);
}

void
util_unregister_unique_dialog (GtkWidget *dialog)
{
  UniqueDialogKey *udkeyp;
  gint i;

  if (!unique_dialog_inited)
    util_unique_dialog_init ();

  for (i = unique_dialog_array->len - 1; i >= 0; i--)
    {
      udkeyp = &g_array_index (unique_dialog_array, UniqueDialogKey, i);
      if (udkeyp->dialog == dialog)
	break;
    }

  if (i >= 0)
    g_array_remove_index (unique_dialog_array, i);
}

/* activate (or raise) a unique dialog into view */
gboolean
util_activate_unique_dialog (gchar *strkey, gint key2)
{
  GtkWidget *dialog;

  if ((dialog = util_lookup_unique_dialog (strkey, key2)))
    {
      gdk_window_raise (GTK_WIDGET (dialog)->window);
      return (TRUE);
    }

  return (FALSE);
}

/* run gtk_main loop until the GtkObject data property "action" is !=
   NULL, mates nicely with util_quick_popup, returns value of
   "action".  Useful for complex routines that require a lot of user
   dialog interaction.  Rather than having to save state info and exit
   and return to a routine, a call to this routine can be made which
   will wait for the user's choice and return the index of the button
   (or other user specified value), -1 if the widget was destroyed or
   -2 if gtk_main_quit was called */
gpointer
util_waitfor_widget_action (GtkWidget *widg)
{
  GQuark quark;
  gpointer val = NULL;
  gboolean destroyed = FALSE;
  guint sigid;

  /* initialize the action variable to NULL */
  quark = gtk_object_data_force_id ("action");
  gtk_object_set_data_by_id (GTK_OBJECT (widg), quark, NULL);

  /* already passing one variable to destroy signal handler, so bind this one
     as a GtkObject data item, will notify us if widget was destroyed */
  gtk_object_set_data (GTK_OBJECT (widg), "_destroyed", &destroyed);

  /* val is set to "action" by util_cb_waitfor_widget_destroyed if the
     widget we are waiting for gets killed */
  sigid =
    gtk_signal_connect (GTK_OBJECT (widg), "destroy",
			GTK_SIGNAL_FUNC (util_cb_waitfor_widget_destroyed),
			&val);
  do
    {
      if (gtk_main_iteration ()) /* run the gtk main loop, wait if no events */
	val = GINT_TO_POINTER (-2); /* gtk_main_quit was called, return -2 */
      else if (val == NULL)	/* check the "action" data property */
	val = gtk_object_get_data_by_id (GTK_OBJECT (widg), quark);
    }
  while (val == NULL);		/* loop until "action" is set */

  if (!destroyed)
    gtk_signal_disconnect (GTK_OBJECT (widg), sigid);

  return (val);
}

static void
util_cb_waitfor_widget_destroyed (GtkWidget *widg, gpointer data)
{
  gpointer *val = data;
  gpointer action;
  gboolean *destroyed;

  action = gtk_object_get_data (GTK_OBJECT (widg), "action");
  destroyed = gtk_object_get_data (GTK_OBJECT (widg), "_destroyed");

  *destroyed = TRUE;

  if (action)
    *val = action;
  else *val = GINT_TO_POINTER (-1);
}

/* a callback for widgets (buttons, etc) within a "parent" widget used by
   util_waitfor_widget_action, sets "action" to the specified "value" */
void
util_widget_action (GtkWidget *cbwidg, gpointer value)
{
  GtkWidget *parent;

  parent = gtk_object_get_data (GTK_OBJECT (cbwidg), "parent");
  gtk_object_set_data (GTK_OBJECT (parent), "action", value);
}

GtkWidget *
util_create_pixmap (gchar ** xpmdata)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  pixmap_get (xpmdata, &pixmap, &mask);

  return (gtk_pixmap_new (pixmap, mask));
}

/* log_view - Display log view window */
void
log_view (gchar * title)
{
  GtkWidget *popdog;
  GtkWidget *hbox;
  GtkWidget *msgarea;
  GtkAdjustment *adj;
  GtkWidget *vscrollbar;
  GtkWidget *btn;

  if (log_viewactive) return;

  if (title) popdog = gtk_popdog_new (title);
  else popdog = gtk_popdog_new (_("Smurf log"));

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (GTK_POPDOG (popdog)->vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  msgarea = gtk_text_new (NULL, NULL);
  gtk_widget_set_usize (msgarea, 400, 100);
  gtk_text_set_editable (GTK_TEXT (msgarea), FALSE);
  gtk_text_set_word_wrap (GTK_TEXT (msgarea), FALSE);
  gtk_text_insert (GTK_TEXT (msgarea), NULL, NULL, NULL, log_buf->str, -1);
  gtk_box_pack_start (GTK_BOX (hbox), msgarea, TRUE, TRUE, 0);
  gtk_widget_show (msgarea);

  adj = GTK_TEXT (msgarea)->vadj;	/* get the message area's vert adj */

  vscrollbar = gtk_vscrollbar_new (adj);
  gtk_box_pack_start (GTK_BOX (hbox), vscrollbar, FALSE, FALSE, 0);
  gtk_widget_show (vscrollbar);

  btn = gtk_button_new_with_label (_("OK"));
  gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
    (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (popdog));
  gtk_box_pack_start (GTK_BOX (GTK_POPDOG (popdog)->action_area), btn,
    FALSE, FALSE, 0);
  gtk_widget_show (btn);

  gtk_signal_connect_object (GTK_OBJECT (popdog), "destroy",
    GTK_SIGNAL_FUNC (log_view_cb_destroy), NULL);

  gtk_widget_show (popdog);

  log_viewactive = TRUE;
  log_view_widg = NULL;
}

/* reset dialog active variables */
static void
log_view_cb_destroy (void)
{
  log_viewactive = FALSE;
  log_view_widg = NULL;
}

/* Logging function, returns FAIL to use as a return value in calling funcs */
gint
logit (gint lvl, gchar * fmt, ...)
{
  va_list args;
  gchar *s;
  gboolean syserr;

  syserr = (lvl & LogErrno) > 0;

  lvl &= LogLevelMask;

  if (lvl < 0 || lvl >= LogLast)
    return (FAIL);

  /* if a log group is active and this level exceeds group's highest.. */
  if (log_groups > 0 && log_grp_highest_lvl < lvl)
    log_grp_highest_lvl = lvl;	/* new highest level */

  if (syserr)
    s = g_strdup_printf ("%s: (errno = %d) %s", fmt, errno, g_strerror (errno));
  else s = g_strdup (fmt);

  va_start (args, fmt);
  log_vargs (s, args);
  va_end (args);

  g_free (s);

  /* if no log groups and log level is above log_poplevel, then popup log
     viewer */
  if (log_groups == 0 && lvl >= log_poplevel)
    log_view (NULL);

  return (FAIL);
}

void
log_vargs (const gchar * message, va_list args)
{
  gchar *s;

  if (!log_buf) log_buf = g_string_new (NULL);

  s = g_strdup_vprintf (message, args);
  log_buf = g_string_append (log_buf, s);
  log_buf = g_string_append (log_buf, "\n");
  g_free (s);

  if (log_buf->len > log_maxsize)
    {				/* log exceeded max size? */
      gint i = 0;
      gchar c;

      /* look for first line that satisfies max size */
      while ((c = log_buf->str[i]))
	{
	  i++;
	  if (c == '\n' && (log_buf->len - i) <= log_maxsize)
	    break;
	}
      g_string_erase (log_buf, 0, i);	/* erase old lines */
    }
}

void
log_message (const gchar * message, ...)
{
  va_list args;

  va_start (args, message);
  log_vargs (message, args);
  va_end (args);
}

/* starts an error group, which esentially waits until log_group_leave to pop up
   an error dialog and then only if an error occured at or above popup_lvl */
void
log_group_enter (gint popup_lvl)
{
  if (log_groups == 0)
    {
      log_grp_popup_lvl = popup_lvl;
      log_grp_highest_lvl = -1;	/* reset highest occured log level */
    }

  log_groups++;
}

/* leave the error group, popping dialog if an error of sufficient level
   occured (set by log_group_enter) */
void
log_group_leave (void)
{
  g_return_if_fail (log_groups <= 0);

  log_groups--;

  if (log_grp_highest_lvl == -1) return;

  if (log_grp_highest_lvl >= log_grp_popup_lvl)
    log_view (NULL);
}

gint
log_length (void)
{
  return (log_buf->len);
}

void
log_clear (void)
{
  log_buf = g_string_truncate (log_buf, 0);
}

gint
safe_fread (void *buf, gint count, FILE * fd)
{
  if (fread (buf, count, 1, fd) != 1)
    {				// size_t = count, nmemb = 1
      if (feof (fd))
	logit (LogFubar, _("EOF while attemping to read %d bytes"), count);
      else
	logit (LogFubar | LogErrno, _("File read failed"));
      return (FAIL);
    }
  return (OK);
}

gint
safe_fwrite (void *buf, gint count, FILE * fd)
{
  if (fwrite (buf, count, 1, fd) != 1)
    {
      logit (LogFubar | LogErrno, _("File write failed"));
      return (FAIL);
    }
  return (OK);
}

gint
safe_fseek (FILE * fd, long ofs, gint whence)
{
  if (fseek (fd, ofs, whence) == -1)
    return (logit (LogFubar | LogErrno, _("File seek failed with offset = %ld"
					  " and whence = %d"), ofs, whence));
  return (OK);
}

void *
safe_malloc (size_t size)
{
  void *ptr;

  if (!(ptr = malloc (size)))
    logit (LogFubar, _("Failed to allocate %d bytes"), (gint) size);
  return (ptr);
}

gchar *
str_crlf2lf (gchar * str)
{				/* convert '\r\n' to '\n' in str */
  gchar *newstr, *s;

  newstr = g_new (gchar, strlen (str) + 1);
  s = newstr;

  while (*str != '\0')
    {
      if (*str != '\r' || *(str + 1) != '\n')
	*(s++) = *str;
      str++;
    }
  *s = '\0';

  return (newstr);
}

gchar *
str_lf2crlf (gchar * str)
{				/* convert '\n' to '\r\n' in str */
  GString *gs;
  gchar *s;

  gs = g_string_sized_new (sizeof (str));

  while (*str != '\0')
    {
      if (*str != '\n')
	gs = g_string_append_c (gs, *str);
      else
	gs = g_string_append (gs, "\r\n");
      str++;
    }
  s = gs->str;
  g_string_free (gs, FALSE);	/* character segment is not free'd */

  return (s);
}

/* returns TRUE if "sub" is found in "str" */
gint
substrcmp (gchar * sub, gchar * str)
{
  gchar *s, *s2;

  if (!*sub)
    return (TRUE);		/* null string, matches */

  while (*str)
    {
      if (tolower (*str) == tolower (*sub))
	{
	  s = sub + 1;
	  s2 = str + 1;
	  while (*s && *s2)
	    {
	      if (tolower (*s) != tolower (*s2))
		break;
	      s++;
	      s2++;
	    }
	  if (!*s)
	    return (TRUE);
	}
      str++;
    }
  return (FALSE);
}
