/*==================================================================
 * midi_alsaraw.c - ALSA Raw Midi Thru driver
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

#ifdef ALSA_SUPPORT

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <glib.h>
#include <sys/poll.h>
#include "i18n.h"
#include "midi.h"
#include "smurfcfg.h"
#include "util.h"
#include "alsa.h"
#include "seq_alsa.h"

#include <sys/asoundlib.h>

static snd_rawmidi_t *midi_alsaraw_handle; /* Handle for ALSA raw device */
static gint midi_alsaraw_card;
static gint midi_alsaraw_dev;
static GArray *inptags = NULL; /* array of GDK input watch tags */

void
midi_alsaraw_load_config (void)
{
  midi_alsaraw_card = smurfcfg_get_val (SMURFCFG_MIDI_ALSACARD)->v_int;
  midi_alsaraw_dev = smurfcfg_get_val (SMURFCFG_MIDI_ALSADEVICE)->v_int;
}

/* init for ALSA RAW midi driver */
gint
midi_alsaraw_init (void)
{
  gint err;
  gint tag;

#ifdef NEW_ALSA
  gchar *s;
  gint count, i;
  struct pollfd *pfd = NULL;
#else
  gint fd;
#endif

#ifdef NEW_ALSA
  s = g_strdup_printf ("hw:%d,%d", midi_alsaraw_card, midi_alsaraw_dev);
  err = snd_rawmidi_open (&midi_alsaraw_handle, NULL, s, SND_RAWMIDI_NONBLOCK);
  g_free (s);

  if (err < 0)
    return (logit (LogFubar, _("Failed to open ALSA RAW MIDI, card %d device %d"
			      " for reading: %s"), midi_alsaraw_card,
		  midi_alsaraw_dev, snd_strerror (err)));

  /* get # of MIDI file descriptors */
  count = snd_rawmidi_poll_descriptors_count (midi_alsaraw_handle);

  if (count > 0)		/* make sure there are some */
    {
      pfd = g_malloc (sizeof (struct pollfd) * count);

      /* grab file descriptor POLL info structures */
      count = snd_rawmidi_poll_descriptors (midi_alsaraw_handle, pfd, count);
    }

  i = 0;
  while (i < count)		/* loop over file descriptors */
    {
      if (pfd[i].events & POLLIN) /* use only the input FDs */
	{
	  tag = gdk_input_add (pfd[i].fd, GDK_INPUT_READ,
			       (GdkInputFunction)midi_read, NULL);
	  if (!inptags)		/* create GDK input tag array if needed */
	    inptags = g_array_new (FALSE, FALSE, sizeof (gint));

	  g_array_append_val (inptags, tag); /* append GDK input tag */
	}
      i++;
    }

  if (pfd)
    g_free (pfd);

  if (count <= 0 || !inptags->len) /* no INPUT FDs? */
    {
      snd_rawmidi_close (midi_alsaraw_handle);
      return (logit (LogFubar,
		    _("Failed to get ALSA RAW MIDI file descriptor")));
    }

#else  /* !NEW_ALSA */

  if ((err = snd_rawmidi_open (&midi_alsaraw_handle, midi_alsaraw_card,
			       midi_alsaraw_dev,
			       SND_RAWMIDI_OPEN_INPUT
			       | SND_RAWMIDI_OPEN_NONBLOCK)) < 0)
    return (logit (LogFubar, _("Failed to open ALSA RAW MIDI, card %d device %d"
			      " for reading: %s"), midi_alsaraw_card,
		  midi_alsaraw_dev, snd_strerror (err)));

  if ((err = snd_rawmidi_block_mode (midi_alsaraw_handle, FALSE)) < 0)
    {
      snd_rawmidi_close (midi_alsaraw_handle);
      return (logit (LogFubar, _("Failed to set ALSA RAW MIDI device to"
			        " non-blocking mode: %s"), snd_strerror(err)));
    }

  /* get the file descriptor for the opened midi handle */
  if ((fd = snd_rawmidi_file_descriptor (midi_alsaraw_handle)) < 0)
    {
      snd_rawmidi_close (midi_alsaraw_handle);
      return (logit (LogFubar, _("Failed to get ALSA RAW file descriptor: %s"),
		    snd_strerror (fd)));
    }

  tag = gdk_input_add (fd, GDK_INPUT_READ, (GdkInputFunction)midi_read, NULL);

  /* old ALSA 0.5.x API only uses one FD, but we do things like
     the new API to unify close routine */
  if (!inptags)
    inptags = g_array_new (FALSE, FALSE, sizeof (gint));

  g_array_append_val (inptags, tag);

#endif /* NEW_ALSA */

  return (OK);
}

/* close routine for ALSA midi driver */
void
midi_alsaraw_close (void)
{
  gint i;

  if (inptags)
    {
      for (i = 0; i < inptags->len; i++)
	{
	  gdk_input_remove (g_array_index (inptags, gint, i));
	}

      g_array_free (inptags, TRUE); /* free input tag array and its elements */
    }

  snd_rawmidi_close (midi_alsaraw_handle);
}

gint
midi_alsaraw_read (guint8 * buf, guint size)
{
  gint i;

  if ((i = snd_rawmidi_read (midi_alsaraw_handle, buf, size)) < 0)
    logit (LogWarn, _("Failed to read from ALSA MIDI device: %s"),
      snd_strerror (i));

  return (i);
}

#endif /* #ifdef ALSA_SUPPORT */
