#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "eval.h"


typedef struct spam_cond_expr spam_cond_expr_t;
typedef spam_eval_t spam_cond_expr_func (spam_cond_t *cond, spam_context_t *ctx);

struct spam_cond_expr
{
	spam_cond_type type;
	spam_cond_expr_func *expr;

	spam_cond_expr_t *next_expr;
};

static spam_eval_t spam_cond_expr_number(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_and(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_or(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_not(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_var(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_eq(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_approx_eq(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_string(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_assign(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_func_call(spam_cond_t *cond, spam_context_t *ctx);

static spam_eval_t spam_cond_expr_lt(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_gt(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_le(spam_cond_t *cond, spam_context_t *ctx);
static spam_eval_t spam_cond_expr_ge(spam_cond_t *cond, spam_context_t *ctx);

static spam_cond_expr_t *spam_lookup_expr( spam_cond_type type );

static spam_cond_expr_t expr_ge =
{
	SPAM_COND_GE,
	spam_cond_expr_ge,
	NULL
};

static spam_cond_expr_t expr_le =
{
	SPAM_COND_LE,
	spam_cond_expr_le,
	&expr_ge
};

static spam_cond_expr_t expr_gt =
{
	SPAM_COND_GT,
	spam_cond_expr_gt,
	&expr_le
};

static spam_cond_expr_t expr_lt =
{
	SPAM_COND_LT,
	spam_cond_expr_lt,
	&expr_gt
};

static spam_cond_expr_t expr_func_call =
{
	SPAM_COND_FUNC_CALL,
	spam_cond_expr_func_call,
	&expr_lt
};

static spam_cond_expr_t expr_assign =
{
	SPAM_COND_ASSIGN,
	spam_cond_expr_assign,
	&expr_func_call
};

static spam_cond_expr_t expr_str =
{
	SPAM_COND_STRING,
	spam_cond_expr_string,
	&expr_assign
};

static spam_cond_expr_t expr_approx_eq =
{
	SPAM_COND_APPROX_EQ,
	spam_cond_expr_approx_eq,
	&expr_str
};

static spam_cond_expr_t expr_eq =
{
	SPAM_COND_EQ,
	spam_cond_expr_eq,
	&expr_approx_eq
};

static spam_cond_expr_t expr_number =
{
	SPAM_COND_NUMBER,
	spam_cond_expr_number,
	&expr_eq,
};

static spam_cond_expr_t expr_var =
{
	SPAM_COND_VARIABLE,
	spam_cond_expr_var,
	&expr_number
};

static spam_cond_expr_t expr_not =
{
	SPAM_COND_NOT,
	spam_cond_expr_not,
	&expr_var
};

static spam_cond_expr_t expr_and =
{
	SPAM_COND_AND,
	spam_cond_expr_and,
	&expr_not
};

static spam_cond_expr_t expr_or =
{
	SPAM_COND_OR,
	spam_cond_expr_or,
	&expr_and
};

static spam_cond_expr_t *expr_list = &expr_or;

static spam_cond_expr_t *spam_lookup_expr( spam_cond_type type )
{
	spam_cond_expr_t *expr = expr_list;
	while ( expr != NULL ) {
		if ( expr->type == type ) {
			break;
		}
		expr = expr->next_expr;
	}
	return expr;
}

spam_eval_t spam_cond_eval(spam_cond_t *cond, spam_context_t *ctx)
{
	spam_cond_expr_t *expr = spam_lookup_expr( cond->type );
	assert( expr != NULL );
	return expr->expr( cond, ctx );
}

static spam_eval_t spam_cond_expr_number( spam_cond_t *cond, spam_context_t *ctx)
{
	return spam_eval_build_number( cond->node.number );
}

static spam_eval_t spam_cond_expr_and(spam_cond_t *cond, spam_context_t *ctx )
{
	int left = spam_cond_eval( cond->children, ctx ).node.number;
	int right = spam_cond_eval( cond->children->next_sibling, ctx ).node.number;

	return spam_eval_build_bool( left && right );
}

static spam_eval_t spam_cond_expr_or(spam_cond_t *cond, spam_context_t *ctx )
{
	int left = spam_cond_eval( cond->children, ctx ).node.number;
	int right = spam_cond_eval( cond->children->next_sibling, ctx ).node.number;

	return spam_eval_build_bool( left || right );
}

static spam_eval_t spam_cond_expr_not(spam_cond_t *cond, spam_context_t *ctx )
{
	int child = spam_cond_eval( cond->children, ctx ).node.number;

	return spam_eval_build_bool( !child );
}

static spam_eval_t spam_cond_expr_var(spam_cond_t *cond, spam_context_t *ctx )
{
	return ctx->variable( cond->node.name, ctx );
}

static spam_eval_t spam_cond_expr_eq(spam_cond_t *cond, spam_context_t *ctx )
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );

	switch ( left.type ) {
	case SPAM_EVAL_BOOL:
	case SPAM_EVAL_NUMBER:
		return spam_eval_build_bool( left.node.number == right.node.number );

	case SPAM_EVAL_STRING:
		return spam_eval_build_bool( ! strcmp( left.node.string, right.node.string ));

	default:
		fprintf( stderr, "Unknown eval type: %d\n", left.type );
		assert( 0 );
	}


}

static spam_eval_t spam_cond_expr_approx_eq(spam_cond_t *cond, spam_context_t *ctx )
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );
	assert( left.type == SPAM_EVAL_STRING );
	return spam_eval_build_bool( strstr( left.node.string, right.node.string ) != NULL );

}

static spam_eval_t spam_cond_expr_string( spam_cond_t *cond, spam_context_t *ctx )
{
	return spam_eval_build_string( cond->node.name );
};

static spam_eval_t spam_cond_expr_assign( spam_cond_t *cond, spam_context_t *ctx )
{
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );
	return ctx->assign( cond->children->node.name, ctx, &right );
}

static spam_eval_t spam_cond_expr_func_call( spam_cond_t *cond, spam_context_t *ctx)
{
	int argsize = 0, i;
	spam_cond_t *func = cond;
	spam_cond_t *arg_list;
	spam_eval_t **arg_eval_list;
	
	for( argsize=0, arg_list = cond->next_sibling; arg_list != NULL;
				argsize++, arg_list = arg_list->next_sibling );	

	arg_eval_list =
		(spam_eval_t **)malloc(sizeof(spam_eval_t *)*(argsize+1));

	for( i = 0, arg_list = cond->next_sibling;
			i < argsize; i++, arg_list = arg_list->next_sibling )
	{
		arg_eval_list[i] = (spam_eval_t *)malloc(sizeof(spam_eval_t));
		*(arg_eval_list[i]) = spam_cond_eval( arg_list, ctx );
	}
	arg_eval_list[argsize] = NULL;

	assert( ctx->func_call != NULL );

	ctx->func_call( func->node.name, ctx, arg_eval_list );

	for ( i = 0; i < argsize; i++ ) {
		free( arg_eval_list[i] );
	}

	free( arg_eval_list );

	return spam_eval_build_bool( 1 );	
}

static spam_eval_t spam_cond_expr_lt( spam_cond_t *cond, spam_context_t *ctx)
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );
	assert( left.type == SPAM_EVAL_NUMBER );

	return spam_eval_build_bool( left.node.number < right.node.number );
}

static spam_eval_t spam_cond_expr_gt( spam_cond_t *cond, spam_context_t *ctx)
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );
	assert( left.type == SPAM_EVAL_NUMBER );

	return spam_eval_build_bool( left.node.number > right.node.number );
}

static spam_eval_t spam_cond_expr_le( spam_cond_t *cond, spam_context_t *ctx)
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );
	assert( left.type == SPAM_EVAL_NUMBER );

	return spam_eval_build_bool( left.node.number <= right.node.number );
}

static spam_eval_t spam_cond_expr_ge( spam_cond_t *cond, spam_context_t *ctx)
{
	spam_eval_t left = spam_cond_eval( cond->children, ctx );
	spam_eval_t right = spam_cond_eval( cond->children->next_sibling, ctx );

	assert( left.type == right.type );
	assert( left.type == SPAM_EVAL_NUMBER );

	return spam_eval_build_bool( left.node.number >= right.node.number );
}

int spam_do_action( spam_action_t *action, spam_context_t *ctx )
{
	return action->action( ctx, action->arg );
}

int spam_ruleset_eval( spam_context_t *ctx, spam_def_t *def, int type )
{
	int rval = 0;
	spam_eval_t eval;
	spam_rule_t *rule = def->rule_list;
	spam_action_t *action;

	while ( rule != NULL ) {
		if ( rule->type == type || rule->type == SPAM_RULE_ALL ) {
			eval = spam_cond_eval( rule->cond_tree, ctx );
			if ( eval.node.number ) {
				action = rule->action_list;
				while ( action != NULL ) {
					rval = spam_do_action( action, ctx );
					action = action->next_action;
				}
			}
		}
		rule = rule->next_rule;
	}

	rule = def->rule_list;
	while ( rule != NULL ) {
		if ( rule->type == SPAM_RULE_RESULT ) {
			eval = spam_cond_eval( rule->cond_tree, ctx );
			if ( eval.node.number ) {
				action = rule->action_list;
				while ( action != NULL ) {
					rval = spam_do_action( action, ctx );
					action = action->next_action;
				}
			}
		}
		rule = rule->next_rule;
	}

	return rval;
}


