/*
 * $Date: 2005/03/04 17:17:02 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2005 Brian Carrier.  All rights reserved
 *
 * raw
 *
 * This file is part of imgtools
 *
 * imgtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * imgtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "img_tools.h"
#include "raw.h"



static OFF_T
raw_read_random(IMG_INFO * img_info, char *buf, OFF_T len, OFF_T offset)
{
    OFF_T cnt;
    IMG_RAW_INFO *raw_info = (IMG_RAW_INFO *) img_info;

    if (verbose)
	fprintf(logfp,
		"raw_read_random: byte offset: %" PRIuOFF " len: %" PRIuOFF
		"\n", offset, len);

    // is there another layer?
    if (img_info->next) {
	return img_info->next->read_random(img_info->next, buf, len,
					   offset);
    }

    // Read the data
    else {
	OFF_T tot_offset = offset + img_info->offset;

	if (raw_info->seek_pos != tot_offset) {
	    if (lseek(raw_info->fd, tot_offset, SEEK_SET) != tot_offset) {
		return 0;
	    }
	    raw_info->seek_pos = tot_offset;
	}

	cnt = read(raw_info->fd, buf, len);
	raw_info->seek_pos += cnt;
	return cnt;
    }
}

OFF_T
raw_get_size(IMG_INFO * img_info)
{
    return img_info->size;
}

void
raw_imgstat(IMG_INFO * img_info, FILE * hFile)
{
    fprintf(hFile, "IMAGE FILE INFORMATION\n");
    fprintf(hFile, "--------------------------------------------\n");
    fprintf(hFile, "Image Type: raw\n");
    fprintf(hFile, "\nSize in bytes: %" PRIuOFF "\n", img_info->size);
    return;
}

void
raw_close(IMG_INFO * img_info)
{
    IMG_RAW_INFO *raw_info = (IMG_RAW_INFO *) img_info;
    close(raw_info->fd);
}


IMG_INFO *
raw_open(OFF_T offset, const char **images, IMG_INFO * next)
{
    IMG_RAW_INFO *raw_info;
    IMG_INFO *img_info;

    raw_info = (IMG_RAW_INFO *) mymalloc(sizeof(IMG_RAW_INFO));
    memset((void *) raw_info, 0, sizeof(IMG_RAW_INFO));

    img_info = (IMG_INFO *) raw_info;

    img_info->itype = IMG_RAW;
    img_info->offset = offset;
    img_info->read_random = raw_read_random;
    img_info->get_size = raw_get_size;
    img_info->close = raw_close;
    img_info->imgstat = raw_imgstat;

    if (next) {
	img_info->next = next;
	img_info->size = next->get_size(next);
    }

    /* Open the file */
    else {
	if ((raw_info->fd = open(images[0], O_RDONLY)) < 0)
	    error("raw_open: open %s: %m", images[0]);

	img_info->size = lseek(raw_info->fd, 0, SEEK_END);
	lseek(raw_info->fd, 0, SEEK_SET);
	raw_info->seek_pos = 0;
    }

    return img_info;
}
