/* -*- c++ -*-
 * __BEGIN_COPYRIGHT
 * SimpleDB API
 * 
 * Copyright (C) 2005 Eminence Technology Pty Ltd
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You can view the GNU Lesser General Public Licence at
 * http://www.gnu.org/licenses/lgpl.html or you can write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Eminence Technology Pty Ltd can be contacted by writing to
 * Eminence Technology, PO Box 118, Moorooka QLD 4105, Australia.
 * Alternatively, you may email opensource [at] eminence [dot] com [dot] au
 * __END_COPYRIGHT
 */

#ifndef __SIMPLEDB_DATABASE_H_
#define __SIMPLEDB_DATABASE_H_ 1

#include <string>
#include <sql.h>
#include <sqlext.h>
#include <iostream>
#include "Query.h"
#include "Exception.h"

namespace SimpleDB {

  /** This class represents a database connection.
   *
   * \todo: Overload the default assignment operator so that
   *        database objects can be assigned without causing
   *        problems with connection handles
   */
  class Database {

  public:

    /** This class is thrown in case of any database exception.
     * A description (if available) is included.
     */
    class Exception : public SimpleDB::Exception {
    public:
      Exception(const std::string& message) : SimpleDB::Exception(message) {};
    };

    /** Exception class thrown when no data is available.
     * Thrown by intQuery function.
     */
    class NoDataException : public Exception {
    public:
      
      /** No parameters are passed to the constructor.
       * A default error message is included.
       */
      NoDataException() : Exception("No data returned.") {
      }
    };
  
    /** Constructor creates the database connection.
     *
     * @param dsn The ODBC data source to connect to.
     */
    Database(std::string dsn);
  
    /** Creates a query object for this database.
     *
     * @return Query The new query object.
     */
    Query newQuery() const;

    /** void Query, query the database and don't worry about return types or
     * if its a successful query.
     * @param query The query to execute.
     */
    void voidQuery (const std::string& query) const;
    
    /** long Query, query the database for a single integer result.
     *
     * \exception SimpleDB::Database::Exception Thrown if error with syntax
     *            etc.
     * \exception SimpleDB::Database::NoDataException Thrown if now rows
     *            are returned by the query.
     *
     * @param query the query to execute
     * @return the integer result from the query
     */
    long intQuery (const std::string& query) const;
    
    /** A Query interface that returns a string result
     * @param query The query to execute.
     * @param bufSize The max size of the string buffer expected to be returned.
     * @return resultant string of the query.
     */
    std::string strQuery (const std::string& query, long bufSize = 1000) const;
    
    /** Escapes special characters in a string to be sent as
     * part of a database query.
     *
     * @param input The input string to be escaped.
     * @return The escaped version of the input string.
     */
    static const std::string escapeString(const std::string& input);
    
    /** Converts a boolean input to the equivalent database value.
     * @param in The boolean to convert
     * @return the converted string.
     */
    static const std::string  boolToStr (const bool in);
    
    /** Function to retrieve the database connection handle.
     *
     * @return The database connection handle.
     */
    const SQLHDBC getConnectionHandle() const;
    
    /** Destructor
     *
     */
    virtual ~Database() ;
  protected:
  
    /** ODBC connection handle
     */
    SQLHDBC connectionHandle;

    /** ODBC environment handle.
     * This handle is created by the constructor and
     * freed in the destructor.
     */
    SQLHENV odbc_environment_handle;
  };

}

inline
std::ostream& operator<<(std::ostream& strm, const std::exception &e) {
  strm << "Database::Exception: " << e.what();
  return strm;
}

#endif
