/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "imageloader.h"
#include "imagelisteview.h"
#include "imagefileinfo.h"

#include <kurl.h>
#include <kapp.h>
#include <klocale.h>
#include <kprocess.h>
#include <kpixmapio.h>

#include <qstring.h>
#include <qimage.h>
#include <qlabel.h>
#include <qdir.h>

#include <iostream.h>
#include <sys/stat.h>
#include <unistd.h>

/* XPM */
static const char *info_xpm[] = {
"18 18 4 1",
". c #000000","a c #4699cd","b c #a8dcff","# c #ffffff",
"..................",".################.",".######aaa#######.",".#####abbaa######.",".#####abaaa######.",
".#####abaaa######.",".#####aaaaa######.",".#####aaaaa######.",".######aaa#######.",".######aaa#######.",
".#######a########.",".################.",".######aaa#######.",".#####abbaa######.",".#####abaaa######.",
".######aaa#######.",".################.",".................."};

class ImageListeView;

void *
__thread_start (void *arg)
{
	pthread_cleanup_push (__thread_cleanup, arg);
	((ImageLoader *) arg)->thread_start ();
	pthread_cleanup_pop (0);
	pthread_detach (pthread_self ());
	return 0;
}

void
__thread_cleanup (void *arg)
{
	((ImageLoader *) arg)->thread_cleanup ();
}


ImageLoader::ImageLoader (QWidget * parent, const char *name):
QObject (parent, name)
{
	EventList.setAutoDelete (true);
	installEventFilter (this);
	Loading = Running = false;

	setThumbnailSize(QSize(80,60));
	info=QPixmap(info_xpm);
}

ImageLoader::~ImageLoader ()
{
	stopLoading (true);
}


void
ImageLoader::setThumbnailSize(QSize newSize)
{
	if(size==newSize)
		return;
	else
		size=newSize;
}


QSize
ImageLoader::getThumbnailSize()
{
	return size;
}


void
ImageLoader::loadMiniImage (QFileInfo * fi, QWidget * w, bool threaded)
{
	bool cont=true;
	QFileInfo thumb = QFileInfo (QDir::homeDirPath () + "/.showimg/cache/" + fi->absFilePath ());
	
	if (thumb.exists ())
	{
		if(fi->lastModified() < thumb.lastModified())   
		{
			QImage im (QDir::homeDirPath () + "/.showimg/cache/" + fi->absFilePath (),
					 "JPEG");
			double
			 wexpand = (double) im.width () / (double)getThumbnailSize().width(),
			 hexpand = (double) im.height () / (double)getThumbnailSize().height();
			
			if (wexpand != 1.0 && hexpand != 1.0) // don't expand small images  !
			{
	 			int neww, newh;
	 			float     s;

	 			if (wexpand > hexpand)
	 			{
	 	  			neww = (int) (im.width () / wexpand );
	 	  			newh = (int) (im.height () / wexpand );
	 	  			s = 1 / wexpand;
	 			}
	 			else
	 	 		{
	 	 			neww = (int) (im.width () / hexpand );
	 	 			newh = (int) (im.height () / hexpand );
	 	 			s = 1 / hexpand;
	 			 }
				 
				im=im.smoothScale(neww,newh);
			}
			cont =false;
			QPixmap pix = KPixmapIO().convertToPixmap(im);
			QPainter p(&pix);
				p.drawRect(0,0,pix.width(),pix.height());
				//if(ImageFileInfo(fi->absFilePath(), IMAGE, false).hasInfo())
				//	p.drawPixmap(0,0, info);
				p.end();
			((ImageListeView *) w)->slotSetPixmap (&pix);
		}
	}
	      
	if(cont)
	{
		ImageLoadEvent *e = new ImageLoadEvent (fi, w, threaded);
		EventList.append (e);
		if (EventList.count () > 0 && !Running)
		{
			Running = true;
			startTimer (1);
			nextImage ();
		}
	}
}

void
ImageLoader::startLoading ()
{
	Running = true;
	ImageLoadEvent *e = ((int) (EventList.count ()) > 0 ? EventList.take (0) : 0);
	if (!e)
	{
		Running = Loading = false;
		killTimers ();
	
		return;      
	}
	if (!initLoading (e))
	{
		cantLoad (e);
		return;
	}
	Loading = true;
	loadImageInternal (e);
}


void
ImageLoader::stopLoading (bool clean)
{
	if (Loading)
	{
		pthread_cancel (ThreadID);
		pthread_join (ThreadID, NULL);
		Loading = Running = false;
		killTimers ();
		ImageLoadedList.clear ();
	}

	if (clean)
	{
		EventList.clear ();
	}
}

void
ImageLoader::cantLoad (ImageLoadEvent * e)
{
	(void)kapp->postEvent (e->widget (), e);
	// Try to load next image
	Loading = false;
	nextImage ();
}

bool
ImageLoader::eventFilter (QObject *, QEvent * e)
{
	switch (e->type ())
	{
		case Event_NextImage:
			startLoading ();
			return true;
		case Event_ImageLoad:
		{
		 	 Loading = false;
		 	 ImageLoadEvent *ev = new ImageLoadEvent (*((ImageLoadEvent *) e));
		 	 finishLoading (ev);
		 	 (void)kapp->postEvent (ev->widget (), ev);
		 	 nextImage ();
		 	 return true;
		}
		default : return false;
	}
	return false;
}


void
ImageLoader::nextImage ()
{
	if (!Loading)
	{
		NextImageEvent *e = new NextImageEvent;
		(void)kapp->postEvent (this, e);
	}
}


void
ImageLoader::thread_start ()
{
	InternalImage.load (InternalPath.data ());
	ImageLoadedList.append (InternalEvent);
}


void
ImageLoader::thread_cleanup ()
{
}


void
ImageLoader::timerEvent (QTimerEvent * e)
{
	while (ImageLoadedList.count () > 0)
	{
		ImageLoadEvent *e = ImageLoadedList.take (0);
		(void)kapp->postEvent (this, e);
	}
}


bool
ImageLoader::initLoading (ImageLoadEvent * e)
{ 
	QFileInfo *fi = e->fileInfo ();
	image_path = QString (fi->absFilePath ());
	image_url = KURL (image_path);

	if (!mini_image_file_exists || mini_image_outdated)
		return true;
	return false;
}


void
ImageLoader::finishLoading (ImageLoadEvent * e)
{
	QFileInfo *fi = e->fileInfo ();
	QImage image = InternalImage, ditheredImage;

	if (image.isNull ())
	{
		image = BarIcon("file_broken",48).convertToImage();
	}	
	
	double
	  wexpand = (double) image.width () / (double)getThumbnailSize().width() ,
	  hexpand = (double) image.height () / (double)getThumbnailSize().height();

	if (wexpand >= 1.0 || hexpand >= 1.0) // don't expand small images  !
	{
		int neww, newh;
		float	  s;
		if (wexpand > hexpand)
		{
		 	 neww = (int) (image.width () / wexpand);
		 	 newh = (int) (image.height () / wexpand);
		 	 s = 1 / wexpand;
		}
		else
		{
			neww = (int) (image.width () / hexpand);
			newh = (int) (image.height () / hexpand);
			s = 1 / hexpand;
		}
		image = image.smoothScale (neww, newh);
		
		QCString  dest = QCString (QDir::homeDirPath () + "/.showimg/");
		QDir	  dirDest = QDir (dest);
		if (!dirDest.exists ())
		{
			QDir ().mkdir (dest);
		}
		dest = dest + "cache/";
		dirDest = QDir (dest);
		if (!dirDest.exists ())
		{
			QDir ().mkdir (dest);
		}
		QCString  res = QCString (fi->absFilePath ());
		QCString  ssrep;
		int  pos = res.find ("/");
		while (pos != -1)
		{
			dest = dest + res.left (pos) + "/";
			dirDest = QDir (dest);
			if (!dirDest.exists ())
			{
				QDir ().mkdir (dest);
			}
			res.setStr (res.right (res.length () - pos - 1));
			pos = res.find ("/");
		}		
		image.save (QString (dest) + fi->fileName (), "JPEG");
	}		
	mini_image = KPixmapIO().convertToPixmap(image);	
	if(!mini_image.isNull ())
	{
		QPainter p(&mini_image);
			p.drawRect(0,0,mini_image.width(),mini_image.height());
			//if(ImageFileInfo(fi->absFilePath(), IMAGE, false).hasInfo())
			//	p.drawPixmap(0,0, info);
			p.end();
		((ImageListeView *) e->widget ())->slotSetPixmap (&mini_image);
	}
	else
		((ImageListeView *) e->widget ())->slotSetPixmap (new QPixmap(BarIcon("file_broken",48)));
}


void
ImageLoader::loadImageInternal (ImageLoadEvent * e)
{
	InternalPath = image_url.path ();
	InternalEvent = e;
	InternalImage.reset ();
	// Problem here with xpm and xbm image, interfere with X system and can cause crash.
	// As it is probably a small image, it's probably not a problem to load it synchronously.

	if (!e->threaded ())
		thread_start ();
	else
		if (pthread_create (&ThreadID, 0, __thread_start, this) == 0);
	else
		qWarning ("%s %d  ImageLoader::loadImageInternal (ImageLoadEvent * e) : unable to start loading thread",__FILE__,__LINE__);

}
