/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* imageliste.cpp*/


#include "imagelisteview.h"

#include "mainwindow.h"
#include "kexifpropsplugin.h"
#include "describe.h"
#include "qtfileicondrag.h"

#include <qmessagebox.h>
#include <qevent.h>
#include <qmime.h>

#include <kiconloader.h>
#include <kmessagebox.h>
#include <kpropsdlg.h>
#include <kopenwith.h>
#include <klineeditdlg.h>
#include <kurlrequesterdlg.h>
#include <kglobalsettings.h>
#include <kio/job.h>
#include <kfiledialog.h>

#include <stdlib.h>



ImageListeView::ImageListeView (QWidget * parent, const char *name,
				MainWindow * mw)
	:KFileIconView (parent, name)
{
	this->mw = mw;

	popup = new KPopupMenu ();

	setItemTextPos( Bottom );
	setArrangement(LeftToRight);	
	setWordWrapIconText (false);	
	setShowToolTips(false);
	setResizeMode (Adjust);	
	QIconView::setSelectionMode (Extended);

	il = new ImageLoader ();	

	setThumbnailSize(QSize(80,60));
	setSpacing (5);
	setIconSize(48);

	connect(this, SIGNAL(loadFinish()), SLOT(slotLoadNext()));
	connect(this, SIGNAL(itemRenamed(QIconViewItem*)), SLOT(endRenamed(QIconViewItem*)));
	connect(this, SIGNAL(selectionChanged()), SLOT(selectionChanged()));
	
	setFocusPolicy (QWidget::NoFocus);	
	sortMode=0;	
	
	loop=true;
	p=true;
	random=false;
}
bool ImageListeView::doPreload(){return p;}
void ImageListeView::setPreload(bool p){this->p=p;}
bool ImageListeView::doLoop(){return loop;}
void ImageListeView::setLoop(bool loop){this->loop=loop;}
bool ImageListeView::doRandom(){return random;}
void ImageListeView::setRandom(bool ran){this->random=ran;}

void 
ImageListeView::setThumbnailSize(QSize newSize)
{
	if(itemTextPos()==Right)
		setGridX(newSize.width()+200);
	else
		setGridX(newSize.width()+10);
	
	setGridY(newSize.height()+20);
	
	il->setThumbnailSize(newSize);
}

void  
ImageListeView::slotUpdate ()
{
	QIconView::slotUpdate();
}

FileItem* 
ImageListeView::firstItem ()
{
	return (FileItem *) QIconView::firstItem ();
}

FileItem* 
ImageListeView::currentItem()
{
	return (FileItem *)QIconView::currentItem ();
}
FileItem* 
ImageListeView::lastItem()
{
	return (FileItem *)QIconView::lastItem ();
}

void 
ImageListeView::slotResetThumbnail()
{
	stopLoading();
	for (FileItem * item = firstItem (); item != 0; item = item->nextItem ())
	{
		item->setPixmap(BarIcon ("image", 48), false);
	}
	arrangeItemsInGrid();
}


void 
ImageListeView::slotRename()
{
	if(QIconView::currentItem())
	{
		bool ok;
		QString name=((ImageFileItem*)QIconView::currentItem())->text(0);
		const QString newName(KLineEditDlg::getText(QString().sprintf(i18n("Rename '<b>%s</b>' :"),(const char*)name),
					 name,
					 &ok, mw->iv).stripWhiteSpace());
		if(ok && newName!="")
		{
			((ImageFileItem*)QIconView::currentItem())->setText(newName);			
		}
	}
	
};

void
ImageListeView::endRenamed(QIconViewItem*)
{
}

void
ImageListeView::initMenu(KActionCollection *actionCollection)
{
	this->actionCollection=actionCollection;

	KPopupMenu *popupsort = new KPopupMenu ();
	actionCollection->action("by name")->plug(popupsort);
	actionCollection->action("by extension")->plug(popupsort);
	actionCollection->action("by size")->plug(popupsort);
	actionCollection->action("by date")->plug(popupsort);
	popup->insertItem(i18n("Sorting"), popupsort);
	
	KPopupMenu *popupar = new KPopupMenu ();
		KPopupMenu *popupIc = new KPopupMenu ();
		actionCollection->action("From left to right")->plug(popupIc);
		actionCollection->action("From top to bottom")->plug(popupIc);
		popupar->insertItem(i18n("Icons"), popupIc);
		
		KPopupMenu *popupTxt = new KPopupMenu ();
		actionCollection->action("on the right")->plug(popupTxt);
		actionCollection->action("at the bottom")->plug(popupTxt);
		popupar->insertItem(i18n("Text"), popupTxt);		
	popup->insertItem(i18n("Arrangement"), popupar);

	
	actionCollection->action("FullScreen")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("editcut")->plug(popup);
	actionCollection->action("editcopy")->plug(popup);
	actionCollection->action("editpaste")->plug(popup);
	actionCollection->action("rename")->plug(popup);
	actionCollection->action("edittrash")->plug(popup);
	actionCollection->action("editdelete")->plug(popup);
	actionCollection->action("editshred")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("copyFilesTo")->plug(popup);
	actionCollection->action("moveFilesTo")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("Open with Gimp")->plug(popup);
	actionCollection->action("Open with")->plug(popup);
	popup->insertSeparator();
	actionCollection->action("Properties")->plug(popup);
	actionCollection->action("Image Info")->plug(popup);
	actionCollection->action("Edit File Type")->plug(popup);
}


void
ImageListeView::refresh ()
{
}


QString 
ImageListeView::currentItemName()
{
	if(currentItem())
		return currentItem()->text();
	else
		return QString();
	
}
 
  
void 
ImageListeView::setCurrentItemName(QString itemName, bool select)
{
	 QIconView::setCurrentItem(findItem(itemName));
	 if(currentItem())
	 {
	 	QIconView::setSelected(currentItem(), true, false);
	 	QIconView::ensureItemVisible(currentItem());
	 	if(select) currentItem()->setSelected(true);
	}
}


bool
ImageListeView::hasImages()
{
	return QIconView::count()!=0;
}


bool
ImageListeView::hasImageSelected()
{
	if(!hasImages())
		return false;		
	for (FileItem *i = firstItem(); i; i=i->nextItem() )
	{
		if(i->isSelected())
			return true;
	}
	return false;
}


void 
ImageListeView::selectionChanged()
{
	if(hasImageSelected())
		mw->setHasImageSelected(true);
	else
		mw->setHasImageSelected(false);
}


void
ImageListeView::slotLoadFirst()
{
	stopLoading();
  	if (mw->preview())
	{
		mw->slotReset ();
		imageLoading = firstItem ();
		
		while(imageLoading && imageLoading->hasPreview())
			imageLoading=imageLoading->nextItem ();	
						
		if(imageLoading)
			slotLoadNext ();
		
		else
			mw->slotDone ();
	}
	else
	{
		return;
	}
}


void
ImageListeView::slotLoadFirst(FileItem *item)
{
	stopLoading();
	if (mw->preview())
	{
		mw->slotReset (false);
		imageLoading = item;
		if(imageLoading)
			slotLoadNext ();
		else
			mw->slotDone ();
	}
	else
	{
		return;
	}
}

void
ImageListeView::slotLoadNext ()
{
	if(imageLoading)
      		il->loadMiniImage (new QFileInfo(imageLoading->fullName()), this,  true);
}


void
ImageListeView::slotSetPixmap (const QPixmap * pm)
{
	if(imageLoading)
	{
		if(pm)
		{
			imageLoading->setPixmap (*pm, true);
		}
	}
	else
	{
		slotLoadFirst();
		return;
	}
	mw->slotPreviewDone ();
	if (imageLoading)  
		imageLoading = imageLoading->nextItem ();
	if (imageLoading)
	{
		emit loadFinish();  
	}
	else
	{
		mw->slotDone();
		arrangeItemsInGrid();
	}
};


void
ImageListeView::next ()
{
	if(!hasImages())
		return;
	
	FileItem *item=NULL;
	if(doRandom())
	{
		srand(time(NULL));
		while(item==NULL)
		{
			int x = (int)(rand()/(RAND_MAX+1.0)*contentsWidth());
			int y = (int)(rand()/(RAND_MAX+1.0)*contentsHeight());
			item=(FileItem *)QIconView::findItem((const QPoint&)QPoint(x,y));
		}
	}
	else
	{
		item=currentItem();
		if(!item)
			item=firstItem();
		else
			if(!currentItem()->isSelected())
			item=firstItem();
		else
			item=item->nextItem();
	}
	if(item)
	{
	 	 QIconView::ensureItemVisible (item);
	 	 QIconView::setCurrentItem (item);
	 	 QIconView::setSelected (item, true, false);
	 	 item ->setSelected (true);
	}
	else
	if(loop)
		first();
	
}


void
ImageListeView::previous ()
{
	QIconViewItem *item=currentItem();
	if(!item)
		item=QIconView::firstItem();
	else
	if(!currentItem()->isSelected())
		item=QIconView::firstItem();
	else
		item=item->prevItem();
	if(item)
	{
	 	 QIconView::ensureItemVisible (item);
	 	 QIconView::setCurrentItem (item);
	 	 QIconView::setSelected (item, true, false);
	 	 item ->setSelected (true);
	}
	else
	if(loop)
		last();

}


void 
ImageListeView::focusInEvent ( QFocusEvent * )
{
}


void
ImageListeView::keyPressEvent (QKeyEvent * e)
{
	e->ignore();
}


void 
ImageListeView::showEvent( QShowEvent * )
{
	QIconView::show();
	if(currentItem())
		QIconView::ensureItemVisible(currentItem());
}


void
ImageListeView::contentsMousePressEvent (QMouseEvent * e)
{
	QIconView::contentsMousePressEvent (e);
}


void
ImageListeView::contentsMouseReleaseEvent (QMouseEvent * e)
{
	if (e->button () == LeftButton)
	{
		QIconView::contentsMouseReleaseEvent ( e );

		int nbs=0;
		for (FileItem * item = firstItem (); item != 0; item = item->nextItem ())
		{
			if (item->isSelected () )
			{
				nbs++;
				if(nbs==2)
					return;
			}
		}
	}
	mousePress(e);
}


void
ImageListeView::mousePress (QMouseEvent * e)
{
	if (e->button () == LeftButton)
	{
		FileItem *item  = currentItem();
		if(item)
		{
			item ->setSelected (true);
		}
	}
	else if (e->button () == RightButton)
	{
		popup->popup (e->globalPos ());
		popup->show ();
	}
	else if (e->button () == MidButton)
	{
		if (currentItem ())
		{
			mw->slotFullScreen ();
			currentItem ()->setSelected (true);
		}
	}
} 

  
void
ImageListeView::contentsMouseMoveEvent ( QMouseEvent * e )
{
  QIconView::contentsMouseMoveEvent ( e );
}


void 
ImageListeView::sort()
{
	switch(sortMode)
	{
		case 0:slotByName(); break;
		case 1:slotByExtension(); break;
		case 2:slotBySize(); break;
		case 3:slotByDate(); break;
	}
}


void
ImageListeView::slotByName()
{
	sortMode=0;
	FileItem *i;
	for (i=(FileItem*)QIconView::firstItem(); i; i = (FileItem*)i->nextItem() )
	{
		i->setKey("name");
	}
	QIconView::sort();
}


void
ImageListeView::slotByExtension()
{	
	sortMode=1;
	FileItem *i;
	for (i=(FileItem*)QIconView::firstItem(); i; i = (FileItem*)i->nextItem() )
	{
		i->setKey("type");
	}
	QIconView::sort();
}


void
ImageListeView::slotBySize()
{
	sortMode=2;

	FileItem *i;
	for (i=(FileItem*)QIconView::firstItem(); i; i = (FileItem*)i->nextItem() )
	{
		i->setKey("size");
	}
	QIconView::sort();
}


void
ImageListeView::slotByDate()
{
	sortMode=3;

	FileItem *i;
	for (i=(FileItem*)QIconView::firstItem(); i; i = (FileItem*)i->nextItem() )
	{
		i->setKey("date");
	}
	QIconView::sort();
	QIconView::ensureItemVisible(currentItem());
}


QDragObject *
ImageListeView::dragObject ()
{
    if ( !currentItem() )
	return 0;

	QPoint orig = viewportToContents( viewport()->mapFromGlobal( QCursor::pos() ) );
	QtFileIconDrag *drag = new QtFileIconDrag( viewport() );

	drag->setPixmap( *currentItem()->pixmap(),
				QPoint( currentItem()->pixmapRect().width() / 2, currentItem()->pixmapRect().height() / 2 ) );

	for ( FileItem *item = firstItem(); item; item = item->nextItem() )
	{
		if ( item->isSelected() )
		{
			QIconDragItem id;
		
			id.setData(QUriDrag::localFileToUri(item->fullName()));
			drag->append( id,
				QRect( item->pixmapRect( FALSE ).x() - orig.x(),
				item->pixmapRect( FALSE ).y() - orig.y(),
				item->pixmapRect().width(), item->pixmapRect().height() ),
				QRect( item->textRect( FALSE ).x() - orig.x(),
				item->textRect( FALSE ).y() - orig.y(),
				item->textRect().width(), item->textRect().height() ),
				QUriDrag::localFileToUri(item->fullName()));
		}
	}
    return drag;
}


void
ImageListeView::slotWallpaper ()
{
	FileItem *item = currentItem ();
	if (!item)
		return;
	currentItem ()->setWallpaper ();
}


void
ImageListeView::slotFileProperty()
{
	if(!currentItem())
		return;
	QApplication::setOverrideCursor (waitCursor);	
	KFileItem *item = new KFileItem( *((FileItem*)currentItem())->fileInfo() );
        KPropertiesDialog *prop	= new KPropertiesDialog( item,
		mw->iv, "KPropertiesDialog",
		true, false);
	KEXIFPropsPlugin *exifProp = new KEXIFPropsPlugin(prop, ((FileItem*)currentItem())->fullName());
	prop->insertPlugin(exifProp);	
	QApplication::restoreOverrideCursor ();
	prop->exec(); 
}


void
ImageListeView::slotImageProperty()
{

}


void 
ImageListeView::slotImageInfo()
{
 	if(!currentItem())
		return;
		
	QApplication::setOverrideCursor (waitCursor);	
	Describe dscr(mw->iv, ((FileItem*)currentItem())->fullName(), "ImageInfo", true);
	QApplication::restoreOverrideCursor ();	
	dscr.exec();	
}


void
ImageListeView::slotGimp ()
{
	FileItem *item = currentItem ();
	if (!item)
		return;

	KShellProcess *proc = new KShellProcess ();
	connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(slotEndGimp(KProcess *)));
	QString com;
	com.sprintf ("gimp-remote -n '%s' >/dev/null 2>/dev/null", (const char *) item->fullName());
	*proc << com;
	if(! proc->start ())
	{
		KRun::run("gimp",KURL::List("file:/"+item->fullName()));
	}
}


void
ImageListeView::slotEndGimp (KProcess *proc)
{
	if(proc->exitStatus()!=0)
	{
		FileItem* item = currentItem ();
		if (!item)
			return;
		KRun::run("gimp",KURL::List("file:/"+item->fullName()));
	}
}



void
ImageListeView::slotSupprimmer ()
{
	
	if(checkDelete())
	if(KMessageBox::questionYesNo(mw->iv, i18n("Do you really want to delete the selected file(s) ?"), i18n("Confirm deletion")) == KMessageBox::No)
		return;

	KURL::List list;
	FileItem *nextItem = 0;
	for (FileItem * item = firstItem ();
	     item != 0; item = item->nextItem ())
	{
		if (item->isSelected ())
		{
			nextItem = item->nextItem ();
			if (item->text (3) == "file")
			{
				list.append(KURL("file:/"+item->fullName()));
			}
			else
			{
			}
		}
	}
	KIO::DeleteJob *del=KIO::del( list, false, true );
	connect(del, SIGNAL(result( KIO::Job *)), this, SLOT(deletionDone( KIO::Job *)));
	
	if (nextItem)
	{
		QIconView::setCurrentItem (nextItem);
		nextItem->setSelected (true);
		QIconView::setSelected (currentItem(), true, true);
		ensureItemVisible (currentItem()->fileInfo());
	}
}


void
ImageListeView::deletionDone( KIO::Job *)
{
	mw->slotRefresh();
}


void
ImageListeView::slotMoveToTrash()
{
	if(checkTrash())
	if(KMessageBox::questionYesNo(mw->iv, i18n("Do you really want to move to trash the selected file(s) ?"), i18n("Confirm deletion")) == KMessageBox::No)
		return;
	FileItem *nextItem = 0;
	KURL::List list;
	for (FileItem * item = firstItem (); item != 0; item = item->nextItem ())
	{
		if (item->isSelected ())
		{
			nextItem = item->nextItem ();
			if (item->text (3) == "file")
			{
				list.append(KURL("file:/"+item->fullName()));
			}
			else
			{
			}
		}
	}
	
	KIO::Job *m=KIO::move( list, KURL("file:/"+KGlobalSettings::trashPath()), true );
	connect(m, SIGNAL(result( KIO::Job *)), this, SLOT(deletionDone( KIO::Job *)));
	
	if (nextItem)
	{
		QIconView::setCurrentItem (nextItem);
		nextItem->setSelected (true);
		QIconView::setSelected (currentItem(), true, true);
		ensureItemVisible (currentItem()->fileInfo());
	}
}


void
ImageListeView::slotShred()
{
	if(checkShred())
	if(KMessageBox::questionYesNo(mw->iv, i18n("Do you really want to shred the selected file(s) ?"), i18n("Confirm deletion")) == KMessageBox::No)
		return;
	KURL::List list;
	FileItem *nextItem = 0;
	for (FileItem * item = firstItem ();
	    item != 0; item =  item->nextItem ())
	{
		if (item->isSelected ())
		{
			nextItem = item->nextItem ();
			if (item->text (3) == "file")
			{
				list.append(KURL("file:/"+item->fullName()));
			}
			else
			{
			}
		}
	}
	KIO::DeleteJob *del=KIO::del( list, true, true );
	connect(del, SIGNAL(result( KIO::Job *)), this, SLOT(deletionDone( KIO::Job *)));
	if (nextItem)
	{
		QIconView::setCurrentItem (nextItem);
		nextItem->setSelected (true);
		QIconView::setSelected (currentItem(), true, true);

		ensureItemVisible (currentItem()->fileInfo());
	}
}


void ImageListeView::stopLoading ()
{
	il->stopLoading (true);
}


void ImageListeView::first ()
{
	if(!hasImages())
	{
		mw->setEmptyImage();
		return;	
	}	
	FileItem *item=firstItem();
	
	QIconView::ensureItemVisible (item);
	QIconView::setCurrentItem (item);
	QIconView::setSelected (item, true, false);
	qApp->processEvents();
	item ->setSelected (true);
}
 


void ImageListeView::last ()
{
	if(!hasImages())
		return;
		
	FileItem *item=lastItem();	
	QIconView::ensureItemVisible (item);
	QIconView::setCurrentItem (item);
	QIconView::setSelected (item, true, false);
	qApp->processEvents();
	item ->setSelected (true);	
}	



FileItem*
ImageListeView::findItem (const QString & text) const
{
	FileItem *i;
	for (i=(ImageFileItem*)QIconView::firstItem(); i; i = (ImageFileItem*)i->nextItem() )
	{
		if(((ImageFileItem*)i)->text(0)==text)
			return i;
	}
	return NULL;
}




void ImageListeView::slotOpenWith()
{
	FileItem *item = currentItem ();
  	if (!item)
    		return;
		
	if(mw->fullScreen())
		mw->slotFullScreen();
		
	KURL::List url(currentItem()->fullName());
	KOpenWithDlg dialog(url, mw);
	if(dialog.exec()!=0)
	{
		KRun::run(dialog.text(), KURL::List("file:/"+item->fullName()));
	}
};

void ImageListeView::slotUnselectAll()
{
	clearSelection ();
}
void ImageListeView::slotInvertSelection()
{	
	QIconView::invertSelection () ;
}
void ImageListeView::slotSelectAll()
{	
	QIconView::selectAll(true);
}

bool
ImageListeView::checkTrash()
{
	return trash;
}
void
ImageListeView::setCheckTrash(bool trash)
{
	this->trash=trash;
}
bool
ImageListeView::checkDelete()
{
	return del;
}
void
ImageListeView::setCheckDelete(bool del)
{
	this->del=del;
}
 bool
ImageListeView::checkShred()
{
	return shred;
}
void
ImageListeView::setCheckShred(bool shred)
{
	this->shred=shred;
}

void
ImageListeView::setItemTextPos ( ItemTextPos pos )
{
	if(itemTextPos()==pos)
		return;
	if(pos==Bottom)
	{
		setGridX(gridX()-200+10);
		setWordWrapIconText (false);	
}
	else
	{
		setGridX(gridX()+200-10);
		setWordWrapIconText (false);	
	}
	QIconView::setItemTextPos(pos);
	arrangeItemsInGrid ();
}


void 
ImageListeView::slotFilesMoveTo()
{
	QStrList uris;
	for (FileItem* item = firstItem (); item != 0; item =  item->nextItem ())
	{
		if (item->isSelected () )
		{
			uris.append(QUriDrag::localFileToUri(item->fullName()));
		}
	}
	if(!uris.isEmpty())
	{
		QString destDir=KFileDialog::getExistingDirectory(mw->currentURL(), mw, i18n("Move selected files to ..."));
		if(!destDir.isEmpty())
		{
			mw->moveFilesTo(uris, destDir+"/");	
		}
	}
}

void 
ImageListeView::slotFilesCopyTo()
{
	QStrList uris;
	for (FileItem* item = firstItem (); item != 0; item = item->nextItem())
	{
		if (item->isSelected () )
		{
			uris.append(QUriDrag::localFileToUri(item->fullName()));
		}
	}
	if(!uris.isEmpty())
	{
		QString destDir=KFileDialog::getExistingDirectory(mw->currentURL(), mw, i18n("Move selected files to ..."));
		if(!destDir.isEmpty())
		{
			mw->copyFilesTo(uris, destDir+"/");	
		}
	}
}

void
ImageListeView::highlight(const KFileViewItem *item)
{
	KFileView::highlight(item);
}

