#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sg_include.h"
#include "sg_err.h"

/* Test code for D. Gilbert's extensions to the Linux OS SCSI generic ("sg")
   device driver.
*  Copyright (C) 1999 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program uses the SCSI command READ BUFFER on the given sg
   device, first to find out how big it is and then to read that
   buffer. The '-q' option skips the data transfer from the kernel
   DMA buffers to the user space. The '-b=num' option allows the
   buffer size (in KBytes) to be specified (default is to use the
   number obtained from READ BUFFER (descriptor) SCSI command).
   The '-s=num' option allows the total size of the transfer to be
   set (in megabytes, the default is 200 MB).

   Version 0.71 (20010321)

*/
        

#define OFF sizeof(struct sg_header)
#define RB_MODE_DESC 3
#define RB_MODE_DATA 2
#define RB_DESC_LEN 4
#define RB_MB_TO_READ 200

#ifndef SG_MAX_SENSE
#define SG_MAX_SENSE 16
#endif



int main(int argc, char * argv[])
{
    int sg_fd, res, j, m;
    unsigned int k, num;
    unsigned char rbCmdBlk [10] = {0x3C, 0, 0, 0, 0, 0, 0, 0, 
                                   0, 0};
    unsigned char * rbBuff = malloc(OFF + sizeof(rbCmdBlk) + 512);
    int rbInLen = OFF + sizeof(rbCmdBlk);
    int rbOutLen;
    unsigned char * buffp = rbBuff + OFF;
    struct sg_header * rsghp = (struct sg_header *)rbBuff;
    int buf_capacity = 0;
    int do_quick = 0;
    int buf_size = 0;
    unsigned int total_size_mb = RB_MB_TO_READ;
    char * file_name = 0;

    for (j = 1; j < argc; ++j) {
        if (0 == strncmp("-b=", argv[j], 3)) {
            m = 3;
            num = sscanf(argv[j] + m, "%d", &buf_size);
            if ((1 != num) || (buf_size <= 0)) {
                printf("Couldn't decode number after '-b' switch\n");
                file_name = 0;
                break;
            }
            buf_size *= 1024;
        }
        else if (0 == strncmp("-s=", argv[j], 3)) {
            m = 3;
            num = sscanf(argv[j] + m, "%u", &total_size_mb);
            if (1 != num) {
                printf("Couldn't decode number after '-s' switch\n");
                file_name = 0;
                break;
            }
        }
        else if (0 == strcmp("-q", argv[j]))
            do_quick = 1;
        else if (*argv[j] == '-') {
            printf("Unrecognized switch: %s\n", argv[j]);
            file_name = 0;
            break;
        }
        else
            file_name = argv[j];
    }
    if (0 == file_name) {
        printf("Usage: 'sg_rbuf [-q] [-b=num] [-s=num] <generic_device>'\n");
        printf("  where: -q       quick, don't xfer to user space\n");
        printf("         -b=num   num is buff size to use (in KBytes)\n");
        printf("         -s=num   num is total size to read (in MBytes)\n");
        printf("                    default total size is 200 MBytes\n");
        printf("                    max total size is 4000 MBytes\n");
        if (rbBuff) free(rbBuff);
        return 1;
    }
    
    sg_fd = open(file_name, O_RDWR);
    if (sg_fd < 0) {
        perror("sg_rbuf: open error");
        if (rbBuff) free(rbBuff);
        return 1;
    }
    /* Don't worry, being very careful not to write to a none-sg file ... */
    res = ioctl(sg_fd, SG_GET_TIMEOUT, 0);
    if (res < 0) {
        /* perror("ioctl on generic device, error"); */
        printf("sg_rbuf: not a sg device, or wrong driver\n");
        if (rbBuff) free(rbBuff);
        return 1;
    }
    
    rbOutLen = OFF + RB_DESC_LEN;
    rsghp->pack_len = 0;                /* don't care */
    rsghp->pack_id = 0;
    rsghp->reply_len = rbOutLen;
    rsghp->twelve_byte = 0;
    rsghp->result = 0;
#ifndef SG_GET_RESERVED_SIZE
    rsghp->sense_buffer[0] = 0;
#endif
    memcpy(rbBuff + OFF, rbCmdBlk, sizeof(rbCmdBlk));
    rbBuff[OFF + 1] = RB_MODE_DESC;
    rbBuff[OFF + 8] = RB_DESC_LEN;

    res = write(sg_fd, rbBuff, rbInLen);
    if (res < 0) {
        perror("sg_rbuf: write (desc) error");
        if (rbBuff) free(rbBuff);
        return 1;
    }
    if (res < rbInLen) {
        printf("sg_rbuf: wrote less (desc), ask=%d, got=%d\n", rbInLen, res);
        if (rbBuff) free(rbBuff);
        return 1;
    }
    
    memset(buffp, 0, RB_DESC_LEN);
    res = read(sg_fd, rbBuff, rbOutLen);
    if (res < 0) {
        perror("sg_rbuf: read (desc) error");
        if (rbBuff) free(rbBuff);
        return 1;
    }
    if (res < rbOutLen) {
        printf("sg_rbuf: read less (desc), ask=%d, got=%d\n", rbOutLen, res);
        if (rbBuff) free(rbBuff);
        return 1;
    }
#ifdef SG_GET_RESERVED_SIZE
    if (! sg_chk_n_print("sg_rbuf: desc", rsghp->target_status, 
                         rsghp->host_status, rsghp->driver_status, 
                         rsghp->sense_buffer, SG_MAX_SENSE)) {
        printf("sg_rbuf: perhaps %s doesn't support READ BUFFER\n",
               file_name);
        if (rbBuff) free(rbBuff);
        return 1;
    }
#else
    if ((rsghp->result != 0) || (0 != rsghp->sense_buffer[0])) {
        printf("sg_rbuf: read(desc) result=%d\n", rsghp->result);
        if (0 != rsghp->sense_buffer[0])
            sg_print_sense("sg_rbuf: desc", rsghp->sense_buffer, 
                           SG_MAX_SENSE);
        printf("sg_rbuf: perhaps %s doesn't support READ BUFFER\n",
               file_name);
        if (rbBuff) free(rbBuff);
        return 1;
    }
#endif
    buf_capacity = ((buffp[1] << 16) | (buffp[2] << 8) | buffp[3]);
    printf("READ BUFFER reports: buffer capacity=%d, offset boundary=%d\n", 
           buf_capacity, (int)buffp[0]);
    
    if (0 == buf_size)
        buf_size = buf_capacity; 
    else if (buf_size > buf_capacity) {
        printf("Requested buffer size=%d exceeds reported capacity=%d\n",
               buf_size, buf_capacity);
        if (rbBuff) free(rbBuff);
        return 1;
    }
#ifdef SG_DEF_RESERVED_SIZE
    res = ioctl(sg_fd, SG_SET_RESERVED_SIZE, &buf_size);
    if (res < 0)
        perror("sg_rbuf: SG_SET_RESERVED_SIZE error");
#endif

    if (rbBuff) free(rbBuff);
    rbBuff = malloc(OFF + sizeof(rbCmdBlk) + buf_size);
    buffp = rbBuff + OFF;
    rsghp = (struct sg_header *)rbBuff;
    
    num = (total_size_mb * 1024U * 1024U) / (unsigned int)buf_size;
    for (k = 0; k < num; ++k) {
        rbOutLen = OFF + buf_size;
        rsghp->pack_len = 0;                /* don't care */
        rsghp->reply_len = rbOutLen;
        rsghp->twelve_byte = 0;
        rsghp->result = 0;
        memcpy(rbBuff + OFF, rbCmdBlk, sizeof(rbCmdBlk));
        rbBuff[OFF + 1] = RB_MODE_DATA;
        rbBuff[OFF + 6] = 0xff & (buf_size >> 16);
        rbBuff[OFF + 7] = 0xff & (buf_size >> 8);
        rbBuff[OFF + 8] = 0xff & buf_size;

        rsghp->pack_id = k;
        res = write(sg_fd, rbBuff, rbInLen);
        if (res < 0) {
            perror("sg_rbuf: write (data) error");
            if (rbBuff) free(rbBuff);
            return 1;
        }
        if (res < rbInLen) {
            printf("sg_rbuf: wrote less (data), ask=%d, got=%d\n", 
                   rbInLen, res);
            if (rbBuff) free(rbBuff);
            return 1;
        }
        
        res = read(sg_fd, rbBuff, do_quick ? OFF : rbOutLen);
        if (res < 0) {
            perror("sg_rbuf: read (data) error");
            if (rbBuff) free(rbBuff);
            return 1;
        }
        if ((! do_quick) && (res < rbOutLen)) {
            printf("sg_rbuf: read less (data), ask=%d, got=%d\n", 
                   rbOutLen, res);
            if (rbBuff) free(rbBuff);
            return 1;
        }
#ifdef SG_GET_RESERVED_SIZE
        if (! sg_chk_n_print("sg_rbuf: data", rsghp->target_status, 
                             rsghp->host_status, rsghp->driver_status, 
                             rsghp->sense_buffer, SG_MAX_SENSE)) {
            if (rbBuff) free(rbBuff);
            return 1;
        }
#else
        if ((rsghp->result != 0) || (0 != rsghp->sense_buffer[0])) {
            printf("sg_rbuf: data result=%d\n", rsghp->result);
            if (0 != rsghp->sense_buffer[0])
                sg_print_sense("sg_rbuf: data", rsghp->sense_buffer, 
                               SG_MAX_SENSE);
            if (rbBuff) free(rbBuff);
            return 1;
        }
#endif
    }
    printf("Read %u MBytes (actual %u MB, %u bytes), buffer size=%d KBytes\n",
    	   total_size_mb, (num * buf_size) / 1048576, num * buf_size,
	   buf_size / 1024);

    if (rbBuff) free(rbBuff);
    res = close(sg_fd);
    if (res < 0) {
        perror("sg_rbuf: close error");
        return 1;
    }
    return 0;
}
