
/*
#    Sfront, a SAOL to C translator    
#    This file: Network library -- constants and externs
#    Copyright (C) 1999  Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


/*************************/
/* network include files */
/*************************/

#ifndef NSYS_NET_INCLUDE

#define NSYS_NET_INCLUDE

#ifndef NSYS_NET

#include <stdio.h> 
#include <stdlib.h> 
#include <errno.h> 
#include <string.h> 
#include <ctype.h>
#include "net_local.h"

#endif

/* harmful on Solaris so we delete functionality */

#if (!defined(sun) && !defined(__sun__))
#include <sys/ioctl.h>
#endif

#include <unistd.h> 
#include <fcntl.h>
#include <sys/time.h>  
#include <sys/types.h> 
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <netinet/in.h> 
#include <net/if.h>

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                           Constants                                    */
/*________________________________________________________________________*/

/***********************/
/* debug/warning flags */
/***********************/

#define NSYS_WARN_NONE     0    /* never warn about anything */
#define NSYS_WARN_STANDARD 1    /* print connect messages    */
#define NSYS_WARN_UNUSUAL  2    /* print unusual events      */

#define NSYS_WARN   NSYS_WARN_UNUSUAL


#define NSYS_JOURNAL_DEBUG_OFF 0
#define NSYS_JOURNAL_DEBUG_ON 1

#define NSYS_JOURNAL_DEBUG NSYS_JOURNAL_DEBUG_OFF

#define NSYS_LATENOTES_DEBUG_OFF 0
#define NSYS_LATENOTES_DEBUG_ON 1

#define NSYS_LATENOTES_DEBUG NSYS_LATENOTES_DEBUG_OFF

/********************/
/* network constants */
/********************/

#define NSYS_MAXRETRY 256
#define NSYS_DONE  0
#define NSYS_ERROR 1

#define NSYS_NONBLOCK 0
#define NSYS_BLOCK    1

#define NSYS_BUFFSIZE 1024

/*****************************/
/* default system parameters */
/*****************************/

#ifndef SO_BSDCOMPAT
#define SO_BSDCOMPAT 0
#endif

#ifndef NSYS_MSETS
#define NSYS_MSETS 5
#endif

#ifndef ACYCLE
#define ACYCLE 42L
#endif

#ifndef ARATE
#define ARATE 44100.0F
#endif

#ifndef APPNAME
#define APPNAME "unknown"
#endif

#ifndef APPVERSION
#define APPVERSION "0.0"
#endif

#ifndef CSYS_MIDI_NUMCHAN
#define CSYS_MIDI_NUMCHAN  16
#endif

#ifndef CSYS_MIDI_NOOP
#define CSYS_MIDI_NOOP     0x70
#endif

#ifndef CSYS_MIDI_POWERUP
#define CSYS_MIDI_POWERUP  0x73
#endif

#ifndef CSYS_MIDI_NOTEOFF
#define CSYS_MIDI_NOTEOFF  0x80
#endif

#ifndef CSYS_MIDI_NOTEON 
#define CSYS_MIDI_NOTEON   0x90
#endif

#ifndef CSYS_MIDI_PTOUCH  
#define CSYS_MIDI_PTOUCH   0xA0
#endif

#ifndef CSYS_MIDI_CC  
#define CSYS_MIDI_CC       0xB0
#endif

#ifndef CSYS_MIDI_PROGRAM 
#define CSYS_MIDI_PROGRAM  0xC0
#endif

#ifndef CSYS_MIDI_CTOUCH 
#define CSYS_MIDI_CTOUCH   0xD0
#endif

#ifndef CSYS_MIDI_WHEEL 
#define CSYS_MIDI_WHEEL    0xE0
#endif

#ifndef CSYS_MIDI_SYSTEM
#define CSYS_MIDI_SYSTEM   0xF0
#endif

#ifndef CSYS_MIDI_CC_BANKSELECT_MSB 
#define CSYS_MIDI_CC_BANKSELECT_MSB  0x00
#endif

#ifndef CSYS_MIDI_CC_MODWHEEL_MSB
#define CSYS_MIDI_CC_MODWHEEL_MSB    0x01
#endif

#ifndef CSYS_MIDI_CC_CHANVOLUME_MSB
#define CSYS_MIDI_CC_CHANVOLUME_MSB  0x07
#endif

#ifndef CSYS_MIDI_CC_BANKSELECT_LSB 
#define CSYS_MIDI_CC_BANKSELECT_LSB  0x20
#endif

#ifndef CSYS_MIDI_CC_SUSTAIN
#define CSYS_MIDI_CC_SUSTAIN         0x40
#endif

#ifndef CSYS_MIDI_CC_ALLSOUNDOFF
#define CSYS_MIDI_CC_ALLSOUNDOFF     0x78
#endif

#ifndef CSYS_MIDI_CC_ALLNOTESOFF
#define CSYS_MIDI_CC_ALLNOTESOFF     0x7B
#endif

#ifndef NSYS_DISPLAY_RTCP
#define NSYS_DISPLAY_RTCP         0
#endif

#ifndef NSYS_DISPLAY_RTCP_HDR
#define NSYS_DISPLAY_RTCP_HDR     0
#endif

#ifndef NSYS_DISPLAY_RTCP_SRINFO
#define NSYS_DISPLAY_RTCP_SRINFO  0
#endif

#ifndef NSYS_DISPLAY_RTCP_SDES 
#define NSYS_DISPLAY_RTCP_SDES    0
#endif

#ifndef NSYS_DISPLAY_RTCP_RRINFO
#define NSYS_DISPLAY_RTCP_RRINFO  0
#endif

#ifndef NSYS_DISPLAY_RTCP_RRTCOMP
#define NSYS_DISPLAY_RTCP_RRTCOMP 0
#endif

/***************/
/* rtp defines */
/***************/

/* SSRC hash table constants */

#define NSYS_HASHSIZE 32         /* table size */
#define NSYS_HASHMASK 31         /* table mask */

/* preferred value for bidirectional RTP port */

#define NSYS_RTP_PORT     5004
#define NSYS_RTP_MAXPORT 65535

/* Based on Ethernet default MTU of 1500 bytes */

#define NSYS_UDPMAXSIZE 1472  
#define NSYS_RTPMAXSIZE 1460

/* masks for combing usec and sec for srand() seed */

#define NSYS_USECMASK   (0x000FFFFF)   
#define NSYS_SECMASK    (0xFFF00000)
#define NSYS_SECSHIFT   20

/* positions of interesting parts of RTP header */

#define NSYS_RTPLOC_BYTE1   0
#define NSYS_RTPLOC_PTYPE   1
#define NSYS_RTPLOC_SEQNUM  2
#define NSYS_RTPLOC_TSTAMP  4
#define NSYS_RTPLOC_SSRC    8

/* extended sequence number constants */

#define NSYS_RTPSEQ_MAXDIFF  4000        /* max number of lost packets */
#define NSYS_RTPSEQ_HIGHEST  0xFFFFFFFF  /* highest sequence number    */
#define NSYS_RTPSEQ_LOWLIMIT 4095        /* lo edge of 16-bit sequence */
#define NSYS_RTPSEQ_HILIMIT  61440       /* hi edge of 16-bit sequence */
#define NSYS_RTPSEQ_EXMASK   0xFFFF0000  /* masks out extension        */
#define NSYS_RTPSEQ_LOMASK   0x0000FFFF  /* masks out 16-bit seqnum    */
#define NSYS_RTPSEQ_EXINCR   0x00010000  /* increments extension       */

#define NSYS_RTPSEQ_MAXLOSS   8388607    /* max number of lost packets */
#define NSYS_RTPSEQ_MINLOSS  -8388608    /* min number of lost packets */
#define NSYS_RTPSEQ_FMASK    0x00FFFFFF  /* bit mask to add fraction   */
#define NSYS_RTPSEQ_TSIGN    0x00800000  /* bit mask for sign-extend   */

/* other RTP constants */

#define NSYS_RTPVAL_BYTE1   0x80  /* V=2, P=X=CC=0   */
#define NSYS_RTPVAL_SETMARK 0x80  /* | to set marker */
#define NSYS_RTPVAL_CLRMARK 0x7F  /* & to clr marker */
#define NSYS_RTPVAL_CHKMARK 0x80  /* & to chk marker */

#define NSYS_RTPLEN_HDR  12

/* RTP known payload set size */

#define NSYS_RTP_PAYSIZE 1

/* condition codes for a new RTP packet */

#define NSYS_RTPCODE_NORMAL   0   /* seqnum directly follows last packet */
#define NSYS_RTPCODE_LOSTONE  1   /* one packet has been lost            */
#define NSYS_RTPCODE_LOSTMANY 2   /* many packets have been lost         */
#define NSYS_RTPCODE_DISCARD  3   /* too late to use, or duplicate       */
#define NSYS_RTPCODE_SECURITY 4   /* possible replay attack, discard     */

/* number of bytes of digest to append */

#define NSYS_RTPSIZE_DIGEST   4

/**************/
/* rtcp sizes */
/**************/

/* sizes of complete RTCP packets */

#define NSYS_RTCPLEN_BYE      8    /* BYE packet */
#define NSYS_RTCPLEN_RREMPTY  8    /* RR with no  report    */
#define NSYS_RTCPLEN_RR      32    /* RR with one report    */
#define NSYS_RTCPLEN_SREMPTY 28    /* SR with no  reciept report */
#define NSYS_RTCPLEN_SR      52    /* SR with one receipt report */
#define NSYS_RTCPLEN_MINIMUM  8    /* the minimum packet size */

/* sizes of RTCP headers */

#define NSYS_RTCPLEN_SRHDR    8
#define NSYS_RTCPLEN_RRHDR    8
#define NSYS_RTCPLEN_SDESHDR  4
#define NSYS_RTCPLEN_BYEHDR   4

/* sizes of RTCP segments */

#define NSYS_RTCPLEN_SENDER  20
#define NSYS_RTCPLEN_REPORT  24

/* sizes of SDES headers */

#define NSYS_RTCPLEN_SDES_CHUNKHDR  4
#define NSYS_RTCPLEN_SDES_ITEMHDR   2

/******************/
/* rtcp locations */
/******************/

/* interesting shared locations in RR/SR headers */

#define NSYS_RTCPLOC_BYTE1  0
#define NSYS_RTCPLOC_PTYPE  1
#define NSYS_RTCPLOC_LENGTH 2
#define NSYS_RTCPLOC_SSRC   4

/* locations of SR sender info */

#define NSYS_RTCPLOC_SR_NTPMSB  8
#define NSYS_RTCPLOC_SR_NTPLSB 12
#define NSYS_RTCPLOC_SR_TSTAMP 16
#define NSYS_RTCPLOC_SR_PACKET 20
#define NSYS_RTCPLOC_SR_OCTET  24

/* interesting offsets into RR sender info */

#define NSYS_RTCPLOC_RR_SSRC       0
#define NSYS_RTCPLOC_RR_FRACTLOSS  4
#define NSYS_RTCPLOC_RR_NUMLOST    5
#define NSYS_RTCPLOC_RR_HISEQ      8
#define NSYS_RTCPLOC_RR_JITTER    12
#define NSYS_RTCPLOC_RR_LASTSR    16
#define NSYS_RTCPLOC_RR_DELAY     20

/* interesting locations in an SDES packet ITEM */

#define NSYS_RTCPLOC_SDESITEM_TYPE   0
#define NSYS_RTCPLOC_SDESITEM_LENGTH 1

/* interesting locations in RTCP BYE packet */

#define NSYS_RTCPLEN_BYE_SSRC       4

/***************/
/* rtcp values */
/***************/

#define NSYS_RTCPVAL_BYTE1   0x80        /* V=2, P=XC=0   */

#define NSYS_RTCPVAL_COUNTMASK   0x1F    /* mask for RC value  */
#define NSYS_RTCPVAL_COOKIEMASK  0xE0    /* mask for VP bits  */

#define NSYS_RTCPVAL_SR   0xC8    /* RTCP sender */
#define NSYS_RTCPVAL_RR   0xC9    /* RTCP receiver */
#define NSYS_RTCPVAL_SDES 0xCA    /* RTCP source desc */
#define NSYS_RTCPVAL_BYE  0xCB    /* RTCP bye */
#define NSYS_RTCPVAL_APP  0xCC    /* RTCP APP packet */

#define NSYS_RTCPVAL_SDES_CNAME   0x01  /* CNAME for SDES */
#define NSYS_RTCPVAL_SDES_NAME    0x02  /* NAME  for SDES */
#define NSYS_RTCPVAL_SDES_EMAIL   0x03  /* EMAIL for SDES */
#define NSYS_RTCPVAL_SDES_PHONE   0x04  /* PHONE for SDES */
#define NSYS_RTCPVAL_SDES_LOC     0x05  /* LOC for SDES */
#define NSYS_RTCPVAL_SDES_TOOL    0x06  /* TOOL for SDES */
#define NSYS_RTCPVAL_SDES_NOTE    0x07  /* NOTE for SDES */
#define NSYS_RTCPVAL_SDES_PRIV    0x08  /* PRIV for SDES */

#define NSYS_RTCPVAL_SDES_SIZE   9      /* number of SDES types */

/****************/
/* rtcp control */
/****************/

/* RTCP and SIP INFO timer constants */

#define NSYS_RTCPTIME_INCR       5      /* 5 seconds between RTCP sends  */
#define NSYS_RTCPTIME_SKIP       2      /* threshold for retransmit skip */
#define NSYS_RTCPTIME_TIMEOUT   25      /* 25 second SSRC timeout        */
#define NSYS_RTCPTIME_MLENGTH  300      /* 5 minute limit for msession   */
#define NSYS_SIPINFO_TRIGGER     6      /* every 5 RTCP send a SIP INFO  */

/* RTCP-monitored exceptions */

#define NSYS_RTCPEX_RTPSIP       1     /* no response on SIP RTP channel */
#define NSYS_RTCPEX_RTPNEXT      2     /* skip RTP retransmission cycle  */
#define NSYS_RTCPEX_RTCPSIP      4     /* no response on SIP RTP channel */
#define NSYS_RTCPEX_RTCPNEXT     8     /* skip RTCP retransmission cycle */
#define NSYS_RTCPEX_NULLROOT     16    /* if nsys_srcroot == NULL        */
#define NSYS_RTCPEX_SRCDUPL      32    /* SSRC clash                     */

/****************/
/* mpeg4-samidi */
/****************/

/*+++++++++++++++++++++++*/
/*  midi command payload */
/*+++++++++++++++++++++++*/

#define NSYS_SM_MLENMAX     63  /* max length of MIDI section */
#define NSYS_SM_EXPANDMAX  186  /* worst case expansion       */

/*++++++++++++++++++++++++++++*/
/* recovery journal constants */
/*++++++++++++++++++++++++++++*/

#define NSYS_SM_SETS 128  /* | to set S header bit   */
#define NSYS_SM_CLRS 127  /* & to clear S header bit */
#define NSYS_SM_CHKS 128  /* & to check S header bit */

#define NSYS_SM_SETH 128  /* | to set H history bit   */
#define NSYS_SM_CLRH 127  /* & to clear H history bit */
#define NSYS_SM_CHKH 128  /* & to check H history bit */

#define NSYS_SM_SLISTLEN 128  /* maximum S list size */

/*+++++++++++++++++++++++++*/
/* recovery journal header */
/*+++++++++++++++++++++++++*/

#define NSYS_SM_JH_SIZE 3       /* journal header size */
#define NSYS_SM_JH_LOC_FLAGS 0  /* byte 0: flags       */
#define NSYS_SM_JH_LOC_CHECK 1  /* bytes 1-2: checkpoint packet */

#define NSYS_SM_JH_CHANMASK 0x0F  /* to mask channel number  */

#define NSYS_SM_JH_SETA  64  /* | to set A header bit   */
#define NSYS_SM_JH_CLRA 191  /* & to clear A header bit */
#define NSYS_SM_JH_CHKA  64  /* & to check A header bit   */

#define NSYS_SM_JH_SETK  32  /* | to set K header bit   */
#define NSYS_SM_JH_CLRK 223  /* & to clear K header bit */
#define NSYS_SM_JH_CHKK  32  /* & to check K header bit   */

/*+++++++++++++++++++*/
/* channel header    */
/*+++++++++++++++++++*/

#define NSYS_SM_CH_SIZE 3       /* channel header size */

#define NSYS_SM_CH_LOC_FLAGS 0   /* location of S bit & channel */
#define NSYS_SM_CH_LOC_LEN   0   /* location of 10-bit length */
#define NSYS_SM_CH_LOC_TOC   2   /* location of table of contents */

#define NSYS_SM_CH_CHANMASK 0x78   /* to extract channel number  */
#define NSYS_SM_CH_LENMASK  0x03FF /* to extract channel length  */

#define NSYS_SM_CH_CHANSHIFT 3     /* to align channel number */

#define NSYS_SM_CH_TOC_SETP 128    /* Program Change (0xC) */
#define NSYS_SM_CH_TOC_CLRP 127    
#define NSYS_SM_CH_TOC_CHKP 128    

#define NSYS_SM_CH_TOC_SETW  64    /* Pitch Wheel (0xE) */
#define NSYS_SM_CH_TOC_CLRW 191    
#define NSYS_SM_CH_TOC_CHKW  64    

#define NSYS_SM_CH_TOC_SETN  32    /* NoteOff (0x8), NoteOn (0x9) */
#define NSYS_SM_CH_TOC_CLRN 223    
#define NSYS_SM_CH_TOC_CHKN  32    

#define NSYS_SM_CH_TOC_SETA  16    /* Poly Aftertouch (0xA) */
#define NSYS_SM_CH_TOC_CLRA 239    
#define NSYS_SM_CH_TOC_CHKA  16    

#define NSYS_SM_CH_TOC_SETT   8    /* Channel Aftertouch (0xD) */
#define NSYS_SM_CH_TOC_CLRT 247    
#define NSYS_SM_CH_TOC_CHKT   8    

#define NSYS_SM_CH_TOC_SETC   4    /* Control Change (0xB) */
#define NSYS_SM_CH_TOC_CLRC 251    
#define NSYS_SM_CH_TOC_CHKC   4    

#define NSYS_SM_CH_TOC_SETF   2    /* Application-specific */
#define NSYS_SM_CH_TOC_CLRF 253    
#define NSYS_SM_CH_TOC_CHKF   2    

#define NSYS_SM_CH_TOC_SETR   1    /* Reserved */
#define NSYS_SM_CH_TOC_CLRR 254    
#define NSYS_SM_CH_TOC_CHKR   1    

/*++++++++++++++++++++++++++++*/
/* chapter P (program change) */
/*++++++++++++++++++++++++++++*/

#define NSYS_SM_CP_SIZE 3               /* chapter size */

#define NSYS_SM_CP_LOC_PROGRAM 0        /* PROGRAM byte */
#define NSYS_SM_CP_LOC_BANKCOARSE 1     /* BANK-COARSE byte */
#define NSYS_SM_CP_LOC_BANKFINE   2     /* BANK-FINE byte */

#define NSYS_SM_CP_SETC 128             /* | to set C header bit   */
#define NSYS_SM_CP_CLRC 127             /* & to clear C header bit */
#define NSYS_SM_CP_CHKC 128             /* & to check C header bit */

#define NSYS_SM_CP_SETF 128             /* | to set F header bit   */
#define NSYS_SM_CP_CLRF 127             /* & to clear F header bit */
#define NSYS_SM_CP_CHKF 128             /* & to check F header bit */

/*+++++++++++++++++++++++++*/
/* chapter W (pitch wheel) */
/*+++++++++++++++++++++++++*/

#define NSYS_SM_CW_SIZE 2           /* chapter size */

#define NSYS_SM_CW_LOC_FIRST 0      /* FIRST  byte */
#define NSYS_SM_CW_LOC_SECOND 1     /* SECOND byte */

#define NSYS_SM_CW_SETD 128         /* | to set D header bit   */
#define NSYS_SM_CW_CLRD 127         /* & to clear D header bit */
#define NSYS_SM_CW_CHKD 128         /* & to check D header bit */

/*+++++++++++++++++++*/
/* chapter N (notes) */
/*+++++++++++++++++++*/

#define NSYS_SM_CN_SIZE  256         /* maximum chapter size (front) */
#define NSYS_SM_CB_SIZE   16         /* maximum bitfield size  */

#define NSYS_SM_CN_LOC_LENGTH   0    /* LENGTH  byte */
#define NSYS_SM_CN_LOC_LOWHIGH  1    /* LOWHIGH byte */
#define NSYS_SM_CN_LOC_LOGSTART 2    /* start of note logs */ 

#define NSYS_SM_CN_LOWMASK   0xF0    /* mask for LOW  */
#define NSYS_SM_CN_HIGHMASK  0x0F    /* mask for HIGH */
#define NSYS_SM_CN_LOWSHIFT     4    /* shift align for LOW */

#define NSYS_SM_CN_HDRSIZE 2         /* size of chapter n header */  
#define NSYS_SM_CN_LOGSIZE 2         /* size of each note log    */  

#define NSYS_SM_CN_LOC_NUM 0         /* note number byte in each log  */
#define NSYS_SM_CN_LOC_VEL 1         /* velocity byte in each log */

#define NSYS_SM_CN_SETB 128          /* | to set B header bit   */
#define NSYS_SM_CN_CLRB 127          /* & to clear B header bit */
#define NSYS_SM_CN_CHKB 128          /* & to check B header bit */

#define NSYS_SM_CN_SETY 128          /* | to set Y header bit   */
#define NSYS_SM_CN_CLRY 127          /* & to clear Y header bit */
#define NSYS_SM_CN_CHKY 128          /* & to check Y header bit */

#define NSYS_SM_CN_ARRAYSIZE  128    /* size of note state arrays */

#define NSYS_SM_CN_BFMIN        0    /* minimum bitfield value    */
#define NSYS_SM_CN_BFMAX       15    /* maximum bitfield value    */
#define NSYS_SM_CN_BFSHIFT      3    /* note to bitfield byte shift */
#define NSYS_SM_CN_BFMASK    0x07    /* mask for bit position      */

#define NSYS_SM_CN_MAXDELAY  0.025F   /* lifetime of Y bit (seconds)  */
#define NSYS_SM_CN_RECDELAY  0.10F    /* receiver test for Y lifetime */

/*++++++++++++++++++++++++*/
/* chapter A (poly touch) */
/*++++++++++++++++++++++++*/

#define NSYS_SM_CA_SIZE  257         /* maximum chapter size */

#define NSYS_SM_CA_LOC_LENGTH   0    /* LENGTH byte */
#define NSYS_SM_CA_LOC_LOGSTART 1    /* start of note logs */ 

#define NSYS_SM_CA_HDRSIZE 1         /* size of chapter a header */  
#define NSYS_SM_CA_LOGSIZE 2         /* size of each note log */  

#define NSYS_SM_CA_LOC_NUM 0         /* NOTENUM byte in each log  */
#define NSYS_SM_CA_LOC_PRESSURE 1    /* PRESSURE byte in each log */

#define NSYS_SM_CA_SETF 128          /* | to set F header bit   */
#define NSYS_SM_CA_CLRF 127          /* & to clear F header bit */
#define NSYS_SM_CA_CHKF 128          /* & to check F header bit */

#define NSYS_SM_CA_SETD 128          /* | to set D header bit   */
#define NSYS_SM_CA_CLRD 127          /* & to clear D header bit */
#define NSYS_SM_CA_CHKD 128          /* & to check D header bit */

#define NSYS_SM_CA_ARRAYSIZE  128    /* size of ptouch state arrays */

/*++++++++++++++++++++++++++++++++*/
/* chapter T (channel aftertouch) */
/*++++++++++++++++++++++++++++++++*/

#define NSYS_SM_CT_SIZE 1            /* chapter size */

#define NSYS_SM_CT_LOC_PRESSURE 0    /* PRESSURE byte */

/*++++++++++++++++++++++++++++*/
/* chapter C (control change) */
/*++++++++++++++++++++++++++++*/

#define NSYS_SM_CC_SIZE  257         /* maximum chapter size */

#define NSYS_SM_CC_LOC_LENGTH   0    /* LENGTH byte */
#define NSYS_SM_CC_LOC_LOGSTART 1    /* start of controller logs */ 

#define NSYS_SM_CC_HDRSIZE 1         /* size of chapter c header */  
#define NSYS_SM_CC_LOGSIZE 2         /* size of each controller log */  

#define NSYS_SM_CC_LOC_LNUM 0        /* CONTROLLER byte in each log  */
#define NSYS_SM_CC_LOC_LVAL 1        /* VALUE/COUNT byte in each log */

#define NSYS_SM_CC_SETF 128          /* | to set F header bit   */
#define NSYS_SM_CC_CLRF 127          /* & to clear F header bit */
#define NSYS_SM_CC_CHKF 128          /* & to check F header bit */

#define NSYS_SM_CC_SETD 128          /* | to set D header bit   */
#define NSYS_SM_CC_CLRD 127          /* & to clear D header bit */
#define NSYS_SM_CC_CHKD 128          /* & to check D header bit */

#define NSYS_SM_CC_ARRAYSIZE  128    /* size of controller state arrays */

#define NSYS_SM_CC_MODULO 0x7F       /* for updating COUNT state */

#define NSYS_SM_CC_SETP 128          /* | to set P COUNT state bit */
#define NSYS_SM_CC_CLRP 127          /* & to clear P COUNT state bit */
#define NSYS_SM_CC_CHKP 128          /* & to check P COUNT bit */

/*++++++++++++++++++++++++++*/
/* chapter F (app speciFic) */
/*++++++++++++++++++++++++++*/

#define NSYS_SM_CF_HDRSIZE       2   /* size of chapter f header */

#define NSYS_SM_CF_LOC_LENGTH    0   /* LENGTH field */

#define NSYS_SM_CF_LENMASK  0x03FF   /* mask for length field */

/*++++++++++++++++++++++++*/
/* mpeg4-samidi: receiver */
/*++++++++++++++++++++++++*/

/* number of MIDI channels to pre-allocate per receiver */

#define NSYS_RECVCHAN  2

/* constants to handle flag bit */

#define NSYS_SM_RV_SETF 128        /* | to set flag bit   */
#define NSYS_SM_RV_CLRF 127        /* & to clear flag bit */
#define NSYS_SM_RV_CHKF 128        /* & to check flag bit */

/*++++++++++++++++++++++*/
/* mpeg4-samidi: sender */
/*++++++++++++++++++++++*/

#define NSYS_SM_GUARD_ONTIME     NSYS_SM_CN_MAXDELAY
#define NSYS_SM_GUARD_MINTIME    0.25F
#define NSYS_SM_GUARD_STDTIME    0.10F
#define NSYS_SM_GUARD_MAXTIME    1.0F

#ifndef NSYS_NET
#define NSYS_SM_FEC_NONE     0
#define NSYS_SM_FEC_NOGUARD  1
#define NSYS_SM_FEC_MINIMAL  2  
#define NSYS_SM_FEC_STANDARD 3  
#define NSYS_SM_FEC_EXTRA    4  
#endif

#define NSYS_SM_LATETIME        0.040F     /* maximum lateness */
#define NSYS_SM_LATEWINDOW        3.5F     /* time window for reset */

/*+++++++++++++++++++++++++++++*/
/* mpeg4-samidi: miscellaneous */
/*+++++++++++++++++++++++++++++*/

/* return values for nsys_netin_journal_recovery() */

#define NSYS_JOURNAL_RECOVERED  0   /* must be zero */
#define NSYS_JOURNAL_FILLEDBUFF 1
#define NSYS_JOURNAL_CORRUPTED  2

/*******/
/* SIP */
/*******/

#ifndef NSYS_SIP_IP
#define NSYS_SIP_IP          "128.32.131.152"
#endif

#ifndef NSYS_SIP_RTP_PORT
#define NSYS_SIP_RTP_PORT    5060
#endif

#ifndef NSYS_SIP_RTCP_PORT
#define NSYS_SIP_RTCP_PORT   5061
#endif

#define NSYS_HOSTNAMESIZE 256
#define NSYS_SDPNATSIZE   128
#define NSYS_CNAMESIZE     32

#define NSYS_NETIN_SIPSTATE 0   /* SIP INVITE state machine states */
#define NSYS_NETIN_SDPSTATE 1
#define NSYS_NETIN_EOFSTATE 2
#define NSYS_NETIN_ERRSTATE 3

#define NSYS_SIP_RETRYMAX       5         /* max retry for initial connect */
#define NSYS_SIP_AUTHRETRYMAX   4         /* max retry for authentication  */

#define NSYS_SIP_UNIXTONTP      2208988800UL /* add to UNIX time for NTP (s) */
 

/*******************/
/* Crypto/Security */
/*******************/

#define NSYS_MD5_LENGTH     16   /* array size for an MD5 digest in binary */
#define NSYS_MD5_ROUNDS  10000   /* number of MD5 passes for keydigest  */

#define NSYS_BASE64_LENGTH  25   /* string size of a BASE64 MD5 digest     */
#define NSYS_MKEY_LENGTH    32   /* length of random key for mirror        */

#define NSYS_MAXLATETIME     5400   /* clocks no more than 90 minutes fast */
#define NSYS_MAXSSESIONTIME 21600   /* session maxtime is 6 hours          */

#define NSYS_MSESSION_NAME "mirror"  /* change if sfront hashes names  */

#ifndef NSYS_MSESSION_INTERVAL
#define NSYS_MSESSION_INTERVAL  12   /* pitch shift for mirror         */
#endif

#define NSYS_MSESSION_MAXRTP  9000   /* maximum RTP packets in session  */
#define NSYS_MSESSION_MAXRTCP  200   /* maximum RTCP packets in session */


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                           Variable externs                             */
/*________________________________________________________________________*/


/*************************/
/* rtp and rtcp: network */
/*************************/

extern int nsys_rtp_fd;               /* fd for rtp           */
extern int nsys_rtcp_fd;              /* fd for rtcp          */
extern int nsys_max_fd;               /* fd for select        */

extern unsigned short nsys_rtp_port;  /* actual rtp port      */ 
extern unsigned short nsys_rtcp_port; /* actual rtcp port     */ 

extern unsigned long nsys_rtp_cseq;   /* rtp cseq number      */
extern unsigned long nsys_rtcp_cseq;  /* rtcp cseq number     */

/*************************/
/* rtp and rtcp: packets */
/*************************/

extern unsigned char nsys_netout_rtp_packet[];  /* rtp packet out */

/* rtcp packets and lengths */

extern unsigned char * netout_rtcp_packet_bye;
extern unsigned char * netout_rtcp_packet_rrempty;
extern unsigned char * netout_rtcp_packet_rr;
extern unsigned char * netout_rtcp_packet_srempty;
extern unsigned char * netout_rtcp_packet_sr;

extern int netout_rtcp_len_bye;
extern int netout_rtcp_len_rrempty;
extern int netout_rtcp_len_rr;
extern int netout_rtcp_len_srempty;
extern int netout_rtcp_len_sr;

extern char * sdes_typename[];            /* rtcp debug array */

extern unsigned long nsys_netout_seqnum;  /* rtp output sequence number */
extern unsigned long nsys_netout_tstamp;  /* rtp output timestamp */
extern unsigned char nsys_netout_markbit; /* for setting markerbit */

/************************/
/* rtp payload support  */
/************************/

typedef struct payinfo {

  int pindex;               /* index into nsys_payload_types  */
  unsigned char ptype;      /* RTP payload number */
  char name[16];            /* name of payload    */
  int srate;                /* audio sample rate  */

} payinfo;

extern struct payinfo nsys_payload_types[];

/**************/
/* rtcp state */
/**************/

extern int nsys_sent_last;    /* a packet was sent last RTCP period     */
extern int nsys_sent_this;    /* a packet was sent this RTCP period     */

extern unsigned long nsys_sent_packets;  /* number of packets sent */
extern unsigned long nsys_sent_octets;   /* number of octets sent */

extern time_t nsys_nexttime;  /* time for next RTCP check               */
extern int nsys_rtcp_ex;      /* status flags to check at RTCP time     */

/******************/
/* identification */
/******************/

extern unsigned long nsys_myssrc;     /*  SSRC -- hostorder  */
extern unsigned long nsys_myssrc_net; /*  SSRC -- netorder   */

extern char * nsys_sessionname;
extern char * nsys_sessionkey;

extern unsigned char nsys_keydigest[];        /* digested key */
extern unsigned char nsys_session_base64[];   /* signed session name */
extern int nsys_msession;                     /* mirror session flag */
extern int nsys_msessionmirror;               /* mirror in mirror session */

extern int nsys_feclevel;
extern int nsys_lateplay;
extern float nsys_latetime;

extern char nsys_clientname[];
extern char nsys_clientip[];
extern char * nsys_username;

extern char nsys_cname[];
extern unsigned char nsys_cname_len;

extern int nsys_powerup_mset; 

/***********/
/* logging */
/***********/

extern int nsys_stderr_size;

/*********************/
/* MIDI input buffer */
/*********************/

extern unsigned char nsys_buff[];
extern long nsys_bufflen;
extern long nsys_buffcnt;

/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/* mpeg4-samidi: recovery journal sender state */
/*_____________________________________________*/

/*************************/
/* checkpoint management */
/*************************/

extern unsigned long netout_jsend_checkpoint_seqnum;  /* current checkpoint */
extern unsigned long netout_jsend_checkpoint_changed; /* 1 if changed */

/*********************/
/* S-list management */
/*********************/

extern unsigned char * netout_jsend_slist[];
extern int netout_jsend_slist_size;

/*******************************/
/* guard journal state machine */
/*******************************/

extern int netout_jsend_guard_send;    /* flag variable: send guard packet */
extern int netout_jsend_guard_time;    /* guard packet timer state */
extern int netout_jsend_guard_next;    /* reload value for timer */
extern int netout_jsend_guard_ontime;  /* minimum delay time for noteon */ 
extern int netout_jsend_guard_mintime; /* minimum delay time for (!noteon) */
extern int netout_jsend_guard_maxtime; /* maximum delay time */ 

/***************************/
/* recovery journal header */
/***************************/

extern unsigned char netout_jsend_header[];    /* journal header */

/**************************/
/* chanell journal record */
/**************************/

extern unsigned char netout_jsend_channel[];
extern unsigned char netout_jsend_channel_size;

/******************************************/
/* structure holding sender channel state */
/******************************************/

typedef struct netout_jsend_state {

  unsigned char  chan;  /* MIDI channel */
  unsigned short clen;  /* first two bytes of cheader */

  /* most recent MIDI commands, used for new source init */

  unsigned char history_active;     /* flag for channel activity */
  unsigned char history_cc_bankc;   /* current cc values */
  unsigned char history_cc_bankf;
  unsigned char history_cc_modwheel;
  unsigned char history_cc_volume;

  unsigned char history_program;       /* last program change */
  unsigned char history_program_bankc;
  unsigned char history_program_bankf;

  /* sequence numbers -- 0 or highest seqnum */

  unsigned long cheader_seqnum; 
  unsigned long chapterp_seqnum;
  unsigned long chapterw_seqnum;    
  unsigned long chaptert_seqnum;    
  unsigned long chaptern_seqnum;    
  unsigned long chaptern_seqarray[NSYS_SM_CN_ARRAYSIZE];
  unsigned long chaptera_seqnum;
  unsigned long chaptera_seqarray[NSYS_SM_CA_ARRAYSIZE];
  unsigned long chapterc_seqnum;
  unsigned long chapterc_seqarray[NSYS_SM_CC_ARRAYSIZE];
  
  /* sender state for chapter n */
  
  unsigned char chapterb_low;      
  unsigned char chapterb_high;   
  unsigned long chaptern_logptr[NSYS_SM_CN_ARRAYSIZE]; /* note log position */
  unsigned long chaptern_timer[NSYS_SM_CN_ARRAYSIZE];  /* Y timer values */
  unsigned long chaptern_timernum;                     /* number of values */
  int chaptern_sset;                                   /* S bit indicators */
  
  /* sender state for chapter a */
  
  unsigned long chaptera_logptr[NSYS_SM_CA_ARRAYSIZE];  /* log positions */
  int chaptera_sset;                                    /* S bit indicators */
  
  /* sender state for chapter c */
  
  unsigned long chapterc_logptr[NSYS_SM_CC_ARRAYSIZE];  /* log positions */
  int chapterc_sset;                                    /* S bit indicators */
  
  unsigned char chapterc_allnotes;                    /* COUNT state holders */
  unsigned char chapterc_allsound;
  unsigned char chapterc_sustain;                  /* pcccccccc: p=1-->down */

  /* sizes for dynamic chapters */
  
  int chaptern_size;    /* chapter n */
  int chapterb_size;    /* chapter n - bitfields */
  int chaptera_size;    /* chapter c */
  int chapterc_size;    /* chapter c */
  
  /* holds current packet bytes */
  
  unsigned char cheader[NSYS_SM_CH_SIZE];  /* chapter header */
  unsigned char chapterp[NSYS_SM_CP_SIZE]; /* chapter p  */
  unsigned char chapterw[NSYS_SM_CW_SIZE]; /* chapter w  */
  unsigned char chaptern[NSYS_SM_CN_SIZE]; /* chapter n  */
  unsigned char chapterb[NSYS_SM_CB_SIZE]; /* chapter n - bfields */
  unsigned char chaptera[NSYS_SM_CA_SIZE]; /* chapter a  */
  unsigned char chaptert[NSYS_SM_CT_SIZE]; /* chapter t  */
  unsigned char chapterc[NSYS_SM_CC_SIZE]; /* chapter c  */
  
} netout_jsend_state;

extern netout_jsend_state netout_jsend[];

/********************************************/
/* structure holding receiver channel state */
/********************************************/

typedef struct netout_jrecv_state {

  /* chapter p */

  unsigned char chapterp_program;
  unsigned char chapterp_bankc;
  unsigned char chapterp_bankf;

  /* chapter w */

  unsigned char chapterw_first;
  unsigned char chapterw_second;

  /* chapter n */

  unsigned char chaptern_vel[NSYS_SM_CN_ARRAYSIZE];
  unsigned long chaptern_tstamp[NSYS_SM_CN_ARRAYSIZE];
  unsigned long chaptern_extseq[NSYS_SM_CN_ARRAYSIZE];

  /* chapter a */

  unsigned char chaptera_pressure[NSYS_SM_CC_ARRAYSIZE];

  /* chapter t */

  unsigned char chaptert_pressure;

  /* chapter c */

  unsigned char chapterc_sustain;
  unsigned char chapterc_value[NSYS_SM_CC_ARRAYSIZE];

  /* navigation */

  unsigned char chan;
  struct netout_jrecv_state * next;

} netout_jrecv_state;

extern netout_jrecv_state * nsys_recvfree;  /* receiver free list */

/*******/
/* SIP */
/*******/

extern unsigned char nsys_rtp_invite[];
extern unsigned char nsys_rtcp_invite[];

extern int nsys_rtp_sipretry;                  /* sip server retry counter */
extern int nsys_rtcp_sipretry;                 

extern int nsys_rtp_authretry;                 /* reauthorization counter  */
extern int nsys_rtcp_authretry;

extern struct sockaddr_in nsys_sip_rtp_addr;   /* current SIP RTP channel */
extern char nsys_sip_rtp_ip[];
extern unsigned long nsys_sip_rtp_inet_addr;
extern unsigned short nsys_sip_rtp_port;
extern unsigned short nsys_sip_rtp_sin_port;

extern struct sockaddr_in nsys_sip_rtcp_addr;  /* current SIP RTCP channel */
extern char nsys_sip_rtcp_ip[];
extern unsigned long nsys_sip_rtcp_inet_addr;
extern unsigned short nsys_sip_rtcp_port;
extern unsigned short nsys_sip_rtcp_sin_port;

extern int nsys_graceful_exit;                 /* requests termination    */

extern unsigned char nsys_rtp_info[];          /* SIP INFO packets        */
extern unsigned char nsys_rtcp_info[];

extern int nsys_behind_nat;                    /* 1 if behind a nat        */
extern int nsys_sipinfo_count;                 /* INFO sending timer       */
extern int nsys_sipinfo_toggle;                /* RTP/RTCP toggle          */

/***************/
/* SSRC stack  */
/***************/

typedef struct source {

  /* information */

  int mset;                           /* MIDI extchan set  */
  unsigned long ssrc;                 /* RTP SSRC number   */
  unsigned long birthtime;            /* Time SSRC was born (UNIX)    */

  unsigned long siptime;              /* SIP INVITE sessiontime (UNIX)    */
  struct sockaddr_in * sdp_addr;      /* IP/port in SDP packet, unchanged */

  struct sockaddr_in * rtp_addr;      /* RTP IP/port of source */
  struct sockaddr_in * alt_rtp_addr;  /* RTP IP/port of NATing */

  struct sockaddr_in * rtcp_addr;     /* RTCP IP/port of source */
  struct sockaddr_in * alt_rtcp_addr; /* RTCP IP/port of NATing */
 
  unsigned char ptype;                /* payload type to send/recv */
  int pindex;                         /* index into nsys_payload_types[]  */
  int srate;                          /* SDP-assigned srate        */

  /* navigation */

  struct source * xtra;               /* if hash clashes */
  struct source * prev;               /* doubly-linked list */
  struct source * next;

  /* reception statistics */

  unsigned long received;             /* total num packets received   */
  unsigned long received_prior;       /* total at last SR/RR          */
  unsigned long expected_prior;       /* total expected at last SR/RR */

  unsigned long base_seq;             /* initial extended seq num   */
  unsigned long hi_lobits;            /* highest seq num low 16b    */
  unsigned long hi_ext;               /* highest extended seq num   */

  /* rtcp items */

  int j_delta;                        /* state variable for jitter  */
  unsigned long jitter;               /* current jitter value       */
  unsigned char lsr[4];               /* last SR timestamp received */
  struct timeval arrival;             /* arrival time of last SR    */
  time_t expire_rtcp;                 /* time to expire ssrc entry  */
  char * cname;                       /* canonical name             */
  unsigned long rtcp_received;        /* for DoS detection          */

  /* time model */

  int ontime;                         /* flags on-time RTP packets   */    
  unsigned long tm_convert;           /* local/remote time offset    */
  unsigned long tm_margin;            /* maximum lateness allowed    */
  unsigned long tm_lateflag;          /* congestion detection flag   */
  unsigned long tm_latetime;          /* tstamp of first late packet */

#if (NSYS_LATENOTES_DEBUG == NSYS_LATENOTES_DEBUG_ON)
  unsigned long tm_first;             /* temporary for printfs       */
  FILE * tm_fd;                       /* file pointer for dat files  */
#endif

  /* journal items */

  unsigned long last_hiseq_rec;       /* RTCP extended sequence number   */
  unsigned long last_hiseq_ext;       /* local-adjusted RTCP ext seq num */

  struct netout_jrecv_state
  * jrecv[CSYS_MIDI_NUMCHAN];         /* reciever journal state */

} source;

extern struct source * nsys_srcfree;       /* mset ssrc tokens */
extern struct source * nsys_ssrc[];        /* SSRC hash table  */
extern struct source * nsys_srcroot;       /* points into nsys_ssrc */


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                              Macros                                    */
/*________________________________________________________________________*/


/******************/
/* error-checking */
/******************/

#define  NSYS_ERROR_RETURN(x) do {\
      nsys_stderr_size+=fprintf(stderr, "  Error: %s.\n", x);\
      nsys_stderr_size+=fprintf(stderr, "  Errno Message: %s\n\n", strerror(errno));\
      return NSYS_ERROR; } while (0)

#define  NSYS_ERROR_TERMINATE(x) do {\
      nsys_stderr_size+=fprintf(stderr, "  Runtime Errno Message: %s\n", strerror(errno));\
      epr(0,NULL,NULL, "Network error -- " x );}  while (0)


/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/
/*                        Function Externs                                */
/*________________________________________________________________________*/

/******************/
/* error checking */
/******************/

extern void epr(int linenum, char * filename, char * token, char * message);

/****************/
/* net_siplib.c */
/****************/

extern int nsys_initsip(void);
extern void nsys_netin_reply(int fd, struct sockaddr_in * addr, 
			     unsigned char * packet, unsigned short status);

extern void nsys_netin_invite(int fd, struct sockaddr_in * addr, 
			      unsigned char * packet);

extern void nsys_createsip(int fd, char * method, unsigned char * sip,
			   char * natline, char * nonce);
extern void nsys_sendsip(int fd, struct sockaddr_in * addr, 
			 unsigned char * sip); 

extern unsigned char * nsys_netin_readmethod(int fd, struct sockaddr_in * addr, 
					     unsigned char * packet); 
extern void nsys_netin_ack(int fd, unsigned char * reply, 
			   unsigned long callid, unsigned long cseq);
extern int nsys_netin_replyparse(int fd, unsigned char * packet, 
				 char *nonce, char * natline, 
				 unsigned long * callid, unsigned long * cseq);
extern void nsys_netin_redirect(int fd, unsigned char * packet,
				unsigned short status);

extern int nsys_netin_sipaddr(int fd, char * ip, unsigned short port);
extern int nsys_netin_siporigin(int fd, struct sockaddr_in * addr);
extern int nsys_netin_make_marray(struct payinfo ** marray, char * media);
extern int nsys_netin_set_marray(char * line, struct payinfo marray[],
				 int mlen);

extern int nsys_netin_payvalid(struct payinfo marray[], int mlen, int fd);
extern void nsys_netin_payset(struct source * sptr, struct payinfo marray[],
			      int mlen);
extern int nsys_netin_noreplay(char * ip, unsigned short port, 
			       unsigned long sdp_time);
extern void nsys_sip_shutdown(void);

/****************/
/* net_rtplib.c */
/****************/

extern int nsys_setup(int block);
extern void nsys_shutdown(void);

extern source * nsys_netin_addsrc(int fd, long ssrc,
				  char * ip, unsigned short port);
extern int nsys_netin_rtpstats(source * sptr, unsigned char * packet);
extern void  nsys_warning(int level, char * message);
extern void nsys_terminate_error(char * message); 
extern void nsys_status(source * sptr, char * message);
extern int nsys_netin_ptypevalid(unsigned char ptype);
extern void nsys_netin_ptypeset(struct source * sptr, unsigned char ptype);
extern netout_jrecv_state * nsys_netin_newrecv(unsigned char chan);
extern char * nsys_find_clientip(char * ip);

/****************/
/* net_rtcplib.c */
/****************/

extern source * nsys_netin_rtcp(unsigned char * packet, int len, 
				struct sockaddr_in * ipaddr);
extern int nsys_make_rtcp(void);
extern int nsys_make_rtcpbye(void);
extern void nsys_netout_rtcptime(void);
extern void nsys_delete_ssrc(unsigned long ssrc, char * reason);
extern source * nsys_harvest_ssrc(int fd, struct sockaddr_in * ipaddr);
extern void nsys_late_windowcheck(source * sptr, unsigned long tstamp);
extern char * nsys_netin_newcname(unsigned char * packet, int len);
extern void nsys_netin_bye(unsigned char * packet, int len);
extern int nsys_netout_excheck(void);
extern void nsys_netout_keepalive(void);
extern void nsys_netin_rtcp_display(unsigned char * packet, int len,
				    struct timeval * now);
extern void nsys_netout_rtcpreport(void);
extern void nsys_netout_rtcpsend(unsigned char * p, int len);
extern int nsys_netin_clear_mset(unsigned char * buff, long * fill,
				 long size);

extern void nsys_netout_rtcp_initpackets(void);
extern void nsys_netout_rtcp_initrr(unsigned char * p, int len);
extern void nsys_netout_rtcp_initsr(unsigned char * p, int len);
extern void nsys_netout_rtcp_initsdes(unsigned char * p, int len);
extern void nsys_netout_rtcp_initbye(unsigned char * p);
extern void nsys_netin_rtcp_trunc(int sub);
extern void nsys_netin_latenotes_open(source * sptr);

/***************/
/* net_jsend.c */
/***************/

extern int nsys_netin_journal_create(unsigned char * packet, int len);
extern void nsys_netout_journal_addstate(unsigned char cmd,
					unsigned char ndata,
					unsigned char vdata);
extern void nsys_netout_journal_addhistory(unsigned char cmd,
					   unsigned char ndata,
					   unsigned char vdata);
extern void nsys_netin_journal_trimstate(source * lptr);
extern void nsys_netout_midistate_init(void);
extern void nsys_netout_guard_tick(void);
extern void nsys_netout_journal_changecheck(void);
extern void nsys_netout_journal_addprogram(netout_jsend_state * jsend, 
					   unsigned char ndata);
extern void nsys_netout_journal_addpwheel(netout_jsend_state * jsend, 
					  unsigned char ndata, 
					  unsigned char vdata);
extern void nsys_netout_journal_addnoteoff(netout_jsend_state * jsend, 
					   unsigned char ndata);
extern void nsys_netout_journal_addnoteon(netout_jsend_state * jsend, 
					  unsigned char ndata, 
					  unsigned char vdata);
extern void nsys_netout_journal_addptouch(netout_jsend_state * jsend, 
					   unsigned char ndata,
					   unsigned char vdata);
extern void nsys_netout_journal_addctouch(netout_jsend_state * jsend, 
					  unsigned char ndata);
extern void nsys_netout_journal_addcontrol(netout_jsend_state * jsend, 
					   unsigned char ndata,
					   unsigned char vdata);

extern void nsys_netout_journal_trimchapter(netout_jsend_state * jsend,
					    int channel);
extern void nsys_netout_journal_trimprogram(netout_jsend_state * jsend);
extern void nsys_netout_journal_trimpwheel(netout_jsend_state * jsend);
extern void nsys_netout_journal_trimctouch(netout_jsend_state * jsend);

extern void nsys_netout_journal_trimallcontrol(netout_jsend_state * jsend);
extern void nsys_netout_journal_trimpartcontrol(netout_jsend_state * jsend,
						unsigned long minseq);
extern void nsys_netout_journal_trimallptouch(netout_jsend_state * jsend);
extern void nsys_netout_journal_trimpartptouch(netout_jsend_state * jsend,
						unsigned long minseq);
extern void nsys_netout_journal_trimallnote(netout_jsend_state * jsend);
extern void nsys_netout_journal_trimpartnote(netout_jsend_state * jsend,
					     unsigned long minseq);

/**************/
/* net_recv.c */
/**************/

extern int nsys_netin_journal_recovery(source * sptr, int rtpcode, 
				       unsigned char * packet, int numbytes,
				       unsigned char * buff, long * fill,
				       long size);

extern void nsys_netin_journal_trackstate(source * sptr, unsigned char cmd,
					  unsigned char ndata, 
					  unsigned char vdata);

extern int nsys_netin_journal_addcmd(source * sptr, unsigned char * buff, 
				     long * fill, long size,
				     unsigned char cmd, unsigned char ndata,
				     unsigned char vdata);

extern int nsys_netin_jrec_program(source * sptr, unsigned char * p,
				   netout_jrecv_state * jrecv,
				   unsigned char * buff,  
				   long * fill, long size);

extern int nsys_netin_jrec_wheel(source * sptr, unsigned char * p,
				 netout_jrecv_state * jrecv,
				 unsigned char * buff,  
				 long * fill, long size);

extern int nsys_netin_jrec_notelog(source * sptr, unsigned char * p,
				   netout_jrecv_state * jrecv, 
				   unsigned char many,
				   short loglen, unsigned char * checkptr,
				   unsigned char * buff, 
				   long * fill, long size);

extern int nsys_netin_jrec_bitfield(source * sptr, unsigned char * p,
				    netout_jrecv_state * jrecv, 
				    unsigned char low, unsigned char high,
				    unsigned char * buff, long * fill,
				    long size);

extern int nsys_netin_jrec_ptouch(source * sptr, unsigned char * p,
				  netout_jrecv_state * jrecv, short loglen,
				  unsigned char many, unsigned char * buff, 
				  long * fill, long size);

extern int nsys_netin_jrec_ctouch(source * sptr, unsigned char * p,
				  netout_jrecv_state * jrecv,
				  unsigned char * buff,  
				  long * fill, long size);

extern int nsys_netin_jrec_control(source * sptr, unsigned char * p,
				   netout_jrecv_state * jrecv, short loglen,
				   unsigned char many, unsigned char * buff, 
				   long * fill, long size);


/****************/
/* net_crypto.c */
/****************/

extern unsigned char * nsys_md5(unsigned char * digest, unsigned char * text,
				int len);

extern char * nsys_hmac_md5(unsigned char* text, int text_len, 
			    unsigned char * keydigest, unsigned char * digest);

extern unsigned char * nsys_digest_base64(unsigned char * output, 
					  unsigned char * input);

extern int nsys_digest_syntaxcheck(char * s);

#endif /* NSYS_NET_INCLUDE */

/* end Network library -- constants and externs */

