/* -*- linux-c -*- */

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <linux/types.h>
#include <endian.h>
#include <byteswap.h>
#include "hashtab.h"
#include "psid.h"

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define cpu_to_le32(x) (x)
#define le32_to_cpu(x) (x)
#define cpu_to_le64(x) (x)
#define le64_to_cpu(x) (x)
#else
#define cpu_to_le32(x) bswap_32(x)
#define le32_to_cpu(x) bswap_32(x)
#define cpu_to_le64(x) bswap_64(x)
#define le64_to_cpu(x) bswap_64(x)
#endif

typedef __u32 psid_t;

#if 0
#define DPRINTF(args...) printf(args)
#else
#define DPRINTF(args...)
#endif

/*
 * Subdirectory for the PSID mapping files.
 */
#define PSEC_SECDIR "...security"
#define PSEC_SECDIR_MODE 0700

/*
 * The PSID mapping files
 */
#define PSEC_CONTEXTS   0	/* security contexts */
#define PSEC_INDEX      1	/* psid -> (offset, len) of context */
#define PSEC_INODES     2	/* ino -> psid */
#define PSEC_NFILES	3	/* total number of security files */

static char *psec_sfiles[PSEC_NFILES] = 
{
	"contexts",
	"index", 
	"inodes"
};

#define PSEC_SECFILE_MODE (S_IFREG | 0600)

/*
 * Record structure for entries in index file.
 */
typedef struct 
{
	loff_t ofs;	/* offset within file, in bytes */
	size_t len;	/* length of context, in bytes */
} s_index_t;

#define SINDEX_MASK (sizeof(s_index_t)-1)

typedef struct psidtab_node {
	psid_t psid;
	char *context;
	struct psidtab_node *next;
} psidtab_node_t;

#define PSIDTAB_SLOTS 32
#define BAD_PSIDS 32

struct psidtab {
	psidtab_node_t *slots[PSIDTAB_SLOTS];	/* PSID cache */
        hashtab_t ino2con;
	int    initialized;	/* is the PSID mapping initialized? */
	psid_t next_psid;	/* next PSID to be allocated */
	int files[PSEC_NFILES];  /* mapping files */
#define contexts_fp files[PSEC_CONTEXTS] /* contexts file */
#define index_fp files[PSEC_INDEX]	 /* index file */
#define inodes_fp files[PSEC_INODES]       /* inode file */
	psid_t bad_psids[BAD_PSIDS];     /* bad PSIDs */
	int n_bad;			 /* number of bad PSIDs */
	s_index_t raw_sindex;		 /* PSID 0 */
};
typedef struct psidtab psidtab_t;

#define PSIDTAB_HASH(psid) (psid & (PSIDTAB_SLOTS - 1))


/*
 * Two PSIDs are reserved:  PSID 0
 * refers to the security context to
 * assign to unlabeled objects in the
 * file system and PSID 1 refers to the
 * security context of the file system itself.
 */
#define FILE_PSID 0
#define FS_PSID   1

/*
 * Create a PSID cache.
 */
static int psidtab_create(psidtab_t **tp)
{
	psidtab_t 	*t;


	t = (psidtab_t *) malloc(sizeof(psidtab_t));
	if (!t)
		return -ENOMEM;
	memset(t, 0, sizeof(psidtab_t));
	t->ino2con = hashtab_create();
	*tp = t;
	return 0;
}


/* 
 * Release the PSID mapping files and
 * free the memory used by the PSID cache.
 */
void psidtab_destroy(psidtab_t *t)
{
	psidtab_node_t     *cur, *tmp;
	int             hvalue, i;


	for (i = 0; i < PSEC_NFILES; i++) {
		if (t->files[i]) {
			close(t->files[i]);
		}
	}

	for (hvalue = 0; hvalue < PSIDTAB_SLOTS; hvalue++) {
		cur = t->slots[hvalue];
		while (cur) {
			tmp = cur;
			cur = cur->next;
			free(tmp);
		}
	}

	hashtab_destroy(t->ino2con);

	free(t);
}


/*
 * Insert an entry for the pair (`psid', `context') in the PSID cache.
 */
static int psidtab_insert(psidtab_t *t, psid_t psid, char *context)
{
	psidtab_node_t     *new;
	int             hvalue;


	hvalue = PSIDTAB_HASH(psid);	
	new = (psidtab_node_t *) malloc(sizeof(psidtab_node_t));
	if (!new) {
		return -ENOMEM;
	}
	new->psid = psid;
	new->context = context;
	new->next = t->slots[hvalue];
	t->slots[hvalue] = new;
	return 0;
}


/*
 * Return the context associated with the 
 * PSID `psid' in the PSID cache.
 * Return 0 if no match is found.
 */
static char *psidtab_search_psid(psidtab_t *t, psid_t psid)
{
	psidtab_node_t     *cur;
	int             hvalue;


	hvalue = PSIDTAB_HASH(psid);
	for (cur = t->slots[hvalue]; cur ; cur = cur->next) {
		if (psid == cur->psid)
			return cur->context;
	}

	return 0;
}


/*
 * Return the PSID associated with the 
 * context `context' in the PSID cache.
 * Return 0 if no match is found.
 * This function never returns a reserved PSID.
 */
static psid_t psidtab_search_context(psidtab_t *t, char *context)
{
	psidtab_node_t     *cur;
	int             hvalue;


	for (hvalue = 0; hvalue < PSIDTAB_SLOTS; hvalue++) {
		for (cur = t->slots[hvalue]; cur; cur = cur->next) {
			if ((strcmp(context,cur->context) == 0) && 
			    cur->psid > FS_PSID)
				return cur->psid;
		}
	}
	
	return 0;
}


/*
 * Change the context associated with PSID `psid'
 * to the context `context' in the PSID cache.
 */
static void psidtab_change_psid(psidtab_t *t, 
				psid_t psid, char *context)
{
	psidtab_node_t     *cur;
	int             hvalue;


	hvalue = PSIDTAB_HASH(psid);
	for (cur = t->slots[hvalue]; cur ; cur = cur->next) {
		if (psid == cur->psid) {
			cur->context = context;
			return;
		}
	}

	return;
}

#define HASH_EVAL 0

#if HASH_EVAL
static void psidtab_hash_eval(psidtab_t *t, char *tag)
{
	int             i, nel, chain_len, max_chain_len, slots_used;
	psidtab_node_t     *node;

	nel = 0;
	slots_used = 0;
	max_chain_len = 0;
	for (i = 0; i < PSIDTAB_SLOTS; i++) {
		node = t->slots[i];
		if (node) {
			slots_used++;
			chain_len = 0;
			while (node) {
				nel++;
				chain_len++;
				node = node->next;
			}
			if (chain_len > max_chain_len)
				max_chain_len = chain_len;
		}
	}

	printf("\n%s psidtab:  %d entries and %d/%d buckets used, longest chain length %d\n",
	       tag, nel, slots_used, PSIDTAB_SLOTS, max_chain_len);
}
#endif


/*
 * Allocate and return a new PSID for the 
 * security context associated with the 
 * context `context'.  
 */
static int newpsid(psidtab_t *t,
		   char *context,
		   psid_t *out_psid)
{
	s_index_t raw_sindex;
	psid_t psid;
	off_t off, off2, len;
	int rc;


	DPRINTF("newpsid:  obtaining psid for context %s\n", context);

	len = strlen(context)+1;

	/*
	 * Append the security context to the contexts file.
	 */
	off = lseek(t->contexts_fp, 0, SEEK_END);
	if (off < 0) {
		perror("lseek");
		return off;
	}
	rc = write(t->contexts_fp, context, len);
	if (rc < 0) {
		perror("write");
		return rc;
	}
	DPRINTF("newpsid:  added %s to contexts at %d\n", 
		context, off);

	/*
	 * Write the index record at the location
	 * for the next allocated PSID.
	 */
	raw_sindex.ofs = cpu_to_le64(off);
	raw_sindex.len = cpu_to_le32(len);
	psid = t->next_psid++;
	off2 = psid * sizeof(s_index_t);
	rc = lseek(t->index_fp, off2, SEEK_SET);
	if (rc < 0) {
		perror("lseek");
		return rc;
	}
	rc = write(t->index_fp, &raw_sindex, sizeof(s_index_t));
	if (rc < 0) {
		perror("write");
		return rc;
	}
	DPRINTF("newpsid:  added (%d,%d) to index at %d for new psid %d\n", 
		off, len, off2, psid); 

	/*
	 * Add the (`psid', `context') pair to the PSID cache.
	 */
	rc = psidtab_insert(t, psid, context);
	if (rc)
		return rc;

	DPRINTF("newpsid:  added (%d, %s) to psidtab\n", psid, context);

	*out_psid = psid;
	return 0;
}


/*
 * Initialize the incore data in `t' for a PSID mapping
 * from the PSID mapping files in the  file system `sb'.
 * Set the SID of `sb' and the SIDs of the inodes for the
 * root directory, the mapping directory and the mapping
 * files.
 *
 * If the file system is unlabeled and the 
 * file system is being mounted read-write, then 
 * create a new PSID mapping on it.  
 */
psidtab_t *psidtab_init(char *rootdir, 
			char *file_context, char *fs_context, int reset)
{
        psidtab_t *t;
	char pathname[1024];
	char pathname2[1024];
	int index;
	int need_to_init = 0;
	s_index_t sindex, raw_sindex;
	psid_t file_psid, fs_psid;
	char *cbuf;
	int rc = 0;

	rc = psidtab_create(&t);
	if (rc)
		return NULL;

	sprintf(pathname, "%s/%s", rootdir, PSEC_SECDIR);
	DPRINTF("psidfiles_init:  looking up %s\n", pathname);
	rc = access(pathname, F_OK);
	if (rc < 0) {
		DPRINTF("psidfiles_init:  %s did not exist; creating\n", 
			pathname);

		need_to_init = 1;

		rc = mkdir(pathname, PSEC_SECDIR_MODE);
		if (rc < 0) {
			perror(pathname);
			goto bad;
		}
	} else if (reset) {
		DPRINTF("psidfiles_init:  %s exists; resetting\n", 
			pathname);

		need_to_init = 1;

		for (index = 0; index < PSEC_NFILES; index++) {
			sprintf(pathname, "%s/%s/%s", rootdir, PSEC_SECDIR, psec_sfiles[index]);
			sprintf(pathname2, "%s/%s/%s.old", rootdir, PSEC_SECDIR, psec_sfiles[index]);
			DPRINTF("psidfiles_init:  renaming %s to %s\n", pathname, pathname2);
			rc = rename(pathname,pathname2);
			if (rc < 0) {
				perror(pathname);
				goto bad;
			}
		}
	}

	/* Look up or create each mapping file */
	for (index = 0; index < PSEC_NFILES; index++) {
		sprintf(pathname, "%s/%s/%s", rootdir, PSEC_SECDIR, psec_sfiles[index]);
		DPRINTF("psidfiles_init:  checking for %s\n", pathname);
		rc = open(pathname, O_RDWR);
		if (rc < 0) {
			DPRINTF("psidfiles_init:  %s did not exist; creating\n", 
				pathname);

			rc = open(pathname, O_CREAT|O_RDWR, PSEC_SECFILE_MODE);
			if (rc < 0) {
				perror(pathname);
				goto bad;
			}
		}

		t->files[index] = rc;
	}

	if (need_to_init) {
		rc = newpsid(t, file_context, &file_psid);
		if (rc)
			goto bad;

		if (file_psid) {
			printf("psidtab_init:  default file psid should be zero, is %d\n", file_psid);
			goto bad;
		}
		

		rc = newpsid(t, fs_context, &fs_psid);
		if (rc)
			goto bad;

		if (fs_psid != FS_PSID) {
			printf("psidtab_init:  file system psid should be %d, is %d\n", FS_PSID, fs_psid);
			goto bad;
		}

		t->initialized = 1;

		return t;
	}	

	/*	
	 * The mapping subdirectory and files already existed.
	 * Load the mapping files into the PSID cache.  Obtain
	 * the SIDs for the file system and the root directory
	 * from the mapping.  Set the initialized flag.
	 */
	DPRINTF("psidfiles_init:  reading index and contexts files\n");
	rc = lseek(t->index_fp, 0, SEEK_SET);
	if (rc < 0) {
		perror("lseek");
		return t;
	}
	rc = read(t->index_fp, &raw_sindex, sizeof(s_index_t));
	t->raw_sindex.ofs = raw_sindex.ofs;
	t->raw_sindex.len = raw_sindex.len;
	while (rc > 0) {	    
		sindex.ofs = le64_to_cpu(raw_sindex.ofs);
		sindex.len = le32_to_cpu(raw_sindex.len);

		DPRINTF("psidfiles_init:  read index (%d,%d)\n",
			sindex.ofs, sindex.len);

		cbuf = malloc(sindex.len);
		if (!cbuf) {
			rc = -ENOMEM;
			goto bad;
		}

		rc = lseek(t->contexts_fp, sindex.ofs, SEEK_SET);
		if (rc < 0) {
			perror("lseek");
			return t;
		}
		rc = read(t->contexts_fp, cbuf, sindex.len);
		if (rc < 0) {
			perror("read");
			goto bad;
		}
		
		DPRINTF("psidfiles_init:  psid %d -> context %s\n",
				t->next_psid, cbuf);
		
		rc = psidtab_insert(t, t->next_psid, cbuf);
		t->next_psid++;
		if (rc) {
			goto bad;
		}

		rc = read(t->index_fp, &raw_sindex, sizeof(s_index_t));
	}

	if (rc < 0) {
		perror("read");
		goto bad;
	}

	t->initialized = 1;

	return t;
bad:	
	return NULL;
}

/*
 * Look up the PSID of `ino' in the PSID mapping
 * and return the context associated with it.
 */
char *psid_to_context(psidtab_t *t, unsigned long ino)
{
	char *context;
	psid_t psid, raw_psid;
	int rc;

	context = (char *)hashtab_search(t->ino2con, ino);
	if (context) 
		return context;

	rc = lseek(t->inodes_fp, ino * sizeof(psid_t), SEEK_SET);
	if (rc < 0) {
		perror("lseek");
		return NULL;
	}

	rc = read(t->inodes_fp, &raw_psid, sizeof(psid_t));
	if (rc < 0)
		return NULL;
	if (rc == 0)
		psid = 0;
	else
		psid = le32_to_cpu(raw_psid);

	context = psidtab_search_psid(t, psid);

	if (!context) {
		printf("psid_to_context:  no context for psid %d\n", psid);
		context = psidtab_search_psid(t, 0);
		if (!context) {
			return NULL;
		}
		context_to_psid(t, ino, context);
	}

	hashtab_insert(t->ino2con,
			    (hashtab_key_t) ino,
			    (hashtab_datum_t) context);
	return context;
}

/*
 * Look up the security context `context'
 * in the PSID mapping and set the PSID
 * for the inode accordingly.  If no PSID exists in
 * the PSID mapping for the security context, then
 * allocate a new PSID and assign it to the security
 * context.
 */
int context_to_psid(psidtab_t *t,
		    unsigned long ino,
		    char *context)
{
	psid_t          psid, raw_psid;
	int rc;

	psid = psidtab_search_context(t, context);
	if (!psid) {
		rc = newpsid(t, context, &psid);
		if (rc)	
			return rc;
	}

	raw_psid = cpu_to_le32(psid);

	rc = lseek(t->inodes_fp, ino * sizeof(psid_t), SEEK_SET);
	if (rc < 0) {
		perror("lseek");
		return rc;
	}

	rc = write(t->inodes_fp, &raw_psid, sizeof(psid_t));
	if (rc < 0)
		return rc;

	return hashtab_replace(t->ino2con,
			       (hashtab_key_t) ino,
			       (hashtab_datum_t) context);
}

/*
 * Change the security context associated
 * with PSID `psid' in the PSID mapping to
 * the security context `context'.
 */
int chpsid(psidtab_t *t,
		  psid_t psid,
		  char *context)
{
	s_index_t raw_sindex;
	psid_t clone_psid;
	loff_t off, off2;
	size_t len;
	int rc;
	psidtab_node_t     *cur;
	int             hvalue;

 
	DPRINTF("chpsid:  changing psid %d to context %s\n", psid, context);

	/*
	 * Check to see if there is already a PSID for 
	 * the security context.
	 */
	for (hvalue = 0; hvalue < PSIDTAB_SLOTS; hvalue++) {
		for (cur = t->slots[hvalue]; cur; cur = cur->next) {
			if (strcmp(context, cur->context) == 0)
				goto found;
		}
	}

found:
	if (hvalue < PSIDTAB_SLOTS) {
		/*
		 * There is already a PSID for the security context.
		 * The index record for `psid' can simply be changed 
		 * to be the same as the index record for this PSID.
		 */
		clone_psid = cur->psid;
		DPRINTF("chpsid:  cloning existing psid %d\n", clone_psid);
		off2 = clone_psid * sizeof(s_index_t);
		rc = lseek(t->index_fp, off2, SEEK_SET);
		if (rc < 0) {
			perror("lseek");
			return rc;
		}
		rc = read(t->index_fp, &raw_sindex, sizeof(s_index_t));
		if (rc < 0) {
			perror("read");
			return rc;
		}
		
		off = le64_to_cpu(raw_sindex.ofs);
		len = le32_to_cpu(raw_sindex.len);
	}
	else {
		/*
		 * There is no PSID for the security context.
		 * Add the security context to the contexts file 
		 * and define a new index record for `psid'.
		 */
		DPRINTF("chpsid:  adding new entry to contexts\n");

		rc = lseek(t->contexts_fp, 0, SEEK_END);
		if (rc < 0) {
			perror("lseek");
			return rc;
		}
		rc = write(t->contexts_fp, context, len);
		if (rc < 0) {
			printf("chpsid:  error %d in writing to contexts\n", -rc);
			return rc;
		}

		raw_sindex.ofs = cpu_to_le64(off);
		raw_sindex.len = cpu_to_le32(len);

		DPRINTF("chpsid:  added %s to contexts at %d\n", 
			context, off);
	}

	/*
	 * Change the index record for `psid'.
	 */
	off2 = psid * sizeof(s_index_t);
	rc = lseek(t->index_fp, off2, SEEK_SET);
	if (rc < 0) {
		perror("lseek");
		return rc;
	}
	rc = write(t->index_fp, &raw_sindex, sizeof(s_index_t));
	if (rc < 0) {
		printf("chpsid:  error %d in writing to index\n", -rc);
		return rc;
	}

	DPRINTF("chpsid:  changed index at %d to (%d, %d) for psid %d\n", 
		off2, off, len, psid); 

	/* 
	 * Change the context for `psid' in the PSID cache.
	 */
	psidtab_change_psid(t, psid, context);

	DPRINTF("chpsid:  changed to (%d, %s) in psidtab\n", psid, context);

	return 0;
}

