/*
 * tcp_client
 * 
 * Create a client socket, connect to a specified address,
 * send a message and receive the echo.
 *
 * Permit testing of socket_secure, connect_secure, 
 * sendto_secure, getsockname_secure and getpeername_secure.
 * 
 * Calls socket_secure if '-s socket_context' is specified.
 * Calls connect_secure if '-l listener_context' is specified.
 * Calls sendto_secure if '-p peer_context' is specified.
 *
 * Usage:  tcp_client [-s socket_context] [-l listener_context] [-p peer_context] address port
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>
#include <stdio.h>

#include <socket_secure.h>
#include <ss.h>

void usage(char *progname) 
{
	printf("usage:  %s [-s socket_context] [-l listener_context] [-p peer_context] address port\n", progname);
	exit(1);
}

int main(int argc, char **argv)
{
	struct sockaddr_in sin;
	char buf[1024], context[255];
	int s, sinlen, ret, buflen, c, contextlen;
	security_id_t so_sid = 0, listener_sid = 0, peer_sid = 0, tmp_sid = 0; 


	while ( (c = getopt(argc, argv, "s:l:p:")) != EOF) {
		switch (c) {
		case 's':
		case 'l':
		case 'p':
			ret = security_context_to_sid(optarg,strlen(optarg)+1,&tmp_sid);
			if (ret) {
				perror(optarg);
				exit(ret);
			}
			if (c == 's')
				so_sid = tmp_sid;
			else if (c == 'l')
				listener_sid = tmp_sid;
			else if (c == 'p')
				peer_sid = tmp_sid;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (optind != (argc - 2)) {
		usage(argv[0]);
	}


	/* Create the client socket */
	if (so_sid) 
		s = socket_secure(AF_INET, SOCK_STREAM, 0, so_sid);
	else
		s = socket(AF_INET, SOCK_STREAM, 0);
	if (s == -1) {
		perror("socket");
		exit(1);
	}

	/* Obtain the security context of the client socket */
	sin.sin_family = AF_INET;
	sinlen = sizeof(struct sockaddr_in);
	ret = getsockname_secure(s, &sin, &sinlen, &tmp_sid);
	if (ret) {
		perror("getsockname_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}
	
	printf("Created socket with context %s\n", context);

	/* Connect to the server */
	sin.sin_addr.s_addr = inet_addr(argv[optind++]);
	sin.sin_port = atoi(argv[optind++]);
	sin.sin_port = htons(sin.sin_port);
	if (listener_sid) 
		ret = connect_secure(s, (struct sockaddr *)&sin, sinlen, listener_sid);
	else
		ret = connect(s, (struct sockaddr *)&sin, sinlen);
	if (ret == -1) {
		perror("connect");
		exit(1);
	}

	ret = getpeername_secure(s, (struct sockaddr*)&sin, &sinlen, &tmp_sid);
	if (ret) {
		perror("getpeername_secure");
		exit(1);
	}

	contextlen = sizeof(context);
	ret = security_sid_to_context(tmp_sid, context, &contextlen);
	if (ret) {
		perror("security_sid_to_context");
		exit(1);
	}

	printf("Connected to peer (%s, %d) with context %s\n",
	       inet_ntoa(sin.sin_addr), ntohs(sin.sin_port), context);

	/* Send and receive echo of message */
	strcpy(buf, "hello world");
	buflen = strlen(buf)+1;

	if (peer_sid) 
		ret = sendto_secure(s, buf, buflen, 0, 0, 0, peer_sid, 0);
	else
		ret = send(s, buf, buflen, 0);
	if (ret != buflen) {
		perror("send");
		exit(1);
	}

	buf[0] = 0;
	ret = recv(s, buf, sizeof(buf), 0);
	if (ret == -1) {
		perror("recv");
		exit(1);
	}

	printf("Received ##%s##\n", buf);
}
