#include "get_sid_list.h"

/* query_user_sid - given a list of sids, allow the user to choose one.  The 
 *                  default is the first sid in the list.  Returns 0 on
 *                  success, 1 on failure
 */
int query_user_sid (security_id_t *ordered_sid_list, int length, 
                     security_id_t *user_sid)
{
    security_context_t scontext = 0; /* The default security context       */
    int sconlen = 0;         /* The length of the default security context */
    int retval;              /* used for return values                     */
    char response[10] ;      /* The user's response                        */
    int choice;              /* The index in the list of the sid chosen by
                                the user                                   */

    /* Make sure there are sids in the list */
    if (length <= 0)
        return 1;

    /* We start out assuming we're using the first sid in the list */
    *user_sid = ordered_sid_list[0];

    /* Get the default context */
    retval = security_sid_to_context (ordered_sid_list[0], scontext, &sconlen);
    if (retval != 0)
    {
        scontext = (security_context_t) malloc (sconlen * 
                                                sizeof (security_context_t));
        if (scontext != 0)
            retval = security_sid_to_context (ordered_sid_list[0], scontext, 
                                              &sconlen);
    }

    if (retval == 0)
    {
        printf ("\nYour default context is %s.\n", scontext);
        if (length > 1)
        {
            printf ("Do you want to choose a different one? [n]");
            fflush (stdin);
            fgets (response, sizeof (response), stdin);
            fflush (stdin);

            if ((response[0] == 'y') || (response[0] == 'Y'))
            {
                choice = sid_menu (ordered_sid_list, length);
                if ((choice >= 0) && (choice < length))
                    *user_sid = ordered_sid_list[choice];
            }
        }
    }

    if (scontext != 0)
        free (scontext);
    return retval;
}


/* sid_menu - given a list of sids, presents a menu of security contexts
 *            to the user.  Returns the number (position in the list) of
 *            the user selected sid.
 */
int sid_menu (security_id_t *ordered_sid_list, int length)
{
    int i;              /* array index                        */
    security_context_t scontext = 0; /* the current context   */
    int sconlen = 0;    /* length of scontext                 */
    int choice = 0;     /* index of the user's choice         */
    int retval;         /* the return value                   */
    char response[10];  /* string to hold the user's response */

    printf ("\n\n");
    for (i = 0; i < length; i++)
    {
        /* convert each sid to a security context */
        retval = security_sid_to_context (ordered_sid_list[i], scontext, 
                                          &sconlen);
        if (retval != 0)
        {
            if (scontext != 0)
                free (scontext);
            scontext = (security_context_t) malloc(sconlen 
                                                   *sizeof(security_context_t));
            if (scontext != 0)
                retval = security_sid_to_context (ordered_sid_list[i], scontext,
                                                  &sconlen);
        }
        
        /* Print the context if the conversion was successful */
        if (retval == 0)
        {
            printf ("[%d] %s\n", i+1, scontext);
        }
        else
        {
            printf ("[%d] Unavailable\n", i);
        }
    }

    while ((choice < 1) || (choice > length))
    {
        printf ("Enter number of choice: ");
        fflush (stdin);
        fgets (response, sizeof (response), stdin);
        fflush (stdin);
        choice = strtol (response, NULL, 10);
    }
 
    if (scontext != 0)
        free (scontext);

    return (choice-1);
}


/* get_field - given fieldstr - the "name" of a field, query the user 
 *             and set the new value of the field
 */
void get_field (const char* fieldstr, char* newfield, int newfieldlen)
{
    int done = 0;  /* true if a non-empty field has been obtained */
 
    while (!done)  /* Keep going until we get a value for the field */
    {
        printf ("\tEnter %s ", fieldstr);
        fflush (stdin);
        fgets (newfield, newfieldlen, stdin);
        fflush (stdin);
        newfield[strlen(newfield)-1] = '\0';
 
        if (strlen(newfield) == 0)  
        {
            printf ("You must enter a %s\n", fieldstr);
        }
        else
        {
            done = 1;
        }
    }
}


/* manual_user_enter - provides a way for a user to manually enter a
 *                     context in case the policy doesn't allow a list
 *                     to be obtained.
 *                     given the userid, queries the user and places the
 *                     sid chosen by the user into user_sid.  Returns 0
 *                     on success.
 */
int manual_user_enter (char *user, security_id_t *user_sid)
{
    char response[10];     /* Used to get yes or no answers from user */
    char role[100];        /* The role requested by the user          */
    int  rolelen = 100;
    char type[100];        /* The type requested by the user          */
    int  typelen = 100;

#ifdef CONFIG_FLASK_MLS
    char level[100];       /* The level requested by the user         */
    int  levellen = 100;
#endif

    context_t new_context; /* The new context chosen by the user      */
    char *user_context;    /* String value of the user's context      */
    int done = 0;          /* true if a valid sid has been obtained   */

    while (!done)
    {
        printf ("Would you like to enter a security context? [y]");
        fgets (response, sizeof(response), stdin);
        if ((response[0] == 'n') || (response[0] == 'N'))
            return -1;
 
        /* Initialize the context.  How this is done depends on whether
           or not MLS is enabled                                        */
#ifdef CONFIG_FLASK_MLS
        new_context = context_new ("user:role:type:level");
#else
        new_context = context_new ("user:role:type");
#endif
  
        /* Allow the user to enter each field of the context individually */
        if (context_user_set (new_context, user))
        {
            context_free (new_context);
            return -1;
        }
        get_field ("role", role, rolelen);
        if (context_role_set (new_context, role))
        {
            context_free (new_context);
            return -1;
        }
        get_field ("type", type, typelen);
        if (context_type_set (new_context, type))
        {
            context_free (new_context);
            return -1;
        }
 
#ifdef CONFIG_FLASK_MLS
        getlevel ("level", level, levellen);
        if (context_level_set (new_context, level))
        {
            context_free (new_context);
            return -1;
        }
#endif

        /* Get the string value of the context and see if it maps
           to a valid sid.  If it does, we're done.               */
        user_context = context_str(new_context);
        if (!security_context_to_sid (user_context, 1 + strlen(user_context), 
                                      user_sid))
            done = 1;
        else 
            printf ("Not a valid security context\n");
    }
    context_free (new_context);
    return 0;
}
