#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/string.h"
#include "gw.h"
#include "stategl.h"
#include "textinput.h"
#include "sar.h"


void SARTextInputHandleKey(sar_core_struct *core_ptr, int k);
void SARTextInputMap(
	sar_core_struct *core_ptr,
	const char *prompt,		/* Prompt label. */
	void (*text_input_cb)(void *, const char *)
);
void SARTextInputUnmap(sar_core_struct *core_ptr);
void SARTextInputDraw(sar_core_struct *core_ptr);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))

/*
 *	Handles key and stores it into the text input buffer.
 *	If the core structure is NULL or the core structure's
 *	text_input_cb is NULL, no action will be taken.
 */
void SARTextInputHandleKey(sar_core_struct *core_ptr, int k)
{
	gw_display_struct *display;
	int i, len;
	char *buf;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

	if(core_ptr->text_input_cb == NULL)
	    return;

	buf = core_ptr->text_input_buffer;
	if(buf == NULL)
	    return;

	len = strlen(buf);

	switch(k)
	{
	  case 0x1b:	/* Escape. */
	    SARTextInputUnmap(core_ptr);
	    break;

	  case GWKeyBackSpace:
	  case GWKeyDelete:
	    i = MAX(len - 1, 0);
	    buf[i] = '\0';
	    break;

	  case '\n':
	    core_ptr->text_input_cb(core_ptr, buf);
	    SARTextInputUnmap(core_ptr);
	    break;

	  default:
	    if(isprint(k))
	    {
		i = MAX(len, 0);
		len++;

		buf = (char *)realloc(
		    buf,
		    (len + 1) * sizeof(char)
		);
		if(buf != NULL)
		{
		    buf[i] = ((display->shift_key_state) ?
			toupper((char)k) : (char)k
		    );
		    buf[len] = '\0';
		}
		core_ptr->text_input_buffer = buf;
	    }
	    break;
	}

	return;
}

/*
 *	Maps the text input buffer on the core structure.
 *	The text input buffer will be allocated to one char containing
 *	a null byte.
 */
void SARTextInputMap(
	sar_core_struct *core_ptr,
	const char *prompt,
	void (*text_input_cb)(void *, const char *)
)
{
	if(core_ptr == NULL)
	    return;

	free(core_ptr->text_input_prompt);
	core_ptr->text_input_prompt = StringCopyAlloc(prompt);

	core_ptr->text_input_buffer = (char *)realloc(
	    core_ptr->text_input_buffer,
	    1 * sizeof(char)
	);
	if(core_ptr->text_input_buffer != NULL)
	    core_ptr->text_input_buffer[0] = '\0';

	core_ptr->text_input_cb = text_input_cb;

	return;
}

/*
 *	`Unmaps' the text input prompt and buffer, deallcating it and
 *	setting the core structure's text_input_cb to NULL.
 */
void SARTextInputUnmap(sar_core_struct *core_ptr)
{
        if(core_ptr == NULL)
            return;

	free(core_ptr->text_input_prompt);
	core_ptr->text_input_prompt = NULL;

	free(core_ptr->text_input_buffer);
	core_ptr->text_input_buffer = NULL;

	core_ptr->text_input_cb = NULL;

	return;
}       

/*
 *	Draws input text buffer onto the window.
 */
void SARTextInputDraw(sar_core_struct *core_ptr)
{
	char *prompt, *buf;
	int prompt_len, buf_len, len;
	int xc, yc, fw, fh;
	gw_display_struct *display;
	GWFont *font;
	sar_color_struct *color;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

	font = option.message_font;
	if(font == NULL)
	    return;

	color = &option.message_color;

	prompt = core_ptr->text_input_prompt;
	if(prompt == NULL)
	    return;
	else
	    prompt_len = strlen(prompt);

	buf = core_ptr->text_input_buffer;
	if(buf == NULL)
	    return;
	else
	    buf_len = strlen(buf);


	GWGetFontSize(font, NULL, NULL, &fw, &fh);

	/* Calculate length of prompt plus buffer plus tolorance and
	 * additional characters (7 more chars) in pixels.
	 */
	len = (prompt_len + buf_len + 7) * fw;
	if(len > display->width)
	    xc = display->width - len;
	else
	    xc = (display->width / 2) - (len / 2);
	yc = (int)(0.5 * display->height);

	/* Draw shaded background. */
        StateGLEnable(&display->state_gl, GL_BLEND);
        StateGLBlendFunc(
	    &display->state_gl,
	    GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
	);
        glBegin(GL_QUADS);
        {
            glColor4d(0.0, 0.0, 0.0, 0.5);
            glVertex2d(0, yc - fh - 2);
            glVertex2d(display->width, yc - fh - 2);
            glVertex2d(display->width, yc + 2);
            glVertex2d(0, yc + 2);
        }
        glEnd();
        StateGLDisable(&display->state_gl, GL_BLEND);


	/* Set text color and font. */
        glColor4d(color->r, color->g, color->b, color->a);
        GWSetFont(display, font);

	/* Draw text. */
	GWDrawString(display, xc, yc, prompt);
	xc += (prompt_len * fw);
	GWDrawString(display, xc, yc, ": ");
	xc += (2 * fw);
	GWDrawString(display, xc, yc, buf);
	xc += (buf_len * fw);
        GWDrawString(display, xc, yc, "_");



	return;
}
