#include <stdio.h>  
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <sys/types.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#else
# include <sys/time.h>
# include <unistd.h>
#endif

#include <GL/gl.h>
#include "../include/strexp.h"
#include "../include/string.h"

#include "menu.h"
#include "obj.h"

#include "sar.h"   


#ifdef __MSW__
static double rint(double x);
#endif	/* __MSW__ */

time_t SARGetCurMilliTime(void);
int SARRandom(void);

Boolean SARGetGLVersion(int *major, int *minor, int *release);
char *SARGetGLVendorName(void);
char *SARGetGLRendererName(void);
char **SARGelGLExtensionNames(int *strc);

int SARIsMenuAllocated(sar_core_struct *core_ptr, int n);
int SARMatchMenuByName(
        sar_core_struct *core_ptr,
        const char *name
);
sar_menu_struct *SARGetCurrentMenuPtr(sar_core_struct *core_ptr);
void SARDeleteListItemData( 
        sar_menu_list_item_struct *item_ptr
);

char *SARTimeOfDayString(sar_core_struct *core_ptr, double t);
char *SARDeltaTimeString(sar_core_struct *core_ptr, time_t t);

int SARParseTimeOfDay(
	const char *string,
	int *h, int *m, int *s
);
int SARParseLatitudeDMS(
        const char *string,
        double *dms
);
int SARParseLongitudeDMS(
        const char *string,
        double *dms   
);

void SARDrawMapProcessHits(
        sar_core_struct *core_ptr, GLint hits, GLuint buffer[]
);

int SARDrawMapObjNameListAppend(
        sar_drawmap_objname_struct ***ptr, int *total,
        GLuint gl_name, const char *obj_name
);
void SARDrawMapObjNameListDelete(  
        sar_drawmap_objname_struct ***ptr, int *total
);

int *SARGetGCCHitList(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	int obj_num,
	int *hits
);
int SARGetGHCOverWater(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num, Boolean *got_hit, Boolean *over_water
);

void SARReportGLError(
	sar_core_struct *core_ptr, GLenum error_code
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)


#ifdef __MSW__
static double rint(double x)
{
	if((double)((double)x - (int)x) > (double)0.5)
	    return((double)((int)x + (int)1));
	else
	    return((double)((int)x));
}
#endif	/* __MSW__ */


/*
 *      Returns the current time since midnight in milliseconds from the
 *      system.
 */
time_t SARGetCurMilliTime(void)
{
#ifdef __MSW__
        SYSTEMTIME t;   /* Current time of day structure. */

        GetSystemTime(&t);
        return(
            (time_t)(
                (((((t.wHour * 60.0) + t.wMinute) * 60) + t.wSecond) * 1000) +
                t.wMilliseconds
            )
        );
#else
        struct timeval tv[1];

        if(gettimeofday(tv, NULL) < 0)
            return(-1);

        return(((tv->tv_sec % 86400) * 1000) + (tv->tv_usec / 1000));
#endif
}


/*
 *      Returns a random number from rand().
 */
int SARRandom(void)
{
        srand((unsigned int)cur_millitime);
        return(rand());
}


/*
 *	Returns the version numbers for the GL version, returns
 *	TRUE on success.
 *
 *	If any of the inputs are NULL, then that respective input will
 *	not be set.
 */
Boolean SARGetGLVersion(int *major, int *minor, int *release)
{
	const GLubyte *strptr;
	char **strv;
	int strc;


	/* Reset inputs. */
	if(major != NULL)
	    (*major) = 0;
        if(minor != NULL)
            (*minor) = 0;
        if(release != NULL)
            (*release) = 0;

	/* Get statically allocated '.' character separated string
	 * containing the version numbers. Format can be either
	 * "major.minor" or "major.minor.release".
	 */
	strptr = glGetString(GL_VERSION);
	if(strptr == NULL)
	    return(False);

	/* Explode version string at the '.' characters. */
	strv = strchrexp((const char *)strptr, '.', &strc);
	if(strv == NULL)
	    return(False);

	if((strc > 0) && (major != NULL))
	    (*major) = atoi(strv[0]);

        if((strc > 1) && (minor != NULL))
            (*minor) = atoi(strv[1]);

        if((strc > 2) && (release != NULL))
            (*release) = atoi(strv[2]);

	StringFreeArray(strv, strc);

	return(True);
}

/*
 *      Returns a dynamically allocated string containing the GL
 *      vendor name. The returned string must be free()'ed by the
 *      calling function.
 *
 *      Can return NULL on error.
 */
char *SARGetGLVendorName(void)
{
        const GLubyte *strptr;

        /* Get statically allocated string containing the GL renderer
         * name.
         */
        strptr = glGetString(GL_VENDOR);
        if(strptr == NULL)
            return(NULL);

        /* Return dynamically allocated string. */
        return(strdup((const char *)strptr));
}

/*
 *	Returns a dynamically allocated string containing the GL
 *	renderer name. The returned string must be free()'ed by the
 *	calling function.
 *
 *	Can return NULL on error.
 */
char *SARGetGLRendererName(void)
{
	const GLubyte *strptr;


	/* Get statically allocated string containing the GL renderer
	 * name.
	 */
	strptr = glGetString(GL_RENDERER);
	if(strptr == NULL)
	    return(NULL);

	/* Return dynamically allocated string. */
	return(strdup((const char *)strptr));
}


/*
 *	Returns a dynamically allocated array of strings containing
 *	the list of GL extension names available.
 *
 *	If strc is not NULL then it will be set to the number of
 *	strings returned.
 *
 *	Can return NULL on error.
 */
char **SARGelGLExtensionNames(int *strc)
{
        const GLubyte *strptr;
	char **strv;
	int lstrc;


	/* Reset inputs. */
	if(strc != NULL)
	    (*strc) = 0;

	/* Get statically allocated space separated string containing
	 * the GL extensions list.
	 */
        strptr = glGetString(GL_EXTENSIONS);
	if(strptr == NULL)
	    return(NULL);

	/* Explode space separated string. */
	strv = strexp(strptr, &lstrc);
	if(strv == NULL)
	    return(NULL);

	/* Update number of strings reciefved (hence number of
	 * extensions).
	 */
	if(strc != NULL)
	    (*strc) = lstrc;

	return(strv);
}


/*
 *      Is menu allocated on the core structure.
 */
int SARIsMenuAllocated(sar_core_struct *core_ptr, int n)
{
        if(core_ptr == NULL)
            return(0);
        else if((n < 0) || (n >= core_ptr->total_menus))
            return(0);
        else if(core_ptr->menu[n] == NULL)
            return(0);
        else
            return(1);
}

/*
 *      Matches a menu on the core structure that matches the given
 *      name (case insensitive). Can return -1 for no match.
 */
int SARMatchMenuByName(
        sar_core_struct *core_ptr,
        const char *name
)
{
        int i;
        sar_menu_struct *menu_ptr;
        
        
        if((core_ptr == NULL) ||
           (name == NULL)
        )
            return(-1);
            
        for(i = 0; i < core_ptr->total_menus; i++)
        {
            menu_ptr = core_ptr->menu[i];
            if(menu_ptr == NULL)
                continue;

            if(menu_ptr->name == NULL)
                continue;
            
            if(!strcasecmp(menu_ptr->name, name))
                return(i);
        }
 
        return(-1);
}

/*
 *      Get pointer to currently selected menu, can return NULL on
 *      error or if no menu is selected.
 */
sar_menu_struct *SARGetCurrentMenuPtr(sar_core_struct *core_ptr)
{
        int n;
        
        
        if(core_ptr == NULL)
            return(NULL);
        else
            n = core_ptr->cur_menu;
         
        if((n < 0) || (n >= core_ptr->total_menus))
            return(NULL);
        else
            return(core_ptr->menu[n]);
}

/*
 *      Deletes the client data structure specified on the menu
 *      list object item pointer.
 */
void SARDeleteListItemData(
        sar_menu_list_item_struct *item_ptr
)
{
        sar_menu_list_item_data_struct *li_data_ptr;

        if(item_ptr == NULL)
            return;
        else
            li_data_ptr = item_ptr->client_data;

        free(li_data_ptr->filename);
        free(li_data_ptr->name);

        free(li_data_ptr);

        item_ptr->client_data = NULL;

        return;
}

/*
 *	Returns a statically allocated string containing the time
 *	of day in %h:%m format or by whatever format specified on the
 *	core structure if the core_ptr is not NULL.
 *
 *	Time t is given in seconds from midnight.
 */
char *SARTimeOfDayString(sar_core_struct *core_ptr, double t)
{
	static char str[80];

#define HTOS(h)	((h) * 3600.0)

	/* Sanitize t. */
	while(t >= HTOS(24.0))
	    t -= HTOS(24.0);
	while(t < HTOS(0.0))
	    t += HTOS(24.0);

	/* Reset string. */
	(*str) = '\0';

	/* No core structure given? */
	if(core_ptr == NULL)
	{
	    int	h = (int)MAX((t / 3600.0), 0.0),
		m = (int)MAX(((int)((int)t / 60) % 60), 0.0);

	    sprintf(str, "%i:%2.2i", h, m);
	}
	else
	{
            int h = (int)MAX((t / 3600.0), 0.0),
                m = (int)MAX(((int)((int)t / 60) % 60), 0.0);

            sprintf(str, "%i:%2.2i", h, m);
        }

#undef HTOS
	return(str);
}

/*
 *      Returns a statically allocated string verbosly stating the
 *      delta time t in accordance to the format specified by the
 *	core_ptr. If core_ptr is NULL then a generic format will be
 *	assumed.
 *
 *      This function never returns NULL.
 */
char *SARDeltaTimeString(sar_core_struct *core_ptr, time_t t)
{
#define len 256
        static char s[len];

        if(t <= 0)
            return("none");

        if(t < 60)
        {
            sprintf(s, "%ld second%s",
                t,
                (t == 1) ? "" : "s"
            );
        }
        else if(t < 3600) 
        {
            t = (time_t)rint((double)t / 60.0);
            sprintf(s, "%ld minute%s",
                t,
                (t == 1) ? "" : "s"
            );
        }
        else if(t < 86400)
        {
            t = (time_t)rint((double)t / 3600.0);
            sprintf(s, "%ld hour%s",
                t,
                (t == 1) ? "" : "s"
            );
        }

        return(s);
#undef len
}


/*
 *	Parses the given string which should be in the format
 *	"h:m:s". Two available formats 24 and 12 hour are supported,
 *	where the 12 hour is checked for by searching for an in
 *	string 'p' character (case insensitive).
 *
 *	String is terminated by a null or blank character.
 *
 *	Returns non-zero if parsing failed.
 */
int SARParseTimeOfDay(
        const char *string,
        int *h, int *m, int *s
)
{
	const char *sp;
	int in_len, is_12hourpm = 0;

	int lstring_len = 80;
	char *sp2, *sp3;
	char lstring[80];

	/* Reset inputs. */
	if(h != NULL)
	    (*h) = 0;
	if(m != NULL)
	    (*m) = 0;
	if(s != NULL)
	    (*s) = 0;

	if((string == NULL) ? 1 : ((*string) == '\0'))
	    return(-1);

	/* Check if this is 12 or 24 hour format. */
	sp = string;
	in_len = 0;
	while(((*sp) != '\0') && !ISBLANK(*sp))
	{
	    if(toupper(*sp) == 'P')
		is_12hourpm = 1;

	    sp++;
	    in_len++;
	}

	/* Copy input string to local string. */
	if(in_len > 0)
	    strncpy(
	        lstring, string, MIN(in_len, lstring_len)
	    );
	else
	    (*lstring) = '\0';
	lstring[MIN(in_len, lstring_len - 1)] = '\0';

	/* Fetch hour. */
	sp2 = lstring;
	sp3 = strchr(sp2, ':');
	if(sp3 != NULL)
	{
	    (*sp3) = '\0';
	    if(h != NULL)
	        (*h) = (atoi(sp2) + ((is_12hourpm) ? + 12 : 0)) % 24;
	    sp2 = sp3 + 1;
	}
	else
	{
	    if(h != NULL)
                (*h) = (atoi(sp2) + ((is_12hourpm) ? + 12 : 0)) % 24;
	    sp2 = NULL;
	}
	if(sp2 == NULL)
	    return(0);

	/* Fetch minute. */
        sp3 = strchr(sp2, ':');
        if(sp3 != NULL)
        {
            (*sp3) = '\0';
            if(m != NULL)
                (*m) = atoi(sp2) % 60;
            sp2 = sp3 + 1;
        }       
        else  
        {
            if(m != NULL)
                (*m) = atoi(sp2) % 60;
            sp2 = NULL;
        }
        if(sp2 == NULL)
            return(0);

        /* Fetch seconds. */
        sp3 = strchr(sp2, ':');
        if(sp3 != NULL)
        {
            (*sp3) = '\0';
            if(s != NULL)
                (*s) = atoi(sp2) % 60;
        }
        else
        {   
            if(s != NULL)
                (*s) = atoi(sp2) % 60;
        }

	return(0);
}

/*
 *	Parses the given string which should be in the format
 *	"d'm"s" or "d". Examples (respective) "34'10"59" or "34.1833".
 *
 *	For format "d" a decimal value from .000 to .999 can be used
 * 	to represent decimal degrees.
 *
 *	Degrees is bounded by -90.0 and 90.0.
 *
 *	String is terminated by a null or blank character.
 *
 *	Returns non-zero if parsing failed.
 */
int SARParseLatitudeDMS(
        const char *string,
        double *dms
)
{
	const char *sp;
	int in_len, is_traditional = 0, is_south = 0;

	int lstring_len = 80;
	char *sp2, *sp3;
	char lstring[80];

	/* Reset inputs. */
	if(dms != NULL)
	    (*dms) = 0.0;

	if((string == NULL) ? 1 : ((*string) == '\0'))
	    return(-1);

	/* Check if this is in decimal or traditional notation. */
	sp = string;
	in_len = 0;
	while(((*sp) != '\0') && !ISBLANK(*sp))
	{
	    if(toupper(*sp) == '\'')
		is_traditional = 1;

	    if(toupper(*sp) == 'S')
		is_south = 1;

	    sp++;
	    in_len++;
	}

	/* Copy input string to local string. */
	if(in_len > 0)
	    strncpy(
	        lstring, string, MIN(in_len, lstring_len)
	    );
	else
	    (*lstring) = '\0';
	lstring[MIN(in_len, lstring_len - 1)] = '\0';

	/* Handle by type. */
	if(is_traditional)
	{
	    /* Traditional notation. */
	    int d = 0, m = 0, s = 0;

	    /* Fetch degrees. */
            sp2 = lstring;
	    if(sp2 != NULL)
	    {
		sp3 = strchr(sp2, '\'');
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    d = atoi(sp2);
		    sp2 = sp3 + 1;
		}
		else
		{
		    d = atoi(sp2);
		    sp2 = NULL;
		}
	    }

            /* Fetch minutes. */
            if(sp2 != NULL)
            {
                sp3 = strchr(sp2, '"');
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    m = atoi(sp2);
                    sp2 = sp3 + 1;
                }
                else  
                {
                    m = atoi(sp2);
                    sp2 = NULL;
                }
            }

            /* Fetch seconds. */
            if(sp2 != NULL)
            {
                sp3 = strchr(sp2, '"'); 
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    s = atoi(sp2);
                    sp2 = sp3 + 1;
                }
                else
                {   
                    s = atoi(sp2);
                    sp2 = NULL;   
                }
            }

	    /* Convert and add to dms return. */
	    if(dms != NULL)
	    {
		/* Negative? */
		if(is_south && (d > 0))
		    (*dms) = (double)-d -
			((double)m / 60.0) - ((double)s / 60.0 / 100.0);
		else if(d < 0)
		    (*dms) = (double)d -
			((double)m / 60.0) - ((double)s / 60.0 / 100.0);
		else
		    (*dms) = (double)d +
			((double)m / 60.0) + ((double)s / 60.0 / 100.0);
	    }
	}
	else
	{
	    /* Decimal notation (this is easy). */
	    sp2 = lstring;
	    if(dms != NULL)
	    {
		double d = atof(sp2);

		if(is_south && (d > 0))
		    (*dms) = -d;
		else
		    (*dms) = d;
	    }
	}

	return(0);
}

/*
 *	Parses the given string which should be in the format
 *	"d'm"s" or "d". Examples (respective) "34'10"59" or "34.1833".
 *
 *	For format "d" a decimal value from .000 to .999 can be used
 * 	to represent decimal degrees.
 *
 *	Degrees is bounded by -140.0 and 140.0.
 *
 *	String is terminated by a null or blank character.
 *
 *	Returns non-zero if parsing failed.
 */
int SARParseLongitudeDMS(
        const char *string,
        double *dms
)
{
	const char *sp;
	int in_len, is_traditional = 0, is_west = 0;

	int lstring_len = 80;
	char *sp2, *sp3;
	char lstring[80];

	/* Reset inputs. */
	if(dms != NULL)
	    (*dms) = 0.0;

	if((string == NULL) ? 1 : ((*string) == '\0'))
	    return(-1);

	/* Check if this is in decimal or traditional notation. */
	sp = string;
	in_len = 0;
	while(((*sp) != '\0') && !ISBLANK(*sp))
	{
	    if(toupper(*sp) == '\'')
		is_traditional = 1;

	    if(toupper(*sp) == 'W')
		is_west = 1;

	    sp++;
	    in_len++;
	}

	/* Copy input string to local string. */
	if(in_len > 0)
	    strncpy(
	        lstring, string, MIN(in_len, lstring_len)
	    );
	else
	    (*lstring) = '\0';
	lstring[MIN(in_len, lstring_len - 1)] = '\0';

	/* Handle by type. */
	if(is_traditional)
	{
	    /* Traditional notation. */
	    int d = 0, m = 0, s = 0;

	    /* Fetch degrees. */
            sp2 = lstring;
	    if(sp2 != NULL)
	    {
		sp3 = strchr(sp2, '\'');
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    d = atoi(sp2);
		    sp2 = sp3 + 1;
		}
		else
		{
		    d = atoi(sp2);
		    sp2 = NULL;
		}
	    }

            /* Fetch minutes. */
            if(sp2 != NULL)
            {
                sp3 = strchr(sp2, '"');
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    m = atoi(sp2);
                    sp2 = sp3 + 1;
                }
                else  
                {
                    m = atoi(sp2);
                    sp2 = NULL;
                }
            }

            /* Fetch seconds. */
            if(sp2 != NULL)
            {
                sp3 = strchr(sp2, '"'); 
                if(sp3 != NULL)
                {
                    (*sp3) = '\0';
                    s = atoi(sp2);
                    sp2 = sp3 + 1;
                }
                else
                {   
                    s = atoi(sp2);
                    sp2 = NULL;   
                }
            }

	    /* Convert and add to dms return. */
	    if(dms != NULL)
	    {
		/* Negative? */
		if(is_west && (d > 0))
		    (*dms) = (double)-d -
			((double)m / 60.0) - ((double)s / 60.0 / 100.0);
		else if(d < 0)
		    (*dms) = (double)d -
			((double)m / 60.0) - ((double)s / 60.0 / 100.0);
		else
		    (*dms) = (double)d +
			((double)m / 60.0) + ((double)s / 60.0 / 100.0);
	    }
	}
	else
	{
	    /* Decimal notation (this is easy). */
	    sp2 = lstring;
	    if(dms != NULL)
	    {
		double d = atof(sp2);

		if(is_west && (d > 0))
		    (*dms) = -d;
		else
		    (*dms) = d;
	    }
	}

	return(0);
}


/*
 *	Called by SARDrawMap() in sardraw.c to process hits and record
 *	them on the core structure's draw map object names list.
 *
 *	Inputs assumed valid.
 */
void SARDrawMapProcessHits(
        sar_core_struct *core_ptr, GLint hits, GLuint buffer[] 
)
{
	int i, j;
	GLuint gl_name, names, *ptr;


	/* Itterate through each hit. */
	for(i = 0, ptr = (GLuint *)buffer; i < hits; i++)
	{
	    /* Itterate through each name on this hit. */
	    names = (*ptr);
	    ptr += 3;
	    for(j = 0; j < names; j++)
	    {
		gl_name = *ptr++;

		SARDrawMapObjNameListAppend(
		    &core_ptr->drawmap_objname,
		    &core_ptr->total_drawmap_objnames,
		    gl_name,
		    NULL		/* No obj names strings for now. */
		);
	    }
	}

	return;
}


/*
 *	Appends a new entry to the sar_drawmap_objname_struct list.
 *
 *	Returns the index to the newly appended entry or -1 on error.
 */
int SARDrawMapObjNameListAppend(
        sar_drawmap_objname_struct ***ptr, int *total,
	GLuint gl_name, const char *obj_name
)
{
        int n;
        sar_drawmap_objname_struct *dmon_ptr;

        if((ptr == NULL) || (total == NULL))
            return(-1);

	/* Sanitize total. */
	if((*total) < 0)
	    (*total) = 0;

	/* Increment total and get append index. */
	n = (*total);
	(*total) = n + 1;

	/* Increase pointer array allocation. */
	(*ptr) = (sar_drawmap_objname_struct **)realloc(
	    *ptr,
	    (*total) * sizeof(sar_drawmap_objname_struct *)
	);
	if((*ptr) == NULL)
	{
	    (*total) = 0;
	    return(-1);
	}

	/* Allocate new structure. */
	(*ptr)[n] = dmon_ptr = (sar_drawmap_objname_struct *)calloc(
	    1, sizeof(sar_drawmap_objname_struct)
	);
	if(dmon_ptr == NULL)
	{
	    (*total) = n;
	    return(-1);
	}

	/* Set values on newly allocated structure. */
	dmon_ptr->gl_name = gl_name;
	dmon_ptr->obj_name = ((obj_name == NULL) ?
	    NULL : strdup(obj_name)
	);

	return(n);
}

/*
 *	Deallocates the list of sar_drawmap_objname_struct structures.
 */
void SARDrawMapObjNameListDelete(
        sar_drawmap_objname_struct ***ptr, int *total
)
{
	int i;
	sar_drawmap_objname_struct *dmon_ptr;

	if((ptr == NULL) || (total == NULL))
	    return;

	/* Deallocate each entry. */
	for(i = 0; i < (*total); i++)
	{
	    dmon_ptr = (*ptr)[i];
	    if(dmon_ptr == NULL)
		continue;

	    /* Deallocate resources. */
	    free(dmon_ptr->obj_name);

	    /* Deallocate structure itself. */
	    free(dmon_ptr);
	}
	/* Deallocate pointer array. */
	if((*ptr) != NULL)
	{
	    free(*ptr);
	    (*ptr) = NULL;
	}
	(*total) = 0;

	return;
}

/*
 *	Convience function that draws the map for ground contact checks
 *	tablulate hit results on the core structure and returns an 
 *	allocated array of object index numbers that were matched.
 *
 *	Note that special numbers have the base offset of (*total)
 *	added to them. So any index that is a special number is
 *	si = i - (*total).
 *
 *	The given obj_num (or -1 for the player object) will never be in 
 *	the returned list of object numbers.
 *
 *	Calling function must deallocate the return.
 */
int *SARGetGCCHitList(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num,
        int *hits
)
{
	int i, n, hit_obj_num, *hit_list = NULL;
	sar_drawmap_objname_struct *dmon_ptr;


	if(hits != NULL)
	    (*hits) = 0;

	if((core_ptr == NULL) || (scene == NULL) || (ptr == NULL) ||
	   (total == NULL)
	)
	    return(hit_list);

	/* Call map drawing routine with draw_for_gcc set to True, this
	 * will draw the map with an isolated camera angle looking 
	 * directly downward with respect to the given object (or player
	 * object if the obj_num is -1). The map draw object names list
	 * on the core structure will be cleared and updated with the
	 * hits list.
	 */
	SARDrawMap(core_ptr, True, False, obj_num);

	/* Get hits list. */
	for(i = 0; i < core_ptr->total_drawmap_objnames; i++)
	{
	    dmon_ptr = core_ptr->drawmap_objname[i];
	    if(dmon_ptr == NULL)
		continue;

	    /* Get the hit object number. */
	    hit_obj_num = (int)dmon_ptr->gl_name;

	    /* Skip if the hit object number is the same as given. */
	    if(hit_obj_num == obj_num)
		continue;

	    /* Allocate more hits. */
	    if(hits != NULL)
	    {
		n = (*hits);
		(*hits) = n + 1;

		hit_list = (int *)realloc(
		    hit_list,
		    (*hits) * sizeof(int)
		);
		hit_list[n] = hit_obj_num;
	    }
	}

	return(hit_list);
}

/*
 *	Checks for ground hit contact, relative to the given obj_num.
 *
 *	Checks if there is any object directly under obj_num that is
 *	drawn. If nothing is drawn directly under the object then
 *	got_hit will be set to False, otherwise True.
 *
 *	If the scene's base type is water, then over_water will be set
 *	True if (and only if) got_hit is False.
 *
 *	Returns 0 on success or non-zero on error.
 */
int SARGetGHCOverWater(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num, Boolean *got_hit, Boolean *over_water
)
{
	if(got_hit != NULL)
	    (*got_hit) = False;
	if(over_water != NULL)
	    (*over_water) = False;


        if((core_ptr == NULL) || (scene == NULL) || (ptr == NULL) ||
           (total == NULL)
        )
            return(-1);

        /* Call map drawing routine with draw_for_ghc set to True, this
         * will draw the map with an isolated camera angle looking   
         * directly downward with respect to the given object (or player
         * object if the obj_num is -1). The drawmap_ghc_result on the
         * structure will be updated.
         */
        SARDrawMap(core_ptr, False, True, obj_num);

	/* Got hit? */
	if(core_ptr->drawmap_ghc_result >= 0.5)
	{
	    /* Mark that we got a hit. */
	    if(got_hit != NULL)
		(*got_hit) = True;

	    /* Definatly not over water. */
	    if(over_water != NULL)
                (*over_water) = False;
	}
	else
	{
	    /* Mark that we did not get a hit. */
	    if(got_hit != NULL)   
                (*got_hit) = False;

	    /* Scene's base is water? */
	    if(scene->base_flags & SAR_SCENE_BASE_FLAG_IS_WATER)
	    {
		if(over_water != NULL)
                    (*over_water) = True;
	    }
	    else
	    {
                if(over_water != NULL)
                    (*over_water) = False;
	    }
	}

	return(0);
}


/*
 *	Reports the given GL error code, does not check if the
 *	error is actually an error and prints explicitly even if
 *	global options specify not to print errors.
 */
void SARReportGLError(
        sar_core_struct *core_ptr, GLenum error_code 
)
{
	const char *error_mesg = NULL, *error_type = NULL;

	switch(error_code)
	{
	  case GL_INVALID_ENUM:
	    error_type = "GL_INVALID_ENUM";
	    error_mesg =
		"Invalid GLenum argument (possibly out of range)";
	    break;

	  case GL_INVALID_VALUE:
	    error_type = "GL_INVALID_VALUE";
	    error_mesg =
                "Numeric argument out of range";
            break;

	  case GL_INVALID_OPERATION:
	    error_type = "GL_INVALID_OPERATION";
	    error_mesg =
                "Operation illegal in current state";
            break;

	  case GL_STACK_OVERFLOW:
	    error_type = "GL_STACK_OVERFLOW";
	    error_mesg =
                "Command would cause stack overflow";
            break;

	  case GL_STACK_UNDERFLOW:
	    error_type = "GL_STACK_UNDERFLOW";
	    error_mesg =
                "Command would cause a stack underflow";
            break;

          case GL_OUT_OF_MEMORY:
            error_type = "GL_OUT_OF_MEMORY";
            error_mesg =
                "Not enough memory left to execute command";
            break;

	  default:
	    error_type = "*UNKNOWN*";
	    error_mesg = "Undefined GL error";
	    break;
	}

	fprintf(
	    stderr,
	    "GL Error code %i: %s: %s\n",
	    error_code, error_type, error_mesg
	);

	return;
}
