#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <math.h>

#include "../include/os.h"

#include "sound.h"
#include "obj.h"
#include "sarsound.h"
#include "sar.h"
#include "config.h"


#ifdef __MSW__
static double rint(double x);
#endif	/* __MSW__ */

void SARSoundEngineVolumeMute(
        snd_recorder_struct *recorder,
        snd_play_struct *inside_snd_play,
        snd_play_struct *outside_snd_play 
);
void SARSoundAdjustEngineVolume(
        snd_recorder_struct *recorder,
        snd_play_struct *inside_snd_play,
        snd_play_struct *outside_snd_play,
        int is_inside,
        int engine_state,
        double throttle,
        double distance_to_camera
);

void SARSoundPlayPredefined(
	snd_recorder_struct *recorder,
	int type,
	sar_scene_struct *scene,
	double distance_to_camera
);


#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))


#ifdef __MSW__
static double rint(double x)
{
	if((double)((double)x - (int)x) > (double)0.5)
	    return((double)((int)x + (int)1));
	else
	    return((double)((int)x));
}
#endif	/* __MSW__ */


/*
 *	Basically calls SARSoundAdjustEngineVolume() with no throttle
 *	and engine state as SAR_ENGINE_STATE_OFF.
 */
void SARSoundEngineVolumeMute(
        snd_recorder_struct *recorder,
        snd_play_struct *inside_snd_play,
        snd_play_struct *outside_snd_play 
)
{
	SARSoundAdjustEngineVolume(
	    recorder, inside_snd_play, outside_snd_play,
	    0,				/* In cockpit? */
	    SAR_ENGINE_STATE_OFF,	/* Engine state? */
	    0.0,			/* Throttle? */
	    0.0
	);

	return;
}

/* 
 *      Adjusts the engine volume and sample rate.
 */
void SARSoundAdjustEngineVolume(
        snd_recorder_struct *recorder,
        snd_play_struct *inside_snd_play,	/* Inside engine sound. */
        snd_play_struct *outside_snd_play,	/* Outside engine sound. */
        int is_inside,				/* Is inside cockpit. */
        int engine_state,			/* One of SAR_ENGINE_STATE_*. */
        double throttle,			/* Throttle coeff 0.0 to 1.0. */
        double distance_to_camera		/* In meters. */
)
{
        int sample_rate;			/* In Hz. */
        double volume;				/* 0.0 to 1.0. */
        double audiable_radius = option.aircraft_engine_sound_distance;


        /* Not connected to sound server? */
        if(recorder == NULL)
            return;

        /* Turn all sounds to 0 volume if engine and throttle are off,
	 * this improves efficiency since no sound would be mixed by
	 * the sound server if volume is 0 but the sound object will
	 * still remain valid.
	 */
        if((engine_state != SAR_ENGINE_STATE_ON) &&
           (throttle <= 0.0)
	)
        {
            SoundChangePlayVolume(
                recorder, inside_snd_play,
                0.0, 0.0
            );
            SoundChangePlayVolume(
                recorder, outside_snd_play,
                0.0, 0.0
            );
            return;
        }


        /* Calculate sample rate based on throttle, increase to a
         * faster sample rate as throttle increases.
         */
        sample_rate = recorder->sample_rate;
        sample_rate = (int)rint(((double)sample_rate +
            ((double)sample_rate * throttle * 1.6)
        ) / 1000);
	sample_rate = (int)(sample_rate * 1000);

        /* Calculate volume based on (1 - (x^0.5)) curve. */
        volume = CLIP(1.0 -
            pow(distance_to_camera / audiable_radius, 0.5),
            0.0, 1.0
        );

        /* Is camera inside cockpit? */
        if(is_inside)
        {
            /* Inside cockpit. */
            SoundChangePlayVolume(
                recorder, inside_snd_play,
                volume, volume
            );
            SoundChangePlaySampleRate(
                recorder, inside_snd_play,
                sample_rate
            );
            SoundChangePlayVolume(
                recorder, outside_snd_play,
                0.0, 0.0
            );
            SoundChangePlaySampleRate(
                recorder, outside_snd_play,
                sample_rate
            );
        }
        else
        {
            /* Outside. */
            SoundChangePlayVolume(
                recorder, inside_snd_play,
                0.0, 0.0
            );
            SoundChangePlaySampleRate(
                recorder, inside_snd_play,
                sample_rate
            );
            SoundChangePlayVolume(
                recorder, outside_snd_play,
                volume, volume
            );
            SoundChangePlaySampleRate(
                recorder, outside_snd_play,
                sample_rate
            );
        }

        return;
}

/*
 *	Plays a predefined sound object based on the given information.
 *
 *	The sound object path will be taken from the given scene
 *	structure.
 */
void SARSoundPlayPredefined(
        snd_recorder_struct *recorder,
        int type,			/* One of SAR_SOUND_PREDEFINED_*. */
        sar_scene_struct *scene,
        double distance_to_camera	/* In meters. */
)
{
	double dist_max, vol;
	sar_option_struct *opt = &option;

	if((recorder == NULL) || (scene == NULL))
	    return;

	switch(type)
	{
	  case SAR_SOUND_PREDEFINED_LAND_BELLY:
	    dist_max = 1200.0;
	    if((distance_to_camera < dist_max) && opt->event_sounds)
	    {
		vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
		SoundStartPlayVoid(
		    recorder,
		    scene->land_belly_sndpath,
		    vol, vol,
		    0, 0
		);
	    }
	    break;

          case SAR_SOUND_PREDEFINED_LAND_SKI:
            dist_max = 100.0;
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->land_ski_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;

          case SAR_SOUND_PREDEFINED_LAND_SKI_SKID:
            dist_max = 600.0;
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->land_ski_skid_sndpath,
                    vol, vol,
                    0, 0
                );  
            }     
            break;

          case SAR_SOUND_PREDEFINED_LAND_WHEEL_SKID:
            dist_max = 600.0;
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->land_wheel_skid_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;

          case SAR_SOUND_PREDEFINED_CRASH_GROUND:
            dist_max = 3281.0;		/* Two miles. */
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->crash_ground_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;

          case SAR_SOUND_PREDEFINED_CRASH_OBSTRUCTION:
            dist_max = 3281.0;		/* Two miles. */
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->crash_obstruction_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;

          case SAR_SOUND_PREDEFINED_SPLASH_AIRCRAFT:
            dist_max = 2421.0;          /* About 1.5 miles. */
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->splash_aircraft_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;

          case SAR_SOUND_PREDEFINED_SPLASH_HUMAN:
            dist_max = 300.0;
            if((distance_to_camera < dist_max) && opt->event_sounds)
            {
                vol = MAX(1.0 - (distance_to_camera / dist_max), 0.0);
                SoundStartPlayVoid(
                    recorder,
                    scene->splash_human_sndpath,
                    vol, vol,
                    0, 0
                );
            }
            break;
	}

	return;
}
