#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/os.h"

#include "gw.h"
#include "menu.h"
#include "sound.h"
#include "sar.h"


static sar_menu_spin_struct *SAROptionsGetSpinByID(
	sar_menu_struct *m, int id_code, int *n
);
static sar_menu_switch_struct *SAROptionsGetSwitchByID(
	sar_menu_struct *m, int id_code, int *n
);
#ifdef JS_SUPPORT
static int SARGetJSButtonFromSpinAction(
        sar_menu_spin_struct *act_spin_ptr, int id_code
);
#endif	/* JS_SUPPORT */

void SARApplyOptionsFromMenus(sar_core_struct *core_ptr);
void SARFetchOptionsToMenus(sar_core_struct *core_ptr);

void SARMenuOptionsSwitchCB(
	void *object, void *client_data,
	int id_code, Boolean state
);
void SARMenuOptionsSpinCB(
	void *object, void *client_data,
        int id_code, char *value
);


/*
 *	Searches for the first spin object on menu m with a matching id
 *	code, returns the pointer to the spin object or NULL on no match.
 */
static sar_menu_spin_struct *SAROptionsGetSpinByID(
	sar_menu_struct *m, int id_code, int *n
)
{
	int i;
	void *ptr;
	sar_menu_spin_struct *spin;


        if(n != NULL) 
            (*n) = -1;

	if(m == NULL)
	    return(NULL);

	for(i = 0; i < m->total_objects; i++)
	{
	    ptr = m->object[i];
	    if(ptr == NULL)
		continue;

	    if(*(int *)ptr == SAR_MENU_OBJECT_TYPE_SPIN)
		spin = ptr;
	    else
		continue;

	    if(spin->id_code == id_code)
	    {
		if(n != NULL)
		    (*n) = i;

		return(spin);
	    }
	}

	return(NULL);
}

/*
 *	Searches for the first switch object on menu m with a matching id
 *      code, returns the pointer to the switch object or NULL on no match.
 */
static sar_menu_switch_struct *SAROptionsGetSwitchByID(
        sar_menu_struct *m, int id_code, int *n
)
{
        int i;
        void *ptr;
        sar_menu_switch_struct *switch_ptr;


	if(n != NULL)
	    (*n) = -1;

        if(m == NULL)
            return(NULL);

        for(i = 0; i < m->total_objects; i++)
        {
            ptr = m->object[i];
            if(ptr == NULL)
                continue;
 
            if(*(int *)ptr == SAR_MENU_OBJECT_TYPE_SWITCH)
                switch_ptr = ptr;
            else
                continue;

            if(switch_ptr->id_code == id_code)
	    {
                if(n != NULL)
                    (*n) = i;

                return(switch_ptr);
	    }
        }

        return(NULL);
}

#ifdef JS_SUPPORT
/*
 *	Returns the current joystick button mapping for the given
 *	joystick action spin's current value.
 *
 *	The given id_code must be one of SAR_OPT_SELECT_JS#_BUTTON_ACTION
 *	where # is the joystick number.
 *
 *	Can return -1 on error or if the value is not set. Calling
 *	function must add 1 to offset when setting to the button spin's
 *	current value since the button spin's value 0 means button number
 *	-1 (none/invalid).
 */
static int SARGetJSButtonFromSpinAction(
	sar_menu_spin_struct *act_spin_ptr, int id_code
)
{
	int val = -1;
	sar_option_struct *opt = &option;


	if(act_spin_ptr == NULL)
	    return(val);

	/* First joystick (js0). */
	if(id_code == SAR_OPT_SELECT_JS0_BUTTON_ACTION)
	{
	    switch(act_spin_ptr->cur_value)
	    {
	      case 0:	/* Rotate Modifier. */
		val = opt->js0_btn_rotate;
		break;
	      case 1:	/* Air Brakes. */
		val = opt->js0_btn_air_brakes;
		break;
	      case 2:	/* Wheel Brakes. */
		val = opt->js0_btn_wheel_brakes;
		break;
	      case 3:	/* Zoom In. */
		val = opt->js0_btn_zoom_in;
		break;
	      case 4:	/* Zoom out. */
		val = opt->js0_btn_zoom_out;
		break;
	      case 5:	/* Hoist Up. */
		val = opt->js0_btn_hoist_up;
		break;
	      case 6:	/* Hoist Down. */
		val = opt->js0_btn_hoist_down;
		break;
/* When adding a new button mapping, besure to add it to the setting of
   the button mappings to option structure in SARMenuOptionsSpinCB().

   Also be sure to add support for EACH joystick!
 */
	    }
	}
	/* Second joystick (js1). */
        else if(id_code == SAR_OPT_SELECT_JS1_BUTTON_ACTION)
        {
            switch(act_spin_ptr->cur_value) 
            {
              case 0:   /* Rotate Modifier. */
                val = opt->js1_btn_rotate;
                break;
              case 1:   /* Air Brakes. */
                val = opt->js1_btn_air_brakes;
                break;
              case 2:   /* Wheel Brakes. */
                val = opt->js1_btn_wheel_brakes;
                break;
              case 3:   /* Zoom In. */
                val = opt->js1_btn_zoom_in;
                break;
              case 4:   /* Zoom out. */
                val = opt->js1_btn_zoom_out;
                break;
              case 5:   /* Hoist Up. */
                val = opt->js1_btn_hoist_up;
                break;
              case 6:   /* Hoist Down. */
                val = opt->js1_btn_hoist_down;
                break;
/* When adding a new button mapping, besure to add it to the setting of   
   the button mappings to option structure in SARMenuOptionsSpinCB().

    Also be sure to add support for EACH joystick!
 */
            }
	}

	if(val < 0)
	    val = -1;

	return(val);
}
#endif	/* JS_SUPPORT */

/*
 *	Sets global option structure values based on values from the
 *	options menu widget/objects.
 */
void SARApplyOptionsFromMenus(sar_core_struct *core_ptr)
{
	int i;
	sar_menu_struct *m;


	if(core_ptr == NULL)
	    return;

	/* Options general. */
	i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS);
	m = ((i < 0) ? NULL : core_ptr->menu[i]);
	if(m != NULL)
	{

	}

        /* Options controller. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {

        }

        /* Options graphics. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_GRAPHICS);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {

        }

        /* Options sounds. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_SOUND);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {

        }

	return;
}

/*
 *	Gets values from global options and sets them to the objects in
 *	the widget/objects on the options menus.
 */
void SARFetchOptionsToMenus(sar_core_struct *core_ptr)
{
        int i;
        sar_menu_struct *m;
	sar_menu_spin_struct *spin_ptr;
	sar_menu_switch_struct *switch_ptr;


        if(core_ptr == NULL)
            return;

        /* Options: General */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {

	}

        /* Options: Simulation (contact sizes, flight dynamics, damage) */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_SIMULATION);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {
            /* Hoist contact, the expansion coefficient from hoist rescue
	     * basket's actual size.
	     */
            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_HOIST_CONTACT, NULL
	    );
            if(spin_ptr != NULL)
            {
                if(option.hoist_contact_expansion_coeff >= 4.0)
		    spin_ptr->cur_value = 0;
                else if(option.hoist_contact_expansion_coeff >= 2.0)
                    spin_ptr->cur_value = 1;
                else if(option.hoist_contact_expansion_coeff >= 1.0)
                    spin_ptr->cur_value = 2;
		else
		    spin_ptr->cur_value = 0;
	    }

            /* Damage resistance, how much of an impact can player
	     * aircraft tolorate from actual coefficient.
	     */
            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_DAMAGE_RESISTANCE, NULL
	    );
            if(spin_ptr != NULL)
            {
		if(option.damage_resistance_coeff >= 2.0)
		    spin_ptr->cur_value = 0;
		else if(option.damage_resistance_coeff >= 1.5)
		    spin_ptr->cur_value = 1;
                else
                    spin_ptr->cur_value = 2;
            }

            /* Flight dynamics dificulty (easy to realistic). */
            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_FLIGHT_PHYSICS, NULL
	    );
            if(spin_ptr != NULL)
            {
		switch(option.flight_physics)
		{
		  case FLIGHT_PHYSICS_REALISTIC:
		    spin_ptr->cur_value = 2;
		    break;

		  case FLIGHT_PHYSICS_MODERATE:
		    spin_ptr->cur_value = 1;
                    break;

		  default:
		    spin_ptr->cur_value = 0;
		    break;
		}
            }
	}

        /* Options: Controller (joysticks) */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {
	    gctl_flags_t gctl_options = option.gctl_options;

	    /* First joystick axises (js0). */
            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_JS0_AXISES, NULL
	    );
            if(spin_ptr != NULL)
            {
		/* These value codes should match those in SARBuildMenus()
		 * that added these values.
		 */
                if(gctl_options & GCTL_OPT_JS0_INIT)
                {
		    if(gctl_options & GCTL_OPT_JS0_AS_THROTTLE_AND_RUDDER)
			spin_ptr->cur_value = 9;
                    else if((gctl_options & GCTL_OPT_JS0_THROTTLE) &&
                            (gctl_options & GCTL_OPT_JS0_HEADING) &&
                            (gctl_options & GCTL_OPT_JS0_HAT)
                    )
                        spin_ptr->cur_value = 8;
                    else if((gctl_options & GCTL_OPT_JS0_HEADING) &&
                            (gctl_options & GCTL_OPT_JS0_HAT)
                    )
                        spin_ptr->cur_value = 7;
                    else if((gctl_options & GCTL_OPT_JS0_HEADING) &&
                            (gctl_options & GCTL_OPT_JS0_THROTTLE)
                    )
                        spin_ptr->cur_value = 6;
                    else if(gctl_options & GCTL_OPT_JS0_HEADING)
                        spin_ptr->cur_value = 5;
                    else if((gctl_options & GCTL_OPT_JS0_THROTTLE) &&
                            (gctl_options & GCTL_OPT_JS0_HAT)
                    )
                        spin_ptr->cur_value = 4;
                    else if(gctl_options & GCTL_OPT_JS0_HAT)
                        spin_ptr->cur_value = 3;
                    else if(gctl_options & GCTL_OPT_JS0_THROTTLE) 
                        spin_ptr->cur_value = 2;
                    else
                        spin_ptr->cur_value = 1;
                }
                else
                {
                    spin_ptr->cur_value = 0;
                }
            }

	    /* Second joystick axises (js1). */
            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_JS1_AXISES, NULL
	    );
            if(spin_ptr != NULL)
            {
                /* These value codes should match those in SARBuildMenus()
                 * that added these values.
                 */
                if(gctl_options & GCTL_OPT_JS1_INIT)
                {
                    if(gctl_options & GCTL_OPT_JS1_AS_THROTTLE_AND_RUDDER)
                        spin_ptr->cur_value = 9;
                    else if((gctl_options & GCTL_OPT_JS1_THROTTLE) &&
                            (gctl_options & GCTL_OPT_JS1_HEADING) &&
                            (gctl_options & GCTL_OPT_JS1_HAT)
                    )
                        spin_ptr->cur_value = 8;
                    else if((gctl_options & GCTL_OPT_JS1_HEADING) &&
                            (gctl_options & GCTL_OPT_JS1_HAT)
                    )
                        spin_ptr->cur_value = 7;
                    else if((gctl_options & GCTL_OPT_JS1_HEADING) &&
                            (gctl_options & GCTL_OPT_JS1_THROTTLE)
                    )
                        spin_ptr->cur_value = 6;
                    else if(gctl_options & GCTL_OPT_JS1_HEADING)
                        spin_ptr->cur_value = 5;
                    else if((gctl_options & GCTL_OPT_JS1_THROTTLE) &&
                            (gctl_options & GCTL_OPT_JS1_HAT)
                    )
                        spin_ptr->cur_value = 4;
                    else if(gctl_options & GCTL_OPT_JS1_HAT)
                        spin_ptr->cur_value = 3;
                    else if(gctl_options & GCTL_OPT_JS1_THROTTLE)
                        spin_ptr->cur_value = 2;
                    else
                        spin_ptr->cur_value = 1;
                }
		else
		{
		    spin_ptr->cur_value = 0;
		}
            }
        }

#ifdef JS_SUPPORT
        /* Options: Controller joystick button mappings. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {
	    sar_menu_spin_struct *act_spin_ptr;


            /* First joystick (js0) button actions. */
            act_spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_JS0_BUTTON_ACTION, NULL
	    );
	    spin_ptr = SAROptionsGetSpinByID(
                m, SAR_OPT_SELECT_JS0_BUTTON_NUMBER, NULL
            );
	    if((spin_ptr != NULL) && (act_spin_ptr != NULL))
	    {
		spin_ptr->cur_value = SARGetJSButtonFromSpinAction(
		    act_spin_ptr, SAR_OPT_SELECT_JS0_BUTTON_ACTION
		) + 1;
		if(spin_ptr->cur_value >= spin_ptr->total_values)
		    spin_ptr->cur_value = 0;
	    }

            /* Second joystick (js1) button actions. */
            act_spin_ptr = SAROptionsGetSpinByID(  
                m, SAR_OPT_SELECT_JS1_BUTTON_ACTION, NULL
            );
            spin_ptr = SAROptionsGetSpinByID(
                m, SAR_OPT_SELECT_JS1_BUTTON_NUMBER, NULL
            );
            if((spin_ptr != NULL) && (act_spin_ptr != NULL))
            {
		spin_ptr->cur_value = SARGetJSButtonFromSpinAction(
                    act_spin_ptr, SAR_OPT_SELECT_JS1_BUTTON_ACTION
                ) + 1;
                if(spin_ptr->cur_value >= spin_ptr->total_values)
                    spin_ptr->cur_value = 0;
	    }
	}
#endif	/* JS_SUPPORT */

        /* Options graphics. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_GRAPHICS);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {
            switch_ptr = SAROptionsGetSwitchByID(
		m, SAR_OPT_SELECT_GROUND_TEXTURE, NULL
	    );
            if(switch_ptr != NULL)
                switch_ptr->state = option.textured_ground;

            switch_ptr = SAROptionsGetSwitchByID(
		m, SAR_OPT_SELECT_OBJECT_TEXTURE, NULL
	    );
            if(switch_ptr != NULL)
                switch_ptr->state = option.textured_objects;

            switch_ptr = SAROptionsGetSwitchByID(
		m, SAR_OPT_SELECT_CLOUDS, NULL
	    );
            if(switch_ptr != NULL)
                switch_ptr->state = option.textured_clouds;

	    switch_ptr = SAROptionsGetSwitchByID(
		m, SAR_OPT_SELECT_ATMOSPHERE, NULL
	    );
            if(switch_ptr != NULL)
                switch_ptr->state = option.atmosphere;

            switch_ptr = SAROptionsGetSwitchByID(
                m, SAR_OPT_SELECT_DUAL_PASS_DEPTH, NULL
            );
            if(switch_ptr != NULL)
                switch_ptr->state = option.dual_pass_depth;


            spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_MAX_VISIBILITY, NULL
	    );
            if(spin_ptr != NULL)
            {
		int i = option.visibility_max;

		if((i >= 0) && (i < spin_ptr->total_values))
		    spin_ptr->cur_value = i;
		else
		    spin_ptr->cur_value = 0;
            } 
        }

        /* Options sounds. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_OPTIONS_SOUND);
        m = ((i < 0) ? NULL : core_ptr->menu[i]);
        if(m != NULL)
        {
	    /* Sounds level. */
	    spin_ptr = SAROptionsGetSpinByID(
		m, SAR_OPT_SELECT_SOUND_LEVEL, NULL
	    );
	    if(spin_ptr != NULL)
	    {
		/* No recorder (implies all sounds off)? */
		if(core_ptr->recorder == NULL)
		{
		    spin_ptr->cur_value = 0;
		}
		/* All sounds on? */
		else if(option.engine_sounds &&
                        option.event_sounds &&
                        option.voice_sounds
		)
		{
		    spin_ptr->cur_value = 3;
		}
		/* Engine sounds only? */
		else if(option.engine_sounds)
		{
		    spin_ptr->cur_value = 2;
		}
		/* Event sounds only? */
		else if(option.event_sounds)
		{
		    spin_ptr->cur_value = 1;
		}
		/* All else no sounds. */
		else
		{
		    spin_ptr->cur_value = 0;
		}
	    }

	    /* Music. */
            switch_ptr = SAROptionsGetSwitchByID(
		m, SAR_OPT_SELECT_MUSIC, NULL
	    );
	    if(switch_ptr != NULL)
		switch_ptr->state = option.music;
        }

	return;
}

/*         
 *      Options menu select switch callback.
 */
void SARMenuOptionsSwitchCB(
        void *object, void *client_data,
        int id_code, Boolean state
)
{
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        sar_menu_switch_struct *switch_ptr = (sar_menu_switch_struct *)object;


        if((core_ptr == NULL) || (switch_ptr == NULL))
            return;

        /* Handle by switch's ID code. */
        switch(id_code)
        {
          /* ****************************************************** */
          case SAR_OPT_SELECT_GROUND_TEXTURE:
            option.textured_ground = state;
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_OBJECT_TEXTURE:
            option.textured_objects = state;
	    if(!state)
	    {
		/* Print warning about object texture if off. */
		GWOutputMessage(
		    core_ptr->display, GWOutputMessageTypeWarning,
		    "Object Texture Disabling Warning",
"Disabling object texturing may affect ground plot\n\
objects and make flight visualization extremely difficult.",
"Ground plot objects and visualization important objects\n\
will not be texturized. It will also be extremely difficult\n\
to visually interprite heightfield objects."
		);
	    }
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_CLOUDS:
            option.textured_clouds = state;
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_ATMOSPHERE:
            option.atmosphere = state;  
            break;
 
          /* ****************************************************** */
          case SAR_OPT_SELECT_DUAL_PASS_DEPTH:
            option.dual_pass_depth = state;
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_MUSIC:
            option.music = state;
            SARUpdateBackgroundMusic(core_ptr);
            break;
        }
}
 
/*
 *      Options menu select spin callback.
 */
void SARMenuOptionsSpinCB(
        void *object, void *client_data,
        int id_code, char *value
)
{               
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        sar_menu_spin_struct *spin_ptr = (sar_menu_spin_struct *)object;
        sar_menu_struct *m;
        gctl_flags_t prev_gctl_options;
        const char *sound_server_connect_arg = "127.0.0.1:9433";


        if((core_ptr == NULL) ||
           (spin_ptr == NULL)
        )
            return;

	/* Recorder address available? */
	if(core_ptr->recorder_address != NULL)
	    sound_server_connect_arg = (const char *)core_ptr->recorder_address;

	/* Get current menu pointer (must be valid). */
	m = SARGetCurrentMenuPtr(core_ptr);
	if(m == NULL)
	    return;

        /* Handle by spin's ID code. Also note that the value's
         * index position is used, not the actual value so the
         * order of values placed into the spin object is important.
         */
        switch(id_code)
        {
          /* ****************************************************** */
          case SAR_OPT_SELECT_HOIST_CONTACT:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
            switch(spin_ptr->cur_value)
            {
	      case 1:	/* Moderate. */
		option.hoist_contact_expansion_coeff = 2.0;
		break;

	      case 2:	/* Authentic/difficult. */
		option.hoist_contact_expansion_coeff = 1.0;
		break;

	      default:	/* Easy. */
		option.hoist_contact_expansion_coeff = 4.0;
		break;
	    }
	    break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_DAMAGE_RESISTANCE:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
            switch(spin_ptr->cur_value)
            {
              case 1:   /* Nominal. */
		option.damage_resistance_coeff = 1.5;
                break;

              case 2:   /* Authentic/difficult. */
		option.damage_resistance_coeff = 1.0;
                break;
                    
              default:	/* Strong/easy. */
		option.damage_resistance_coeff = 2.0;
                break;
            }   
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_FLIGHT_PHYSICS:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
            switch(spin_ptr->cur_value)
            {
              case 1:   /* Moderate. */
		option.flight_physics = FLIGHT_PHYSICS_MODERATE;
                break;

              case 2:   /* Realistic/difficult. */
		option.flight_physics = FLIGHT_PHYSICS_REALISTIC;
                break;

              default:  /* Easy. */
		option.flight_physics = FLIGHT_PHYSICS_EASY;
                break;
            }
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_JS0_AXISES:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
            option.gctl_options = 0;
            switch(spin_ptr->cur_value)
            {
              case 1:   /* 2D. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH
                );
                break;

              case 2:   /* 2D with throttle. */ 
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_THROTTLE
                );
                break;

              case 3:   /* 2D with hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_HAT
                );
                break;

              case 4:   /* 2D with throttle & hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_THROTTLE | GCTL_OPT_JS0_HAT  
                );
                break;
            
              case 5:   /* 3D. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_HEADING
                );
                break;
                
              case 6:   /* 3D with throttle. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT | 
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_HEADING |
                    GCTL_OPT_JS0_THROTTLE
                );
                break;
        
              case 7:   /* 3D with hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |
                    GCTL_OPT_JS0_HEADING |
                    GCTL_OPT_JS0_HAT
                );
                break;
              
              case 8:   /* 3D with throttle & hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT |
                    GCTL_OPT_JS0_BANK | GCTL_OPT_JS0_PITCH |  
                    GCTL_OPT_JS0_HEADING |
                    GCTL_OPT_JS0_THROTTLE | GCTL_OPT_JS0_HAT
                );
                break;
                
              case 9:   /* As throttle & rudder. */
                option.gctl_options |= (
                    GCTL_OPT_JS0_INIT | GCTL_OPT_JS0_AS_THROTTLE_AND_RUDDER
                );
                break;
            }   

#ifdef JS_SUPPORT              
            /* If all joysticks not initialized, then switch to
             * keyboard as controller. Otherwise if one or more joysticks
	     * is marked as the game controller then set controler as
	     * joystick.
             */
            if((option.gctl_options & GCTL_OPT_JS0_INIT) ||
               (option.gctl_options & GCTL_OPT_JS1_INIT)
            )
                option.gctl_controller_type = GCTL_CONTROLLER_JOYSTICK;
            else
                option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;

            /* Reinitialize game controller to the new type. */
            GWSetInputBusy(core_ptr->display);
            SARInitGCTL(core_ptr);	/* Realize changes. */
            GWSetInputReady(core_ptr->display);
#else	/* JS_SUPPORT */
	    /* Print warning about no joystick support. */
	    GWOutputMessage(
		core_ptr->display, GWOutputMessageTypeError,
		"No Joystick Support Available",
"There is no joystick support available.",
"This program was not compiled with joystick support and/or\n\
joystick support was not enabled on your operating system."
	    );

	    /* Reinitialize game controler back to keyboard as needed. */
	    if(option.gctl_controller_type != GCTL_CONTROLLER_KEYBOARD)
	    {
	        option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;
                SARInitGCTL(core_ptr);	/* Realize changes. */
	    }
#endif	/* JS_SUPPORT */
            break;
                    
          /* ****************************************************** */
          case SAR_OPT_SELECT_JS1_AXISES:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
            prev_gctl_options = option.gctl_options;
            option.gctl_options = 0;   
            switch(spin_ptr->cur_value)
            {
              case 1:   /* 2D. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH
                );
                break;

	      case 2:   /* 2D with throttle. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_THROTTLE
                );
                break;
            
              case 3:   /* 2D with hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_HAT
                );
                break;
          
              case 4:   /* 2D with throttle & hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |  
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |  
                    GCTL_OPT_JS1_THROTTLE | GCTL_OPT_JS1_HAT
                );
                break;
            
              case 5:   /* 3D. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_HEADING
                );
                break;
                
              case 6:   /* 3D with throttle. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT | 
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_HEADING |
                    GCTL_OPT_JS1_THROTTLE
                );
                break;
            
              case 7:   /* 3D with hat. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_HEADING |
                    GCTL_OPT_JS1_HAT
                );
                break;
              
              case 8:   /* 3D with throttle & hat. */
                option.gctl_options |= ( 
                    GCTL_OPT_JS1_INIT |
                    GCTL_OPT_JS1_BANK | GCTL_OPT_JS1_PITCH |
                    GCTL_OPT_JS1_HEADING |
                    GCTL_OPT_JS1_THROTTLE | GCTL_OPT_JS1_HAT
                );
                break;
                
              case 9:   /* As throttle & rudder. */
                option.gctl_options |= (
                    GCTL_OPT_JS1_INIT | GCTL_OPT_JS1_AS_THROTTLE_AND_RUDDER
                );
                break;
            }   
#ifdef JS_SUPPORT
            /* If all joysticks not initialized, then switch to
             * keyboard as controller.
             */
            if((option.gctl_options & GCTL_OPT_JS0_INIT) ||
               (option.gctl_options & GCTL_OPT_JS1_INIT)
            )
                option.gctl_controller_type = GCTL_CONTROLLER_JOYSTICK;
            else
                option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;

            /* Reinitialize game controller to the new type, note that
	     * even if game controller type is the same (not changed)
	     * certain parameters for the game controller may have changed
	     * so a reinitialize is needed.
	     */
	    GWSetInputBusy(core_ptr->display);
            SARInitGCTL(core_ptr);
	    GWSetInputReady(core_ptr->display);

#else   /* JS_SUPPORT */
            /* Print warning about no joystick support. */
            GWOutputMessage(
                core_ptr->display, GWOutputMessageTypeError,
                "No Joystick Support Available",
"There is no joystick support available.",
"This program was not compiled with joystick support and/or\n\
joystick support was not enabled on your operating system."
            );

            /* Reinitialize back to keyboard. */
            if(option.gctl_controller_type != GCTL_CONTROLLER_KEYBOARD)
            {
                option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;
                SARInitGCTL(core_ptr);
            }
#endif  /* JS_SUPPORT */
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_JS0_BUTTON_ACTION:
#ifdef JS_SUPPORT
	    if(1)
	    {
		int n;
		sar_menu_spin_struct *btn_spin_ptr;

		btn_spin_ptr = SAROptionsGetSpinByID(
		    m, SAR_OPT_SELECT_JS0_BUTTON_NUMBER, &n
		);
		if(btn_spin_ptr != NULL)
		{
		    btn_spin_ptr->cur_value = SARGetJSButtonFromSpinAction(
			spin_ptr, SAR_OPT_SELECT_JS0_BUTTON_ACTION
		    ) + 1;
                    if(btn_spin_ptr->cur_value >= btn_spin_ptr->total_values)
                        btn_spin_ptr->cur_value = 0;
		    SARMenuDrawObject(core_ptr->display, m, n);
		    GWSwapBuffer(core_ptr->display);
		}
	    }
#endif	/* JS_SUPPORT. */
	    break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_JS0_BUTTON_NUMBER:
#ifdef JS_SUPPORT
            if(1)
            {
		sar_option_struct *opt = &option;
                sar_menu_spin_struct *act_spin_ptr = SAROptionsGetSpinByID(
                    m, SAR_OPT_SELECT_JS0_BUTTON_ACTION, NULL
                );
		if(act_spin_ptr != NULL)
		{
		    int new_val = spin_ptr->cur_value - 1;

		    /* Set option for button mapping based on which value
		     * the action spin is on.
		     */
		    switch(act_spin_ptr->cur_value)
		    {
		      case 0:	/* Rotate Modifier. */
			opt->js0_btn_rotate = new_val;
			break;
                      case 1:	/* Air Brakes. */
                        opt->js0_btn_air_brakes = new_val;
                        break;
                      case 2:	/* Wheel Brakes. */
                        opt->js0_btn_wheel_brakes = new_val;
                        break;
                      case 3:	/* Zoom In. */
                        opt->js0_btn_zoom_in = new_val;
                        break;
                      case 4:	/* Zoom out. */
                        opt->js0_btn_zoom_out = new_val;
                        break;
                      case 5:	/* Hoist Up. */
                        opt->js0_btn_hoist_up = new_val;
                        break;
                      case 6:	/* Hoist Down. */
                        opt->js0_btn_hoist_down = new_val;
                        break;
/* When adding new button mappings, be sure to add support for them
 * in SARGetJSButtonFromSpinAction().
 *
 * Also remember to add support for EACH joystick!
 */
		    }
                }
            }
#endif  /* JS_SUPPORT. */
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_JS1_BUTTON_ACTION:
#ifdef JS_SUPPORT
            if(1)
            {
                int n;
                sar_menu_spin_struct *btn_spin_ptr;

                btn_spin_ptr = SAROptionsGetSpinByID(
                    m, SAR_OPT_SELECT_JS1_BUTTON_NUMBER, &n 
                );
                if(btn_spin_ptr != NULL)
                {
                    btn_spin_ptr->cur_value = SARGetJSButtonFromSpinAction(
                        spin_ptr, SAR_OPT_SELECT_JS1_BUTTON_ACTION
                    ) + 1;
                    if(btn_spin_ptr->cur_value >= btn_spin_ptr->total_values)
                        btn_spin_ptr->cur_value = 0;
                    SARMenuDrawObject(core_ptr->display, m, n);
                    GWSwapBuffer(core_ptr->display);
                }
            }
#endif  /* JS_SUPPORT. */
            break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_JS1_BUTTON_NUMBER:
#ifdef JS_SUPPORT
            if(1)
            {
                sar_option_struct *opt = &option;
                sar_menu_spin_struct *act_spin_ptr = SAROptionsGetSpinByID(
                    m, SAR_OPT_SELECT_JS1_BUTTON_ACTION, NULL
                );
                if(act_spin_ptr != NULL)
                {
                    int new_val = spin_ptr->cur_value - 1;

                    /* Set option for button mapping based on which value
                     * the action spin is on.
                     */
                    switch(act_spin_ptr->cur_value)
                    {
                      case 0:   /* Rotate Modifier. */
                        opt->js1_btn_rotate = new_val;
                        break;
                      case 1:   /* Air Brakes. */
                        opt->js1_btn_air_brakes = new_val;
                        break;
                      case 2:   /* Wheel Brakes. */
                        opt->js1_btn_wheel_brakes = new_val;
                        break;
                      case 3:   /* Zoom In. */
                        opt->js1_btn_zoom_in = new_val;
                        break;
                      case 4:   /* Zoom out. */
                        opt->js1_btn_zoom_out = new_val;
                        break;
                      case 5:   /* Hoist Up. */
                        opt->js1_btn_hoist_up = new_val;
                        break;
                      case 6:   /* Hoist Down. */
                        opt->js1_btn_hoist_down = new_val;
                        break;
/* When adding new button mappings, be sure to add support for them
 * in SARGetJSButtonFromSpinAction().
 * 
 * Also remember to add support for EACH joystick!
 */
                    }
                }
            }
#endif  /* JS_SUPPORT. */
            break;

          /* ****************************************************** */
	  case SAR_OPT_SELECT_MAX_VISIBILITY:
            /* These values should correspond to those used to
             * set the values in SARBuildMenus().
             */
	    option.visibility_max = spin_ptr->cur_value;
	    break;

          /* ****************************************************** */
          case SAR_OPT_SELECT_SOUND_LEVEL:
/* Macro to turn sound on as needed by checking if the core structure's
 * recorder is initialized. If the recorder is NULL then an attempt to
 * connect to the sound server will be made using the argument
 * specified by sound_server_connect_arg.
 */
#define SND_ON_AS_NEEDED        \
if(core_ptr->recorder == NULL) \
{ \
 GWSetInputBusy(core_ptr->display); \
 core_ptr->recorder = SoundInit( \
  SNDSERV_TYPE_Y, \
  sound_server_connect_arg, \
  NULL		/* Do not start sound server. */ \
 ); \
 if(core_ptr->recorder == NULL) \
 { \
  char *buf; \
  int buf_len; \
 \
  buf_len = 512 + ((sound_server_connect_arg == NULL) ? \
   0 : strlen(sound_server_connect_arg) \
  ); \
  buf = (char *)malloc(buf_len * sizeof(char)); \
  if(buf != NULL) \
   sprintf(buf, \
"Unable to connect to sound server at:\n\n    %s\n\n\
Make sure that the address is correct,\n\
use the argument --recorder to specify\n\
an alternate address as needed.\n\n\
To start the sound server in most cases\n\
use `starty'", \
    sound_server_connect_arg \
   ); \
  GWOutputMessage( \
   core_ptr->display, \
   GWOutputMessageTypeError, \
   "Sound initialization failed!", \
   buf, \
"Please check to make sure that your sound server\n\
is running and available at the specified address.\n\
Also make sure that you have sufficient permission\n\
to connect to it. To start in most cases run `starty'." \
   ); \
   free(buf); \
 } \
 else \
 { \
  SoundChangeMode( \
   core_ptr->recorder, core_ptr->audio_mode_name \
  ); \
 } \
 GWSetInputReady(core_ptr->display); \
}

#define SND_OFF_AS_NEEDED       \
if(core_ptr->recorder != NULL) \
{ \
 GWSetInputBusy(core_ptr->display); \
 SoundShutdown(core_ptr->recorder); \
 core_ptr->recorder = NULL; \
 GWSetInputReady(core_ptr->display); \
}

            switch(spin_ptr->cur_value)
            {
              case 1:   /* On, events only. */
                option.engine_sounds = False;
		option.event_sounds = True;
		option.voice_sounds = False;
                SND_ON_AS_NEEDED
                break;

              case 2:   /* On, events and engine. */
                option.engine_sounds = True;
		option.event_sounds = True;
		option.voice_sounds = False;
                SND_ON_AS_NEEDED  
                break;
 
              case 3:   /* On, events, engine, and voice (all on). */
		option.engine_sounds = True;
		option.event_sounds = True;
		option.voice_sounds = True;
		SND_ON_AS_NEEDED
                break;
    
              default:  /* All else assume off. */
		option.engine_sounds = False;
		option.event_sounds = False;
		option.voice_sounds = False;
                SND_OFF_AS_NEEDED
                break;
            }
#undef SND_ON_AS_NEEDED

#if !defined(HAVE_Y)
	    /* Print warning about no sound support. */
	    GWOutputMessage(
		core_ptr->display, GWOutputMessageTypeWarning,
		"Sound Support Not Available",
"There is no sound support available.",
"This program was not compiled with sound support and/or\n\
sound support was not enabled on your operating system."
	    );
#endif

            break;

        }
 
        return;
}
