#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/os.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/string.h"

#include "gctl.h"
#include "sar.h"

#include "config.h"


int SAROptionsLoadFromFile(sar_option_struct *opt, const char *filename);
int SAROptionsSaveToFile(sar_option_struct *opt, const char *filename);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


/*
 *	Loads options/preferances from file and stores them on opt.
 */
int SAROptionsLoadFromFile(sar_option_struct *opt, const char *filename)
{
	FILE *fp;
	char *buf = NULL;
	struct stat stat_buf;
	double value[10];


	if((opt == NULL) || (filename == NULL))
	    return(-1);

	if(stat(filename, &stat_buf))
	{
	    fprintf(stderr, "%s: No such file.\n", filename);
	    return(-1);
	}
#ifdef S_ISREG
	if(!S_ISREG(stat_buf.st_mode))
	{
	    fprintf(stderr,
                "%s: Not a file.\n",
		filename
            );
            return(-1);
	}
#endif	/* S_ISREG */

	fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
	    fprintf(stderr, "%s: Cannot open.\n", filename);
	    return(-1);
	}

	do
	{
	    buf = FSeekNextParm(
		fp,
		buf,
		SAR_COMMENT_CHAR,
		SAR_CFG_DELIM_CHAR
	    );
	    if(buf == NULL)
	        break;

	    /* Version. */
	    if(!strcasecmp(buf, "Version"))
	    {
		FGetValuesF(fp, value, 2);



	    }

	    /* Show menu backgrounds. */
	    else if(!strcasecmp(buf, "MenuBackgrounds"))
            {
                FGetValuesF(fp, value, 1);
 
		/* Omit this, let command line set it. */         
            }

	    /* Textured ground base. */
	    else if(!strcasecmp(buf, "TexturedGround"))
            {
                FGetValuesF(fp, value, 1);
		opt->textured_ground = (((int)value[0]) ? True : False);
            }
	    /* Textured objects. */
            else if(!strcasecmp(buf, "TexturedObjects"))
            {
                FGetValuesF(fp, value, 1);
                opt->textured_objects = (((int)value[0]) ? True : False);
            }
            /* Textured clouds. */
            else if(!strcasecmp(buf, "TexturedClouds"))
            {
                FGetValuesF(fp, value, 1);
                opt->textured_clouds = (((int)value[0]) ? True : False);
            }
            /* Atmosphere. */
            else if(!strcasecmp(buf, "Atmosphere"))
            {
                FGetValuesF(fp, value, 1);
                opt->atmosphere = (((int)value[0]) ? True : False);
            }
            /* Dual pass depth. */
            else if(!strcasecmp(buf, "DualPassDepth"))
            {
                FGetValuesF(fp, value, 1);
                opt->dual_pass_depth = (((int)value[0]) ? True : False);
            }
            /* Prop wash. */
            else if(!strcasecmp(buf, "PropWash"))
            {
                FGetValuesF(fp, value, 1);
                opt->prop_wash = (((int)value[0]) ? True : False);
            }
            /* Smoke trails. */
            else if(!strcasecmp(buf, "SmokeTrails"))
            {
                FGetValuesF(fp, value, 1);
                opt->smoke_trails = (((int)value[0]) ? True : False);
            }
	    /* GL polygon offset factor. */
            else if(!strcasecmp(buf, "GLPolygonOffsetFactor"))
            {
                FGetValuesF(fp, value, 1);
                opt->gl_polygon_offset_factor = value[0];
            }
            /* Maximum visibility. */
            else if(!strcasecmp(buf, "VisibilityMax"))
            {
                FGetValuesF(fp, value, 1);
                opt->visibility_max = (int)value[0];
            }

            /* Engine sounds. */
            else if(!strcasecmp(buf, "EngineSounds"))
            {
                FGetValuesF(fp, value, 1);
                opt->engine_sounds = (((int)value[0]) ? True : False);
            }
            /* Event sounds. */
            else if(!strcasecmp(buf, "EventSounds"))
            {
                FGetValuesF(fp, value, 1);
                opt->event_sounds = (((int)value[0]) ? True : False);
            }
            /* Voice sounds. */
            else if(!strcasecmp(buf, "VoiceSounds"))
            {
                FGetValuesF(fp, value, 1);
                opt->voice_sounds = (((int)value[0]) ? True : False);
            }
            /* Music. */
            else if(!strcasecmp(buf, "Music"))
            {
                FGetValuesF(fp, value, 1);
                opt->music = (((int)value[0]) ? True : False);
            }
	    /* Aircraft engine sound distance (in meters). */
	    else if(!strcasecmp(buf, "AircraftEngineSoundDistance"))
            {
		FGetValuesF(fp, value, 1);
                opt->aircraft_engine_sound_distance = value[0]; 
            }

            /* HUD color. */
            else if(!strcasecmp(buf, "HUDColor"))
            {
                FGetValuesF(fp, value, 4);

		/* ARGB */
                opt->hud_color.a = MAX(MIN(value[0], 1.0), 0.0);
                opt->hud_color.r = MAX(MIN(value[1], 1.0), 0.0);
                opt->hud_color.g = MAX(MIN(value[2], 1.0), 0.0);
                opt->hud_color.b = MAX(MIN(value[3], 1.0), 0.0);
            }
            /* Message color. */
            else if(!strcasecmp(buf, "MessageColor"))
            {
                FGetValuesF(fp, value, 4);

                /* ARGB */
                opt->message_color.a = MAX(MIN(value[0], 1.0), 0.0);
                opt->message_color.r = MAX(MIN(value[1], 1.0), 0.0);
                opt->message_color.g = MAX(MIN(value[2], 1.0), 0.0);
                opt->message_color.b = MAX(MIN(value[3], 1.0), 0.0);
            }

            /* Smoke puff spawn interval (in miliseconds). */
            else if(!strcasecmp(buf, "SmokeSpawnInterval"))
            {
                FGetValuesF(fp, value, 1);
/* Depreciated. */
            }
            /* Explosion frame increment interval (in milliseconds). */
            else if(!strcasecmp(buf, "ExplosionFrameInterval"))
            {
                FGetValuesF(fp, value, 1);
                opt->explosion_frame_int = (time_t)value[0];

                if(opt->explosion_frame_int < 1)
                {
                    fprintf(stderr,
 "%s: ExplosionFrameInterval: warning: Value out of range, suggest %i.\n",
                        filename, SAR_DEF_EXPLOSION_FRAME_INT
                    );
                }
            }
	    /* Splash frame increment interval (in milliseconds). */
	    else if(!strcasecmp(buf, "SplashFrameInterval"))
            {
                FGetValuesF(fp, value, 1);
		opt->splash_frame_int = (time_t)value[0];
		
		if(opt->splash_frame_int < 1)
		{
		    fprintf(stderr,
 "%s: SplashFrameInterval: warning: Value out of range, suggest %i.\n",
			filename, SAR_DEF_SPLASH_FRAME_INT
		    );
		}
            }

            /* Smoke puff life span (in milliseconds). */
            else if(!strcasecmp(buf, "SmokeLifeSpan"))
            {
                FGetValuesF(fp, value, 1);
/* Depreciated. */
	    }
            /* Crash caused explosion life span (in milliseconds). */
            else if(!strcasecmp(buf, "CrashExplosionLifeSpan"))
            {
                FGetValuesF(fp, value, 1);
                opt->crash_explosion_life_span = (time_t)value[0];

                if(opt->crash_explosion_life_span < 1)
                {
                    fprintf(stderr,
 "%s: CrashExplosionLifeSpan: warning: Value out of range, suggest %i.\n",
                        filename, SAR_DEF_CRASH_EXPLOSION_LIFE_SPAN
                    );
                }
            }
            /* Dropped fuel tanks life span (after landing on ground). */
            else if(!strcasecmp(buf, "FuelTankLifeSpan"))
            {
                FGetValuesF(fp, value, 1);
                opt->fuel_tank_life_span = (time_t)value[0];

                if(opt->fuel_tank_life_span < 1)
                {
                    fprintf(stderr,
 "%s: FuelTankLifeSpan: warning: Value out of range, suggest %i.\n",
                        filename, SAR_DEF_FUEL_TANK_LIFE_SPAN
                    );
                }
            }

            /* Rotor wash visiblity coefficient (0.0 to 1.0). */
            else if(!strcasecmp(buf, "RotorWashVisCoeff"))
            {
                FGetValuesF(fp, value, 1);
                if(value[0] < 0.0)
		    value[0] = 0.0;
		if(value[0] > 1.0)
		    value[0] = 1.0;
                opt->rotor_wash_vis_coeff = value[0];
            }

	    /* Game controller type. */
	    else if(!strcasecmp(buf, "GameControllerType"))
            {
                FGetValuesF(fp, value, 1);
		opt->gctl_controller_type = (int)value[0];
#if !defined(JS_SUPPORT)
		if(opt->gctl_controller_type == GCTL_CONTROLLER_JOYSTICK)
		{
		    fprintf(stderr,
"%s: Warning: Joystick support not compiled, GameControllerType should \
not be set to joystick.\n",
			filename
		    );
		}
#endif	/* !JS_SUPPORT */
            }
	    /* Joystick #1 (js0) button mappings. */
	    else if(!strcasecmp(buf, "Joystick0ButtonMappings") ||
                    !strcasecmp(buf, "JoystickButtonMappings")
	    )
            {
		/* There are 7 values:
		 *
		 * <rotate> <air brakes> <wheel brakes> <zoom in>
		 * <zoom out> <hoist up> <hoist_down>
		 */
                FGetValuesF(fp, value, 7);

		opt->js0_btn_rotate = (int)value[0];
                opt->js0_btn_air_brakes = (int)value[1];
                opt->js0_btn_wheel_brakes = (int)value[2];
                opt->js0_btn_zoom_in = (int)value[3];
                opt->js0_btn_zoom_out = (int)value[4];
		opt->js0_btn_hoist_up = (int)value[5];
                opt->js0_btn_hoist_down = (int)value[6];
	    }
            /* Joystick #2 (js1) button mappings. */
            else if(!strcasecmp(buf, "Joystick1ButtonMappings"))
	    {
                /* There are 7 values:
                 *
                 * <rotate> <air brakes> <wheel brakes> <zoom in>
                 * <zoom out> <hoist up> <hoist_down>
                 */
                FGetValuesF(fp, value, 7);

                opt->js1_btn_rotate = (int)value[0];
                opt->js1_btn_air_brakes = (int)value[1];  
                opt->js1_btn_wheel_brakes = (int)value[2];
                opt->js1_btn_zoom_in = (int)value[3]; 
                opt->js1_btn_zoom_out = (int)value[4];  
                opt->js1_btn_hoist_up = (int)value[5];  
                opt->js1_btn_hoist_down = (int)value[6];
            }
	    /* Game controller options. */
	    else if(!strcasecmp(buf, "GameControllerOptions"))
	    {
		/* Value is stored as bits (unsigned long). */
		FGetValuesF(fp, value, 1);

		opt->gctl_options = (gctl_flags_t)value[0];
	    }

	    /* Hoist contact expansion coefficient. */
	    else if(!strcasecmp(buf, "HoistContactExpansionCoefficient"))
            {
                FGetValuesF(fp, value, 1);
                opt->hoist_contact_expansion_coeff = MAX(value[0], 1.0);
	    }
            /* Damage resistance coefficient. */
            else if(!strcasecmp(buf, "DamageResistanceCoefficient"))
            {
                FGetValuesF(fp, value, 1);
                opt->damage_resistance_coeff = MAX(value[0], 1.0);
            }
            /* Flight physics. */
            else if(!strcasecmp(buf, "FlightPhysics"))
            {
                FGetValuesF(fp, value, 1);
                opt->flight_physics = (int)value[0];
            }

	    /* Last selected mission. */
	    else if(!strcasecmp(buf, "LastSelectedMission"))
            {
                FGetValuesF(fp, value, 1);
                opt->last_selected_mission = (int)value[0];
            }
            /* Last selected free flight scene. */
            else if(!strcasecmp(buf, "LastSelectedFFScene"))
            {
                FGetValuesF(fp, value, 1);
                opt->last_selected_ffscene = (int)value[0];
            }
            /* Last selected free flight aircraft. */
            else if(!strcasecmp(buf, "LastSelectedFFAircraft"))
            {
                FGetValuesF(fp, value, 1);
                opt->last_selected_ffaircraft = (int)value[0];
            }
	    /* Last selected free flight weather preset. */
            else if(!strcasecmp(buf, "LastSelectedFFWeather"))
	    {
                FGetValuesF(fp, value, 1);
		opt->last_selected_ffweather = (int)value[0];
            }
            /* Last toplevel window position. */
            else if(!strcasecmp(buf, "LastToplevelPosition"))
            {
                FGetValuesF(fp, value, 4);
                opt->last_x = (int)value[0];
                opt->last_y = (int)value[1];
                opt->last_width = (int)value[2];
                opt->last_height = (int)value[3];
            }


	    else
	    {
	        fprintf(stderr,
	            "%s: Unsupported parameter `%s'.\n",
	            filename, buf
	        );
		FSeekNextLine(fp);
	    }

	} while(1);

	FClose(fp);

	return(0);
}

/*
 *	Saves options/preferances to the file specified by filename from
 *	the data in opt.
 */
int SAROptionsSaveToFile(sar_option_struct *opt, const char *filename)
{
	FILE *fp;
	struct stat stat_buf;


	if((opt == NULL) ||
	   (filename == NULL)
	)
	    return(-1);

	if(!stat(filename, &stat_buf))
	{
#ifdef S_ISDIR
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		fprintf(stderr,
"%s: Cannot write configuration, object exists as a directory.\n",
		    filename
		);
		return(-1);
	    }
#endif	/* S_ISDIR */
	}

	/* Open file for writing. */
	fp = FOpen(filename, "wb");
	if(fp == NULL)
	    return(0);

#define PUTCR	fputc('\n', fp);

	/* Write header. */
	fprintf(fp,
"# %s - Configuration\n\
#\n\
#       Contains selections, game play, graphics, and sound preferances.\n\
#\n\
#       This file is automatically generated, the settings can be\n\
#       modified from the program's options menu.\n\
#\n\
#       You may manually modified this file with a text editor if you\n\
#       know what you are doing.\n\
#\n\
\n",
	    PROG_NAME
	);

	/* Version. */
	fprintf(fp,
	    "Version = %i %i",
	    PROG_VERSION_MAJOR, PROG_VERSION_MINOR
	);
	PUTCR

	/* Show menu backgrounds. */
	fprintf(fp,
	    "MenuBackgrounds = %i",
	    ((opt->menu_backgrounds) ? 1 : 0)
	);
        PUTCR

	/* Textured ground base. */
	fprintf(fp,
	    "TexturedGround = %i",
	    ((opt->textured_ground) ? 1 : 0)
	);
        PUTCR
        /* Textured objects. */
        fprintf(fp,
            "TexturedObjects = %i",
            ((opt->textured_objects) ? 1 : 0)
        );
        PUTCR
        /* Textured clouds. */
        fprintf(fp,
            "TexturedClouds = %i",
            ((opt->textured_clouds) ? 1 : 0)
        );
        PUTCR
        /* Atmosphere. */
        fprintf(fp,
            "Atmosphere = %i",
            ((opt->atmosphere) ? 1 : 0)
        );
        PUTCR
	/* Dual pass depth. */
	fprintf(fp,
            "DualPassDepth = %i",
            ((opt->dual_pass_depth) ? 1 : 0)
        );
        PUTCR
        /* Prop wash. */
        fprintf(fp,
            "PropWash = %i",
            ((opt->prop_wash) ? 1 : 0)
        );
        PUTCR
        /* Smoke trails. */
        fprintf(fp,
            "SmokeTrails = %i",
            ((opt->smoke_trails) ? 1 : 0)
        );
        PUTCR
	/* GL polygon offset factor. */
	fprintf(fp,
            "GLPolygonOffsetFactor = %f",
            opt->gl_polygon_offset_factor
        );   
        PUTCR
	/* Maximum visibility. */
        fprintf(fp,
"# Maximum visibility is computed: miles = 3 + (visibility_max * 3)"
        );
        PUTCR
	fprintf(fp,
            "VisibilityMax = %i",
            opt->visibility_max
        );
        PUTCR
        PUTCR

        /* Engine sounds. */
        fprintf(fp,
            "EngineSounds = %i",
            ((opt->engine_sounds) ? 1 : 0)
        );
        PUTCR
        /* Event sounds. */
        fprintf(fp,
            "EventSounds = %i",
            ((opt->event_sounds) ? 1 : 0)
        );
        PUTCR
        /* Voice sounds. */
        fprintf(fp,
            "VoiceSounds = %i",
            ((opt->voice_sounds) ? 1 : 0)
        );
        PUTCR
        /* Music. */
        fprintf(fp,
            "Music = %i",
            ((opt->music) ? 1 : 0)
        );
        PUTCR
	/* Aircraft engine sound distance (in meters). */
	fprintf(fp,
	    "AircraftEngineSoundDistance = %f",
	    opt->aircraft_engine_sound_distance
	);
        PUTCR
	PUTCR

        /* HUD color. */
        fprintf(fp,
            "HUDColor = %f %f %f %f",
	    opt->hud_color.a,
            opt->hud_color.r,
            opt->hud_color.g,
            opt->hud_color.b
        );
        PUTCR
	/* Message color. */
	fprintf(fp,
            "MessageColor = %f %f %f %f",
            opt->message_color.a,
            opt->message_color.r,
            opt->message_color.g,
            opt->message_color.b
        );
        PUTCR
        PUTCR

        /* Smoke puff spawn interval (in miliseconds). */
/* SmokeSpawnInterval is depreciated. */
        /* Explosion frame increment interval (in milliseconds). */
        fprintf(fp,
            "ExplosionFrameInterval = %ld",
            opt->explosion_frame_int
        );
        PUTCR
	/* Splash frame increment interval (in milliseconds). */
        fprintf(fp,
            "SplashFrameInterval = %ld",
            opt->splash_frame_int
        );
	PUTCR
        PUTCR

        /* Smoke puff life span (in milliseconds). */
/* SmokeLifeSpan is depreciated. */
        /* Crash caused explosion life span (in milliseconds). */
        fprintf(fp,
            "CrashExplosionLifeSpan = %ld",
            opt->crash_explosion_life_span
        );
	PUTCR
	/* Dropped fuel tanks, after landing on ground (in milliseconds). */
        fprintf(fp,
            "FuelTankLifeSpan = %ld",
            opt->fuel_tank_life_span
        );
        PUTCR
        PUTCR

	/* Rotor wash visibility coefficient. */
	fprintf(fp,
	    "RotorWashVisCoeff = %f",
	    opt->rotor_wash_vis_coeff
	);
	PUTCR

        /* Game controller type. */
	fprintf(fp,
            "GameControllerType = %i",
	    opt->gctl_controller_type
	);
        PUTCR
        /* Joystick button mappings (7 values). */
	fprintf(fp,
"\
# Joystick button mappings: <rot> <air_brk> <whl_brk> <zm_in> <zm_out>\n\
#                           <hoist_up> <hoist_dn>"
	);
	PUTCR
        fprintf(fp,
            "Joystick0ButtonMappings = %i %i %i %i %i %i %i",
            opt->js0_btn_rotate,
            opt->js0_btn_air_brakes,
            opt->js0_btn_wheel_brakes,
            opt->js0_btn_zoom_in,
            opt->js0_btn_zoom_out,
	    opt->js0_btn_hoist_up,
            opt->js0_btn_hoist_down
        );
        PUTCR
        fprintf(fp,
            "Joystick1ButtonMappings = %i %i %i %i %i %i %i",
            opt->js1_btn_rotate,
            opt->js1_btn_air_brakes,
            opt->js1_btn_wheel_brakes,
            opt->js1_btn_zoom_in,
            opt->js1_btn_zoom_out,
            opt->js1_btn_hoist_up,
            opt->js1_btn_hoist_down
        );
        PUTCR
        /* Game controller options. */
        fprintf(fp,
            "GameControllerOptions = %ld",
	    opt->gctl_options
	);
        PUTCR
        PUTCR


        /* Hoist contact expansion coefficient. */
        fprintf(fp,
            "HoistContactExpansionCoefficient = %f",
            opt->hoist_contact_expansion_coeff
	);
	PUTCR
        /* Damage resistance coefficient. */
	fprintf(fp,
            "DamageResistanceCoefficient = %f",
	    opt->damage_resistance_coeff
	);
        PUTCR
        /* Flight physics. */
        fprintf(fp,
            "FlightPhysics = %i",
            opt->flight_physics
	);
        PUTCR
	PUTCR

        /* Last selected mission. */
        fprintf(fp,
            "LastSelectedMission = %i",
            opt->last_selected_mission
        );
        PUTCR
        /* Last selected free flight scene. */
        fprintf(fp,
            "LastSelectedFFScene = %i",
            opt->last_selected_ffscene
        );
        PUTCR
        /* Last selected free flight aircraft. */
        fprintf(fp,
            "LastSelectedFFAircraft = %i",
            opt->last_selected_ffaircraft
        );
        PUTCR
	/* Last selected free flight weather preset. */
	fprintf(fp,
            "LastSelectedFFWeather = %i",
	    opt->last_selected_ffweather
	);
	PUTCR
	/* Last toplevel window position. */
	fprintf(fp,
            "LastToplevelPosition = %i %i %i %i",
	    opt->last_x, opt->last_y, opt->last_width, opt->last_height
	);
        PUTCR




	/* Change permissions so that only owner may read and write. */
#ifdef __MSW__

#else
	if(fchmod(fileno(fp), S_IRUSR | S_IWUSR))
	{
	    fprintf(stderr,
		"%s: Cannot set permissions.\n",
		filename
	    );
	}
#endif

	/* Close file. */
	FClose(fp);

	return(0);
}
