#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/os.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/string.h"

#include "sound.h"
#include "musiclistio.h"
#include "sar.h"
#include "config.h"


static int SARMusicNameToCode(const char *name);
sar_music_ref_struct *SARMusicMatch(
        sar_music_ref_struct **ml, int total,
        int ref_code 
);

void SARMusicListDeleteAll(
        sar_music_ref_struct ***ml, int *total
);
int SARMusicListLoadFromFile(
        const char *filename,
	sar_music_ref_struct ***ml, int *total
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Returns the referance code with respect to the given name.
 *
 *	Can return SAR_MUSIC_CODE_NONE on failed match.
 */
static int SARMusicNameToCode(const char *name)
{
	if(name == NULL)
	    return(SAR_MUSIC_CODE_NONE);

	if(!strcasecmp(name, "Splash"))
	    return(SAR_MUSIC_CODE_SPLASH);
	else if(!strcasecmp(name, "Menus"))
	    return(SAR_MUSIC_CODE_MENUS);
	else
	    return(SAR_MUSIC_CODE_NONE);
/* Need to finish this. */
}

/*
 *	Matches the music file by matching the given ref_code with the 
 *	list of music files' names.
 */
sar_music_ref_struct *SARMusicMatch(
        sar_music_ref_struct **ml, int total,
        int ref_code
)
{
	int i;
	sar_music_ref_struct *mr_ptr;

	if((ml == NULL) || (total < 1))
	    return(NULL);

	for(i = 0; i < total; i++)
	{
	    mr_ptr = ml[i];
	    if(mr_ptr == NULL)
		continue;

	    if(SARMusicNameToCode(mr_ptr->name) == ref_code)
		return(mr_ptr);
	}

	return(NULL);
}


/*
 *      Deletes music files referance list.
 */
void SARMusicListDeleteAll(
        sar_music_ref_struct ***ml, int *total
)
{
        int i;
	sar_music_ref_struct *mr_ptr;


        if((ml == NULL) || (total == NULL))
            return;

        if(option.runtime_debug)
            printf(
"SARMusicListDeleteAll(): Deallocating music file referances.\n"
            );

        for(i = 0; i < (*total); i++)
        {
            mr_ptr = (*ml)[i];
            if(mr_ptr == NULL)
                continue;

	    free(mr_ptr->name);
            free(mr_ptr->filename);
	    free(mr_ptr->next_name);

            free(mr_ptr);
        }

        if((*ml) != NULL)
        {
            free(*ml);
            (*ml) = NULL;
        }
        (*total) = 0;

	return;
}

/*
 *      Loads music file referances list from file.
 *
 *      Given pointer array should be set to valid values. It will be
 *      deallocated first before the file is loaded.
 */
int SARMusicListLoadFromFile(
        const char *filename,
        sar_music_ref_struct ***ml, int *total
)
{
        int i;
        FILE *fp;
        char *buf = NULL;
        struct stat stat_buf;  
        double value[10];
	sar_music_ref_struct *mr_ptr = NULL;


        if((filename == NULL) || (ml == NULL) || (total == NULL))
            return(-1);

        /* Deallocate any existing music referances in the list. */
        SARMusicListDeleteAll(ml, total);

        if(stat(filename, &stat_buf))
        {
            fprintf(stderr, "%s: No such file.\n", filename);
            return(-1);
        }
#ifdef S_ISREG
        if(!S_ISREG(stat_buf.st_mode))
        {
            fprintf(stderr,
                "%s: Not a file.\n",
                filename
            );
            return(-1);
        }
#endif  /* S_ISREG */

        /* Open file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)   
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            return(-1);
        }

        if(option.runtime_debug)
            printf(
"SARMusicListLoadFromFile(): Loading music file referances from %s.\n",
                filename
            );

        do
        {
            buf = FSeekNextParm(
                fp,
                buf,
                SAR_COMMENT_CHAR,
                SAR_CFG_DELIM_CHAR
            );
            if(buf == NULL)
                break;

            if(!strcasecmp(buf, "Version"))
            {
                FGetValuesF(fp, value, 2);

            }

            else if(!strcasecmp(buf, "MusicAdd"))
            {
                char *strptr = FGetString(fp);

                mr_ptr = NULL;


                i = (*total);
                (*total) = (*total) + 1;
                (*ml) = (sar_music_ref_struct **)realloc(
                    *ml,
                    (*total) * sizeof(sar_music_ref_struct *)
                );
                if((*ml) == NULL)
                {
                    (*total) = 0;
                }
                else
                {
                    mr_ptr = (sar_music_ref_struct *)calloc(
                        1, sizeof(sar_music_ref_struct)
                    );
                    (*ml)[i] = mr_ptr;
                }


                /* Set msuic referance code. */
                if(mr_ptr != NULL)
		{
		    free(mr_ptr->name);
		    mr_ptr->name = StringCopyAlloc(strptr);
		}

		free(strptr);
            }
            else if(!strcasecmp(buf, "MusicFileName"))
            {
                char *strptr = FGetString(fp);

                if(mr_ptr != NULL)
                {
                    free(mr_ptr->filename);
                    mr_ptr->filename = strptr;
                }
                else
                {
                    free(strptr);
                }
            }
            else if(!strcasecmp(buf, "MusicFlags"))
            {
		char *strptr = FGetString(fp);
		const char *cstrptr;

		cstrptr = (const char *)strptr;
                while((cstrptr == NULL) ? 0 : ((*cstrptr) != '\0'))
                {
                    if(strcasepfx(cstrptr, "repeating"))
			mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_REPEAT;
		    else if(strcasepfx(cstrptr, "fade_in") ||
                            strcasepfx(cstrptr, "fadein")
		    )
                        mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_FADE_IN;
                    else if(strcasepfx(cstrptr, "fade_out") || 
                            strcasepfx(cstrptr, "fadeout")
                    )
                        mr_ptr->flags |= SAR_MUSIC_REF_FLAGS_FADE_OUT;
		}

		free(strptr);
            }
            else if(!strcasecmp(buf, "MusicNextName"))
            {
                char *strptr = FGetString(fp);

                if(mr_ptr != NULL)
                {
                    free(mr_ptr->next_name);
                    mr_ptr->next_name = strptr;
                }
                else
                {
                    free(strptr); 
                }
            }

            else
            {
                fprintf(stderr,
                    "%s: Unsupported parameter `%s'.\n",
                    filename, buf
                );
                FSeekNextLine(fp);
            }

        } while(1);

        /* Close file. */
        FClose(fp);

	return(0);
}
