/*
  scsiadd - add/remove scsi devices from the linux scsi subsystem
  ----------------------------------------------------------------
  Copyright (C) 2000-2007 by Dirk Jagdmann <doj@cubic.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

//#define DEBUG

/// maximum SCSI ID which is scanned
static int maxid = 15;

/// the modes in which the program should operate
typedef enum
  {
    SINGLE,
    SCAN
  } scsiaddmode_t;

/// action the program should commit
typedef enum
  {
    ADD,
    REMOVE
  } scsiaddaction_t;

/// print the help message and terminate program
static void help()
{
  printf(
	 "scsiadd " VER " - add and remove devices from the scsi subsystem\n"
	 "---------------------------------------------------------------\n"
	 "syntax: scsiadd {-a|-r} <id>\n"
	 "        scsiadd {-a|-r} <id> <lun>\n"
	 "        scsiadd {-a|-r} <host> <id> <lun>\n"
	 "        scsiadd {-a|-r} <host> <channel> <id> <lun>\n"
	 "        scsiadd [-i maxid] -s <host>\n"
	 "        scsiadd [-i maxid] -s <host> <channel>\n"
         "        scsiadd -p\n"
	 "parameters not given are assumed 0\n"
	 "-a: add a device (default if no command given)\n"
	 "-r: remove device\n"
	 "-s: scan for devices\n"
         "-p: print scsi status\n"
         "-h: print this help message\n"
         "-i: maximum SCSI ID which is scanned\n"
         "\nhttp://llg.cubic.org/tools - ftp://ftp.cubic.org/pub/llg\n"
	 );

  exit(1);
}

/**
    check if the calling user is root. If not try to make the program run as root.
    If program can not be made root, terminate.
*/
void becomeRoot()
{
  uid_t me;

  me = getuid();
  if(me!=0)
    {
#ifdef DEBUG
      fprintf(stderr, "Aha, I am not root. Therefore I will now try to become root\n");
#endif
      if(geteuid()!=0)
	{
	  fprintf(stderr,
		  "I can not become root. scsiadd was most likely started by a normal user.\n"
		  "If you want to make this program suid root use make install-suidroot when\n"
		  "installing or (as root) type: chmod +s %s\n",
		  PATH
		  );
	  exit(1);
	}
      setuid(0);
    }
}

/**
    perform the command to add/remove a scsi device.
    @param action action that will be performed. currently supported are ADD and REMOVE.
    @param host host number
    @param channel channel number
    @param id id number
    @param lun lun number
    @return 0 on success. -1 on error
*/
static int scsiadd(const scsiaddaction_t action,
		   const int host,
		   const int channel,
		   const int id,
		   const int lun)
{
  char *actionstr, buf[128];
  int fd, len, ret=0, w;

  switch(action)
    {
    case ADD:
      actionstr="add";
      break;
    case REMOVE:
      actionstr="remove";
      break;
    default:
      fprintf(stderr, "scsiadd:scsiadd(): oops, action error\n");
      return -1;
    }

  snprintf(buf, sizeof(buf), "scsi %s-single-device %i %i %i %i\n", actionstr, host, channel, id, lun);
  len=strlen(buf);

  fd=open("/proc/scsi/scsi", O_WRONLY);
  if(fd < 0)
    {
      perror("scsiadd:scsiadd(): could not open /proc/scsi/scsi (w)");
      return -1;
    }

  errno=0;
  w=write(fd, buf, len);
  /* check if the command did not succeed */
  if(errno && w != len)
    {
      /* ignore errors in IDs > 8 and EINVAL, which are triggered in scan mode on adapters not support Wide adressing */
      if(!(id>=9 && errno==EINVAL))
	{
	  fprintf(stderr, "could not %s device %i %i %i %i : %s\n", actionstr, host, channel, id, lun, strerror(errno));
	  ret=-1;
	}
    }

  if(close(fd) < 0)
    {
#ifdef DEBUG
      perror("scsiadd:scsiadd():could not close /proc/scsi/scsi (w)");
#endif
    }

  return ret;
}

/**
    prints the contents of /proc/scsi/scsi to output stream
    @param os the Output Stream
    @return 0 on success. -1 on error
*/
static int scsidump(FILE *os)
{
  FILE *f=0;

  if(os==NULL)
    return -1;

  f=fopen("/proc/scsi/scsi", "r");
  if(f == NULL)
    {
      perror("scsiadd:scsidump():could not open /proc/scsi/scsi (r)");
      return -1;
    }

  while(!feof(f))
    {
      char buf[129];
      int i=fread(buf, 1, sizeof(buf)-1, f);
      buf[i]=0;
      fprintf(os, "%s", buf);
    }

  if(fclose(f) < 0)
    {
#ifdef DEBUG
      perror("scsiadd:scsidump():could not close /proc/scsi/scsi (r)");
#endif
    }

  return 0;
}

/**
   main function.
   @param argc number of command line items
   @param argv pointer to command line items
   @return 0 on success. 1 on error
*/
int main(int argc, char **argv)
{
  int host=0, channel=0, id=0, lun=0;
  scsiaddmode_t scanmode=SINGLE;
  scsiaddaction_t mode=ADD;
  int ret=0;
  int c=0;
  int i=0;

  // parse command line
  while((c=getopt(argc, argv, "ahi:prs"))!=EOF)
    {
      switch(c)
	{
	case 'a': mode=ADD; break;
	case 'p': scsidump(stdout); return 0;
	case 'r': mode=REMOVE; break;
	case 's': scanmode=SCAN; break;
	case 'i':
	  if(optarg==NULL) break;
	  maxid=strtol(optarg, NULL, 0);
	  if(maxid<0)
	    {
	      fprintf(stderr, "error: max SCSI ID < 0\n");
	      return 1;
	    }
	  break;
	case '?':
	case 'h':
	default: help();
	}
    }

  // convert the supplied numbers
  switch(scanmode)
    {
    case SINGLE:
      switch(argc-optind)
	{
	case 1:
	  id=atoi(argv[optind]);
	  break;
	case 2:
	  id=atoi(argv[optind]);
	  lun=atoi(argv[optind+1]);
	  break;
	case 3:
	  host=atoi(argv[optind]);
	  id=atoi(argv[optind+1]);
	  lun=atoi(argv[optind+2]);
	  break;
	case 4:
	  host=atoi(argv[optind]);
	  channel=atoi(argv[optind+1]);
	  id=atoi(argv[optind+2]);
	  lun=atoi(argv[optind+3]);
	  break;
	default:
	  help();
	}
      break;
    case SCAN:
      switch(argc-optind)
	{
	case 1:
	  host=atoi(argv[optind]);
	  break;
	case 2:
	  host=atoi(argv[optind]);
	  channel=atoi(argv[optind+1]);
	  break;
	}
      break;
    default:
      fprintf(stderr, "scsiadd: oops! scanmode error.\n");
      return 1;
    }

#ifdef DEBUG
  printf("command args: %i %i %i %i %i\n", mode, host, channel, id, lun);
#endif

  becomeRoot();

  switch(scanmode)
    {
    case SINGLE:
      ret=scsiadd(mode, host, channel, id, lun);
      break;
    case SCAN:
      for(i=0; i<maxid; i++)
	{
#ifdef DEBUG
	  printf("scan: %i %i %i %i %i\n", mode, host, channel, id, lun);
#endif
	  ret=scsiadd(mode, host, channel, i, lun);
	}
      break;
    default:
      fprintf(stderr, "scsiadd: oops! scanmode error\n");
      return 1;
    }

  if(ret>=0)
    ret=scsidump(stdout);

  return -ret;
}
