/*  Screem:  spell.c,
 *  handles spell checking of pages, no longer uses the Gnome spell check
 *  widget as it is going to be removed from gnome-libs 2.0
 *
 *  Copyright (C) 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>

#include "editor.h"
#include "site.h"
#include "page.h"
#include "preferences.h"

extern GtkWidget *app;
extern Site *current_site;
extern Preferences *cfg;

#define ACCEPT_BUTTON 2
#define SKIP_BUTTON 3
#define REPLACE_BUTTON 4
#define INSERT_BUTTON 5
#define INSERT_CASE_BUTTON 6

typedef struct Word {
	gchar *word;
	gint pos;
} Word;

static GList *words;

static GList* split_text( gchar *text ); /* returns a list of the individual
					    words, ignoring HTML tags */

GladeXML *xml;

static void start_spell_check( GtkWidget *dialog, gint in, gint out );
void spell_check( void );


void spell_check()
{
     	static GtkWidget *dialog = NULL;
	Page *page;

	int p[ 2 ];
	int q[ 2 ];

	pid_t pid;
	gint status;

	gchar *cmd = "ispell";
	gchar *args[16] = {
		"screem-spellchecker",
		"-a",
		NULL
	};
	
	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );
      
	/* try and fork an ispell process */
	pipe( p );
	pipe( q );
	switch( ( pid = fork() ) ) {
	case -1:  /* error */
		break;
	case 0: /* child */
		close( 0 );
		dup( p[ 0 ] );
		close( p[ 1 ] );
		close( 1 );
		dup( q[ 1 ] );
		close( q[ 0 ] );
		execvp( cmd, args );
		_exit( 1 );
		break;
	default:  /* parent */
		close( p[ 0 ] );
		close( q[ 1 ] );

		/* need to split p->data up into individual words, 
		   ignoring tags, although it needs to check things like alt 
		   attributes inside tags */
		screem_editor_buffer_text();
		words = split_text( screem_page_get_data( page ) );

		xml = glade_xml_new( cfg->glade_path, "spellcheck_dialog" );
		dialog = glade_xml_get_widget( xml, "spellcheck_dialog" );

		glade_xml_signal_autoconnect( xml );
		
		start_spell_check( dialog, q[ 0 ], p[ 1 ] );

		/* kill ispell as it should still be there */
		kill( pid, SIGTERM );

		waitpid( pid, &status, 0 );

		gtk_widget_destroy( dialog );

		close( q[ 0 ] );
		close( p[ 1 ] );
		break;
	}
}

/* splits text up into its individual words, ignoring html tags,
   FIXME: should really handle any text within tag attributes such as alt
   in an <img> tag */
static GList* split_text( gchar *text )
{
	GList *words;
	GList *list;
	gchar c;
	gchar *start;
	gchar *word;
	gint len;
	gint i;
	gint wordLen;
	Word *w;

	words = g_list_alloc();
	list = words;
	len = strlen( text );

       	for( i = 0, start = text; i < len; i ++ ) {
		c = *( text + i );
		if( ( c == ' ' ) || ( c == '<' ) || ( c == '\n' ) ||
		    ( c == '&' ) || ( c == ';' ) || ( c == '/' ) ||
		    ( c == '.' ) ) {
			if( start != text + i ) {
				/* we have come to the end of a word */
				wordLen = text + i - start;
				word = g_strndup( start, wordLen );
				w = g_new( Word, 1 );
				w->word = word;
				w->pos = start - text;
				list->data = w;
				g_list_append( words, NULL );
				list = list->next;
			}
			start = text + i + 1;
		}

		/* stopped at the start of a tag, advance forward until we
		   hit the end of the tag */
		if( c == '<' ) {
			while( ( c != '>' ) && ( i < len ) ) {
				i ++;
				c = *( text + i );
			}
			start = text + i + 1;
		} else if( c == '&' ) {
			/* we stopped at a char encoding element */
			while( ( c != ';' ) && ( i < len ) ) {
				i ++;
				c = *( text + i );
			}
			start = text + i + 1;
		}
	}

	return words;
}

static void start_spell_check( GtkWidget *dialog, gint in, gint out )
{
	Word *word;
	gboolean result = FALSE;
	gchar buffer[ BUFSIZ + 1 ];
	gchar *tbuf;
	gchar *temp;
	gint size;

	GtkWidget *word_entry;
	GtkWidget *alt_list;

	GList *alts = NULL;
	GList *list;
	gchar *item[ 2 ] = { NULL, NULL };

	gchar c;
	gint num;
	gchar *curword;

	gint button;
	gboolean replace;

	gint len;
	gint offset = 0;

	word_entry = glade_xml_get_widget( xml, "word" );
	alt_list = glade_xml_get_widget( xml, "alt_list" );

	/* get rid of the ispell header thingy */
	if( read( in, buffer, BUFSIZ ) < 0 ) {
		perror( "read" );
		return;
	}

	for( ;words->data; words = words->next, alts = NULL, result = FALSE ) {
		word = words->data;

		/* send word to ispell */
		temp = g_strconcat( "^", word->word, "\n", NULL );

		if( write( out, temp, strlen( temp ) ) < 0 ) {
			perror( "write" );
			g_free( temp );
			break;
		}

		g_free( temp );

		/* now read the result */
		if( ( size = read( in, buffer, BUFSIZ ) ) < 0 ) {
			perror( "read" );
			break;
		}
		buffer[ size ] = '\0';
		tbuf = NULL;

		switch( buffer[ 0 ] ) {
		case '*':
			/* word is ok */
			break;
		case '+':
			/* word found */
			break;
		case '-':
			/* found as compound */
			break;
		case '\n':
			/* ignore the word */
			break;
		case '#':
			/* not found, no idea what it is */
			result = TRUE;
			break;
		case '?':
			/* not found, but we have guesses */
		case '&':
			/* not found, but here is some near misses */
			result = TRUE;

			/* format:
			   <symbol> <word> <number> <word number>: <word>, ...
			 */
			tbuf = temp = g_malloc( BUFSIZ );
			sscanf( buffer, "%c %s %d", &c, temp, &num );
			temp = strchr( buffer, ':' );
			temp += 2;

			/* temp points to the next word */
			while( num ) {
				curword = temp;
				temp = strchr( curword, ',' );
				if( temp ) {
					*temp = '\0';
					temp += 2;
				}
				alts = g_list_append( alts, curword );
				num --;
			}
			break;
		default:
			/* probably a number */
			break;
		}

		if( result ) {
			/* we need to interact with the user, first
			   fill in the fields in the dialog */
			gtk_entry_set_text( GTK_ENTRY( word_entry ),
					    word->word );
			gtk_clist_clear( GTK_CLIST( alt_list ) );
			for( list = alts; list; list = list->next ) {
				item[ 0 ] = list->data;
				gtk_clist_append( GTK_CLIST( alt_list ),
						  item );
			}

			len = strlen( word->word );

			screem_editor_set_pos( word->pos + offset );
			screem_editor_select_region( word->pos + offset,
						     word->pos + offset + 
						     len );

			button = gnome_dialog_run( GNOME_DIALOG( dialog ) );
			c = '\0';
			replace = FALSE;
			switch( button ) {
				/* ok what to do */
			case ACCEPT_BUTTON:
				/* ispell @ command */
				c = '@';
				break;
			case REPLACE_BUTTON:
				replace = TRUE;
				break;
			case INSERT_BUTTON:
				/* ispell * command + replace */
				c = '+';
				replace = TRUE;
				break;
			case INSERT_CASE_BUTTON:
				/* ispell & command + replace */
				c = '&';
				replace = TRUE;
				break;
			case SKIP_BUTTON:
				/* do nothing */
			default:
				break;
			}
			curword = gtk_entry_get_text( GTK_ENTRY(word_entry) );
			if( c != '\0' ) {
				/* FIXME: remove command characters from 
				   curword */

				temp = g_strdup_printf( "%c%s\n", c,curword );
				write( out, temp, strlen( temp ) );
				g_free( temp );
			}
			if( replace ) {
				/* replace the word in the editor at
				   position word->pos + offset */
				gint pos = word->pos + offset;
				screem_editor_delete_forward( pos, len );
				screem_editor_insert( pos, curword );
				offset += strlen( curword ) - len;
				/* FIXME: why do we loose the next selection
				   after we have done the above? its something
				   to do with the order events are being
				   processed in, as if we wait for a while
				   it stays, the colour wizard has the
				   same problem */
			}
			g_list_free( alts );
		} else
			button = -1;

		g_free( tbuf );

		if( button == 0 )
			break;
	}
}

void spell_check_select_replacement( GtkCList *clist, gint row, gint col,
				     GdkEventButton *event, gpointer data )
{
	gchar *info;
	GtkWidget *word_entry;

	gtk_clist_get_text( clist, row, 0, &info );
	word_entry = glade_xml_get_widget( xml, "word" );
	gtk_entry_set_text( GTK_ENTRY( word_entry ), info );
}

void spell_check_button( GtkWidget *widget, gchar *type )
{
	GtkWidget *dialog;
	gint button = -1;

	dialog = glade_xml_get_widget( xml, "spellcheck_dialog" );

	if( ! strcmp( "accept", type ) )
		button = ACCEPT_BUTTON;
	else if( ! strcmp( "skip", type ) )
		button = SKIP_BUTTON;
	else if( ! strcmp( "replace", type ) )
		button = REPLACE_BUTTON;
	else if( ! strcmp( "insert", type ) )
		button = INSERT_BUTTON;
	else if( ! strcmp( "insert case", type ) )
		button = INSERT_CASE_BUTTON;

	gtk_signal_emit_by_name( GTK_OBJECT( dialog ), "clicked",
				 button, NULL );
}
