/*  Screem:  pageDruid.c,
 *  A page druid for guiding users through the process of creating a page
 *
 *  Copyright (C) 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <dirent.h>

#include <glade/glade.h>

#ifdef HAVE_GTKHTML
#include <gtkhtml/gtkhtml.h>
#endif

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "page.h"
#include "pageUI.h"
#include "site.h"
#include "htmlfuncs.h"
#include "fileops.h"
#include "siteTreeUI.h"
#include "preferences.h"
#include "dtd.h"

static GladeXML *xml;
static Page *page;
static gboolean block;

extern Site *current_site;
extern Preferences *cfg;

typedef struct PageDetails {
	gchar *path;
	gchar *stylesheet;
	gchar *doctype;
	
	gchar *title;
	gchar *author;
	gchar *charset;
	gchar *keywords;

	gboolean prepared;
	gchar *template;

	gboolean overide;
	gboolean is_frameset;
	guint8 bg[ 4 ]; guint8 t[ 4 ]; guint8 l[ 4 ]; guint8 v[ 4 ]; 
	guint8 a[ 4 ];
	gchar *image;
} PageDetails;

static PageDetails *details = NULL;

void screem_page_druid_canceled( GtkWidget *widget );
void screem_page_druid_finish( GnomeDruidPage *widget, GtkWidget *druid );
gboolean screem_page_druid_details( GnomeDruidPage *druid_page,
				    GtkWidget *druid );
gboolean screem_page_druid_templates( GnomeDruidPage *druid_page,
				      GtkWidget *druid );
gboolean screem_page_druid_colours( GnomeDruidPage *druid_page,
				    GtkWidget *druid );
void screem_page_druid_details_prepare( GnomeDruidPage *druid_page, 
					GtkWidget *druid );
void screem_page_druid_templates_prepare( GnomeDruidPage *druid_page, 
					  GtkWidget *druid );
void screem_page_druid_template_standard( GtkToggleButton *toggle );
void screem_page_druid_template_custom( GtkToggleButton *toggle );
GtkWidget *screem_page_druid_preview( void );


/**
 * screem_page_druid:
 * @page:  the page struct to use
 *
 * opens the page druid dialog
 *
 * return values: none
 */
void screem_page_druid( Page *p )
{
	xml = glade_xml_new( cfg->glade_path, "page_druid" );
	glade_xml_signal_autoconnect( xml );

	page = p;
	details = g_new0( PageDetails, 1 );
	block = FALSE;
}

void screem_page_druid_canceled( GtkWidget *widget )
{
	GtkWidget *window;

	window = glade_xml_get_widget( xml, "page_druid" );

	gtk_widget_destroy( window );
	screem_page_destroy( page );
}

void screem_page_druid_finish( GnomeDruidPage *widget, GtkWidget *druid )
{
	const gchar *data;
	GNode *node;
	GNode *root;
	GNode *html;
	GNode *head;
	GNode *title;
	gchar *temp;
	gchar *temp2;
	GtkWidget *window;
	ScreemDTD *dtd;
	
	window = glade_xml_get_widget( xml, "page_druid" );

	/* all details gathered, build page */
	screem_page_set_pathname( page, details->path );

	if( details->template ) {
		if( ! copy_file( details->template, 
				 screem_page_get_pathname( page ) ) )  {
			screem_page_destroy( page );
			return;
		}
		/* succesfuly copying, load the template */
		screem_page_load( page );

		/* we now need to edit the page and fill in the details the 
		   user gave */
		data = screem_page_get_data( page );
		dtd = screem_get_doctype( data );
		node = root = screem_html_build_parse_tree( dtd, data, 0 );

		html = screem_html_parse_tree_find_node( node, "html" );
		head = screem_html_parse_tree_find_node( node, "head" );
		title = screem_html_parse_tree_find_node( node, "title" );

		screem_html_destroy_parse_tree( root );
		g_free( details->template );
	} else {
		if( strlen( details->doctype ) != 0 )
			temp = g_strconcat( details->doctype,  
					    "\n<html>\n<head>\n<title>",
					    details->title,
					    "</title>\n", NULL );
		else
			temp = g_strconcat( "<html>\n<head>\n<title>",
					    details->title, "%s</title>\n", 
					    NULL );
		if( strlen( details->author ) != 0 ) {
			temp2 = temp;
			temp = g_strconcat( temp2, 
					    "<meta name=\"author\" content=\"",
					    details->author, "\">\n", NULL );
			g_free( temp2 );
		}
		if( strlen( details->charset ) != 0 ) {
			temp2 = temp;
			temp = g_strconcat(temp2, 
					   "<meta name=\"charset\" content=\"",
					   details->charset, "\">\n", NULL );
			g_free( temp2 );
		}
		if( strlen( details->keywords ) != 0 ) {
			temp2 = temp;
			temp = g_strconcat(temp2,
					   "<meta name=\"keywords\" content=\""
					   , details->keywords, "\">\n", NULL);
			g_free( temp2 );
		}
		temp2 = temp;
		temp = g_strconcat(temp2, 
				   "<meta name=\"generator\" content=\"Screem "
				   , VERSION, "\">\n", NULL);
		g_free( temp2 );
		if( strlen( details->stylesheet ) != 0 ) {
			temp2 = temp;
			temp = g_strconcat( temp2,
					    "<link rel=\"stylesheet\" href=\"",
					    details->stylesheet, 
					    "\" type=\"text/css\">\n", NULL );
			g_free( temp2 );
		}
		temp2 = temp;
		if( details->is_frameset )
			temp = g_strconcat( temp2, 
					    "</head>\n<noframes>\n<body",
					    NULL );
		else
			temp = g_strconcat( temp2, "</head>\n<body", NULL );
		g_free( temp2 );

		if( strlen( details->image ) != 0 ) {
			temp2 = temp;
			temp = g_strconcat( temp, " background=\"", 
					    details->image, "\"", NULL );
			g_free( temp2 );
		}
		if( details->bg[ 0 ] ) {
			temp2 = temp;
			temp = g_strdup_printf( "%s bgcolor=\"#%.2x%.2x%.2x\"",
						temp,
						details->bg[ 1 ],
						details->bg[ 2 ], 
						details->bg[ 3 ] );
			g_free( temp2 );
		}
		if( details->t[ 0 ] ) {
			temp2 = temp;
			temp = g_strdup_printf( "%s text=\"#%.2x%.2x%.2x\"",
						temp,
						details->t[ 1 ], 
						details->t[ 2 ], 
						details->t[ 3 ] );
			g_free( temp2 );
		}
		if( details->l[ 0 ] ) {
			temp2 = temp;
			temp = g_strdup_printf( "%s link=\"#%.2x%.2x%.2x\"",
						temp,
						details->l[ 1 ],
						details->l[ 2 ], 
						details->l[ 3 ] );
			g_free( temp2 );
		}
		if( details->v[ 0 ] ) {
			temp2 = temp;
			temp = g_strdup_printf( "%s vlink=\"#%.2x%.2x%.2x\"",
						temp,
						details->v[ 1 ],
						details->v[ 2 ], 
						details->v[ 3 ] );
			g_free( temp2 );
		}
		if( details->a[ 0 ] ) {
			temp2 = temp;
			temp = g_strdup_printf( "%s alink=\"#%.2x%.2x%.2x\"",
						temp,
						details->a[ 1 ],
						details->a[ 2 ], 
						details->a[ 3 ] );
			g_free( temp2 );
		}
		temp2 = temp;
		if( details->is_frameset )
			temp = g_strconcat(temp2, 
					   ">\n</noframes>\n</body>\n</html>\n"
					   , NULL );
		else
			temp = g_strconcat( temp2, ">\n\n</body>\n</html>\n",
					    NULL );
		g_free( temp2 );
		screem_page_set_data( page, temp );
		g_free( temp );
	}

	/* save the page */
	screem_page_set_changed( page, TRUE );
	if( ! screem_page_save( page ) ) {
		screem_page_destroy( page );
		return;
	}

	temp = g_strdup( screem_page_get_data( page ) );
	screem_page_destroy( page );

	/* is there already a page with that name in the site? */
	if( !(page = screem_site_locate_page( current_site, details->path )) ){
		screem_site_add_page( current_site, details->path );
		page = screem_site_locate_page( current_site, details->path );
	} else if( page == screem_site_get_current_page( current_site ) ){
		screem_page_set_data( page, temp );
	} else {
		screem_page_set_data( page, temp );
	}
	g_free( temp );
	page = screem_site_locate_page( current_site, details->path );
	refresh_file_tree();
	screem_page_insert( page );

	gtk_widget_destroy( window );
}

gboolean screem_page_druid_details( GnomeDruidPage *druid_page,
				    GtkWidget *druid )
{
	GtkWidget *widget;

	/* get the basic details */
	widget = glade_xml_get_widget( xml, "page_path" );
	details->path = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_style" );
	details->stylesheet = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_doctype" );
	details->doctype = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* details->doctype is the type, we want the complete DTD,
	   lets do a lookup */
	details->doctype = screem_dtd_build_doctype_tag( details->doctype );

	/* get meta information */
	widget = glade_xml_get_widget( xml, "page_title" );
	details->title = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_author" );
	details->author = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_charset" );
	details->charset = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_keywords" );
	details->keywords = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* must have a path and title */
	if( ( strlen( details->path ) == 0 ) || 
	    ( strlen( details->title ) == 0 ) ) {
		glade_xml_new( cfg->glade_path, "pmessage1" );
		return TRUE;
	}

	/* path may already exist, confirm overwrite */
	return (! overwrite( details->path ) );
}

gboolean screem_page_druid_templates( GnomeDruidPage *druid_page,
				      GtkWidget *druid )
{
	GtkWidget *widget;
	GtkWidget *widget2;
	const gchar *tpath;

	widget = glade_xml_get_widget( xml, "use_standard" );
	widget2 = glade_xml_get_widget( xml, "use_custom" );

	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		/* standard template is in use */
		widget = glade_xml_get_widget( xml, "template_tree" );
		tpath = RESOURCE_PATH"Templates";
		details->template = g_strconcat( tpath, G_DIR_SEPARATOR_S, 
						 details->template, NULL );
		

 	} else if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget2)) ) {
		/* custom template */
		widget = glade_xml_get_widget( xml, "custom_template" );
		details->template = gtk_entry_get_text( GTK_ENTRY( widget ) );
		details->template = g_strdup( details->template );
	}

	return FALSE;
}

gboolean screem_page_druid_colours( GnomeDruidPage *druid_page,
				    GtkWidget *druid )
{
	GtkWidget *widget;
	gint8 junk;   /* not interested in the alpha value from the
			  colour picker */

	widget = glade_xml_get_widget( xml, "overide_template" );
	details->overide = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "is_frameset" );
	details->is_frameset = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "use_bgcolor" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "bgcolor" );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
					   &details->bg[ 1 ],&details->bg[ 2 ],
					   &details->bg[ 3 ], &junk );
		details->bg[ 0 ] = 1;
	} else 
		details->bg[ 0 ] = 0;

	widget = glade_xml_get_widget( xml, "use_text" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "text" );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
					   &details->t[ 1 ], &details->t[ 2 ], 
					   &details->t[ 3 ], &junk );
		details->t[ 0 ] = 1;
	} else 
		details->t[ 0 ] = 0;

	widget = glade_xml_get_widget( xml, "use_link" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "link" );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
					   &details->l[ 1 ], &details->l[ 2 ], 
					   &details->l[ 3 ], &junk );
		details->l[ 0 ] = 1;
	} else 
		details->l[ 0 ] = 0;

	widget = glade_xml_get_widget( xml, "use_vlink" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "vlink" );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
					   &details->v[ 1 ], &details->v[ 2 ], 
					   &details->v[ 3 ], &junk );
		details->v[ 0 ] = 1;
	} else 
		details->v[ 0 ] = 0;

	widget = glade_xml_get_widget( xml, "use_alink" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "alink" );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
					   &details->a[ 1 ], &details->a[ 2 ], 
					   &details->a[ 3 ], &junk );
		details->a[ 0 ] = 1;
	} else 
		details->a[ 0 ] = 0;

	widget = glade_xml_get_widget( xml, "background" );
	details->image = gtk_entry_get_text( GTK_ENTRY( widget ) );

	return FALSE;
}

void screem_page_druid_details_prepare( GnomeDruidPage *druid_page, 
					GtkWidget *druid )
{
	GtkWidget *widget;

	/* put in the users name as the author */
	widget = glade_xml_get_widget( xml, "page_author" );

	/* put the DTD types in */
	widget = glade_xml_get_widget( xml, "dtd_combo" );
	screem_dtd_fill_combo( widget );
	widget = glade_xml_get_widget( xml, "page_doctype" );
	gtk_entry_set_text( GTK_ENTRY( widget ), cfg->mpage->default_dtd );

	/* put in the default page charset */
	widget = glade_xml_get_widget( xml, "page_charset" );
	gtk_entry_set_text( GTK_ENTRY( widget ), cfg->mpage->default_charset );
}

void screem_page_druid_templates_prepare( GnomeDruidPage *druid_page, 
					  GtkWidget *druid )
{
	/* if we are working on a site and the site has a default
	   template then set to use custom template and set the
	   template path to be the sites default template */
	const gchar *path;
	GtkWidget *widget;

	const gchar *tpath = RESOURCE_PATH"Templates";
	struct dirent **namelist;
	const gchar *mime_type;
	gint num;
	gint i;

	gchar *item[ 2 ] = { NULL, NULL };
	GtkCTreeNode *sibling = NULL;
	GString *template_file;
	GString *template_info;
	gchar *temp;

	/* we want to scan RESOURCE_PATH/Templates/ for possible page templates */
	widget = glade_xml_get_widget( xml, "template_tree" );
	gtk_clist_clear( GTK_CLIST( widget ) );

	num = scandir( tpath, &namelist, 0, alphasort );

	for( i = 0; i < num; i ++ ) {
		path = namelist[ i ]->d_name;
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type( path );
#else
		mime_type = gnome_vfs_mime_type_from_name( path );
#endif
		/* we support subdirectories so templates can be better
		   organised */
		if( g_file_test( path, G_FILE_TEST_ISDIR) ) {
			/* it is a directory, not yet supported */

			g_free( namelist[ i ] );
			continue;
		}
		/* html templates (and I suppose we had better do xml as well)
		   only */
		if( strcmp( "text/html", mime_type ) &&
		    strcmp( "text/xml", mime_type ) ) {
			g_free( namelist[ i ] );
			continue;
		}
		/* we need to load the template file so we can get the
		   template information */
		template_file = load_text_file_with_dir( tpath, 
							 namelist[ i ]->d_name );
		if( ! strncmp( "<!-- Screem Template -->",
			       template_file->str,
			       strlen( "<!-- Screem Template -->" ) ) ) {
			/* it is a valid screem template, get the info
			   comment */
			temp = template_file->str + 
				strlen( "<!-- Screem Template -->" );
			/* a bit arbitrary, assume next comment is the template
			   info */
			temp = strstr( temp, "<!--" );

			/* if ! temp then we have no template info, I suppose
			   we can allow this */
			if( temp ) {
				temp += strlen( "<!--" );
				template_info = g_string_new( "" );
				while( *temp != '\0' ) {
					if( !strncmp( temp,"-->",strlen( "-->" ) ) )
						break;
					g_string_append_c( template_info, *temp );
					temp ++;
				}
			} else {
				template_info = g_string_new(_( "No information" ));
			}
			/* add node to the tree */
			item[ 0 ] = namelist[ i ]->d_name;
			sibling = gtk_ctree_insert_node( GTK_CTREE( widget ), NULL,
							 sibling, item, 3,
							 NULL, NULL, NULL, NULL,
							 TRUE, FALSE );	

			/* set the template info as row data */
			gtk_ctree_node_set_row_data_full( GTK_CTREE( widget ),
							  sibling,
							  template_info->str,
							  (GtkDestroyNotify)g_free);

			g_string_free( template_info, FALSE );
		}
		
		g_string_free( template_file, TRUE );
		g_free( namelist[ i ] );
	}

	if( details->prepared )
		return;

	path = screem_site_get_template_path( current_site );
	if( path ) {
		widget = glade_xml_get_widget( xml, "use_custom" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      TRUE );
		widget = glade_xml_get_widget( xml, "custom_template" );
		gtk_entry_set_text( GTK_ENTRY( widget ), path );
	}

	details->prepared = TRUE;
}

void screem_page_druid_template_select( GtkCTree *tree, GtkCTreeNode *row,
					gint column )
{
	gchar *data = gtk_ctree_node_get_row_data( tree, row );
	GtkWidget *label = glade_xml_get_widget( xml, "template_info" );

	gtk_ctree_node_get_pixtext( tree, row, column, &details->template,
				    NULL, NULL, NULL );

	gtk_label_set( GTK_LABEL( label ), data );
}

void screem_page_druid_template_standard( GtkToggleButton *toggle )
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( xml, "standard_template_frame" );
	gtk_widget_set_sensitive( widget,
				  gtk_toggle_button_get_active( toggle ) );

	if( ! block ) {
		block = TRUE;
		widget = glade_xml_get_widget( xml, "use_custom" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
					      FALSE );
		block = FALSE;
	}
}

void screem_page_druid_template_custom( GtkToggleButton *toggle )
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( xml, "custom_template_frame" );
	gtk_widget_set_sensitive( widget,
				  gtk_toggle_button_get_active( toggle ) );

	if( ! block ) {
		block = TRUE;
		widget = glade_xml_get_widget( xml, "use_standard" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		block = FALSE;
	}
}

GtkWidget *screem_page_druid_preview()
{
	GtkWidget *widget;
	GtkWidget *sw;
	const gchar *text = "\
<body bgcolor=\"black\" text=\"white\" link=\"orange\">\
Here is some text\
<br>\
<a href=\"http://www.screem.org/\">Here is a link</a>\
<br>\
</body>";

#ifndef HAVE_GTKHTML
	widget = gtk_label_new( "GtkHTML widget will go here" );
	return widget;
#else
        GtkHTMLStream *handle;

	widget = gtk_html_new();
	sw = gtk_scrolled_window_new( GTK_LAYOUT( widget )->hadjustment,
                                      GTK_LAYOUT( widget )->vadjustment );
        gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ), 
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );

        gtk_container_add( GTK_CONTAINER( sw ), widget );

	handle = gtk_html_begin( GTK_HTML( widget ) );
        gtk_html_write( GTK_HTML( widget ), handle, text, strlen( text ) );
        gtk_html_end( GTK_HTML( widget ), handle, GTK_HTML_STREAM_OK );

	gtk_widget_show( widget );

	return sw;
#endif
}
