/*  Screem:  editor_guile.c,
 *  Guile glue functions for accessing the screem editor
 * 
 *  Copyright (C) 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#ifdef HAVE_GUILE
#include <guile/gh.h>
#endif

#include <gnome.h>

#include "editor.h"
#include "editor_guile.h"

/* "scheme name", C function name, required args, optional args,
   takes rest list */
#ifdef HAVE_GUILE
static gpointer editor_procedures[] = {
	"screem-editor-insert", (gpointer)scm_screem_editor_insert,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-clear", (gpointer)scm_screem_editor_clear,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-get-length", (gpointer)scm_screem_editor_get_length,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-get-text", (gpointer)scm_screem_editor_get_text,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-has-selection", 
	(gpointer)scm_screem_editor_has_selection,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-select-region", 
	(gpointer)scm_screem_editor_select_region,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-cut", (gpointer)scm_screem_editor_cut,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-copy",(gpointer)scm_screem_editor_copy,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-paste",(gpointer)scm_screem_editor_paste, 
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-clear-selection", 
	(gpointer)scm_screem_editor_clear_selection,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-set-pos",(gpointer)scm_screem_editor_set_pos,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-get-pos",(gpointer)scm_screem_editor_get_pos, 
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-delete-forward", 
	(gpointer)scm_screem_editor_delete_forward,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-enable", (gpointer)scm_screem_editor_enable,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-disable", (gpointer)scm_screem_editor_disable,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-buffer-text", (gpointer)scm_screem_editor_buffer_text, 
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-insert-markup", 
	(gpointer)scm_screem_editor_insert_markup,
	GINT_TO_POINTER( 2 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-insert-attribute", 
	(gpointer)scm_screem_editor_insert_attribute,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-insert-file", (gpointer)scm_screem_editor_insert_file,
	GINT_TO_POINTER( 1 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-undo", (gpointer)scm_screem_editor_undo,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-redo", (gpointer)scm_screem_editor_redo,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	"screem-editor-goto-line", (gpointer)scm_screem_editor_goto_line,
	GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ), GINT_TO_POINTER( 0 ),
	NULL
};
#endif

void screem_editor_init_guile()
{
	gint i;
#ifdef HAVE_GUILE
      	for( i = 0; editor_procedures[ i ]; i += 5 ) {
		gh_new_procedure( editor_procedures[ i ],
				  editor_procedures[ i + 1 ],
				  (gint)editor_procedures[ i + 2 ],
				  (gint)editor_procedures[ i + 3 ],
				  (gint)editor_procedures[ i + 4 ] );
	}
#endif
}

#ifdef HAVE_GUILE
SCM scm_screem_editor_insert( SCM pos, SCM text )
{
	gint position;
	gchar *t;
	gint len;

	position = gh_scm2int( pos );
	t = gh_scm2newstr( text, &len );

	screem_editor_insert( position, t );

	return SCM_EOL;
}

SCM scm_screem_editor_clear()
{
	screem_editor_clear();

	return SCM_EOL;
}

SCM scm_screem_editor_get_length()
{
	gint len;

	len = screem_editor_get_length();

	return gh_int2scm( len );
}

SCM scm_screem_editor_get_text( SCM from, SCM len )
{
	gint f;
	gint l;
	gchar *text;

	f = gh_scm2int( from );
	l = gh_scm2int( len );

	text = screem_editor_get_text( f, l );

	return gh_str2scm( text, strlen( text ) );
}

/* how do we do the return pointer parameters? */
SCM scm_screem_editor_has_selection()
{
	gint start;
	gint end;
	gboolean ret;

	ret = screem_editor_has_selection( &start, &end );

	return gh_bool2scm( ret ); 
}

SCM scm_screem_editor_select_region( SCM start, SCM len )
{
	gint s;
	gint l;

	s = gh_scm2int( start );
	l = gh_scm2int( len );

	screem_editor_select_region( s, l );

	return SCM_EOL;
}

SCM scm_screem_editor_cut()
{
	screem_editor_cut();

	return SCM_EOL;
}

SCM scm_screem_editor_copy()
{
	screem_editor_copy();

	return SCM_EOL;
}

SCM scm_screem_editor_paste()
{
	screem_editor_paste();

	return SCM_EOL;
}

SCM scm_screem_editor_clear_selection()
{
	screem_editor_clear_selection();

	return SCM_EOL;
}

SCM scm_screem_editor_set_pos( SCM pos )
{
	gint p;

	p = gh_scm2int( pos );

	screem_editor_set_pos( p );

	return SCM_EOL;
}

SCM scm_screem_editor_get_pos()
{
	gint p;

	p = screem_editor_get_pos();

	return gh_int2scm( p );
}

SCM scm_screem_editor_delete_forward( SCM pos, SCM len )
{
	gint p;
	gint l;

	p = gh_scm2int( pos );
	l = gh_scm2int( len );

	screem_editor_delete_forward( p, l );

	return SCM_EOL;
}


SCM scm_screem_editor_enable()
{
	screem_editor_enable();

	return SCM_EOL;
}

SCM scm_screem_editor_disable()
{
	screem_editor_disable();

	return SCM_EOL;
}

SCM scm_screem_editor_buffer_text()
{
	screem_editor_buffer_text();

	return SCM_EOL;
}

SCM scm_screem_editor_insert_markup( SCM open_element, SCM close_element )
{
	gchar *open;
	gchar *close;
	gint len;

	open = gh_scm2newstr( open_element, &len );
	close = gh_scm2newstr( close_element , &len );

	screem_editor_insert_markup( open, close );

	return SCM_EOL;
}

SCM scm_screem_editor_insert_attribute( SCM attribute )
{
	gchar *attr;
	gint len;

	attr = gh_scm2newstr( attribute, &len );

	screem_editor_insert_attribute( attr );

	return SCM_EOL;
}

SCM scm_screem_editor_insert_file( SCM filename )
{
	gchar *file;
	gint len;

	file = gh_scm2newstr( filename, &len );

	screem_editor_insert_file( file );

	return SCM_EOL;
}

SCM scm_screem_editor_undo()
{
	screem_editor_undo();

	return SCM_EOL;
}

SCM scm_screem_editor_redo()
{
	screem_editor_redo();

	return SCM_EOL;
}

SCM scm_screem_editor_goto_line()
{
	screem_editor_goto_line();

	return SCM_EOL;
}

#endif
