/*  Screem:  cvs.c,
 *  CVS features for Screem
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <gnome.h>

#include "cvs.h"
#include "editor.h"
#include "fileops.h"
#include "page.h"
#include "pageUI.h"
#include "site.h"
#include "siteTreeUI.h"
#include "siteUI.h"
#include "support.h"

extern Site *current_site;

static gboolean cvs_flags_dialog( gchar **extra_flags, gchar **log_text );

static GtkWidget *settings_dialog( gboolean message );
static GtkWidget *checkout_dialog( void );

gboolean cvs_delete_with_filename( gchar *path )
{
	Site *site;
   	const gchar *cvs_root;
	gchar *command;

	gchar *extra_flags;
	gboolean ret;

	site = current_site;

	g_return_val_if_fail( site != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, NULL ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s delete %s %s 2>&1",
				   cvs_root, extra_flags, path );
	
	ret = execute_command( command );
	g_free( command );

	return (ret == 0);
}

gboolean cvs_add_with_filename( gchar *path )
{
	Site *site;
   	const gchar *cvs_root;
	gchar *command;

	gchar *log_text;
	gchar *extra_flags;
	gboolean ret;

	site = current_site;

	g_return_val_if_fail( site != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, &log_text ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s add %s -m \"%s\" %s 2>&1",
				   cvs_root, extra_flags, log_text, path );
	
	ret = execute_command( command );
	g_free( command );

	return (ret==0);
}

gboolean cvs_update_page()
{
	Site *site;
	Page *page;
	const gchar *cvs_root;
	const gchar *path;
	gchar *command;

  	gchar *extra_flags;
	gboolean ret;

	site = current_site;

	page = screem_site_get_current_page( site );
	
	g_return_val_if_fail( page != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );
	path = screem_page_get_pathname( page );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, NULL ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s update %s %s 2>&1",
				   cvs_root, extra_flags, path );

	ret = execute_command( command );

	g_free( command );

	/* reload the page now */
	if( ret == 0 ) {
		screem_page_revert( page );  /* does a reload */
		screem_page_insert( page );
	}

	return (ret==0) ;
}

gboolean cvs_commit_page()
{
	Site *site;
	Page *page;
	const gchar *cvs_root;
	const gchar *path;
	gchar *command;

	gchar *log_text;
	gchar *extra_flags;
	gboolean ret;

	site = current_site;

	page = screem_site_get_current_page( site );
	
	g_return_val_if_fail( page != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );
	path = screem_page_get_pathname( page );

	/* saves the page prior to commiting */
	screem_editor_buffer_text();
	screem_page_save( page );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, &log_text ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s commit %s -m \"%s\" %s 2>&1",
				   cvs_root, extra_flags, log_text, path );

	ret = execute_command( command );

	g_free( command );

	return (ret==0);
}

gboolean cvs_checkout_site()
{
	GtkWidget *dialog;
	GtkWidget *module_name;
	GtkWidget *cvs_flags;
	GtkWidget *cvs_repository;
	gint button;
	gchar *module;
	gchar *extra_flags;
	gchar *cvs_root;
	gchar *command;
	gboolean ret;

	dialog = checkout_dialog();

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	module_name = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( dialog ),"module_name");
	cvs_flags = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( dialog ), "cvs_flags" );
	cvs_repository = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( dialog ), "cvs_root" );

	module = gtk_editable_get_chars( GTK_EDITABLE( module_name ), 0, -1 );
	extra_flags = g_strdup( gtk_entry_get_text( GTK_ENTRY( cvs_flags ) ) );
	cvs_root = g_strdup( gtk_entry_get_text( GTK_ENTRY( cvs_repository ) ) );

	gtk_widget_destroy( dialog );

	if( button == 1 ) {
		g_free( module );
		g_free( extra_flags );
		g_free( cvs_root );
		return FALSE;
	}

	/* ok'ed */
	if(strlen(cvs_root) != 0)
		command = g_strdup_printf( "cvs -d %s -z3 checkout %s \"%s\" 2>&1",
					   cvs_root, extra_flags, module );
	else
		command = g_strdup_printf( "cvs -z3 checkout %s \"%s\" 2>&1",
					   extra_flags, module );
     	g_free( extra_flags );
     	g_free( cvs_root );
	ret = execute_command( command );
	g_free( command );

	if( ret != 0 ) {
		g_free( module );
		return FALSE;
	}

	/* successful checkout, open the site */
	ret = screem_site_open_with_filename( module );
	g_free( module );

	return (ret==0);
}

gboolean cvs_update_site()
{
	Site *site;
	const gchar *cvs_root;
	const gchar *path;
	gchar *command;

	gchar *extra_flags;
	gboolean ret;

	site = current_site;

      	g_return_val_if_fail( site != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );
	path = screem_site_get_pathname( site );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, NULL ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s update %s %s 2>&1",
				   cvs_root, extra_flags, path );

	ret = execute_command( command );

	g_free( command );

	return (ret==0);
}

gboolean cvs_import_site()
{
	Site *site;
	const gchar *cvs_root;
	const gchar *path;
	const gchar *name;
	gchar *command;

	gchar *log_text;
	gchar *extra_flags;
	gboolean ret;

	gchar *backup_path;
	gboolean cpok;

	GList *list;
	gchar *new_path;
	const gchar *pathname;
	Page *page;
	gchar cwd[16384];

	site = current_site;

      	g_return_val_if_fail( site != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );
	path = screem_site_get_pathname( site );
	name = screem_site_get_name( site );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, &log_text ) )
		return FALSE;

	/* must be ok */
	chdir( path );
	command = g_strdup_printf( "\
cvs -d %s import %s -m \"%s\" \"%s\" screem start 2>&1",
				   cvs_root, extra_flags, log_text, name );
	
	ret = execute_command( command );

	g_free( command );

	if( ret != 0 ) {
		chdir( screem_site_get_pathname( site ) );
		return FALSE;
	}


	/* backup directory and perform a checkout on the imported site */
	backup_path = tmpnam( NULL );

	cpok = copy_dir( path, backup_path, FALSE );

	if( cpok ) {
		chdir( ".." );
		/* backup succeeded */
		delete_dir( path );
		/* perform cvs checkout */
		command = g_strdup_printf("cvs -d %s -z3 checkout \"%s\" 2>&1",
					  cvs_root, name );
		cpok = ( execute_command( command ) == 0 );
		g_free( command );
		if( ! cpok ) {
			/* failed checkout: restore from backup */
			copy_dir( backup_path, path, FALSE );
		}
		
		/* delete backup */
		delete_dir( backup_path );
	}

	if( cpok ) {
		/* the checkout was successful, set the site pathname
		   to the correct path */
		chdir( name );
		getcwd( cwd, 16384 );
		screem_site_set_pathname( site, cwd );
		/* modify the pathnames of all the pages */
		for( list = screem_site_get_pages( site ); list; 
		     list = list->next ) {
			page = (Page*)list->data;
			pathname = screem_page_get_pathname( page );
			new_path = g_strdup_printf( "%s%c%s", cwd, 
						    G_DIR_SEPARATOR,
						    pathname + strlen( path ));
			screem_page_set_pathname( page, new_path );
			g_free( new_path );
		}
		refresh_file_tree();
		screem_site_save( site );
	}

	chdir( screem_site_get_pathname( site ) );

	return cpok;
}

gboolean cvs_commit_site()
{
	Site *site;
	const gchar *cvs_root;
	const gchar *path;
	gchar *command;

	gchar *log_text;
	gchar *extra_flags;
	gboolean ret;

	site = current_site;

      	g_return_val_if_fail( site != NULL, FALSE );

	cvs_root = screem_site_get_cvs_root( site );
	path = screem_site_get_pathname( site );

	g_return_val_if_fail( cvs_root != NULL, FALSE );

	if( ! cvs_flags_dialog( &extra_flags, &log_text ) )
		return FALSE;
	
	/* must be ok */
	command = g_strdup_printf( "cvs -d %s commit %s -m \"%s\" %s 2>&1",
				   cvs_root, extra_flags, log_text, path );

	ret = execute_command( command );

	g_free( command );

	return (ret==0);
}

static gboolean cvs_flags_dialog( gchar **extra_flags, gchar **log_text )
{
	GtkWidget *dialog;
	gint button;
	GtkWidget *message;
	GtkWidget *cvs_flags;

 	g_return_val_if_fail( extra_flags != NULL, FALSE );
  
	dialog = settings_dialog( (gboolean)log_text );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	message = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( dialog ), "message" );
	cvs_flags = (GtkWidget*)
		gtk_object_get_data( GTK_OBJECT( dialog ), "cvs_flags" );

	if( log_text )
		*log_text = gtk_editable_get_chars( GTK_EDITABLE( message ),
						    0, -1 );

	*extra_flags = g_strdup( gtk_entry_get_text( GTK_ENTRY( cvs_flags ) ));

	gtk_widget_destroy( dialog );

	/* if button == cancel then return */
	if( button == 1 ) {
		if( log_text )
			g_free( *log_text );
		g_free( *extra_flags );
		return FALSE;
	}

	return TRUE;
}

static GtkWidget *settings_dialog( gboolean allow_message )
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox1;
  GtkWidget *frame1;
  GtkWidget *message;
  GtkWidget *frame2;
  GtkWidget *entry1;
  GtkWidget *cvs_flags;
  GtkWidget *dialog_action_area1;
  GtkWidget *button1;
  GtkWidget *button2;
 
  dialog = gnome_dialog_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog", dialog);
  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE);

  dialog_vbox1 = GNOME_DIALOG (dialog)->vbox;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox1", dialog_vbox1);
  gtk_widget_show (dialog_vbox1);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_ref (vbox1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox1", vbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

  frame1 = gtk_frame_new (_("Log Message"));
  gtk_widget_ref (frame1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "frame1", frame1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame1);
  gtk_box_pack_start (GTK_BOX (vbox1), frame1, TRUE, TRUE, 4);

  message = gtk_text_new (NULL, NULL);
  gtk_widget_ref (message);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "message", message,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (message);
  gtk_container_add (GTK_CONTAINER (frame1), message);
  gtk_text_set_editable (GTK_TEXT (message), TRUE);

  if( ! allow_message )
	  change_state( frame1, NULL );

  frame2 = gtk_frame_new (_("Extra CVS Flags"));
  gtk_widget_ref (frame2);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "frame2", frame2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame2);
  gtk_box_pack_start (GTK_BOX (vbox1), frame2, FALSE, FALSE, 0);

  entry1 = gnome_entry_new ("cvs_flags_hist");
  gtk_widget_ref (entry1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "entry1", entry1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry1);
  gtk_container_add (GTK_CONTAINER (frame2), entry1);

  cvs_flags = gnome_entry_gtk_entry (GNOME_ENTRY (entry1));
  gtk_widget_ref (cvs_flags);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "cvs_flags", cvs_flags,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (cvs_flags);

  dialog_action_area1 = GNOME_DIALOG (dialog)->action_area;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_action_area1", dialog_action_area1);
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
  button1 = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
  gtk_widget_ref (button1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "button1", button1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button1);
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);
  button2 = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
  gtk_widget_ref (button2);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "button2", button2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button2);
  GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);

  return dialog;
}

static GtkWidget *checkout_dialog()
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox1;
  GtkWidget *frame1;
  GtkWidget *entry2;
  GtkWidget *module_name;
  GtkWidget *frame2;
  GtkWidget *entry1;
  GtkWidget *cvs_flags;
  GtkWidget *frame3;
  GtkWidget *entry3;
  GtkWidget *cvs_root;
  GtkWidget *dialog_action_area1;
  GtkWidget *button1;
  GtkWidget *button2;
 
  dialog = gnome_dialog_new (NULL, NULL);
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog", dialog);
  gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, FALSE, FALSE);

  dialog_vbox1 = GNOME_DIALOG (dialog)->vbox;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox1", dialog_vbox1);
  gtk_widget_show (dialog_vbox1);

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_ref (vbox1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox1", vbox1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

  frame1 = gtk_frame_new (_("CVS Module"));
  gtk_widget_ref (frame1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "frame1", frame1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame1);
  gtk_box_pack_start (GTK_BOX (vbox1), frame1, TRUE, TRUE, 4);

  entry2 = gnome_entry_new ("cvs_module_hist");
  gtk_widget_ref (entry2);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "entry2", entry2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry2);
  gtk_container_add (GTK_CONTAINER (frame1), entry2);

  module_name = gnome_entry_gtk_entry (GNOME_ENTRY (entry2));
  gtk_widget_ref (module_name);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "module_name", module_name,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (module_name);

  frame2 = gtk_frame_new (_("Extra CVS Flags"));
  gtk_widget_ref (frame2);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "frame2", frame2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame2);
  gtk_box_pack_start (GTK_BOX (vbox1), frame2, FALSE, FALSE, 0);

  entry1 = gnome_entry_new ("cvs_flags_hist");
  gtk_widget_ref (entry1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "entry1", entry1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry1);
  gtk_container_add (GTK_CONTAINER (frame2), entry1);

  cvs_flags = gnome_entry_gtk_entry (GNOME_ENTRY (entry1));
  gtk_widget_ref (cvs_flags);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "cvs_flags", cvs_flags,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (cvs_flags);

  frame3 = gtk_frame_new (_("CVS Root"));
  gtk_widget_ref (frame3);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "frame3", frame3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (frame3);
  gtk_box_pack_start (GTK_BOX (vbox1), frame3, FALSE, FALSE, 0);

  entry3 = gnome_entry_new ("cvs_root_hist");
  gtk_widget_ref (entry3);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "entry3", entry3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (entry3);
  gtk_container_add (GTK_CONTAINER (frame3), entry3);

  cvs_root = gnome_entry_gtk_entry (GNOME_ENTRY (entry3));
  gtk_widget_ref (cvs_root);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "cvs_root", cvs_root,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (cvs_root);

  dialog_action_area1 = GNOME_DIALOG (dialog)->action_area;
  gtk_object_set_data (GTK_OBJECT (dialog), "dialog_action_area1", dialog_action_area1);
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (dialog_action_area1), 8);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
  button1 = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
  gtk_widget_ref (button1);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "button1", button1,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button1);
  GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

  gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);
  button2 = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
  gtk_widget_ref (button2);
  gtk_object_set_data_full (GTK_OBJECT (dialog), "button2", button2,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button2);
  GTK_WIDGET_SET_FLAGS (button2, GTK_CAN_DEFAULT);

  return dialog;
}
