/*  Screem:  imageWizard.c,
 * A new improved image wizard
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "site.h"
#include "page.h"
#include "editor.h"
#include "fileops.h"

#include "image.xpm"

extern GtkWidget *app;
extern Site *current_site;

static GladeXML *xml;
static GtkWidget *dialog = NULL;


void imageWizard();
void image_wizard_preview( GtkEntry *widget );
void image_wizard_closed( GtkWidget *widget, GdkEvent *event, gpointer data );
void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data );

#define TAG "img"
#define WIZARD_NAME "Image"

static gboolean popup = FALSE;
static gchar *unknown_attributes;

/* the following is for identifying this as a wizard for the <a> tag, this
   allows it to be opened from the popup menu, and have its entries filled
   with the current settings */
G_MODULE_EXPORT const gchar *html_tag( gchar **wizard_name )
{
	*wizard_name = g_strdup( WIZARD_NAME ); 
	return g_strdup( TAG );
}

G_MODULE_EXPORT void popup_wizard( GtkMenuItem *menu, GList *data )
{
	GList *list;

	gint start, len;
	gchar *name;
	gchar *value;
	gchar *temp;

  	GtkWidget *widget;

	g_return_if_fail( data != NULL );

	popup = TRUE;
	unknown_attributes = g_strdup( "" );

	/* display wizard */
	imageWizard();

	widget = glade_xml_get_widget( xml, "thumb" );
	change_state( widget );

	/* force modal */
	gtk_window_set_modal( GTK_WINDOW( dialog ), TRUE );
	/* fill in fields: to avoid using a struct the list is formatted as
	   follows:

	   list->data = start pos;
	   list->data = length;

	   from then on:

	   list->data = attribute 1 name
	   list->next->data = attribute 1 value
	   list->next->next->data = attribute 2 name
	   list->next->next->next->data = attribute 2 value
	   .
	   .
	   .
	   etc.
	*/
	start = (gint)data->data;
	data = data->next;
	len = (gint)data->data;

	screem_editor_select_region( start, len );

	for( list = data->next; list; list = list->next ) {
		name = list->data;
		list = list->next;
		value = list->data;
		if( ! strcmp( name, "src" ) ) {
			widget = glade_xml_get_widget(xml, "image_path_entry");
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "alt" ) ) {
			widget = glade_xml_get_widget(xml, "alt_entry" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "width" ) ) {
			widget = glade_xml_get_widget(xml, "width_entry" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "height" ) ) {
			widget = glade_xml_get_widget(xml, "height_entry" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "align" ) ) {
			widget = glade_xml_get_widget(xml, "align_entry" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "border" ) ) {
			widget = glade_xml_get_widget(xml, "border" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "hspace" ) ) {
			widget = glade_xml_get_widget(xml, "hspace" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! strcmp( name, "vspace" ) ) {
			widget = glade_xml_get_widget(xml, "vspace" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else {
			/* not handled by the wizard, add to
			   unknown_attributes variable */
			temp = unknown_attributes;
			if( ! value ) {
				unknown_attributes = g_strconcat( temp, " ", 
								  name, NULL );
			} else {
				unknown_attributes = g_strconcat( temp, " ", 
								  name, "=\"",
								  value, "\"",
								  NULL);
			}
			g_free( temp );
		}
	}

	/* now we want to sit in a loop allowing gtk to continue.
	   We don't want to return as it will remove the highlight on the
	   text */
	while( dialog ) {
		if( gtk_events_pending() )
			gtk_main_iteration();
	}

	g_free( unknown_attributes );
	unknown_attributes = NULL;
	popup = FALSE;
}

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
        g_print("imageWizard: check-init\n");
        return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
        g_print( "imageWizard: unloaded\n" );
}

G_MODULE_EXPORT void init() 
{
        GtkWidget *imageButton;
	GtkWidget *toolbar;

        GnomeUIInfo menuinfo[] = { 
                {
                        GNOME_APP_UI_ITEM, N_("Image..."),
                        N_("Insert an Image"),
                        imageWizard, NULL, NULL,
                        GNOME_APP_PIXMAP_STOCK,
                        GNOME_STOCK_MENU_BLANK,
                        0,
                        GDK_CONTROL_MASK, NULL
                },
                GNOMEUIINFO_END
        };

   	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );
        /* place menu item after image under insert */
        gnome_app_insert_menus( GNOME_APP( app ),
                                _("_Insert/"), menuinfo);

        /* place a button on the wizards toolbar */
        imageButton = gnome_pixmap_new_from_xpm_d( IMAGE_ICON );
        gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
                                 _("Image Wizard"), "", imageButton ,
                                 imageWizard, 0 );

        g_print( "imageWizard: initialised\n" );
}

void imageWizard()
{
	Page *page;

	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );
        
        if( dialog ) {
                gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/imageWizard.glade",
			     "imagewizard_dialog" );
	glade_xml_signal_autoconnect( xml );

	dialog = glade_xml_get_widget( xml, "imagewizard_dialog" );
}

void image_wizard_preview( GtkEntry *widget )
{
	GtkPixmap *image;
	gchar *filename;
	GtkWidget *frame;
	GtkWidget *width;
	GtkWidget *height;

	gchar *w;
	gchar *h;

	GdkPixbuf *pixbuf;
	GdkPixbuf *temp;
    	GdkPixmap *pixmap;
	GdkBitmap *mask;

	GtkWidget *junk;

	GtkWidget *iw;
	GtkWidget *ih;
	GtkWidget *fs;

	struct stat s;
	gchar *size;

	frame = glade_xml_get_widget( xml, "frame1" );
	image = gtk_object_get_data( GTK_OBJECT( frame ), "image" );
	width = glade_xml_get_widget( xml, "width_entry" );
	height = glade_xml_get_widget( xml, "height_entry" );

	iw = glade_xml_get_widget( xml, "initial_width" );
	ih = glade_xml_get_widget( xml, "initial_height" );
	fs = glade_xml_get_widget( xml, "file_size" );

	filename = gtk_entry_get_text( widget );

	if( ! filename )
		return;

	/* does the filename exist?  if it doesn't then we don't want
	   to try loading it as imlib spews loads warning message and
	   attempts to load the image with imagemagick etc */
       if( stat( filename, &s ) < 0 )
	       return;
       /* if its a directory we aren't interested */
       if( g_file_test( filename, G_FILE_TEST_ISDIR ) )
	       return;

	pixbuf = gdk_pixbuf_new_from_file( filename );

	if( ! pixbuf )
		return;

	/* set the width/height fields */
	w = g_strdup_printf( "%i", gdk_pixbuf_get_width( pixbuf ) );
	h = g_strdup_printf( "%i", gdk_pixbuf_get_height( pixbuf ) );
	size = g_strdup_printf( "%i bytes", (gint)s.st_size );
	gtk_entry_set_text( GTK_ENTRY( width ), w );
	gtk_entry_set_text( GTK_ENTRY( height ), h );

	gtk_label_set( GTK_LABEL( iw ), w );
	gtk_label_set( GTK_LABEL( ih ), h );
	gtk_label_set( GTK_LABEL( fs ), size );

	g_free( w );
	g_free( h );
	g_free( size );

	temp = gdk_pixbuf_scale_simple( pixbuf, 160, 120, GDK_INTERP_NEAREST );
	gdk_pixbuf_render_pixmap_and_mask( temp, &pixmap, &mask, 100 );
	gdk_pixbuf_unref( temp );
	gdk_pixbuf_unref( pixbuf );

	if( ! image ) {
		junk = glade_xml_get_widget( xml, "junk" );
		gtk_widget_destroy( junk );
		image = GTK_PIXMAP( gtk_pixmap_new( pixmap, mask ) );
		gtk_widget_show( GTK_WIDGET( image ) );
                gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET(image));
                gtk_object_set_data (GTK_OBJECT (frame), "image", image);
	}

	gtk_pixmap_set( GTK_PIXMAP( image ), pixmap, mask );
	gdk_pixmap_unref( pixmap );
	gdk_bitmap_unref( mask );
	gtk_widget_show( GTK_WIDGET( image ) );
}

void image_wizard_closed( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	dialog = NULL;
}

void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data )
{
	GtkWidget **d;
	GtkWidget *fileEntry;
	GtkWidget *altEntry;
	GtkWidget *widthEntry;
	GtkWidget *heightEntry;
	GtkWidget *alignEntry;
	GtkWidget *copy;
	GtkWidget *thumb;
	gchar *filename;
	gchar *alt;
	gdouble width;
	gdouble height;
	gchar *align;
	GtkWidget *border;
	gint b;
	GtkWidget *hspace;
	gint hs;
	GtkWidget *vspace;
	gint vs;
       	gboolean copyNeeded;
	gboolean makeThumb;

	gchar *dest;
	gchar *dfile;
	gchar *tfile = NULL;
	Page *page;

	GdkImlibImage *i, *i2;

	gchar *imgTag;
	gchar *aTag;

	gint pos;

	gchar *dir;

	const gchar *pathname;

	page = screem_site_get_current_page( current_site );
	
	g_return_if_fail( page != NULL );

	pathname = screem_page_get_pathname( page );

	fileEntry = glade_xml_get_widget( xml, "image_path_entry" );
	altEntry = glade_xml_get_widget( xml, "alt_entry" );
	widthEntry = glade_xml_get_widget( xml, "w_num_entry" );
	heightEntry = glade_xml_get_widget( xml, "h_num_entry" );
	alignEntry = glade_xml_get_widget( xml, "align_entry" );
	copy = glade_xml_get_widget( xml, "copy" );
	thumb = glade_xml_get_widget( xml, "thumb" );
	border = glade_xml_get_widget( xml, "border" );
	hspace = glade_xml_get_widget( xml, "hspace" );
	vspace = glade_xml_get_widget( xml, "vspace" );

	if( button < 2 ) { /* ok or applied */
		/* get the filename */
		filename = gtk_entry_get_text( GTK_ENTRY( fileEntry ) );

		/* get the alt text */
		alt = gtk_entry_get_text( GTK_ENTRY( altEntry ) );
		
		/* get the width value */
		width = gnome_number_entry_get_number( GNOME_NUMBER_ENTRY( widthEntry ) );
		/* get the height value */
		height = gnome_number_entry_get_number( GNOME_NUMBER_ENTRY( heightEntry ) );

		/* get the alignment */
		align = gtk_entry_get_text( GTK_ENTRY( alignEntry ) );

		/* get the border size */
		b = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(border));
		
		/* get hspace size */
		hs = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(hspace));

		/* get vspace size */
		vs = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(vspace));

		/* do we need to copy the file to the pages directory? */
		copyNeeded = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( copy ) );
		/* do we need to make a thumbnail? */
		makeThumb = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( thumb ) );

		/* copy the file if we need to */
		if( copyNeeded ) {
			dest = g_strndup( pathname, 
					  g_filename_index( pathname ) );
			dfile = g_strdup_printf( "%s%s", dest, filename + g_filename_index( filename ) );
			g_free( dest );
			copy_file( filename, dfile );
			filename = dfile;
		}

		/* make the thumbnail if we need to */
		if( makeThumb ) {
			dfile = g_strndup( filename, g_extension_pointer( filename ) - filename - 1);
			tfile = g_strdup_printf("%s-thumb.%s", dfile,
						g_extension_pointer(filename));
			g_free( dfile );
		
			i = gdk_imlib_load_image( filename );
			i2 = gdk_imlib_clone_scaled_image( i, (gint)width,
							   (gint)height );
			gdk_imlib_destroy_image( i );
			gdk_imlib_save_image( i2, tfile, NULL );
			gdk_imlib_destroy_image( i2 );

			dfile = tfile;
			tfile = filename;
			filename = dfile;
		}
		dir = g_dirname( pathname );
		chdir( dir );
		g_free( dir );
		/* now construct the tag we are going to insert */
		if( ! popup ) {
			imgTag = g_strdup_printf( "<img src=\"%s\" alt=\"%s\" width=\"%i\" height=\"%i\" align=\"%s\" border=\"%i\" hspace=\"%i\" vspace=\"%i\">", relative_path(filename, NULL), alt, (gint)width, (gint)height, align, b, hs, vs );
		} else {
			imgTag = g_strdup_printf( "<img src=\"%s\" alt=\"%s\" width=\"%i\" height=\"%i\" align=\"%s\" border=\"%i\" hspace=\"%i\" vspace=\"%i\"%s>", relative_path(filename, NULL), alt, (gint)width, (gint)height, align, b, hs, vs, unknown_attributes );	
		}

		aTag = imgTag;
			
		/* if its a thumbnail we need to wrap this up in a href tag */
		if( makeThumb ) {
			dfile = filename;
			filename = tfile;
			tfile = filename;
		
			aTag = g_strdup_printf( "<a href=\"%s\">%s</a>", 
						relative_path(filename, NULL),
						imgTag );
			g_free( imgTag );
		}

		/* if we have been called by right clicking on a tag
		   then we erase the selected text */
		if( popup ) {
			screem_editor_clear_selection();
		} 
			pos = screem_editor_get_pos();
		
		/* now we insert it at position pos */
		screem_editor_insert( pos, aTag );

		g_free( aTag );
	}

	if( button == 2 || button == 0 ) { /* ok or close clicked */
		gtk_widget_destroy( GTK_WIDGET( widget ) );
		dialog = NULL;
	}
}
